/*
        Adsmith 1.8 :  An Efficient Object-Based DSM Environment on PVM
    
                        Author:  William W. Y. Liang

           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW

                   Copyright (C) 1996 All Rights Reserved

                                  NOTICE

      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.

      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

#ifndef __ADSMCVTAB_H__
#define __ADSMCVTAB_H__

#include "pack.h"

// -- Coherence Verstion Table, including coherence vector for each Host --
class CohVerTab {
  short nhost,nproc;
  unsigned long **vertab;	// version table, size of nhost x nproc
public:

  CohVerTab() : nhost(0), nproc(0), vertab(NULL) {};
  ~CohVerTab();

  void set_nhost(short nh);
  void set_nproc(short np);

  short get_nhost() { return nhost; }
  short get_nproc() { return nproc; }

  int size() { return nhost*nproc*sizeof(long); }

  void set_ver(short hn,short pn,unsigned long v);

  unsigned long get_ver(short hn,short pn) { return vertab[hn][pn]; }
  unsigned long *get_ver(short hn) { return vertab[hn]; }

  void CohVerTab::upk_writeinfo(AdsmRecvPacket& rpkt);
};

// Methods of CohVerTab class

CohVerTab::~CohVerTab() { 
  if (vertab!=NULL) {
    for (int i=0; i<nhost; i++) delete [] vertab[i]; 
    delete [] vertab;
  }
}

// set host number
void CohVerTab::set_nhost(short nh) {
  if (nh<=nhost) return;

  unsigned long **newtab=new unsigned long*[nh];	// allocate new table

  // copy pointers from old table to new table
  int i;
  for (i=0; i<nhost; i++) newtab[i]=vertab[i];
  if (nproc>0)
    for (; i<nh; i++) {	// new vectors, set to 0s
      newtab[i]=new unsigned long[nproc];
      for (int j=0; j<nproc; j++) newtab[i][j]=0;
    } 
  else
    for (; i<nh; i++) newtab[i]=0;

  delete [] vertab;	// delete old table
  vertab=newtab;	// replace with new table
  nhost=nh;
}

// set process number
void CohVerTab::set_nproc(short np) {
  if (np<=nproc) return;

  for (int i=0; i<nhost; i++) {	// for each host
    unsigned long *newvec=new unsigned long[np]; // allocate new process vector
    
    // copy versions from old vector to new vector
    int j;
    for (j=0; j<nproc; j++) newvec[j]=vertab[i][j];
    for (; j<np; j++) newvec[j]=0;

    delete [] vertab[i];	// delete old vector
    vertab[i]=newvec;	// replace with new vector
  }

  nproc=np;
}

// set version number for specified host and process
inline void CohVerTab::set_ver(short hn,short pn,unsigned long v) {
#ifdef ADSMDEBUG
  if (hn<0||hn>=nhost) {  
    cerr<<"CohVerTab::set_ver(): host number out of range"<<endl;
    Exit(__FILE__,__LINE__,0);
  }
  if (pn<0||pn>=nproc) {  
    cerr<<"CohVerTab::set_ver(): process number out of range"<<endl;
    Exit(__FILE__,__LINE__,0);
  }
#endif

  vertab[hn][pn]=v;
}

// receive a write info (from daemon)
inline void CohVerTab::upk_writeinfo(AdsmRecvPacket& rpkt) {
  short np;	// current number of processes
  short dn;	// daemon hostno
  rpkt>>np>>dn;

  unsigned long newvervec[np];	// new version vector for dn
  rpkt.upkulong(newvervec,(int)np);

  set_nproc(np);	// set process number

  // set into writeinfo's version table
  for (int i=0; i<np; i++) 
    if (newvervec[i]>get_ver(dn,i)) set_ver(dn,i,newvervec[i]);
}

// pack total write info
inline AdsmSendPacket& operator<<(AdsmSendPacket& spkt,CohVerTab& wi) {
  spkt<<wi.get_nhost();	// host number know by me
  spkt<<wi.get_nproc();	// process number know by me
  for (short i=0; i<wi.get_nhost(); i++)
    spkt.pkulong(wi.get_ver(i),wi.get_nproc());

  return spkt;
}

// unpack total write info and apply on existing one
inline AdsmRecvPacket& operator>>(AdsmRecvPacket& rpkt,CohVerTab& wi) {
  short nh,np;
  rpkt>>nh; 
  rpkt>>np;

// #ifdef ADSMDEBUG
//   if (nh<wi.get_nhost()) {
//     cerr<<"AdsmRecvPacket.<<(CohVerTab&): host number out of range"<<endl;
//     Exit(__FILE__,__LINE__,0);
//   }
//   if (np<wi.get_nproc()) {
//     cerr<<"AdsmRecvPacket.<<(CohVerTab&): process number out of range"<<endl;
//     Exit(__FILE__,__LINE__,0);
//   }
// #endif

  wi.set_nhost(nh);
  wi.set_nproc(np);

  for (int i=0; i<nh; i++) {
    unsigned long tmpvec[np];
    rpkt.upkulong(tmpvec,np);	// get a version vector

    // apply to wi
    for (int j=0; j<np; j++)
      if (wi.get_ver(i,j)<tmpvec[j]) 	// newer version
        wi.set_ver(i,j,tmpvec[j]);
  }

  return rpkt;
}

#endif // __ADSMCVTAB_H__
