/*
        Adsmith 1.8 :  An Efficient Object-Based DSM Environment on PVM
 
                        Author:  William W. Y. Liang
 
           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW
 
                   Copyright (C) 1996 All Rights Reserved

                                  NOTICE

      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.

      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iostream.h>
#include "pvm3.h"
#include "adsm.h"
#include "common.h"
#include "adsmd.h"
#include "dynarr.h"
#include "hash.h"
#include "pack.h"
#include "cohver.h"
#include "unoffic.h"
#include "sysdep.h"

// -- Global Variables --

// system control flags
unsigned long _adsm_ctrlflags=0;

Timing _adsm_total_time;

#ifdef ADSMCTRL

unsigned long _adsm_timestamp=0;

Timing _adsm_modset_time;
Timing _adsm_pkt_time;
Timing _adsm_idle_time;

int _adsm_msgnum_system=0; 
int _adsm_msgsiz_system=0; 
int _adsm_msgnum_consist=0; 
int _adsm_msgsiz_consist=0; 
int _adsm_msgnum_malloc=0;
int _adsm_msgsiz_malloc=0;
int _adsm_msgsiz_access=0;      
int _adsm_msgnum_access=0;      
int _adsm_msgsiz_sync=0;            
int _adsm_msgnum_sync=0;            

// statistics info
static int total_msgnum=0;
static int total_msgsize=0;
static int total_msgnum_system=0; 
static int total_msgsiz_system=0; 
static int total_msgnum_consist=0; 
static int total_msgsiz_consist=0; 
static int total_msgnum_malloc=0;
static int total_msgsiz_malloc=0;
static int total_msgsiz_access=0;      
static int total_msgnum_access=0;      
static int total_msgsiz_sync=0;            
static int total_msgnum_sync=0;            

#endif

// system packet for send and receive
static AdsmRecvPacket rpkt;
static AdsmSendPacket spkt;

// command
static char command;

// system info
static int      parent,mytid,master_dtid;
static int      i_am_master=0;

// buffer information
static int stid;
static short srcno; // source process number

// the deamons host table
static short nhost;
static short hostno;
static int *htab_dtids;

// process list
static DynArray<Process> proclist;
static short procnum=0;
static short nproc=0;

// global shared data table
static int      glb_table_size=GLB_TABLE_SIZE;
static int      glb_index=0;
static int	objnum=0;
static GlbVarTab   **glb_vtab;

// hash table for variables
static VarHashTab hash_tab;

// aggregate list 
static DynArray<int> loadlist; // for replying load requests
static DynArray<int> *colist;  // to aggreate coherence messages

// coherence version/timestamp for this daemon
static CohVerTab cohvertab; // coherent version table known here

// Message Head

struct MsgHead { int useless; };

static ostream& operator<<(ostream& s, MsgHead& m) {
   m.useless++;

#ifdef ADSMCTRL
   s<<_adsm_timestamp<<":\t";
#endif

   return s<<'D'<<hostno<<":\t";
}

static MsgHead msghead;

// -- Main Program --

main() {
  sysdep_set_new_handler(); // set new exception handler

  // initializations
  mytid=pvm_mytid();
  parent=pvm_parent();

  // wait for command
  while (1) {
    char checkcode;
    rpkt.recv(-1,ADSM_CHANNEL_CMD,&stid);
    rpkt>>checkcode;	// basic check, no ADSMDEBUG comment option

    if (checkcode!=(char)CHECKCODE) {
      cerr<<msghead<<"Error: check code not match"<<endl;
      Exit(__FILE__,__LINE__);
    }

    rpkt>>srcno>>command;
    switch (command) {
      case ADSMD_CMD_INIT: 
        adsmd_cmd_init();
	break;
      case ADSMD_CMD_ENROLL:
        adsmd_cmd_enroll();
        break;
      case ADSMD_CMD_PROCTIDS:
        adsmd_cmd_proctids();
        break;
      case ADSMD_CMD_HANDSHAKE:
        adsmd_cmd_handshake();
        break;
      case ADSMD_CMD_REGISTER:
        adsmd_cmd_register();
        break;
      case ADSMD_CMD_ATTACH:
        adsmd_cmd_attach();
        break;
      case ADSMD_CMD_UNDECL:
        adsmd_cmd_undecl();
        break;
      case ADSMD_CMD_MASTERUNDECL:
        adsmd_cmd_masterundecl();
        break;
      case ADSMD_CMD_REFRESH:
      case ADSMD_CMD_REFRESH_NOW:
        adsmd_cmd_refresh();
        break;
      case ADSMD_CMD_FLUSH:
      case ADSMD_CMD_FLUSH_NOW:
        adsmd_cmd_flush();
        break;
      case ADSMD_CMD_ACK: 
        adsmd_cmd_acksim();
        break;
      case ADSMD_CMD_P:
        adsmd_cmd_p();
        break;
      case ADSMD_CMD_LOCK:
        adsmd_cmd_lock();
        break;
      case ADSMD_CMD_BARRIER:
        adsmd_cmd_barrier();
        break;
      case ADSMD_CMD_V:
        adsmd_cmd_v();
        break;
      case ADSMD_CMD_UNLOCK:
        adsmd_cmd_unlock();
        break;
      case ADSMD_CMD_ATOMIC_BEGIN:
        adsmd_cmd_atomic_begin();
        break;
      case ADSMD_CMD_ATOMIC_END:
        adsmd_cmd_atomic_end();
        break;
      case ADSMD_CMD_ATOMIC:
        adsmd_cmd_atomic();
        break;
      case ADSMD_CMD_EXIT: 
        if (adsmd_cmd_exit()) sysdep_exit(0);
        break;
      case ADSMD_CMD_SHUTDOWN: 
        adsmd_cmd_shutdown();
        return 0;
      default:
        cerr<<"adsmd: Unknown command "<<(int)command<<endl;
        break;
    }
  }
}

// --- inline functions ---

inline void adsmd_check_table_size(int index) {
  if (index>=glb_table_size) adsmd_inc_table_size(index);
}

// manage free list
inline int adsmd_get_gindex() {
  int next=glb_index;
  glb_index+=nhost; // for parallel allocate
  adsmd_check_table_size(glb_index);
  return next;
}

inline int adsmd_load(char cmd,int gindex,short lpno,int conti) {
  loadlist.insert(gindex); // for bulk reply

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<glb_vtab[gindex]->name<<" is loaded by P"<<lpno<<endl;
#endif

  if (glb_vtab[gindex]->hint&AdsmDataCache) { // cached
    if (!glb_vtab[gindex]->cacherlist) 
      glb_vtab[gindex]->cacherlist=new DynArray<short>;

    glb_vtab[gindex]->cacherlist->uniq_insert(lpno); // record requester

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<"P"<<lpno<<" is added to the cacherlist of "
          <<glb_vtab[gindex]->name<<endl;
#endif

  }

  if (conti==1) return 0; // to be continued, ack latter

  adsmd_doneload(cmd,lpno);

  return 1;
}

inline int adsmd_barrier(int gindex,short ttl) {
  // int *total=(int*)glb_vtab[gindex]->data,*count=total+1;

  if (glb_vtab[gindex]->total==0) glb_vtab[gindex]->total=ttl;

  if (ttl!=glb_vtab[gindex]->total)
    cerr<<msghead<<'P'<<srcno<<"'s barrier count "<<ttl
        <<" mismatch with the existing count "<<glb_vtab[gindex]->total<<endl;

  // add to barrlist

  if (!glb_vtab[gindex]->barrlist)
    glb_vtab[gindex]->barrlist=new DynArray<short>;

  glb_vtab[gindex]->barrlist->insert(srcno);

  glb_vtab[gindex]->count++;

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<'P'<<srcno<<" arrives barrier "<<glb_vtab[gindex]->name
        <<" with count "<<glb_vtab[gindex]->count<<endl;
#endif

  if (glb_vtab[gindex]->count<glb_vtab[gindex]->total)
    return 0;
  else {
    glb_vtab[gindex]->total=glb_vtab[gindex]->count=0;
    return 1;
  }
}
            
inline void adsmd_block(int gindex,short requester,char rwtype) {
  // record the requester in a circular list
  Tlock_list *p=new Tlock_list;
  p->procno=requester;
  p->rwtype=rwtype;
  if (glb_vtab[gindex]->blkqueue==NULL) {
    p->next=p;
    glb_vtab[gindex]->blkqueue=p;
  } else {
    p->next=glb_vtab[gindex]->blkqueue->next; // head
    glb_vtab[gindex]->blkqueue->next=p;       // tail
    glb_vtab[gindex]->blkqueue=p;             // point to tail
  } 
}

inline void adsmd_unblock(int gindex) {
  Tlock_list *p=glb_vtab[gindex]->blkqueue->next;
  if (p->next!=p) // exist next locker
    glb_vtab[gindex]->blkqueue->next=p->next;
  else
    glb_vtab[gindex]->blkqueue=NULL;

  delete p;
}

inline void adsmd_check_locker(int gindex) {
  if (glb_vtab[gindex]->blkqueue==NULL) {
    cerr<<"adsmd_check_locker: no locker exist"<<endl;
    return;
  }

  Tlock_list *p=glb_vtab[gindex]->blkqueue->next; // p is current locker
  if (p->procno!=srcno) {
    cerr<<"adsmd_check_locker(): unlocker P"<<srcno
        <<" is not the locker P"<<p->procno<<endl;
    return;
  }
}

inline void adsmd_coherence(int gindex) {
  // aggregating coherence messages

  if (glb_vtab[gindex]->hint&AdsmDataCache) { // cached
    if (!glb_vtab[gindex]->cacherlist) 
      glb_vtab[gindex]->cacherlist=new DynArray<short>;

    if (command==ADSMD_CMD_FLUSH ||
        !(glb_vtab[gindex]->hint&AdsmDataUpdate) ||
        glb_vtab[gindex]->hint&AdsmDataMWriter)
    { // update for nonsync must be sent to writer for update order problem
      // i.e. coh msgs are not needed for Ordinary or Invalidate or Mwriter
      glb_vtab[gindex]->cacherlist->remove(srcno); // exclude the writer
    }

    // if exists cachers, inv/update them; Non cache will skip this part
    if (!glb_vtab[gindex]->cacherlist->is_empty()) {
      short *tidv=glb_vtab[gindex]->cacherlist->get_array();
      int count=glb_vtab[gindex]->cacherlist->get_size();

      // aggregate inv/update
      for (int i=0; i<count; i++) colist[tidv[i]].insert(gindex);

#ifdef ADSMCTRL
      char action[]="invalidate";
#endif

      if (!(glb_vtab[gindex]->hint&AdsmDataUpdate))
        glb_vtab[gindex]->cacherlist->reset();

#ifdef ADSMCTRL
      else // invalidate: reset the cacherlist
        strcpy(action,"update");

      if (_adsm_ctrlflags&AdsmCtrlDbgTrc) {
        cerr<<msghead<<"The above store "<<action<<' ';
        for (int i=0; i<count; i++) cerr<<'P'<<tidv[i]<<' ';
        cerr<<endl;
      }
#endif

    }

    glb_vtab[gindex]->cacherlist->uniq_insert(srcno); // record writer

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<"P"<<srcno<<" is added to the cacherlist of "
          <<glb_vtab[gindex]->name<<endl;
#endif

  }
}

// --- non-inline functions ---

void adsmd_statistics() {
  // collect number of all sent message as the total number of messages

#ifdef ADSMCTRL
  if (i_am_master) {
    if (_adsm_ctrlflags&AdsmCtrlMsgTtl) {
        cerr<<msghead<<"Total number of remote messages = "<<total_msgnum<<endl;
        cerr<<msghead<<"Total bytes  of remote messages = "<<total_msgsize<<endl;
    }

    if (_adsm_ctrlflags&AdsmCtrlMsgCls) {
      cerr<<msghead<<"Total number of system  messages = "
          <<total_msgnum_system<<endl;                   
      cerr<<msghead<<"Total number of consist messages = "
          <<total_msgnum_consist<<endl;                  
      cerr<<msghead<<"Total number of malloc  messages = "
          <<total_msgnum_malloc<<endl;                   
      cerr<<msghead<<"Total number of access  messages = "
          <<total_msgnum_access<<endl;                   
      cerr<<msghead<<"Total number of sync    messages = "
          <<total_msgnum_sync<<endl;                     
      cerr<<msghead<<"Total bytes  of system  messages = "
          <<total_msgsiz_system<<endl;                   
      cerr<<msghead<<"Total bytes  of consist messages = "
          <<total_msgsiz_consist<<endl;                  
      cerr<<msghead<<"Total bytes  of malloc  messages = "
          <<total_msgsiz_malloc<<endl;                   
      cerr<<msghead<<"Total bytes  of access  messages = "
          <<total_msgsiz_access<<endl;                   
      cerr<<msghead<<"Total bytes  of sync    messages = "
          <<total_msgsiz_sync<<endl;                     
    }
  }

  // hash info
  if (_adsm_ctrlflags&AdsmCtrlSttHsh)
    cerr<<msghead<<"Hash Table Info :"<<hash_tab<<endl;

  // misc info
  if (_adsm_ctrlflags&AdsmCtrlSttRst) {
    cerr<<msghead<<"terminates, with statistics:"<<endl
        <<"Message count: "<<spkt.get_msgnum()<<"(send) + "<<rpkt.get_msgnum()
        <<"(recv) = "<<spkt.get_msgnum()+rpkt.get_msgnum()<<" messages"<<endl
        <<"Message size: "<<spkt.get_msgsize()<<"(send) + "<<rpkt.get_msgsize()
        <<"(recv) = "<<spkt.get_msgsize()+rpkt.get_msgsize()<<" bytes"<<endl;
  }

  // modset timing
  if (_adsm_ctrlflags&AdsmCtrlTimMod)
    cerr<<msghead<<"Time in ModifySet = "<<_adsm_modset_time<<" sec"<<endl;

  if (_adsm_ctrlflags&AdsmCtrlTimPkt)
    cerr<<msghead<<"Time in (un)Packing = "<<_adsm_pkt_time<<" sec"<<endl;

  if (_adsm_ctrlflags&AdsmCtrlTimIdl)
    cerr<<msghead<<"Time of Idleness (including time in pvm_recv()) = "
        <<_adsm_idle_time<<" sec"<<endl;
#endif

  // object info
  if (_adsm_ctrlflags&AdsmCtrlSttObj)
    cerr<<msghead<<"Number of objects = "<<objnum<<endl;

  if (_adsm_ctrlflags&AdsmCtrlTimTtl) {
    _adsm_total_time.stop();
    cerr<<msghead<<"Total execution time = "<<_adsm_total_time<<" sec"<<endl;
  }

}

void adsmd_reply_register(DynArray<AllocRec> &existlist) {
  // reply requester those already existing
  int xnum=existlist.get_size();
  AllocRec *xrec=existlist.get_array();
  if (xnum>0) {
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<(char)ADSMD_CMD_REGISTER;
#endif

    spkt<<glb_table_size<<hostno<<xnum;
    for (int i=0; i<xnum; i++) {
      int igdx=xrec[i].get_gindex();
      void *ptr=xrec[i].get_ptr();
      spkt.pkbyte((char*)&ptr,sizeof(void*));
      spkt<<igdx<<glb_vtab[igdx]->hint<<glb_vtab[igdx]->size;
    }
    spkt.send(proclist.get_entry(srcno)->get_tid(),ADSM_CHANNEL_ALLOCATE);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_malloc,_adsm_msgsiz_malloc);
#endif
  }
}

void adsmd_forward_register(DynArray<AllocRec> *forwardlist) {
  for (int hn=0; hn<nhost; hn++) {
    int fnum=forwardlist[hn].get_size();
    if (fnum<=0) continue;
    AllocRec *frec=forwardlist[hn].get_array();

    spkt.reset();
    spkt<<(char)CHECKCODE<<srcno<<(char)ADSMD_CMD_REGISTER<<fnum;
    for (int i=0; i<fnum; i++) {
      int igdx=frec[i].get_gindex();
      void *ptr=frec[i].get_ptr();

      spkt.pkbyte((char*)&ptr,sizeof(void*));
      spkt<<glb_vtab[igdx]->name<<glb_vtab[igdx]->size
          <<glb_vtab[igdx]->hint<<glb_vtab[igdx]->home<<(int)(i+1<fnum);

      if (glb_vtab[igdx]->data) {
        spkt<<1;
        spkt.pkbyte((char*)glb_vtab[igdx]->data,glb_vtab[igdx]->size);
        delete [] glb_vtab[igdx]->data;
        glb_vtab[igdx]->data=NULL;
      } else
        spkt<<0;
    }
    spkt.send(htab_dtids[hn],ADSM_CHANNEL_CMD); // parallel allocate
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_malloc,_adsm_msgsiz_malloc);
#endif

    forwardlist[hn].reset();
  }
}

// enlarge global table size
void adsmd_inc_table_size(int index) {
  int oldsize=glb_table_size;
  glb_table_size+=((index+GLB_TABLE_SIZE-1)/GLB_TABLE_SIZE)*GLB_TABLE_SIZE;
  GlbVarTab **new_glb_vtab=new GlbVarTab*[glb_table_size]; 

  int i;
  for (i=0; i<oldsize; i++) new_glb_vtab[i]=glb_vtab[i];
  for (; i<glb_table_size; i++) new_glb_vtab[i]=NULL;

  delete [] glb_vtab;
  glb_vtab=new_glb_vtab;
}

void adsmd_doneload(char cmd,short lpno) {
  // aggregate reply to the same source
  int lnum=loadlist.get_size();
  int *lgindex=loadlist.get_array();

  int i=0;
  while (i<lnum) {
    int conti=1;
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<cmd;
#endif

    while (conti==1) {
      spkt<<lgindex[i];
      conti=(i+1>=lnum||spkt.suggestsend())?-1:1;
      spkt<<conti; // continue signal
      spkt.pkbyte(glb_vtab[lgindex[i]]->data,glb_vtab[lgindex[i]]->size);
      i++;
    }
    spkt.send(proclist.get_entry(lpno)->get_tid(),
      cmd==(char)ADSMD_CMD_REFRESH?ADSM_CHANNEL_REFRESH:ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif
  }
  loadlist.reset();
}

void adsmd_store(int gindex,int conti,int ack) {
  if (glb_vtab[gindex]->data==NULL)
    glb_vtab[gindex]->data=new char[glb_vtab[gindex]->size];

  // apply storing content to home bucket

  if (glb_vtab[gindex]->hint&AdsmDataMWriter) { 
    // apply modified set and save it for write update 
    rpkt.upkmodset(glb_vtab[gindex]->data); 
    
    if (!glb_vtab[gindex]->lastmodset) 
      glb_vtab[gindex]->lastmodset=new ModifySet;

    glb_vtab[gindex]->lastmodset->dup(rpkt.getmodset()); // latter cover former
  } else 
    rpkt.upkbyte(glb_vtab[gindex]->data,glb_vtab[gindex]->size); 

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc) {
    cerr<<msghead<<"P"<<srcno<<" store ";
    if (_adsm_ctrlflags&AdsmCtrlDbgDat) {
      cerr<<hex;
      for (int j=0; j<glb_vtab[gindex]->size; j++)
        cerr<<(unsigned)glb_vtab[gindex]->data[j];
      cerr<<dec<<" to ";
    }
    cerr<<glb_vtab[gindex]->name;
    if (_adsm_ctrlflags&AdsmCtrlDbgDat)
      if (glb_vtab[gindex]->hint&AdsmDataMWriter)
        cerr<<" by ModifySet "<<(*glb_vtab[gindex]->lastmodset);
    cerr<<endl;
  }
#endif

  adsmd_coherence(gindex);

  if (conti==1) return; // to be continued, ack latter

  adsmd_donestore(ack);
}

int adsmd_donestore(int ack) {
  int cohacknum=0;

  // bulk transfering inv/update messages
  for (short pn=0; pn<nproc; pn++) {
    int cnum=colist[pn].get_size();
    if (cnum<=0) continue; // no msg for pn

    short proc_hno=proclist.get_entry(pn)->get_hostno();
    if (proc_hno<0) continue;

    unsigned long newcohver=cohvertab.get_ver(hostno,pn); // new ver for pn

    int *cgdx=colist[pn].get_array();

    int i=0;
    while (i<cnum) {
      ++newcohver; // increase coherence version/timestamp
      if (newcohver==MAXLONG) {
        cerr<<"Adsmith: timestamp overflow !!"<<endl;
	shutdown();
      }

      int conti2=1;
      spkt.reset();
#ifdef ADSMCTRL
      if (_adsm_ctrlflags&AdsmCtrlOptAck) {
        cohacknum++; // the number of the acks of the coherence messages 
        // send coherence messages to the daemon first
        spkt<<(char)CHECKCODE<<srcno<<(char)ADSMD_CMD_ACK<<pn;
      }
#endif
      spkt<<hostno<<newcohver; 
      while (conti2==1) {
        conti2=(i+1>=cnum||spkt.suggestsend())?-1:1;
        int cogindex=cgdx[i];
        if (glb_vtab[cogindex]->hint&AdsmDataUpdate) {
          spkt<<(int)ADSM_COHERENCE_UPDATE<<cogindex;
          spkt<<conti2;
          if (glb_vtab[cogindex]->hint&AdsmDataMWriter) 
            spkt.pkmodset(glb_vtab[cogindex]->lastmodset); 
          else
            spkt.pkbyte(glb_vtab[cogindex]->data,glb_vtab[cogindex]->size);
        } else {
          spkt<<(int)ADSM_COHERENCE_INV<<cogindex;
          spkt<<conti2;
        }
        i++;
      }

#ifdef ADSMCTRL
      if (_adsm_ctrlflags&AdsmCtrlOptAck)
        spkt.send(htab_dtids[proc_hno],ADSM_CHANNEL_CMD);
      else
#endif
        spkt.send(proclist.get_entry(pn)->get_tid(),ADSM_CHANNEL_COHERENCE);
#ifdef ADSMCTRL
      spkt.classify_message(_adsm_msgnum_consist,_adsm_msgsiz_consist);
#endif
    }
    colist[pn].reset();
    cohvertab.set_ver(hostno,pn,newcohver);	// set newer cohver for pn
  }

  // ack to request process with coherent version and procno list
#ifdef ADSMCTRL
  // Force ack when OptAck is enabled
  if (ack || _adsm_ctrlflags&AdsmCtrlOptAck) {
#else
  if (ack) {
#endif
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<(char)ADSMD_CMD_FLUSH;
#endif

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk)) {
#endif
      spkt<<nproc<<hostno;
      spkt.pkulong(cohvertab.get_ver(hostno),(int)nproc);
#ifdef ADSMCTRL
    }
    spkt<<cohacknum; // ask the writer to wait for that number of acks
#endif
    spkt.send(stid,ADSM_CHANNEL_FLUSH); // all store action assume pipeline
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif
  }

  return cohacknum;
}

void adsmd_cmd_acksim() {
#ifdef ADSMCTRL
  cout<<"adsmd_cmd_acksim(): option Oa (ack) is not available "
      <<"in this version, please execute your program without "
      <<"the flags Oa set in ADSM_FLAGS"<<endl;
/*
  // retrieve the coherence message
  short pn;
  rpkt>>pn;

  int restsize=rpkt.get_restsize();
  char *data=new char[restsize];
  rpkt.upkbyte(data,restsize);
  
  // forward the coherence message
  spkt.reset();
  spkt.pkbyte(data,restsize);
  int sz=spkt.send(proclist.get_entry(pn)->get_tid(),ADSM_CHANNEL_COHERENCE);

  delete [] data;

  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<"forward a coherence message to P"<<pn<<endl;

  // send ack to writer (srcno) first
  spkt.reset();
  spkt.send(proclist.get_entry(srcno)->get_tid(),ADSM_CHANNEL_ACK);
  // count ack msgs
  spkt.classify_message(_adsm_msgnum_consist,_adsm_msgsiz_consist);

  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<"send W-Ack to P"<<srcno<<endl;
*/
#endif
}

void shutdown() {
  spkt.reset();
  spkt<<(char)CHECKCODE<<srcno<<(char)ADSMD_CMD_SHUTDOWN;
  spkt.send(master_dtid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif
}

// --- command functions ---

void adsmd_cmd_init() {
  int i;

  rpkt>>nhost;

  unsigned long ac;
  rpkt>>ac;
  _adsm_ctrlflags=ac;

  if (_adsm_ctrlflags&AdsmCtrlTimTtl) _adsm_total_time.start();

  // allocate table memory
  htab_dtids=new int[nhost];

  // set daemon table
  rpkt.upkint(htab_dtids,nhost);

  // save the master_dtid
  master_dtid=htab_dtids[0];
  if (master_dtid==mytid) i_am_master=1;
          
  // set host number
  for (hostno=0; hostno<nhost; hostno++) 
    if (mytid==htab_dtids[hostno]) break;

  // build connection between hosts: start
  spkt.reset();
  for (i=hostno+1; i<nhost; i++) { // begin from next host
    spkt.send(htab_dtids[i],ADSM_CHANNEL_DANGER);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif
  }

  // initialize global variable table and start allocation index
  glb_vtab=new GlbVarTab*[glb_table_size]; 
  for (i=0; i<glb_table_size; i++) glb_vtab[i]=NULL;
  glb_index=(hostno==0)?nhost:hostno;
  /* glb_vtab[0] is left used because 0 is reserved by hash table
     to indicate that one entry is not used yet. */

  // initialize coherence version table
  cohvertab.set_nhost(nhost);
  cohvertab.set_nproc(1);	// at least the starter

  // set to direct route

  if (!(_adsm_ctrlflags&AdsmCtrlOptPDR)) 
    pvm_setopt(PvmRoute,PvmRouteDirect); 

  if (_adsm_ctrlflags&AdsmCtrlInfVer) 
    cerr<<msghead<<"Adsmith version: "<<VERSION<<endl;

  // build connection between hosts: start
  for (i=0; i<hostno; i++) 
    rpkt.recv(htab_dtids[i],ADSM_CHANNEL_DANGER);
}

void adsmd_cmd_enroll() { 
  // record process tid to process list
  short procno=srcno;
  short hno;
  rpkt>>hno;

  Process enroll_proc(stid,hno);

  if (i_am_master)
    procno=proclist.overwrite(enroll_proc);
  else
    proclist.set_entry(procno,enroll_proc);

  procnum++;

  if (procno>=nproc) nproc=procno+1; // procno start from 0

  // reply process no., with proctids
  if (i_am_master) {
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<(char)ADSMD_CMD_ENROLL;
#endif

    spkt<<procno<<nproc;
    for (int i=0; i<nproc; i++) {
      Process *p=proclist.get_entry(i);
      spkt<<p->get_tid()<<p->get_hostno();
    }
    spkt.send(stid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<'P'<<procno<<"(t"<<hex<<stid<<dec<<") joins Adsmith"<<endl;
#endif

  }

  // resize coherence table
  DynArray<int> *newcolist=new DynArray<int>[nproc];
  delete [] colist;
  colist=newcolist;

  // resize coherence version table against process number
  cohvertab.set_nproc(nproc);

  // check version
  char libver[30];
  rpkt>>libver;
  if (strcmp(libver,VERSION)!=0)
    cerr<<"Warning: Adsmith library version ("<<libver
        <<") does not match with daemon version ("<<VERSION<<")"<<endl;
}

int adsmd_cmd_exit() { 
  if (i_am_master) {

#ifdef ADSMCTRL
    // collect message statistics info from the process
    int msgnum,msgsize,msgnum_system,
        msgsiz_system,msgnum_consist,msgsiz_consist,msgnum_malloc,
        msgsiz_malloc,msgsiz_access,msgnum_access,msgsiz_sync,msgnum_sync;

    rpkt>>msgnum>>msgsize>>msgnum_system
        >>msgsiz_system>>msgnum_consist>>msgsiz_consist>>msgnum_malloc
        >>msgsiz_malloc>>msgnum_access>>msgsiz_access>>msgnum_sync
        >>msgsiz_sync;

    total_msgnum        +=msgnum;
    total_msgsize       +=msgsize;
    total_msgnum_system +=msgnum_system; 
    total_msgsiz_system +=msgsiz_system; 
    total_msgnum_consist+=msgnum_consist; 
    total_msgsiz_consist+=msgsiz_consist; 
    total_msgnum_malloc +=msgnum_malloc;
    total_msgsiz_malloc +=msgsiz_malloc;
    total_msgnum_access +=msgnum_access;      
    total_msgsiz_access +=msgsiz_access;      
    total_msgnum_sync   +=msgnum_sync;            
    total_msgsiz_sync   +=msgsiz_sync;            

    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<'P'<<srcno<<" leaves Adsmith"<<endl;
#endif

  }

  Process exit_proc(stid,-1);
  proclist.set_entry(srcno,exit_proc);
  procnum--;
         
  if (procnum>0) return 0;

  // all processes have terminated, collect statistics
  if (!i_am_master) { // slave daemons: send message info to master
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<(char)ADSMD_CMD_EXIT;
#endif

#ifdef ADSMCTRL
    spkt<<spkt.get_msgnum()<<spkt.get_msgsize()
        <<_adsm_msgnum_system<<_adsm_msgsiz_system
        <<_adsm_msgnum_consist<<_adsm_msgsiz_consist
        <<_adsm_msgnum_malloc<<_adsm_msgsiz_malloc
        <<_adsm_msgnum_access<<_adsm_msgsiz_access
        <<_adsm_msgnum_sync<<_adsm_msgsiz_sync;
#endif
    spkt.send(master_dtid,ADSM_CHANNEL_DANGER);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif
  } else { // master daemon: collect message info from all daemons
#ifdef ADSMCTRL
    // add mine
    total_msgnum +=spkt.get_msgnum();
    total_msgsize+=spkt.get_msgsize();
    total_msgnum_system +=_adsm_msgnum_system; 
    total_msgsiz_system +=_adsm_msgsiz_system; 
    total_msgnum_consist+=_adsm_msgnum_consist; 
    total_msgsiz_consist+=_adsm_msgsiz_consist; 
    total_msgnum_malloc +=_adsm_msgnum_malloc;
    total_msgsiz_malloc +=_adsm_msgsiz_malloc;
    total_msgnum_access +=_adsm_msgnum_access;      
    total_msgsiz_access +=_adsm_msgsiz_access;      
    total_msgnum_sync   +=_adsm_msgnum_sync;            
    total_msgsiz_sync   +=_adsm_msgsiz_sync;            
#endif

    // add others
    for (int i=1; i<nhost; i++) {
      rpkt.recv(htab_dtids[i],ADSM_CHANNEL_DANGER);

#ifdef ADSMDEBUG
      char cmd;
      rpkt>>cmd;
      if (cmd!=ADSMD_CMD_EXIT) {
        cerr<<"adsmd_cmd_exit(): wrong message"<<endl;
        Exit(__FILE__,__LINE__);
      }
#endif

#ifdef ADSMCTRL
      int msgnum,msgsize,msgnum_system,
          msgsiz_system,msgnum_consist,msgsiz_consist,msgnum_malloc,
          msgsiz_malloc,msgsiz_access,msgnum_access,msgsiz_sync,msgnum_sync;

      rpkt>>msgnum>>msgsize>>msgnum_system
          >>msgsiz_system>>msgnum_consist>>msgsiz_consist>>msgnum_malloc
          >>msgsiz_malloc>>msgnum_access>>msgsiz_access>>msgnum_sync
          >>msgsiz_sync;

      total_msgnum        +=msgnum;
      total_msgsize       +=msgsize;
      total_msgnum_system +=msgnum_system; 
      total_msgsiz_system +=msgsiz_system; 
      total_msgnum_consist+=msgnum_consist; 
      total_msgsiz_consist+=msgsiz_consist; 
      total_msgnum_malloc +=msgnum_malloc;
      total_msgsiz_malloc +=msgsiz_malloc;
      total_msgnum_access +=msgnum_access;      
      total_msgsiz_access +=msgsiz_access;      
      total_msgnum_sync   +=msgnum_sync;            
      total_msgsiz_sync   +=msgsiz_sync;            
#endif

    }

    // The above trick is dangerous since master is waiting message 
    // from other daemon, which may cause dead lock. To quarrenty the 
    // safety, general daemons can not wait messages from master!
  }

  adsmd_statistics();
  pvm_exit();

  return 1; // terminate
}

void adsmd_cmd_shutdown() { 
  if (i_am_master) {
    int num=proclist.get_size();
    Process *proc=proclist.get_array();
    for (int i=0; i<num; i++) 
      if (proc[i].get_hostno()>=0 && proc[i].get_tid()!=stid) 
        pvm_kill(proc[i].get_tid());
        
    spkt.reset();
    spkt<<(char)CHECKCODE<<srcno<<(char)ADSMD_CMD_SHUTDOWN;
    spkt.mcast(htab_dtids+1,nhost-1,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif
  }

  pvm_exit();

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc) cerr<<msghead<<"terminates"<<endl;
#endif

}

void adsmd_cmd_proctids() {
  spkt.reset();

#ifdef ADSMDEBUG
  spkt<<(char)ADSMD_CMD_PROCTIDS;
#endif

  spkt<<nproc;
  for (int i=0; i<nproc; i++) {
    Process *p=proclist.get_entry(i);
    spkt<<p->get_tid()<<p->get_hostno();
  }
  spkt.send(stid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_system,_adsm_msgsiz_system);
#endif
}

void adsmd_cmd_handshake() {
  int times;
  rpkt>>times;

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<"handshake with P"<<srcno<<", no. "<<times<<endl;
#endif

  spkt.reset();
  spkt<<hostno<<0<<(int)ADSM_COHERENCE_HANDSHAKE;

#ifdef ADSMDEBUG
  spkt<<command<<times;
#endif
  spkt.send(stid,ADSM_CHANNEL_COHERENCE);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_consist,_adsm_msgsiz_consist);
#endif
}

// bulk register
void adsmd_cmd_register() { 
  char var_name[MAX_NAME_LEN],*dupid,*data;
  int gindex,num,size,toinit,conti;
  short hint,home;
  void *ptr;

  rpkt>>num;

  DynArray<AllocRec> existlist;
  existlist.addsize(num);
  DynArray<AllocRec> *forwardlist=new DynArray<AllocRec>[nhost];
  for (int i=0; i<nhost; i++) forwardlist[i].addsize(num/nhost/3); 
  // expect at most 1/3 object will be so

  do {
    rpkt.upkbyte((char*)&ptr,sizeof(void*));
    rpkt>>var_name>>size>>hint>>home>>conti>>toinit;

    // look up hash table first
    VarChainEntry *the_entry=NULL;
    if (unique(var_name)) // UNIQUE allocattion, force allocation
      gindex=-1;
    else {
      the_entry=hash_tab.insert(var_name);
      gindex=the_entry->get_gindex();
    } 

    if (gindex<=0) { // not found, new entry, first time allocation
      gindex=adsmd_get_gindex(); // select a new gindex
      if (unique(var_name)) 
        dupid=UNIQUE_OBJ; // constant
      else {
        the_entry->set_gindex(gindex); // set gindex
        dupid=the_entry->save_name(); // keep private name
      } 

      // data initialization
      data=new char[size];
      if (toinit) 
        rpkt.upkbyte(data,size);
      else
        memset(data,0,size);

      glb_vtab[gindex]=new GlbVarTab(dupid,size,hint,home,data);
      objnum++;

    } else
       if (toinit) rpkt.upkbyte(NULL,size); // prevent re-initialize

    AllocRec allocrec(gindex,ptr);
    if (glb_vtab[gindex]->home==hostno) { // bulk allocation
      existlist.insert(allocrec);

#ifdef ADSMCTRL
      if (_adsm_ctrlflags&AdsmCtrlDbgTrc) {
        cerr<<msghead<<"P"<<srcno<<" allocate "<<glb_vtab[gindex]->name
            <<" (size "<<glb_vtab[gindex]->size<<") as index "<<gindex
            <<" on H/"<<glb_vtab[gindex]->home<<endl;
      }
#endif

    } else // let real home register
      forwardlist[home].insert(allocrec);

  } while (conti==1);

  adsmd_reply_register(existlist);

  adsmd_forward_register(forwardlist);

  delete [] forwardlist;
}

void adsmd_cmd_attach() {
  char	*name="";
  int gindex,size=0,retcode=0;
  short home=0,hint=0;

  rpkt>>gindex;
  if (glb_vtab[gindex]!=NULL) {
    name=glb_vtab[gindex]->name;
    size=glb_vtab[gindex]->size;
    hint=glb_vtab[gindex]->hint;
    home=glb_vtab[gindex]->home;
    retcode=1;
  }

  spkt.reset();

#ifdef ADSMDEBUG
  spkt<<command;
#endif

  spkt<<glb_table_size<<retcode<<name<<size<<hint<<home;
  spkt.send(stid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_malloc,_adsm_msgsiz_malloc);
#endif
}

void adsmd_cmd_undecl() { 
/* some semantic problem here, nullize it temporarily
  int index;
  rpkt>>index;

  if (glb_vtab[index]!=NULL&&	          // if it is originally validate
      glb_vtab[index]->blkqueue==NULL&&	  // if lock list is empty
      glb_vtab[index]->cacherlist->is_empty()) { // if cacher list is empty
    // ask master to undeclare
    spkt.reset();
    spkt<<(char)CHECKCODE<<srcno<<(char)ADSMD_CMD_MASTERUNDECL<<index;
    spkt.send(master_dtid,ADSM_CHANNEL_CMD);
    // race problem may occure here
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_malloc,_adsm_msgsiz_malloc);
#endif
  }
*/
}

void adsmd_cmd_masterundecl() { 
  int index;
  rpkt>>index;
  delete glb_vtab[index];
  glb_vtab[index]=NULL;
}

void adsmd_cmd_refresh() { 
  int gindex,conti;
  loadlist.addsize(objnum);
  do { 
    rpkt>>gindex>>conti;

#ifdef ADSMDEBUG
    if (conti!=1&&conti!=-1) {
      cerr<<"adsmd_cmd_refresh(): wrong continue/stop flag"<<endl;
      Exit(__FILE__,__LINE__);
    }
#endif

    adsmd_load(command,gindex,srcno,conti);
  } while (conti==1);
}

void adsmd_cmd_flush() { 
  int gindex,conti;
  do { 
    rpkt>>gindex>>conti;

#ifdef ADSMDEBUG
    if (conti!=1&&conti!=-1) {
      cerr<<"adsmd_cmd_flush(): wrong continue/stop flag"<<endl;
      Exit(__FILE__,__LINE__);
    }
#endif

    adsmd_store(gindex,conti);
  } while (conti==1);
}

void adsmd_cmd_p() { 
  int gindex;
  rpkt>>gindex;

  int *semaphore=(int*)glb_vtab[gindex]->data;
  if (*semaphore>0) {
    (*semaphore)--;

    // ack to requester immediately
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<command;
#endif

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    spkt<<cohvertab;	// send coherence vector here
    spkt.send(stid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<'P'<<srcno<<" pass semaphore "
          <<glb_vtab[gindex]->name<<endl;
#endif

    return;
  }

  adsmd_block(gindex,srcno);
}

void adsmd_cmd_lock() {
  int gindex;
  rpkt>>gindex;

  if (glb_vtab[gindex]->blkqueue==NULL) { 
    // lock grant: ack to requester
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<command;
#endif

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    spkt<<cohvertab;	// send coherence vector here
    spkt.send(stid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<'P'<<srcno<<" enter mutex "<<glb_vtab[gindex]->name<<endl;
#endif

  }

  adsmd_block(gindex,srcno);
}
            
void adsmd_cmd_barrier() {
  int gindex;
  short ttl;
  rpkt>>gindex>>ttl;

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    rpkt>>cohvertab;	// receive coherence version table here

  if (adsmd_barrier(gindex,ttl)) {
    int bnum=glb_vtab[gindex]->barrlist->get_size();
    short *bpno=glb_vtab[gindex]->barrlist->get_array();

    // ack to requester immediately
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<command;
#endif

    spkt<<glb_vtab[gindex]->total;

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    spkt<<cohvertab;	// send coherence version table here

    int *btid=new int[bnum];
    for (int i=0; i<bnum; i++) 
      btid[i]=proclist.get_entry(bpno[i])->get_tid();
    spkt.mcast(btid,bnum,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif

    delete [] btid;

    // reset
    glb_vtab[gindex]->barrlist->reset();
  }
}

void adsmd_cmd_v() { 
  int gindex;
  rpkt>>gindex;

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    rpkt>>cohvertab;	// receive coherence version table here

  int *semaphore=(int*)glb_vtab[gindex]->data;

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<'P'<<srcno<<" set semaphore "<<glb_vtab[gindex]->name<<endl;
#endif

  if (glb_vtab[gindex]->blkqueue==NULL)
    (*semaphore)++;
  else {
    // wake up first waiter
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<(char)ADSMD_CMD_P;
#endif

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    spkt<<cohvertab;	// send coherence version table here
    spkt.send(proclist.get_entry(glb_vtab[gindex]->blkqueue->next->procno)->
      get_tid(),ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<'P'<<proclist.get_entry(glb_vtab[gindex]->
                          blkqueue->next->procno)->get_tid()
          <<" pass semaphore "<<glb_vtab[gindex]->name<<endl;
#endif

    adsmd_unblock(gindex);
  }
}

void adsmd_cmd_unlock() {
  int gindex;
  rpkt>>gindex;

#ifdef ADSMCTRL
  if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    rpkt>>cohvertab;	// receive coherence version table here

  // compare and remove locker
  adsmd_check_locker(gindex);
  adsmd_unblock(gindex);

#ifdef ADSMCTRL
  if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
    cerr<<msghead<<'P'<<srcno<<" leave mutex "<<glb_vtab[gindex]->name<<endl;
#endif

  // wake up the waiter process
  if (glb_vtab[gindex]->blkqueue!=NULL) {
    spkt.reset();

#ifdef ADSMDEBUG
    spkt<<(char)ADSMD_CMD_LOCK;
#endif

#ifdef ADSMCTRL
    if (!(_adsm_ctrlflags&AdsmCtrlOptHsk))
#endif
    spkt<<cohvertab;	// send coherence vector here
    spkt.send(proclist.get_entry(glb_vtab[gindex]->blkqueue->next->procno)->
      get_tid(),ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
    spkt.classify_message(_adsm_msgnum_sync,_adsm_msgsiz_sync);
#endif

#ifdef ADSMCTRL
    if (_adsm_ctrlflags&AdsmCtrlDbgTrc)
      cerr<<msghead<<'P'<<proclist.get_entry(glb_vtab[gindex]->
                          blkqueue->next->procno)->get_tid()
          <<" enter mutex "<<glb_vtab[gindex]->name<<endl;
#endif

  }
}

// /*readers*/ in the following are now commented out,
// see comments in libadsm.c for atomic_end(,AtomicRead)

void adsmd_cmd_atomic_begin() {
  int gindex,conti;
  char rw;
  rpkt>>gindex>>conti>>rw;

#ifdef ADSMDEBUG
  if (conti!=-1) {
    cerr<<"adsmd_cmd_atomic_begin(): wrong continue/stop flag "<<conti<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  // Writer First Protocol
  switch (rw) {
    case AdsmAtomicWrite:
      adsmd_block(gindex,srcno,rw); // for checking on non-blocked
      if (glb_vtab[gindex]->status&AdsmdStatusWriting // exist a writer
          // || glb_vtab[gindex]->readers>0 // or readers /*readers*/ // writer enter anyway
         ) {
        glb_vtab[gindex]->status|=AdsmdStatusWriting; 
        // prevent future readers for fairness, even when readers!=0
        return;
      }
      glb_vtab[gindex]->status|=AdsmdStatusWriting;
      break;
    case AdsmAtomicRead:
      if (glb_vtab[gindex]->status&AdsmdStatusWriting) { // exist a writer
        adsmd_block(gindex,srcno,rw);
        return;
      }
      // glb_vtab[gindex]->readers++; /*readers*/
      // not added to lock list
      break;
  }

  // lock phase
  adsmd_load(command,gindex,srcno,-1); // lock grant & refresh
}
            
void adsmd_cmd_atomic_end() {
  int gindex,conti;
  char rw;
  short collectnum;
  rpkt>>gindex>>conti>>rw>>collectnum;

#ifdef ADSMDEBUG
  if (conti!=-1) {
    cerr<<"adsmd_cmd_atomic_end(): wrong continue/stop flag "<<conti<<endl;
    Exit(__FILE__,__LINE__);
  }
#endif

  switch (rw) {
    case AdsmAtomicWrite:
      adsmd_store(gindex,-1,collectnum<2); // ack on collectnum<2
      adsmd_check_locker(gindex); // check if requester==locker on write
      adsmd_unblock(gindex); // remove last writer from lock list
      glb_vtab[gindex]->status&=~AdsmdStatusWriting;
      break;
    // case AdsmAtomicRead:
    //   if (--glb_vtab[gindex]->readers>0) // like V except -- /*readers*/ 
    //     return; /*readers*/ 
    //   break; // continue only on readers == 0
  }

  // wake up the waiter process
  while (glb_vtab[gindex]->blkqueue!=NULL) {
    int quit=0;
    Tlock_list *p=glb_vtab[gindex]->blkqueue->next; // waiting record
    switch (p->rwtype) {
      case AdsmAtomicWrite:
        // if (glb_vtab[gindex]->readers==0) /* readers */
          adsmd_load(ADSMD_CMD_ATOMIC_BEGIN,gindex,p->procno,-1);
        glb_vtab[gindex]->status|=AdsmdStatusWriting; 
        quit=1;
        break;
      case AdsmAtomicRead:
        adsmd_load(ADSMD_CMD_ATOMIC_BEGIN,gindex,p->procno,-1);
        adsmd_unblock(gindex); // remove last writer from lock list
        // glb_vtab[gindex]->readers++; /*readers*/
        break;
      default:
        cerr<<"adsmd_atomic_end(): non-write/read type for "
            <<glb_vtab[gindex]->name<<endl;
    }
    if (quit) break;
  }

  // for collect num > 1, do barrier and reply
  if (collectnum>1 && adsmd_barrier(gindex,collectnum)) {
    int bnum=glb_vtab[gindex]->barrlist->get_size();
    short *bpno=glb_vtab[gindex]->barrlist->get_array();

    // send the data to requesters
    for (int i=0; i<bnum; i++)
      adsmd_load(ADSMD_CMD_ATOMIC_END,gindex,bpno[i],-1);

    // reset
    glb_vtab[gindex]->barrlist->reset();
  }
}

void adsmd_cmd_atomic() { 
  int gindex,el;
  static int exprlen=0;
  static char *expr=NULL;

  rpkt>>gindex>>el;

  // get expression
  if (el>exprlen) {
    delete [] expr;
    expr=new char[exprlen=el];
  }
  rpkt.upkbyte(expr,el);

  char *errmsg=adsm_calc(expr,glb_vtab[gindex]->data); // cacluate
  if (errmsg) cerr<<msghead<<"Expression `"<<expr<<"' Error:"<<errmsg<<endl;
  int retcode=errmsg?-1:0;

  // coherence job
  adsmd_coherence(gindex); // record coherence targets
  adsmd_donestore(0); 
    // send invalidate messages, without ack (see libadsm.c)

  // replay
  spkt.reset();

#ifdef ADSMDEBUG
  spkt<<(char)ADSMD_CMD_ATOMIC<<gindex;
#endif

  spkt<<retcode;
  spkt.pkbyte(glb_vtab[gindex]->data,glb_vtab[gindex]->size);

  spkt.send(stid,ADSM_CHANNEL_CMD);
#ifdef ADSMCTRL
  spkt.classify_message(_adsm_msgnum_access,_adsm_msgsiz_access);
#endif
}
