/*
        Adsmith 1.8 :  An Efficient Object-Based DSM Environment on PVM
 
                        Author:  William W. Y. Liang
 
           Institute of Computer Science & Information Engineering
                   National Taiwan University, Taipei, TW
 
                   Copyright (C) 1996 All Rights Reserved
 
                                  NOTICE
 
      Permission to use, copy, modify, and distribute this software and
      its documentation for any purpose and without fee is hereby granted
      provided that the above copyright notice appear in all copies and
      that both the copyright notice and this permission notice appear in
      supporting documentation.
 
      The author makes no representations about the suitability of this
      software for any purpose.  This software is provided ``as is''
      without express or implied warranty.
*/

#ifndef __ADSM_H__
#define __ADSM_H__


/*=====================================================================*
   Process Control Functions (Primary Feature)
 *=====================================================================*/

// adsm_spawn - process spawn function
  int adsm_spawn(char *prog, char **argv=0, int flags=0,
                 char *where=0, int num=1, int *tids=0); 
  int adsm_spawn(char *prog, int num); 

// adsm_wait - wait a process or a group of processes to finish
  int adsm_wait();
  int adsm_wait(int tid);
  int adsm_wait(int *tids, int num);

/*=====================================================================*
   Shared Object Allocation Functions (Primary Feature)
 *=====================================================================*/

// AdsmHint - the properties of the shared objects
  #define AdsmDataLocal   (short)0x0001
  #define AdsmDataCache   (short)0x0002
  #define AdsmDataUpdate  (short)0x0004
  #define AdsmDataMWriter (short)0x0008
  #define AdsmDataMovable (short)0x0010
  #define AdsmDataDefault (short)0

// adsm_malloc - allocate a share object
  void* adsm_malloc(char *id, int size, short hint=AdsmDataDefault); 
  void* adsm_malloc(char *id, int size, void *initdata,
                    short hint=AdsmDataDefault);
  void* adsm_malloc(char *id, int size, void *initdata, int home, 
                    short hint); // To avoid ambiguity! =AdsmDataDefault);

// adsm_free - free the shared object locally
  void adsm_free(void *data); 

/*=====================================================================*
  Ordinary Accesses (Primary Feature)
 *=====================================================================*/

// adsm_refresh - ordinary load access (buffer refresh function)
  void adsm_refresh(void *data);     

// adsm_flush - ordinary store access (buffer flush function)
  void adsm_flush(void *data);

/*=====================================================================*
  Non-Synchronization Accesses (Primary Feature)
 *=====================================================================*/

// adsm_refresh_now - non-synchronization load access (buffer refresh function)
  void adsm_refresh_now(void *data); 

// adsm_flush_now - non-synchronization store access (buffer flush function)
  void adsm_flush_now(void *data);   

/*=====================================================================*
  Synchronization Accesses (Primary Feature)
 *=====================================================================*/


// AdsmSemaphore - the class of shared objects used as semaphores
  class AdsmSemaphore {
  public:
    AdsmSemaphore(char *id, int val=1);
    ~AdsmSemaphore();

    int get();
    void set(int val);

    void wait();
    void signal();

  private:
    int *value;
  };
 
// AdsmMutex - the class of shared objects used as mutex
  class AdsmMutex {
  public:
    AdsmMutex(char *id);
    ~AdsmMutex();

    void lock();
    void unlock();

  private:
    int *tag;
  };

// AdsmBarrier - the class of shared objects used as barrier
  class AdsmBarrier {
  public:
    AdsmBarrier(char *id);
    ~AdsmBarrier();

    int barrier(int count);

  private:
    int *tag;
  };

/*=====================================================================*
  Pointer Control (Primary Feature)
 *=====================================================================*/

// adsm_gid - global address retrieval function
  int adsm_gid(void *data);        

// adsm_attach - attach a global address retrieval into local process
  void* adsm_attach(int gaddr);     

/*=====================================================================*
  Process number and Host number (Primary Feature)
 *=====================================================================*/

// adsm_procno - get the process number
  int adsm_procno();

// adsm_hostno - get the host number
  int adsm_hostno(int procno=-1);

// adsm_procno2tid, adsm_tid2procno - translate between Adsmith
//                                    process number and PVM task id
  int adsm_procno2tid(int procno);
  int adsm_tid2procno(int tid);

/*=====================================================================*
  Other Functions (Primary Feature)
 *=====================================================================*/

// adsm_shutdown - terminate the parallel application
  void adsm_shutdown(); 

// adsm_version - get the Adsmith version
  const char *adsm_version(); 

// adsm_enroll - force to enroll Adsmith
  void adsm_enroll();

/*=====================================================================*
   Bulk Transfer Type and Delimiter Functions (Advance Feature)
 *=====================================================================*/

// AdsmBulkType - bulk transfer type
  enum AdsmBulkType {
    AdsmBulkBegin,
    AdsmBulkEnd
  };

// AdsmBulk - Bulk transfer delimiter function for adsm_malloc(),
//            adsm_refresh(), adsm_flush(), and adsm_prefresh()
  void adsm_malloc(AdsmBulkType type, int num=0);
  void adsm_refresh(AdsmBulkType type);     
  void adsm_flush(AdsmBulkType type); 
  void adsm_prefresh(AdsmBulkType type);     

/*=====================================================================*
   Aggregated Functions (Advance Feature)
 *=====================================================================*/

// adsm_malloc_array - aggregated function of adsm_flush()'s
  void adsm_malloc_array(char *id,int size,int num,void *arrptr,
                         short hint=AdsmDataDefault);
  void adsm_malloc_array(char *id,int size,int num,void *arrptr,
                         int *distarr, short hint=AdsmDataDefault);
  void adsm_malloc_array(char *id, int size, int begin, int num,
                         void *arrptr, void *init=0, int *distarr=0,
                         short hint=AdsmDataDefault);

// adsm_refresh_array - aggregated function of adsm_refresh()'s
  void adsm_refresh_array(void *data, int len);

// adsm_prefresh_array - aggregated function of adsm_prefresh()'s
  void adsm_prefresh_array(void *data, int len);    

// adsm_flush_array - aggregated function of adsm_flush()'s
  void adsm_flush_array(void *data, int len);

/*=====================================================================*
  Prefetch Accesses (Advance Feature)
 *=====================================================================*/

// adsm_prefresh - prefetch access (buffer contents prefetch function
  void adsm_prefresh(void *data);    

/*=====================================================================*
  Atomic Accesses (Advance Feature)
 *=====================================================================*/

// AdsmAtomicType - atomic type
  enum AdsmAtomicType {
    AdsmAtomicWrite,
    AdsmAtomicRead
  };

// adsm_atomic_begin, adsm_atomic_end - atomic section delimiter function
  void adsm_atomic_begin(void *data, int rw=AdsmAtomicWrite); 
  void adsm_atomic_end(void *data);   

// adsm_atomic - atomic expression delimiter function
  void adsm_atomic(void *data, char *expr);   

/*=====================================================================*
  Collect Accesses (Advance Feature)
 *=====================================================================*/

// adsm_collect_begin, adsm_collect_end - collect access delimiter function
  void adsm_collect_begin(void *data, int num);
  void adsm_collect_end(void *data);

#endif // __ADSM_H__
