//////////////////////////////////////////////////////////////////////////
//
// Sample program for Adsmith
// Purpose: Calculate the summation of values in a vector
// Paradigm: SPMD
//
// Algorithm: 
//   Shared Objects: 
//     the vector and the summation
//   Procedure:
//     0. Declare shared objects and initialize them.
//     1. Spawn a process on each host.
//     2. Each process calculates the partial sum of equal size subvector. 
//     3. Each process add the partial sum into the global sum themselves.
//
// Usage: sample [size]
//
// Author: William W. Y. Liang
// Update: 4/29/1996
//
//////////////////////////////////////////////////////////////////////////


#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <iostream.h>
#include "adsm.h"	// standard libray of Adsmith
#include "adsmutil.h"	// non-standard utilities of Adsmith
#include "adsmtime.h"	// a timing class

void slave();

// system parameters
int seqno;
int nhost;
AdsmBarrier Bsample("sample");	// for barrier synchronization

// initial values
int zero=0;
int deflen=10000;		// default length

// shared variable
int *len;			// length of the vector
int *vec;			// source of the vector
unsigned *sum;			// result of the sum

main(int argc,char *argv[]) {
  // get my sequence no. and host number
  seqno=get_seqno(); // get_seqno() and get_nhost()
  nhost=get_nhost(); // both are in adsmutil.h

  // initialize content of the vector and spawn child processes
  if (seqno==0) { // the 1st process
    // determine the size of vector
    if (argc>1) deflen=atoi(argv[1]);

    cout<<nhost<<" host(s) detected"<<endl;
    cout<<"Length of vector = "<<deflen<<endl;

    // initialize the vector
    int initvec[deflen];
    for (int i=0; i<deflen; i++) initvec[i]=i;
 
    // allocate shared objects
    adsm_malloc(AdsmBulkBegin);	// allocate in aggregate mode
    len=(int*)adsm_malloc("len",sizeof(int),&deflen);
    sum=(unsigned*)adsm_malloc("sum",sizeof(unsigned),&zero);
    vec=(int*)adsm_malloc("vec",(*len)*sizeof(int),initvec);
    adsm_malloc(AdsmBulkEnd);

    // spawn processes on each host
    adsm_spawn(execname(argv[0]),nhost); // execname extracts the program name

    // timing
    Timing Tsample("sample");
    Bsample.barrier(nhost+1);
    Tsample.start(); // timing start
    Bsample.barrier(nhost+1);
    Tsample.stop();   // timing end

    // obtain the final result
    adsm_refresh(sum); 
    cout<<"Sum = "<<(*sum)<<endl;
  } else {	// the workers
    // allocate shared objects
    adsm_malloc(AdsmBulkBegin);	// allocate in aggregate mode
    len=(int*)adsm_malloc("len",sizeof(int));
    sum=(unsigned*)adsm_malloc("sum",sizeof(unsigned));
    adsm_malloc(AdsmBulkEnd);

    // read the parameters
    adsm_refresh(len);

    // allocate shared vector and read the value
    vec=(int*)adsm_malloc("vec",(*len)*sizeof(int));
    adsm_refresh(vec);

    slave();
  }

  adsm_free(len);
  adsm_free(sum);
  adsm_free(vec);
}

void slave() {
  // calculate the range to be computed by this process
  int size=((*len)+nhost-1)/nhost;
  int begin=size*(seqno-1); 
  int end=begin+size; 
  if (end>(*len)) end=(*len);

  // ready to compute
  Bsample.barrier(nhost+1); 

  // calculate partial sum
  int partial=0;
  for (int i=begin; i<end; i++) partial+=vec[i];

  // add partial sum to the total sum

  // Using Mutex

  // AdsmMutex mutex("mutex");
  // mutex.lock();	// lock the mutex
  // adsm_refresh(sum);	// load the content of sum
  // *sum+=partial;	// add partial sum to total sum
  // adsm_flush(sum);	// store the content of sum
  // mutex.unlock();	// unlock the mutex

  // Using Atomic Begin-End

  // atomic access is faster than the above mutex/refresh method
  // adsm_atomic_begin(sum); // begin of an atomic access on `sum'
  // *sum+=partial;        // add partial sum to total sum
  // adsm_atomic_end(sum);   // end of the atomic access

  // Using Atomic Expression

  char expr[50];
  sprintf(expr,"[uint] @+=%d",partial); // 'uint' is 'unsigned int'
  adsm_atomic(sum,expr);

  // wait for all processes done
  Bsample.barrier(nhost+1);
}
