// Program: An example code of Producer-Consumer
//
// Reference Book: Concurrent Programming, Principles and Practice
//                 Gregory R. Andrews, Page 189
//
// Usage: prodcons
//
// Author: William W. Y. Liang
// Date: 1996/5/8

#include <iostream.h>
#include <unistd.h>
#include "adsm.h"
#include "adsmutil.h"

#define BUFSIZE	5 // buffer size
#define PRODNUM	2 // producer number
#define CONSNUM	4 // consumer number
#define RUNTIME	5 // seconds

main() {
  int seqno=get_seqno();
  int tmp=0;
 
  if (seqno==0) {
    cout<<"Buffer size = "<<BUFSIZE<<endl
        <<"Producers = "<<PRODNUM<<endl
        <<"Consumers = "<<CONSNUM<<endl
        <<"Run time = "<<RUNTIME<<" seconds "<<endl;
   
    // spawn child
    adsm_spawn("prodcons",PRODNUM+CONSNUM);

    // this is the controller, shutdown all processes after timeout
    sleep(RUNTIME);
    adsm_shutdown();
    return 0;
  }

  // allocate buffer, indies
  int *buf=(int*)adsm_malloc("buf",sizeof(int)*BUFSIZE,
                             AdsmDataCache|AdsmDataUpdate);
  int *front=(int*)adsm_malloc("front",sizeof(int),&tmp,
                             AdsmDataCache|AdsmDataUpdate);
  int *rear=(int*)adsm_malloc("rear",sizeof(int),&tmp,
                             AdsmDataCache|AdsmDataUpdate);

  // semaphores
  AdsmSemaphore empty("empty",BUFSIZE);
  AdsmSemaphore full("full",0);

  // mutex
  AdsmMutex producer("producer"); // protect between producers
  AdsmMutex consumer("consumer"); // protect between consumers

  // main loop
  int count=0;
  while (1) {
    if (seqno<=PRODNUM) { // producer

      empty.wait(); // produce only when buffer has empty slots
      producer.lock(); // synchronize with other producer for shared 'rear'
        adsm_refresh(rear);

        buf[*rear]=++count; // put data on buffer

        cout<<"Producer["<<seqno<<"] put "<<count
            <<" on "<<*rear<<"'th position of the buffer."<<endl;

        *rear=(*rear+1)%BUFSIZE;

        adsm_flush(rear);
        adsm_flush(buf);
      producer.unlock();
      full.signal(); // signal that an item has been produced

    } else { // consumer

      full.wait(); // consume only when buffer has full slots
      consumer.lock(); // synchronize with other consumer for shared 'front'
        adsm_refresh(buf);
        adsm_refresh(front);

        count=buf[*front]; // get data from buffer

        cout<<"Consumer["<<seqno<<"] get "<<count
            <<" from "<<*front<<"'th position of the buffer."<<endl;

        *front=(*front+1)%BUFSIZE;

        adsm_flush(front);
      consumer.unlock();
      empty.signal(); // signal that an item has been consumed

    }
  }
}
