################################################################################

#  Edit file window.  There is an entry for the file name (default efil) and
#  an entry for the editor if the EDITOR environment variable is not set.
#  The edited file name is put into edfile

proc EditFile {efil {w .editfile}} {
   global penv status edfile prefs colors

   catch {destroy $w}
   toplevel $w
   wm title $w "PADE Edit File"
   wm iconname $w "Edit File"
   wm geometry $w +290+458
   $w config -cursor {arrow red white}
   
   set status 0
   set edfile ""
 
   frame   $w.f2 -bd 1 -relief raised
   frame   $w.f3 -bd 1 -relief raised
   frame   $w.f4 -bd 1 -relief raised

#------------------------------------------------------------------------------
#  if an editor is not defined then get one
   if {$penv(EDITOR)==""} {
      set edtext Edit
      frame   $w.f1 -bd 1 -relief raised
      label $w.f1.msg -text "Editor command:"
      entry   $w.f1.entry -textvariable penv(EDITOR) -relief sunken -width 55
      .editfile.f1.entry delete 0 end
      pack   $w.f1.msg -side left -padx 2m -pady 5m -anchor w
      pack   $w.f1.entry -side right -padx 2m -pady 5m -anchor e
   } else {
      set edtext $penv(EDITOR)
   }

#------------------------------------------------------------------------------
#  file to edit
   label   $w.f2.msg -text "Enter file name:"
   entry   $w.f2.entry -textvariable efil -relief sunken -width 55
   if {$efil!=""} {
      .editfile.f2.entry delete 0 end
      .editfile.f2.entry insert 0 $efil
   }
   pack    $w.f2.msg -side left -expand 1 -padx 2m -pady 5m -anchor w
   pack    $w.f2.entry -side left -expand 1 -padx 2m -pady 5m -anchor e

#------------------------------------------------------------------------------
   text	   $w.f3.text -height 1 -fg $colors(TEXT) -width 1 -state disabled \
      -relief flat
   pack    $w.f3.text -fill x -expand 1

#------------------------------------------------------------------------------
   button $w.f4.clear -text "Clear" -command {
      TextWrite .editfile.f3.text ""
      set efil ""
      .editfile.f2.entry delete 0 end
   }

   button  $w.f4.filefind -text "File Browser" -command {
      TextWrite .editfile.f3.text ""
      set ffile [fileselect "Pick existing file" $prefs(FILE_PATH)/ 1]
      if {$ffile!=""} {
         .editfile.f2.entry delete 0 end
         .editfile.f2.entry insert 0 $ffile
         focus .editfile.f2.entry
         set efil $ffile
      }
   }

#  Editor button
   button $w.f4.edit -text $edtext -command {
      TextWrite .editfile.f3.text ""
      RunEditor $efil
      if {$status} {
         set edfile $efil
         set efil ""
         destroy .editfile
      }
   }

   button $w.f4.cancel -text "Cancel" -command {
      TextWrite .editfile.f3.text ""
      set edfile ""
      set efil ""
      destroy .editfile
   }
   pack   $w.f4.clear $w.f4.filefind $w.f4.edit $w.f4.cancel \
     -side left -expand 1 -padx 3m -pady 4m -ipadx 2m -ipady 1m

#------------------------------------------------------------------------------
#  display editor prompt only if editor is not defined

   if {$penv(EDITOR)==""} {
      pack  $w.f1 -anchor w -fill x
      pack  $w.f2 $w.f3 $w.f4 -fill x
      focus $w.f1.entry
      bind $w.f1.entry <Return> {focus .editfile.f2.entry}
      bind $w.f1.entry <Tab>    {focus .editfile.f2.entry}
      bind $w.f1.entry <Button-2> {
	 .editfile.f1.entry insert insert [selection get]
      }
   } else {
      pack  $w.f2 $w.f3 $w.f4 -fill x
      focus .editfile.f2.entry
   }

#  run editor if <Return> in filename entry
   bind $w.f2.entry <Return> {
      TextWrite .editfile.f3.text ""
      RunEditor $efil
      if {$status} {
         set edfile $efil 
         set efil ""
         destroy .editfile
      }
   }
   bind $w.f2.entry <Button-2> {
      .editfile.f2.entry insert insert [selection get]
   }
   ConfigColorText $w
   update idletasks
   tkwait window $w
}

################################################################################

#  run an editor with file

proc RunEditor {file} {
   global penv status prefs

   if {($penv(EDITOR)!="") && ($file!="")} {
      if {[string compare $penv(EDITOR) xedit]==0} {
         exec xedit $file &
      } elseif {[string compare $penv(EDITOR) nedit]==0} {
         exec nedit $file &
      } elseif {[string compare $penv(EDITOR) textedit]==0} {
         exec textedit $file &
      } else {
         exec $prefs(TERM) -geometry +170+218 -sl 1000 -e $penv(EDITOR) $file &
      }
      set status 1
      return
   }
   set status 0
   if {($penv(EDITOR)=="") && ($file=="")} {
      TextWrite .editfile.f3.text "Editor and file not defined"
   } elseif {$file==""} {
      TextWrite .editfile.f3.text "File not defined"
   } elseif {$penv(EDITOR)==""} {
      TextWrite .editfile.f3.text "Editor not defined"
   }
}

################################################################################

#  displays the contents of a file in a listbox, possibly save it to another file

proc ListFile {listname text} {
   global nwin winnam nline

#  allow for multiple windows
   incr nwin
   set winnam($nwin) .$nwin

   catch {destroy $winnam($nwin)}
   toplevel $winnam($nwin)

   wm title $winnam($nwin) "$text"
   wm iconname $winnam($nwin) "$text"
   wm geometry $winnam($nwin) +170+218
   $winnam($nwin) config -cursor {arrow red white}
   
   frame     $winnam($nwin).f1 -bd 1 -relief raised
   frame     $winnam($nwin).f2 -bd 1 -relief raised

#  listbox with scroll bars
   scrollbar $winnam($nwin).f1.yscroll -relief sunken \
      -command [list $winnam($nwin).f1.list yview]   
   scrollbar $winnam($nwin).f1.xscroll -relief sunken \
      -command [list $winnam($nwin).f1.list xview] -orient horiz
   listbox   $winnam($nwin).f1.list -xscroll [list $winnam($nwin).f1.xscroll set] \
      -yscroll [list $winnam($nwin).f1.yscroll set] -relief sunken -width 80 \
      -height 30 -setgrid 1

#  put listname in listbox
   set file [open $listname r]
   set nline($nwin) -1
   while {[gets $file line] >= 0} {
      regsub -all \t $line " " line
      $winnam($nwin).f1.list insert end $line
      incr nline($nwin)
   }

   pack      $winnam($nwin).f1.xscroll -side bottom -fill x
   pack      $winnam($nwin).f1.yscroll -side right -fill y
   pack      $winnam($nwin).f1.list -expand 1 -fill y

#  close button, figure out which window is being closed
   button    $winnam($nwin).f2.done -text "Close" -command {
      set bpick [WhichListButton done]
      if {$bpick > 0} {destroy $winnam($bpick)}
   }
      
#  save button, figure out which window is being closed
   button    $winnam($nwin).f2.save -text "Save" -command {
      set bpick [WhichListButton save]
      if {$bpick > 0} {
	 set savefile [fileselect "Enter file to Save output to" $prefs(FILE_PATH)/ 0]
	 if {$savefile!=""} {
            set ok 1
	    if {[file isdirectory $savefile]} {
               Dialog .d {PADE Warning} \
        	  "$savefile  is not a file" error 0 {OK}
               set ok 0
	    }
	    if {[file exists $savefile]} {
               set pick [Dialog .d {PADE Question} \
        	  "Overwrite  $savefile  ?" questhead 0 \
        	  {Overwrite} {Cancel}]
               if {$pick} {set ok 0}
	    }
            if {$ok} {
 	       set f_savefile [open "$savefile" w]
	       set i 0
	       while {$i <= $nline($bpick)} {
		  puts $f_savefile [$winnam($bpick).f1.list get $i]
		  incr i
	       }
	       close $f_savefile
            }
	 }
      }
   }

   pack $winnam($nwin).f2.save $winnam($nwin).f2.done \
      -side left -expand 1 -padx 3m -pady 4m -ipadx 2m -ipady 1m
   pack $winnam($nwin).f1 $winnam($nwin).f2 -side top -fill x   

   ConfigColorText $winnam($nwin)

   tkwait window $winnam($nwin)
}

################################################################################

#  figure out which button was picked in the List window

proc WhichListButton {button} {
   global winnam nwin
   
   set bpick 0
   DoLoop i 1 $nwin {
      if {[winfo exists $winnam($i)]} {
         if {[lindex [$winnam($i).f2.$button configure -state] 4] == \
            "active"} {set bpick $i}
      }
   }
   return $bpick
}

################################################################################

#  an entry window with Done, Cancel, Clear, and File Browser buttons
#  x,y - window corner
#  labeltext - value of label
#  defaultentry - default entry value

proc EntryBox {x y labeltext defaultentry {w .entry}} {
   global entry1 defaultentrysave prefs

   catch {destroy $w}
   toplevel $w

   wm title $w "PADE Entry"
   wm iconname $w "Entry"
   wm geometry $w +$x+$y
   $w config -cursor {arrow red white}

   set defaultentrysave ""
   if {$defaultentry!=""} {set defaultentrysave $defaultentry}

   frame   $w.f1 -bd 1 -relief raised
   frame   $w.f2 -bd 1 -relief raised

   label   $w.f1.msg -text $labeltext
   entry   $w.f1.entry -textvariable entry1 -relief sunken -width 55 
   $w.f1.entry delete 0 end
   $w.f1.entry insert 0 $defaultentry
   pack $w.f1.msg $w.f1.entry -side left -expand 1 -padx 2m -pady 4m

   button $w.f2.done -text "Done" -command {
      set entry1 [string trim $entry1]
      destroy .entry
   }
   button $w.f2.cancel -text "Cancel" -command {
      set entry1 $defaultentrysave
      destroy .entry
   }
   button $w.f2.clear -text "Clear" -command {
      set entry1 ""
      .entry.f1.entry delete 0 end
   }
   button  $w.f2.filefind -text "File Browser" -command {
      set ffile [fileselect "Pick file" $prefs(FILE_PATH)/ 0]
      if {$ffile!=""} {
         .entry.f1.entry delete 0 end
         .entry.f1.entry insert 0 $ffile
      }
   }

   pack $w.f2.clear $w.f2.filefind $w.f2.done $w.f2.cancel \
      -side left -expand 1 -padx 3m -pady 4m -ipadx 2m -ipady 1m
   pack $w.f1 $w.f2 -fill x   
   ConfigColorText $w

   focus $w.f1.entry

   bind $w.f1.entry <Return> {
      set entry1 [string trim $entry1]
      destroy .entry
   }
   tkwait window $w
}

################################################################################

#  a dialog, from 'Tcl and the Tk Toolkit' by Ousterhout, page 267

proc Dialog {w title text bitmap default args} {
   global tkPriv

#  1. Create top-level window and divide into top and bottom parts

   catch {destroy $w}
   toplevel $w -class Dialog
   wm title $w $title
   wm iconname $w Dialog
   wm protocol $w WM_DELETE_WINDOW { }
   wm transient $w [winfo toplevel [winfo parent $w]]
   $w config -cursor {arrow red white}
   frame $w.top -relief raised -bd 1
   pack $w.top -side top -fill both
   frame $w.bot -relief raised -bd 1
   pack $w.bot -side bottom -fill both

#  2. Fill top part with bitmap and message

   message $w.top.msg -width 6i -text $text
   pack $w.top.msg -side right -expand 1 -fill both -padx 3m -pady 3m
   if {$bitmap != ""} {
      label $w.top.bitmap -bitmap $bitmap
      pack $w.top.bitmap -side left -padx 3m -pady 3m
   }

#  3. Create row of buttons at bottom of dialog

   set i 0
   foreach but $args {
       button $w.button$i -text $but -command "set tkPriv(button) $i"
       if {$i == $default} {
	   frame $w.default -relief sunken -bd 1
	   raise $w.button$i $w.default
	   pack $w.default -in $w.bot -side left -expand 1 -padx 3m -pady 2m
	   pack $w.button$i -in $w.default -padx 2m -pady 2m
	   bind $w <Return> "$w.button$i flash; set tkPriv(button) $i"
       } else {
	   pack $w.button$i -in $w.bot -side left -expand 1 \
		   -padx 3m -pady 2m
       }
       incr i
   }

# 4. Withdraw the window, then update all the geometry information
# so we know how big it wants to be, then center the window in the
# display and de-iconify it.

   ConfigColorText $w
   wm withdraw $w
   update idletasks
   set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	   - [winfo vrootx [winfo parent $w]]]
   set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	   - [winfo vrooty [winfo parent $w]]]
   wm geom $w +$x+$y
   wm deiconify $w

# 5. Set a grab and claim the focus too.

   set oldFocus [focus]
   grab $w
   tkwait visibility $w
   if {$default >= 0} {
       focus $w.button$default
   } else {
       focus $w
   }

# 6. Wait for the user to respond, then restore the focus and
# return the index of the selected button.  Restore the focus
# before deleting the window, since otherwise the window manager
# may take the focus away so we can't redirect it.

   tkwait variable tkPriv(button)
   catch {focus $oldFocus}
   destroy $w
   return $tkPriv(button)
}

################################################################################

#  write text in win

proc TextWrite {win text} {

   $win configure -state normal
   $win delete 1.0 end
   $win insert end $text
   $win configure -state disabled
}

################################################################################

#  Look for commands in directories of PATH environment variable
#  1 - found, 0 - not found

proc FindCmd {cmd} {
   global env penv

   if {$penv(PATH)!=""} {
      foreach dir [split $env(PATH) ":"] {
         if {[file exists $dir/$cmd]} {
	    if {[file isfile $dir/$cmd] && [file executable $dir/$cmd]} {return 1}
         }
      }
      if {$cmd!=""} {
         puts " $cmd not found in PATH environment variable"
      }
      return 0
   } else {
      return 1
   }
}

################################################################################

#  a do loop, from 'Tcl and the Tk Toolkit' by Ousterhout, page 123

proc DoLoop {varName first last body} {
   global errorInfo errorCode
   
   upvar $varName v
   for {set v $first} {$v <= $last} {incr v} {
      set code [catch {uplevel $body} string]
      if {$code == 1} {
         return -code error -errorinfo $errorInfo \
            -errorcode $errorCode $string
      } elseif {$code == 2} {
         return -code return $string
      } elseif {$code == 3} {
         return
      } elseif {$code > 4} {
         return -code $code $string
      }
   }
}

################################################################################

#  find minimum of a list, return index of minimum

proc MinList {list} {
   set min 0

   DoLoop i 1 [expr [llength $list]-1] {
      if {[lindex $list $i] < [lindex $list $min]} {set min $i}
   } 
   return $min
}

################################################################################

#  return current timestamp

proc TimeStamp {} {

   set tempfile /tmp/pade.temp[pid]
   set f_temp [open $tempfile w]
   puts $f_temp 1
   close $f_temp
   set ts [file mtime $tempfile]
   catch [file delete $tempfile]
   return $ts
}

################################################################################

#  configure all windows for color and text or whatever
#  w is the window name, for example: . or .mywindow

proc ConfigColorText {w} {
   global colors fonts
   
   SetColorText $w
   
#  check if there are any children for the current parent
   if {[winfo children $w] != ""} {
   
#  loop on all the children
      foreach i [winfo children $w] {
         SetColorText $i
         ConfigColorText $i
      }
      
#  no more children
   } else {
      return
   }
}
################################################################################

#  set color and text or whatever

proc SetColorText {i} {
   global colors fonts
   
   set ic [winfo class $i]

#  configure widgets
   if {$ic == "Menubutton" || $ic == "Menu" || $ic == "Button"} {
      catch {$i configure -background grey85}
      catch {$i configure -font $fonts(FONT_SLANT)}
      catch {$i configure -highlightthickness 0}
   } elseif {$ic == "Entry" || $ic == "Listbox"} {
      catch {$i configure -background #d1a0a0}
      catch {$i configure -font $fonts(FONT_NORM)}
      catch {$i configure -highlightthickness 0}
   } else {
      catch {$i configure -background grey90}
      catch {$i configure -font $fonts(FONT_NORM)}
      catch {$i configure -highlightthickness 0}
   } 
}
