/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: pgm.P,v $
 *	$Author: brunner $	$Locker:  $		$State: Exp $
 *	$Revision: 1.1 $	$Date: 1994/10/14 21:24:06 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: pgm.P,v $
 * Revision 1.1  1994/10/14  21:24:06  brunner
 * Initial revision
 *
 ***************************************************************************/


/* Simple Charm++ program for doing a Jacobi method for a 5-point stencil
   problem	*/


const int n = 4 ;
/* nxn is the size of the sub-domain */

#define THRESHOLD   1e-4

const int EAST=0, WEST=1, NORTH=2, SOUTH=3  ;

message BOUNDARY { // message used to send sub-domain boundaries to other PEs
    int from;  
    float boundary[n+2]; 
} ;

message BooleanObject {	// message having the reduction result
    int convergeValue ;
} ;

message dummyMsg {
    int x ;
} ;


extern "C" double sqrt(double);

class TwoDGrid {	// this class encapsulates data and functions
			// associated with the sub-domain
    float A[n+2][n+2] ;

    int init() {
        for (int i=0; i<n+2; i++)
            for(int j=0; j<n+2; j++) A[i][j] = 0.0 ;
	return(0) ;
    }

    public:
    TwoDGrid() 
    {
	init() ;
    }

    void update(TwoDGrid * Q)
    { 
        for(int i=1; i<=n; i++)
            for(int j=1; j<=n; j++)
                Q->A[i][j] = 0.25*(A[i-1][j]+A[i+1][j]+A[i][j-1]+A[i][j+1]);
    }
    void copy(TwoDGrid * Q) 
    { 
        for(int i=1; i<=n; i++)
            for(int j=1; j<=n; j++) A[i][j] = Q->A[i][j];
    }
    
#define myabs(x) ((x<0)?-x:x)
    int LocalConverge(TwoDGrid * Q) 
    {
        float diff ;
    
        for(int i=1; i<=n; i++) {
            for(int j=1; j<=n; j++) {
                if ( A[i][j] == 0.0 )
                    diff = myabs(Q->A[i][j]-A[i][j]) ;
                else 
                    diff = myabs(Q->A[i][j]-A[i][j]) / myabs(A[i][j]) ;
                if ( diff > THRESHOLD ) 
                    return 0  ;
            }
        }
    
        return 1 ;
    }
    
    void PrintResult(int mypenum, int K, int numitns) 
    {  
        for(int i=1; i<=n; i++)
            for(int j=0; j<=n; j++)
                CPrintf("(%d,%d) %f\n", (mypenum/K)*n+i, (mypenum%K)*n+j, A[i][j]);
        CPrintf("Total number of iterations : %d\n",numitns) ;
    }

    void SetRow(int rownum, float value)
    {
        for ( int i=1; i<n; i++ ) 
            A[rownum][i] = value ;
    }

    void SetCol(int colnum, float value)
    {
        for ( int i=1; i<n; i++ ) 
            A[i][colnum] = value ;
    }

    void CopyRow(int rownum, float *vector)
    {
        for ( int i=1; i<n; i++ ) 
            A[rownum][i] = vector[i];
    }

    void CopyCol(int colnum, float *vector)
    {
        for ( int i=1; i<n; i++ ) 
            A[i][colnum] = vector[i];
    }

    void GetRow(int rownum, float *vector)
    {
        for ( int i=1; i<n; i++ ) 
            vector[i] = A[rownum][i] ;
    }

    void GetCol(int colnum, float *vector)
    {
        for ( int i=1; i<n; i++ ) 
            vector[i] = A[i][colnum] ;
    }

} ;



// base class, to which reduction results are sent 
branched chare class Receiver {

    entry :
    virtual void ReceiveResult(BooleanObject *result)
    // This is called in a broadcast when the reduction is completed.
    {
        CPrintf("Default processing,result->convergeValue=%d\n",
                            result->convergeValue) ;
    }
} ;


// this implements a global "AND" reduction
branched chare class ReductionAnd {
    BooleanObject *currentValue ;
    int count ;
    Receiver handle client ;
 
    void init()
    {   currentValue = (BooleanObject *) new_message(BooleanObject) ;
        currentValue->convergeValue = TRUE ;
        count = CNumSpanTreeChildren(CMyPeNum()) + 1 ;
    }

    void AbsorbValue(BooleanObject *value)
    {   currentValue->convergeValue &= value->convergeValue ; 
        if (--count == 0) {
            // This processor is the leaf of the tree
            // (or has already received messages from all its children)
            if (CMyPeNum() == CSpanTreeRoot()) // Broadcast global convergence value 
	    	client[ALL]=>ReceiveResult(currentValue) ;
            else  {
                int parent = CSpanTreeParent(CMyPeNum()) ; 
                thishandle[parent]=>FromChildren(currentValue) ;
            }
	    init() ;   // Be ready for next reduction
        }
    }

    public: 
    void StartRedn(BooleanObject *local, Receiver handle h) 
    // Called by clients to start a global SUM 
    {   client = h ;
        AbsorbValue(local) ; 
    }       

 entry:
    void Initialize(dummyMsg *parameters)
    // This is called when this chare is created by a new_branched_chare call
    {   init() ;
	delete_message(parameters) ;
    }

    void FromChildren(BooleanObject *value)  // Children send their values here
    {       AbsorbValue(value) ;     
    } 
 
} ;

message DOMAINInit { 
    ReductionAnd handle redn;
} ;




branched chare class domain : public Receiver 
{
    TwoDGrid P, Q ;
    int   count,numOfNeighbours;
    int   reductionDone;
    int   valid[4];
    int   K, mypenum;       // KxK is the number of processors 
    ReductionAnd handle rednHandle ;
    int NumItns ;

 entry: 
    void BranchInit(DOMAINInit *msg)
    {   
        rednHandle = msg->redn;
        delete_message(msg);
        NumItns = 0 ;
        initialize();
        sendBoundaries();
        count = numOfNeighbours;
        reductionDone = TRUE;  
    }

    void recvBoundary (BOUNDARY *msg)
    {   
        copyBoundary(msg->from,msg->boundary);
        delete_message(msg);
        if (--count == 0 && reductionDone) iterate() ;  
    }

private: 
    void iterate()
    {   
        P.update(&Q);
        NumItns++ ;

        BooleanObject *local = (BooleanObject *) new_message(BooleanObject) ;
        local->convergeValue = P.LocalConverge(&Q);
    
        // Reinitialize for next iteration
        reductionDone = FALSE;
        count = numOfNeighbours; 

	rednHandle[LOCAL]->StartRedn(local,thishandle) ;	
    }


entry: 
    void ReceiveResult(BooleanObject *global)
    // This is called in a broadcast when the reduction is completed.
    { 
        if ( global->convergeValue )
            P.PrintResult(mypenum,K,NumItns);
        else { 
            reductionDone = TRUE;
            P.copy(&Q);
            sendBoundaries() ;
            if (count == 0) iterate() ; 
        } 
    }

private: 
    void initialize()
    {   
        mypenum = CMyPeNum();
        int maxpe = CMaxPeNum(); 


        K  = (int) sqrt((double) maxpe);
        if (K*K != maxpe) { 
            CPrintf("error. The number of processors must be a square.\n");
            CharmExit();
        }
        int posi  = mypenum / K;   
        int posj  = mypenum % K;
        numOfNeighbours = 4;

	// the valid array tells us which neighbours are valid
        for(int i=0;i<4; i++) 
            valid[i] = TRUE;
        if (posi == 0)   {
            numOfNeighbours--; 
            valid[NORTH] = FALSE;
        }
        if (posi == K-1) {
            numOfNeighbours--; 
            valid[SOUTH] = FALSE;
        }
        if (posj == 0)   {
            numOfNeighbours--; 
            valid[WEST]= FALSE;
        }
        if (posj == K-1) {
            numOfNeighbours--; 
            valid[EAST]= FALSE;
        }
        // initialize external boundaries
        if (posi == 0) P.SetRow(0, 1.0) ;

    }

private: 
    void sendBoundaries()
    { 
        int destPe;

        for (int dir=EAST; dir<=SOUTH; dir++)
            if (valid[dir]) {
                BOUNDARY *msg = (BOUNDARY *) new_message(BOUNDARY);
                switch (dir) {
                case NORTH : 
                    msg->from = SOUTH;  
                    destPe = mypenum-K;
                    P.GetRow(1,msg->boundary) ;
                    break;
                case SOUTH : 
                    msg->from = NORTH;  
                    destPe = mypenum+K;
                    P.GetRow(n,msg->boundary) ;
                    break;
                case EAST  : 
                    msg->from = WEST;   
                    destPe = mypenum+1;
                    P.GetCol(n,msg->boundary) ;
                    break;
                case WEST  : 
                    msg->from = EAST;   
                    destPe = mypenum-1;
                    P.GetCol(1,msg->boundary) ;
                }
                thishandle[destPe]=>recvBoundary(msg) ;
            }
    }

private: 
    void copyBoundary(int from, float boundary[])
    {   
        switch (from) {
        case NORTH: 
            P.CopyRow(0,boundary) ;
            break;
        case SOUTH: 
            P.CopyRow(n+1,boundary) ;
            break;
        case EAST : 
            P.CopyCol(n+1,boundary) ;
            break;
        case WEST : 
            P.CopyCol(0,boundary) ;
        }
    }
} ;



#include "quiescence.h"

chare class main {

entry: 
    void QUIESCENCE(QUIESCENCE_MSG *m) 
    { 
	CPrintf("In Quiescence, exiting\n") ;
	delete_message(m) ;
        CharmExit(); 
    }

public:
    main()
    { 
	dummyMsg *m = (dummyMsg *) new_message(dummyMsg) ;

	// Create the branched chare for doing the reduction
        DOMAINInit *msg = (DOMAINInit *) new_message(DOMAINInit);
        msg->redn  = new_branched_chare(ReductionAnd,&(ReductionAnd::Initialize),m) ; 

	// Now create the actual branched chare for doing the Jacobi method
        domain handle jacobi_boc;
        jacobi_boc = new_branched_chare(domain,&(domain::BranchInit),msg);

	// Start Quiescence detection
	CStartQuiescence(&(main::QUIESCENCE),mainhandle) ;

    }
} ;

