/* @TITLE "version0.c - reading old pattern files" */
/* version0.c: Compatibility code for reading old version 0 pattern files. 
 *   Pattern files and reference strings are lists of block numbers, 
 * instead of lists of (offset, len) references. The format of the file
 * and data structures is slightly different. This module is called whenever
 * reading an old v0 pattern is needed; it reads in the v0 file and 
 * stores it internally in v1 format. We can no longer write v0 files.
 *
 * This file exports functions
 *    ReadV0Pattern
 *    ReadV0Header
 */

/* $Id: version0.c,v 7.1 91/05/09 19:33:24 dfk Tape2 $ */

#include <stdio.h>
#include <usdfk.h>
#include "format.h"			/* new format */
#include "refs.h"
#include "pat-intern.h"

/* @SUBTITLE "Version 0 file format information" */
/* Version 0 File format: */
/* (char) magic number
 * (char) version number
 * (structure) header
 * (array of string headers) string header for each string
 * (array of strings) each of the strings
 *   each string has format:
 *   (array of int) the set of blocks in this string
 *   (array of int) the set of comptimes in this string
 *   (array of int) the portions in this string, if any
 */

#define P0_NAMELEN 39
#define P0_COMLEN 79

/* typedef's used here */
typedef struct pattern0_s P0_HEAD;
typedef struct string0_s P0_STRING;

struct pattern0_s {
    char name[P0_NAMELEN+1];	/* name of pattern */
    char comment[P0_COMLEN+1]; /* comment about pattern */
    unsigned int filesize;	/* "file" size in blocks */
    boolean global;			/* global (T) or local (F) pattern */
    int nprocs;			/* number of strings in this file */
    /* these aren't necessary but tell how it was made */
    unsigned int computation;	/* computation per block */
    boolean computation_fixed; /* is computation a fixed amount? */
    short style;			/* pattern style (eg PAT_LPS) */
};

/* Where to find the string header for string s, s in [0,nprocs) */
#define STRING0_POS(s) ((unsigned int) \
  (2*sizeof(short)+sizeof(P0_HEAD)+(s)*sizeof(P0_STRING)))

/* actually, this did not change from v0 to v1 */
struct string0_s {
    unsigned short numrefs;	/* number of references in this string */
    unsigned short numportions; /* number of portions in this string */
    boolean portion_limit;	/* limit prefetching to this portion? */
    unsigned long location;	/* start of refs array in file */
};

#define V0_CHUNKSIZE 1024	/* all my old tests used 1K blocks */

/* STATIC GLOBAL VARIABLES */
static short *position = NULL; /* string to be read from file (shared) */

static P0_HEAD cur0_head;	/* header for current pattern */

/* LOCAL FUNCTIONS */
static unsigned int *IntList();

/* @SUBTITLE "ReadV0Header: read header for version 0" */
/* We read a version 0 header from the file and store it in cur0_head.
 * We copy the relevant data to a version 1 header and return it
 * to the caller, which is always OpenPattern.
 */
boolean
ReadV0Header(v1_head, pattern_fp)
	P_HEAD *v1_head;		/* v1 head to store information */
	FILE *pattern_fp;		/* file to read from */
{
    if (fread(&cur0_head, sizeof(cur0_head), 1, pattern_fp) != 1)
	 return(FALSE);

    strncpy(v1_head->name, cur0_head.name, P_NAMELEN);
    strncpy(v1_head->comment, cur0_head.comment, P_COMLEN);
    v1_head->filesize = cur0_head.filesize * V0_CHUNKSIZE;
    v1_head->global = cur0_head.global;
    v1_head->nprocs = cur0_head.nprocs;
    v1_head->chunksize = V0_CHUNKSIZE;
    v1_head->writes = FALSE;	
    v1_head->computation = cur0_head.computation;
    v1_head->computation_fixed = cur0_head.computation_fixed;
    v1_head->style = cur0_head.style;			

    return(TRUE);
}

/* @SUBTITLE "ReadV0Pattern: Read one pattern string" */
/* If we are here, then ReadV0Header must have been called above.
 * Thus, we have access to the header in cur0_head. 
 */
/* file pattern_fp is left open */
void
ReadV0Pattern(pos)
	int pos;				/* which pattern? */
{
    P0_STRING string;		/* one string header */
    unsigned int *sectors;
    unsigned int *comptime;
    int i;
    ONEREF *ref;

    OpenPatternFile();

    /* read string header */
    fseek(pattern_fp, STRING0_POS(pos), 0);
    fread(&string, sizeof(string), 1, pattern_fp);

    /* allocate refs structure if necessary */
    if (my_refs == (REFS *) NULL) {
	   my_refs = (REFS *) AllocLocal(sizeof(REFS));
	   bzero(my_refs, sizeof(REFS));
    }

    /* allocate space for lists */
    sectors = IntList(NULL, my_refs->nchunks, string.numrefs);
    comptime = IntList(NULL, my_refs->nchunks, string.numrefs);
    if (string.numportions != 0)
	 my_refs->portions = IntList(my_refs->portions, my_refs->nportions,
						    string.numportions);
    else {
	   if (my_refs->portions) {
		  UsFree(my_refs->portions);
		  my_refs->portions = NULL;
	   }
    }
    my_refs->nportions = string.numportions;

    /* read the string */
    fseek(pattern_fp, string.location, 0);
    fread(sectors, sizeof(unsigned int), string.numrefs, pattern_fp);
    fread(comptime, sizeof(unsigned int), string.numrefs, pattern_fp);
    if (my_refs->portions)
	 fread(my_refs->portions, sizeof(unsigned int), string.numportions, 
		  pattern_fp);

    /* allocate space for new-version references */
    if (my_refs->nchunks < string.numrefs && my_refs->chunks != NULL) {
	   UsFree(my_refs->chunks);
	   my_refs->chunks = NULL;
    }
    if (my_refs->chunks == NULL) {
	   my_refs->chunks = (ONEREF*)AllocLocal(string.numrefs * sizeof(ONEREF));
	   if (my_refs->chunks == NULL) {
		  fprintf(stderr, "node %d Out of memory for reference list\n", 
				UsProc_Node);
		  exit(1);
	   }
    }
    my_refs->nchunks = string.numrefs;

    /* copy old version references to new version */
    for (i = 0; i < my_refs->nchunks; i++) {
	   ref = &(my_refs->chunks[i]);
	   ref->write = FALSE;
	   ref->length = V0_CHUNKSIZE;
	   ref->offset = sectors[i] * V0_CHUNKSIZE;
	   ref->comptime = comptime[i];
    }

    /* initialize other parts of ref structure */
    my_refs->global = cur0_head.global;
    my_refs->chunksize = V0_CHUNKSIZE;
    my_refs->next = 0;
    my_refs->readingchunk = NULL;
    if (my_refs->portions)
	 my_refs->cur_portion_end = my_refs->portions[0];
    else
	 my_refs->cur_portion_end = my_refs->nchunks-1;
    my_refs->portion_limit = string.portion_limit;

    UsFree(sectors);
    UsFree(comptime);
}

/* @SUBTITLE "IntList: reallocate a list of integers" */
static unsigned int *
IntList(list, oldnum, newnum)
	unsigned int *list;
	int oldnum;
	int newnum;
{
    if (list && oldnum < newnum) {
	   UsFree(list);
	   list = NULL;
    }
    if (list == NULL) {
	   list = (unsigned int *) AllocLocal(newnum * sizeof(unsigned int));
	   if (list == (unsigned int *)NULL) {
		  fprintf(stderr, "Out of memory\n");
		  exit(1);
	   }
	   return(list);
    } else
	 return(list);
}
