/* @TITLE "transpat - generate transitional pattern from plain patterns" */
/*
 * transpat.c - Builds a pattern file to be used by the driver, 
 * based on two other pattern files. Basically, a certain number
 * of references from each pattern are picked out and the two concatenated.
 *
 * In the local case, the work must be done in parallel; in the global
 * case, the work is done only on the main 
 * processor. The file itself may be written in parallel.
 *
 * David Kotz 4/90
 */

static char rcsid[] = "$Id: transpat.c,v 7.1 91/05/09 19:33:24 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "format.h"
#include "refs.h"
#include "pat-intern.h"

/* @SUBTITLE "Declarations" */

/* IMPORTED Data, Functions */
extern void FreeRefs();
extern REFS *my_refs;

/* GLOBAL DATA */
char filename1[P_FILENAMELEN+1];	/* first filename */
char filename2[P_FILENAMELEN+1];	/* second filename */
char newfilename[P_FILENAMELEN+1];	/* new pattern filename */
#define STYLE_LEN 10
char stylename[STYLE_LEN+1];

static P_HEAD newhead;		/* the header info for pattern */

int skip;					/* # chunks to skip now */
int skip1, skip2;			/* # chunks to skip in each pattern */
int len1, len2;			/* # of chunks from each old pattern */
int newlen;				/* # of chunks in new pattern */

REFS *newrefs;				/* the new pattern; possibly one per proc */

/* LOCAL FUNCTIONS */
static void Setup();
static void CopyPart();
static void MovePattern();
static void ReadInfo();
static void ShareInfo();

/* @SUBTITLE "main program" */
main(argc, argv)
	int argc;
	char **argv;
{
    P_HEAD *head;			/* header for pattern file */

    InitializeUs();			/* so we can use parallelism, if there */

    ReadInfo(argc, argv);	/* read pattern parameters */

    newlen = len1+len2;		/* new pattern length */

    head = OpenPattern(filename1, "r"); /* open the first file */
    if (head == (P_HEAD *)NULL)
	 exit(1);

    if (head->nprocs != ProcsInUse()) {
	   ClosePattern();
	   fprintf(stderr, "Need exactly %d processors to do this.\n", 
			 head->nprocs);
	   exit(1);
    }
    /* remember the head information */
    bcopy(head, &newhead, sizeof(P_HEAD));
    ClosePattern();
	 
    /* Look at second file */
    head = OpenPattern(filename2, "r"); /* open the second file */
    if (head == (P_HEAD *)NULL)
	 exit(1);

    if (head->global != newhead.global) {
	   ClosePattern();
	   fprintf(stderr, "One pattern is global and one local; not allowed.\n");
	   exit(1);
    }
    if (head->nprocs != newhead.nprocs) {
	   ClosePattern();
	   fprintf(stderr, "The two patterns use different numbers of procs (%d and %d).\n", 
			 newhead.nprocs, head->nprocs);
	   exit(1);
    }
    newhead.filesize = max(newhead.filesize, head->filesize);
    newhead.chunksize = max(newhead.chunksize, head->chunksize);
    newhead.writes = newhead.writes || head->writes;
    ClosePattern();
	 
    /* Set up for parallel action */
    ShareInfo();			/* share parameters */

    GenTaskForEachProc(Setup);

    if (!LoadPattern(filename1, NULL)) { /* open the first file */
	   fprintf(stderr, "Cannot load pattern '%s'\n", filename1);
	   exit(1);
    }
    printf("Skipping %d chunks, taking %d chunks from %s\n",
		 skip1, len1, filename1);
    skip = skip1;
    Share(&skip);
    GenTaskForEachProc(CopyPart, len1);
    /* close file everywhere */
    ClosePattern();

    if (!LoadPattern(filename2, NULL)) { /* open the second file */
	   fprintf(stderr, "Cannot load pattern '%s'\n", filename2);
	   exit(1);
    }
    printf("Skipping %d chunks, taking %d chunks from %s\n", 
		 skip2, len2, filename2);
    skip = skip2;
    Share(&skip);
    GenTaskForEachProc(CopyPart, len2);
    /* close file everywhere */
    ClosePattern();

    newhead.style = PAT_MIX;

    CreateFile(newfilename, &newhead);
    GenTaskForEachProc(MovePattern);
    printf("Writing %d chunks to %s\n", newlen, newfilename);
    /* write the patterns to the file */
    WritePattern();
    /* close file everywhere */
    ClosePattern();

    printf("File %s complete\n", newfilename);

    exit(0);
}

/* @SUBTITLE "Setup: Set up the new pattern" */
static void
Setup()
{
    newrefs = (REFS *)AllocLocal(sizeof(REFS));
    bzero(newrefs, sizeof(REFS));

    /* allocate space for lists */
    newrefs->chunks = (ONEREF *)AllocLocal(newlen * sizeof(ONEREF));
    newrefs->portions = (unsigned int *)AllocLocal(newlen * sizeof(int));

    newrefs->global = newhead.global;
    newrefs->nchunks = 0;
    newrefs->nportions = 0;
    newrefs->next = 0;
    newrefs->cur_portion_end = 0;
    newrefs->chunksize = newhead.chunksize;
    newrefs->portion_limit = FALSE; /* we simply leave this false */
}

/* @SUBTITLE "CopyPart: Copy part of the current pattern" */
static void
CopyPart(len)
	int len;				/* number of chunks to copy */
{
    /* global skip; */
    int base = newrefs->nchunks; /* number already in new pattern */
    int cs, cd;			/* source and destination index for chunks */
    int ps, pd;			/* source and destination index for portions */

    if (skip + len > my_refs->nchunks) {
	   printf("Not enough chunks in pattern: need %d, have %u\n", 
			skip+len, my_refs->nchunks);
	   if (my_refs->nchunks <= skip) {
		  skip = 0;
		  len = my_refs->nchunks;
	   } else {
		  len = my_refs->nchunks - skip;
	   }
	   printf("Adjusted to skip %d, length %d\n", skip, len);
    }

    /* copy chunks from one to the other */
    for (cs = skip, cd = base; cs < skip+len; cs++, cd++)
	 newrefs->chunks[cd] = my_refs->chunks[cs];
    newrefs->nchunks = cd;

    if (my_refs->portions != NULL) {
	   /* end previous portion, if necessary */
	   pd = newrefs->nportions; 
	   if (base > 0 && pd == 0)
		newrefs->portions[pd++] = base-1;
	   /* add portions to the list of portions */
	   for (ps = 0;
		   ps < my_refs->nportions && my_refs->portions[ps] < len;
		   ps++) 
		newrefs->portions[pd++] = my_refs->portions[ps] - skip + base;
	   /* terminate final portion */
	   if (pd == 0 || newrefs->portions[pd-1] != newrefs->nchunks-1)
		newrefs->portions[pd++] = newrefs->nchunks-1;
	   newrefs->nportions = pd;
    } else if (newrefs->nportions > 0) {
	   /* in case a non-portion pattern follows a portion pattern */
	   newrefs->portions[newrefs->nportions++] = newrefs->nchunks-1;
    }
}

/* @SUBTITLE "MovePattern: Move newrefs to my_refs" */
static void
MovePattern()
{
    FreeRefs();			/* free my_refs */
    my_refs = newrefs;		/* now pretend that newrefs is my_refs */

    if (my_refs->nportions == 0) {
	   UsFree(my_refs->portions);
	   my_refs->portions = NULL;
    }
	 
}

/* @SUBTITLE "ReadInfo: Read in the various pattern parameters" */

static void
ReadInfo(argc, argv)
	int argc;
	char **argv;
{
    char line[11];

    printf("Note: %d Procs in use; must match pattern files\n", ProcsInUse());

    if (--argc > 0) {
	   strncpy(filename1, *++argv, P_FILENAMELEN);
	   printf("Filename of FIRST pattern is '%s'\n", filename1);
    } else {
	   printf("Name of FIRST pattern FILE (%d chars): ", P_FILENAMELEN);
	   ReadString(filename1, "Filename", P_FILENAMELEN);
    }	   
    if (*filename1 == '\0') {
	   printf("Must specify filename\n");
	   exit(1);
    }

    if (--argc > 0) {
	   skip1 = atoi(*++argv);
	   printf("Skip %d chunks in first file\n", skip1);
    } else {
	   printf("Number of chunks to SKIP in first file [per proc, if local]: ");
	   ReadNumber("Chunks to skip", &skip1);
    }

    if (--argc > 0) {
	   len1 = atoi(*++argv);
	   printf("Take %d chunks from first file\n", len1);
    } else {
	   printf("Number of chunks from first file [per proc, if local]: ");
	   ReadNumber("Number of chunks", &len1);
    }

    if (--argc > 0) {
	   strncpy(filename2, *++argv, P_FILENAMELEN);
	   printf("Filename of SECOND pattern is '%s'\n", filename2);
    } else {
	   printf("Name of SECOND pattern FILE (%d chars): ", P_FILENAMELEN);
	   ReadString(filename2, "Filename", P_FILENAMELEN);
    }	   
    if (*filename2 == '\0') {
	   printf("Must specify filename\n");
	   exit(1);
    }

    if (--argc > 0) {
	   skip2 = atoi(*++argv);
	   printf("Skip %d chunks in second file\n", skip2);
    } else {
	   printf("Number of chunks to SKIP in second file [per proc, if local]: ");
	   ReadNumber("Chunks to skip", &skip2);
    }

    if (--argc > 0) {
	   len2 = atoi(*++argv);
	   printf("Take %d chunks from second file\n", len2);
    } else {
	   printf("Number of chunks from second file [per proc, if local]: ");
	   ReadNumber("Number of chunks", &len2);
    }

    if (--argc > 0) {
	   strncpy(newfilename, *++argv, P_FILENAMELEN);
	   printf("Filename of NEW pattern is '%s'\n", newfilename);
    } else {
	   printf("Name of NEW pattern FILE (%d chars): ", P_FILENAMELEN);
	   ReadString(newfilename, "Filename", P_FILENAMELEN);
    }	   
    if (*newfilename == '\0') {
	   printf("Must specify filename\n");
	   exit(1);
    }

    if (access(newfilename, 0) == 0) {
	   printf("File '%s' exists. Overwrite [y]? ", newfilename);
	   ReadString(line, "yes or no", 10);
	   switch (*line) {
		  case 'y':
		  case 'Y':
		  case '\0': {
			 break;
		  }
		  case 'n':
		  case 'N':
		  default: {
			 printf("File not touched.\n");
			 exit(0);
		  }
	   }
    }

    if (--argc > 0) {
	   strncpy(newhead.name, *++argv, P_NAMELEN);
	   printf("Pattern name is '%s'\n", newhead.name);
    } else {
	   printf("Name of pattern (%d chars): ", P_NAMELEN);
	   ReadString(newhead.name, "Name", P_NAMELEN);
    }

    if (--argc > 0) {
	   strncpy(newhead.comment, *++argv, P_COMLEN);
	   printf("Comment is '%s'\n", newhead.comment);
    } else {
	   printf("Comment (%d chars): ", P_COMLEN);
	   ReadString(newhead.comment, "Comment", P_COMLEN);
    }

    if (--argc > 0) {
	   strncpy(stylename, *++argv, STYLE_LEN);
	   printf("Style name is '%s'\n", stylename);
    } else {
	   printf("Pattern Style name (eg, lps/lpr): ");
	   ReadString(stylename, "Style", STYLE_LEN);
    }
}

/* @SUBTITLE "ShareInfo: Share the pattern parameters" */
static void
ShareInfo()
{
    Share(&newlen);
    Share(&skip1);
    Share(&skip2);

    ShareBlk(&newhead, sizeof(newhead));
}

