/* @TITLE "read - read patterns from file" */
/*
 * read.c - read the pattern that was generated from the file. For
 * LoadPattern this is done in parallel for local patterns.
 * Any pattern already resident is overwritten.
 *
 * David Kotz 4/89, 8/90
 */

static char rcsid[] = "$Id: read.c,v 7.1 91/05/09 19:33:24 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "format.h"
#include "refs.h"
#include "pat-intern.h"

/* GLOBAL variables */
REFS *my_refs = NULL;		/* my reference string */
static REFS **all_refs = NULL; /* all reference strings (NULL if not shared) */
int pat_procs = 0;			/* how many procs in pattern */

/* STATIC GLOBAL VARIABLES */
static short *position = NULL; /* string to be read from file (shared) */
static unsigned long last_location; /* loc of last pattern read */

static P_HEAD cur_head;		/* header for current pattern */
static int cur_version;		/* which version is the input file? */

extern boolean ReadV0Pattern(); /* from version0.c */
extern boolean ReadV0Header(); /* from version0.c */

/* LOCAL FUNCTIONS */
static void ReadMyString();
static void ReadOneString();
static unsigned int *IntList();
static void ResetEach();
static void FreeAnyPattern();
static void FreeSharedRefs();

/* EXPORTED FUNCTIONS */
void FreeRefs();

/* @SUBTITLE "LoadPattern: Load the pattern from the file" */
/* Will read the pattern in for all processors.
 * The number of processors we have must equal the number
 * in the pattern, if a local pattern.
 * If the filename is the same as one loaded previously, no reload is
 * necessary.
 */
boolean					/* true if successful */
LoadPattern(filename, filesize)
	char *filename;		/* disk file containing pattern */
	unsigned int *filesize;	/* (output) size of file in bytes */
{
    boolean success = FALSE;	/* return value */

    FreeSharedRefs();		/* if shared, free everything */

    /* Same file as last time? Just reset and return. */
    if (my_refs != NULL && SamePatternFile(filename)) {
	   fprintf(stderr, "Resetting pattern %s\n", filename);
	   ResetPattern();
	   if (filesize != NULL)
		*filesize = cur_head.filesize;
	   return(TRUE);
    }

    if (position == (short *)NULL)
	 AllocateShareZero(position, short);
    else
	 *position = 0;

    /* open file and read header into cur_head */
    if (OpenPattern(filename, "r") == (P_HEAD *)NULL)
	 return(FALSE);

    if (cur_head.global) {
	   /* old pattern was local, new one is global */
	   if (my_refs != NULL && !my_refs->global)
		GenTaskForEachProc(FreeRefs, 0);
	   ReadPattern(0);
	   Share(&my_refs);		/* use same my_refs for all procs */
	   success = TRUE;
    } else {
	   if (cur_head.nprocs != ProcsInUse()) {
		  fprintf(stderr, "Pattern requires %u processors\n",
				cur_head.nprocs);
		  success = FALSE;
	   } else {
		  /* old pattern was global, new one is local */
		  if (my_refs != NULL && my_refs->global) {
			 FreeRefs();
			 Share(&my_refs);
		  }
		  ShareBlk(&cur_head, sizeof(cur_head));
		  GenTaskForEachProc(ReadMyString, 0);
		  success = TRUE;
	   }
    }

    ClosePattern();

    /* store filesize for caller */
    if (success && filesize != NULL)
	 *filesize = cur_head.filesize;

    return(success);
}

/* @SUBTITLE "LoadPatternShared: Load the pattern from the file" */
/* Will read the pattern in for all processors.
 * This is different from LoadPattern, since we do not need to 
 * match the number of processes with the nprocs of the pattern.
 * We read all strings into shared memory and record them all in
 * all_refs. Then later, UsePattern will set my_refs from all_refs.
 *
 * If the filename is the same as one loaded previously, no reload is
 * necessary.
 */
boolean					/* true if successful */
LoadPatternShared(filename, headp)
	char *filename;		/* disk file containing pattern */
	P_HEAD *headp;			/* (output) the file header */
{
    /* Free whatever pattern might be out there */
    FreeAnyPattern();

    /* open file and read header into cur_head */
    if (OpenPattern(filename, "r") == (P_HEAD *)NULL)
	 return(FALSE);

    MakePatternShared(cur_head.nprocs);
    ShareBlk(&cur_head, sizeof(cur_head));

    GenOnI(ReadOneString, cur_head.nprocs);

    ClosePattern();

    /* store head for the caller */
    if (headp != NULL)
	 bcopy(&cur_head, headp, sizeof(P_HEAD));

    /* my_refs == NULL still */
    return(TRUE);
}

/* @SUBTITLE "OpenPattern: Open the pattern file and read header" */
P_HEAD *					/* NULL on failure */
OpenPattern(name, mode)
	char *name;			/* file name */
	char *mode;			/* eg "r", "r+" */
{
    int magic;
    int version;

    PatternFileInUse(name, mode);

    OpenPatternFile();

    magic = getc(pattern_fp);
    version = getc(pattern_fp);

    if (magic != PATTERN_MAGIC) {
	   fprintf(stderr, "File '%s' is not a pattern file\n", name);
	   fclose(pattern_fp);
	   pattern_fp = NULL;
	   return(NULL);
    }

    cur_version = version;
    Share(&cur_version);

    if (version == 0) {
	   if (!ReadV0Header(&cur_head, pattern_fp)) {
		  fprintf(stderr, "Pattern file %s: cannot read version 0 header\n",
				name);
		  fclose(pattern_fp);
		  pattern_fp = NULL;
		  return(NULL);
	   }
    } else {
	   if (version != PATTERN_FORMAT_VERSION) {
		  fprintf(stderr, "File '%s' is of unknown format\n", name);
		  fclose(pattern_fp);
		  pattern_fp = NULL;
		  return(NULL);
	   }
	   
	   if (fread(&cur_head, sizeof(P_HEAD), 1, pattern_fp) != 1) {
		  fprintf(stderr, "Pattern file '%s': missing header\n", name);
		  fclose(pattern_fp);
		  pattern_fp = NULL;
		  return(NULL);
	   }
    }

    fprintf(stderr,"Opened pattern '%s' in file '%s', version %d\n",
		  cur_head.name, name, version);

    pat_procs = cur_head.nprocs;
    Share(&pat_procs);

    return(&cur_head);
}

/* @SUBTITLE "ReadOneString: Choose and read a pattern string" */
/* parallel (GenOnI) */
static void
ReadOneString(dummy, index)
	int dummy;
	int index;			/* which string to read */
{
    /* my_refs should be NULL */
    ReadPattern(index);		/* fills in my_refs */
    SetPattern(index);		/* records my_refs in all_refs */
    my_refs = NULL;
}

/* @SUBTITLE "ReadMyString: Choose and read a pattern string" */
/* parallel (GenTaskForEachProc) */
static void
ReadMyString()
{
    /* atomically choose pattern */
    int pos = Atomic_add(position, 1);

    ReadPattern(pos);		/* fills in my_refs */
}

/* @SUBTITLE "ReadPattern: Read one pattern string" */
/* file pattern_fp is left open */
void
ReadPattern(pos)
	int pos;				/* which pattern? */
{
    P_STRING string;		/* one string header */
    
    if (cur_version == 0) {
	   /* Whoops! Old format version: use compatibility function */
	   ReadV0Pattern(pos);
	   return;
    }

    OpenPatternFile();

    /* read string header */
    fseek(pattern_fp, STRING_POS(pos), 0);
    fread(&string, sizeof(string), 1, pattern_fp);

    /* allocate refs structure if necessary */
    if (my_refs == (REFS *) NULL) {
	   my_refs = (REFS *) AllocLocal(sizeof(REFS));
	   bzero(my_refs, sizeof(REFS));
    }

    /* allocate space for lists */
    if (my_refs->nchunks < string.numrefs && my_refs->chunks != NULL) {
	   UsFree(my_refs->chunks);
	   my_refs->chunks = NULL;
    }
    if (my_refs->chunks == NULL) {
	   my_refs->chunks = (ONEREF*)AllocLocal(string.numrefs * sizeof(ONEREF));
	   if (my_refs->chunks == NULL) {
		  fprintf(stderr, "node %d Out of memory for reference list\n", 
				UsProc_Node);
		  exit(1);
	   }
    }
    if (string.numportions != 0)
	 my_refs->portions = IntList(my_refs->portions, my_refs->nportions,
						    string.numportions);
    else {
	   if (my_refs->portions) {
		  UsFree(my_refs->portions);
		  my_refs->portions = NULL;
	   }
    }

    my_refs->nchunks = string.numrefs;
    my_refs->nportions = string.numportions;

    /* read the string */
    fseek(pattern_fp, string.location, 0);
    fread(my_refs->chunks, sizeof(ONEREF), my_refs->nchunks, 
		pattern_fp);
    if (my_refs->portions)
	 fread(my_refs->portions, sizeof(unsigned int), my_refs->nportions, 
		  pattern_fp);

    /* store the location for use by RewritePattern() */
    last_location = string.location;

    /* initialize other parts of ref structure */
    my_refs->global = cur_head.global;
    my_refs->chunksize = cur_head.chunksize;
    my_refs->next = 0;
    my_refs->readingchunk = NULL;
    if (my_refs->portions)
	 my_refs->cur_portion_end = my_refs->portions[0];
    else
	 my_refs->cur_portion_end = my_refs->nchunks-1;

    my_refs->portion_limit = string.portion_limit;
}

/* @SUBTITLE "Use/SetPattern: pick one of the shared strings" */
/* This sets my_refs to one of the shared strings. */
void
UsePattern(index)
	int index;
{
    /* no-op if not in shared mode */
    if (IsSharedPattern())
	 my_refs = all_refs[index];
}

/* This sets one of the shared strings to my_refs. */
void
SetPattern(index)
	int index;
{
    /* no-op if not in shared mode */
    if (IsSharedPattern(NULL))
	 all_refs[index] = my_refs;
}

/* @SUBTITLE "IsSharedPattern: is the pattern a shared pat?" */
/* must be a function since it's called externally */
boolean					/* is the pattern shared */
IsSharedPattern()
{
    return(all_refs != NULL);
}

/* @SUBTITLE "MakePatternShared: make a new pattern, and make it shared" */
boolean
MakePatternShared(nprocs)
	int nprocs;
{
    FreeAnyPattern();

    all_refs = (REFS **)Allocate(nprocs * sizeof(REFS *));
    Share(&all_refs);
    pat_procs = nprocs;
    Share(&pat_procs);
}

/* @SUBTITLE "RewritePattern: Rewrite pattern string to file" */
/* file pattern_fp assumed to be open already, and left open */
/* The location for this position is left in last_location 
 * by ReadPattern.
 */
void
RewritePattern()
{
    if (cur_version != PATTERN_FORMAT_VERSION)
	 fprintf(stderr, "Cannot rewrite version %d pattern file\n",
		    cur_version);
    else {
	   /* write the string */
	   fseek(pattern_fp, last_location, 0);
	   fwrite(my_refs->chunks, sizeof(ONEREF), my_refs->nchunks, 
			pattern_fp);
	   fwrite(my_refs->portions, sizeof(unsigned int), my_refs->nportions,
			pattern_fp);
    }
}

/* @SUBTITLE "RewriteHead: Rewrite pattern head to file" */
/* file pattern_fp assumed to be open already, and left open */
void
RewriteHead(head)
	P_HEAD *head;
{
    if (cur_version != PATTERN_FORMAT_VERSION)
	 fprintf(stderr, "Cannot rewrite version %d pattern file\n",
		    cur_version);
    else {
	   /* seek past magic number and version number */
	   fseek(pattern_fp, 2*sizeof(char), 0);
	   /* write the header */
	   fwrite(head, sizeof(P_HEAD), 1, pattern_fp);
    }
}

/* @SUBTITLE "ClosePattern: Close the pattern file we are using" */
void
ClosePattern()
{
    GenTaskForEachProc(ClosePatternFile, 0);
}

/* @SUBTITLE "IntList: reallocate a list of integers" */
static unsigned int *
IntList(list, oldnum, newnum)
	unsigned int *list;
	int oldnum;
	int newnum;
{
    if (list && oldnum < newnum) {
	   UsFree(list);
	   list = NULL;
    }
    if (list == NULL) {
	   list = (unsigned int *) AllocLocal(newnum * sizeof(unsigned int));
	   if (list == (unsigned int *)NULL) {
		  fprintf(stderr, "Out of memory\n");
		  exit(1);
	   }
	   return(list);
    } else
	 return(list);
}

/* @SUBTITLE "FreeAnyPattern: free whatever is there now" */
static void
FreeAnyPattern()
{
    if (!IsSharedPattern()) {
	   if (my_refs != NULL) {
		  /* had something open in non-shared mode */
		  if (my_refs->global) {
			 FreeRefs();
			 Share(&my_refs);
		  } else
		    GenTaskForEachProc(FreeRefs, 0);
	   }
    } else
	 FreeSharedRefs();

}

/* @SUBTITLE "FreeRefs: Free the local reference string" */
/* may be parallel */
void
FreeRefs()
{
    if (my_refs == NULL)
	 return;

    if (my_refs->chunks != NULL)
	 UsFree(my_refs->chunks);

    if (my_refs->portions != NULL)
	 UsFree(my_refs->portions);

    UsFree(my_refs);

    my_refs = NULL;
    pat_procs = 0;
}

void
FreeSharedRefs()
{
    int r;

    if (all_refs == NULL)
	 return;

    for (r = 0; r < cur_head.nprocs; r++) {
	   UsePattern(r);
	   FreeRefs();
    }
    UsFree(all_refs);

    all_refs = NULL;
    Share(&all_refs);

    pat_procs = 0;
    Share(&pat_procs);

    my_refs = NULL;
    Share(&my_refs);    
}

/* @SUBTITLE "ResetPattern: Resets the reference string after use" */
void
ResetPattern()
{
    if (my_refs && my_refs->global)
	 ResetEach();
    else
	 GenTaskForEachProc(ResetEach,0);
}

static void
ResetEach()
{
    if (my_refs) {
	   my_refs->next = 0;
	   if (my_refs->portions)
		my_refs->cur_portion_end = my_refs->portions[0];
	   else
		my_refs->cur_portion_end = my_refs->nchunks - 1;
    }
}

/* @SUBTITLE "TouchPattern: Touch reference pattern on this proc" */
void
TouchPattern(refs)
	REFS *refs;			/* ref list to touch */
{
    /* if a null argument, touch my_refs */
    if (refs == (REFS *)NULL)
	 refs = my_refs;

    TouchBlock(refs, sizeof(REFS), FALSE);
    if (refs->nchunks != 0)
	 TouchBlock(refs->chunks, refs->nchunks * sizeof(ONEREF), FALSE);
    if (refs->nportions != 0)
	 TouchBlock(refs->portions, refs->nportions * sizeof(int), FALSE);
}

