/* @TITLE "makepat - generate patterns for driver" */
/*
 * makepat.c - Builds a pattern file to be used by the driver.
 * In the local case, the work is done in parallel; in the global case,
 * the work is done only on the main processor and the reference string
 * is shared. The file itself may be written in parallel. Note that we
 * do not know the blocksize - that is independent of the pattern. 
 *
 * usage:
 *   makepat [-#] args
 * where args are described below. The optional -# tells makepat to start up
 * with # processes, regardless of the current cluster size (# should be
 * less than or equal to the cluster size). This is useful for global
 * patterns, where -1 is the correct form.
 *
 * David Kotz 
 *
 *  7/31/90 - added -# option
 *  6/1/90  - extended to arbitrary-size chunks
 *  5/22/89 - added LW1, SEG patterns
 *  5/22/89 - added command-line processing
 *  4/??/89 - created
 */

static char rcsid[] = "$Id: makepat.c,v 7.1 91/05/09 19:33:24 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "format.h"
#include "refs.h"
#include "pat-intern.h"
#include "gen.h"

/* @SUBTITLE "Declarations" */

/* GLOBAL DATA */
#define STYLE_LEN 10
static char stylename[STYLE_LEN+1];	/* the name of the pattern style */
static char filename[P_FILENAMELEN+1];	/* pattern filename */

/* Pattern parameters */
static int Pattern;			/* which pattern style */
int PortionSkip;			/* dist btw starts of portions (chunks) */
int PortionLength;			/* length of portions (chunks) */
unsigned int filesize;		/* size of the file in bytes */
int nprocs;				/* number of processors used for pattern */
int chunks;				/* number of chunks */
int computation;			/* computation time per chunk */
boolean computation_fixed;	/* computation is fixed amount? */
int chunksize;				/* size of chunks (bytes) */
boolean writes;			/* do writes (T), or reads (F) */

/* LOCAL FUNCTIONS */
static boolean ReadInfo();
static void ShareInfo();

/* @SUBTITLE "main program" */
main(argc, argv)
	int argc;
	char **argv;
{
    boolean asked_last;
    P_HEAD genhead;

    if (argc >= 2 && *argv[1] == '-') {
	   /* the -# argument: limit the number of procs */
	   int nprocs = atoi(argv[1]+1);
	   printf("Limiting to %d procs\n", nprocs);
	   SetUsConfig(configProcs, nprocs);
	   argc--; argv++;		/* "scan" off this argument */
    }

    /* so we can use parallelism */
    InitializeUs();

    /* read pattern parameters */
    asked_last = ReadInfo(argc, argv, &genhead);

    ShareInfo();

    /* announce a new pattern, shared mode */
    MakePatternShared(genhead.nprocs);

    /* Generate the pattern for the experiment */
    switch (Pattern) {
	   case PAT_RND: genRND(); break;
	   case PAT_LPR: genLPR(); break;
	   case PAT_LPS: genLPS(); break;
	   case PAT_GPR: genGPR(); break;
	   case PAT_GPS: genGPS(); break;
	   case PAT_GW:  genGW(); break;
	   case PAT_LW:  genLW(); break;
	   case PAT_LW1: genLW1(); break;
	   case PAT_SEG: genSEG(); break;
	   case PAT_GRND: genGRND(); break;
	   default: printf("Bad pattern value %d\n", Pattern); exit(0);
    }

    /* some things may have changed */
    genhead.filesize = filesize;

    if (asked_last) {
	   printf("\nSummary:\n");
	   printf("Filename: %s\n", filename);
	   printf("Name: %s\n", genhead.name);
	   printf("Comment: \n%s\n", genhead.comment);
	   printf("Pattern: %s\n", stylename);
	   printf("File size: %u bytes\n", genhead.filesize);
	   printf("References are %s\n", genhead.writes ? "Writes" : "Reads");
	   printf("Chunks are %u bytes\n", genhead.chunksize);
	   printf("Chunks in pattern: %u\n", chunks);
	   printf("Number of processors: %d\n", genhead.nprocs);
	   printf("Computation: %u msec, %s\n", genhead.computation,
			genhead.computation_fixed ? "Fixed" : "Average");
	   printf("\n");
    }

    /* initialize the file */
    CreateFile(filename, &genhead);

    /* write the patterns to the file */
    WritePattern();

    printf("File %s complete\n", filename);
}

/* @SUBTITLE "ReadInfo: Read in the various pattern parameters" */

static boolean
ReadInfo(argc, argv, head)
	int argc;				/* points to one before first arg */
	char **argv;
	P_HEAD *head;
{
    char line[11];
    boolean asked_last;

    if (--argc > 0) {
	   strncpy(filename, *++argv, P_FILENAMELEN);
	   printf("Filename is '%s'\n", filename);
    } else {
	   printf("Name of pattern FILE (%d chars): ", P_FILENAMELEN);
	   ReadString(filename, "Filename", P_FILENAMELEN);
    }	   

    if (*filename == '\0') {
	   printf("Must specify filename\n");
	   exit(1);
    }

    if (access(filename, 0) == 0) {
	   printf("File '%s' exists. Overwrite [y]? ", filename);
	   ReadString(line, "yes or no", 10);
	   switch (*line) {
		  case 'y':
		  case 'Y':
		  case '\0': {
			 break;
		  }
		  case 'n':
		  case 'N':
		  default: {
			 printf("File not touched.\n");
			 exit(0);
		  }
	   }
    }

    if (--argc > 0) {
	   strncpy(head->name, *++argv, P_NAMELEN);
	   printf("Pattern name is '%s'\n", head->name);
    } else {
	   printf("Name of pattern (%d chars): ", P_NAMELEN);
	   ReadString(head->name, "Name", P_NAMELEN);
    }

    if (--argc > 0) {
	   strncpy(head->comment, *++argv, P_COMLEN);
	   printf("Comment is '%s'\n", head->comment);
    } else {
	   printf("Comment (%d chars): ", P_COMLEN);
	   ReadString(head->comment, "Comment", P_COMLEN);
    }

    if (--argc > 0) {
	   strncpy(stylename, *++argv, STYLE_LEN);
	   printf("Style name is '%s'\n", stylename);
    } else {
	   printf("Pattern Style name: ");
	   ReadString(stylename, "Style", STYLE_LEN);
    }

    if (strcmp(stylename, "rnd") == 0) Pattern = PAT_RND;
    else if (strcmp(stylename, "lpr") == 0) Pattern = PAT_LPR;
    else if (strcmp(stylename, "lps") == 0) Pattern = PAT_LPS;
    else if (strcmp(stylename, "gpr") == 0) Pattern = PAT_GPR;
    else if (strcmp(stylename, "gps") == 0) Pattern = PAT_GPS;
    else if (strcmp(stylename, "gw") == 0) Pattern = PAT_GW;
    else if (strcmp(stylename, "lw") == 0) Pattern = PAT_LW;
    else if (strcmp(stylename, "lw1") == 0) Pattern = PAT_LW1;
    else if (strcmp(stylename, "seg") == 0) Pattern = PAT_SEG;
    else if (strcmp(stylename, "grnd") == 0) Pattern = PAT_GRND;
    else {
	   printf("Bad pattern name '%s'\n", stylename);
	   exit(1);
    }
    head->style = Pattern;

    /* local patterns: number of procs */
    if (Pattern == PAT_LPR || Pattern == PAT_LPS 
	   || Pattern == PAT_LW || Pattern == PAT_LW1
	   || Pattern == PAT_SEG || Pattern == PAT_RND) {
	   head->global = FALSE;
	   if (--argc > 0) {
		  nprocs = atoi(*++argv);
		  printf("Number of procs is %d\n", nprocs);
	   } else {
		  printf("Number of processes: ");
		  ReadNumber("Number of processes", &nprocs);
	   }
    } else {
	   /* global patterns */
	   head->global = TRUE;
	   nprocs = 1;
    }
    head->nprocs = nprocs;

    if (   Pattern == PAT_LPR || Pattern == PAT_LPS
	   || Pattern == PAT_GPR || Pattern == PAT_GPS) {
	   if (--argc > 0) {
		  PortionLength = atoi(*++argv);
		  printf("Portion length is %d chunks\n", PortionLength);
	   } else {
		  printf("Portion Length (chunks): ");
		  ReadNumber("Portion Length", &PortionLength);
	   }
    }

    
    if (Pattern == PAT_LPS || Pattern == PAT_GPS) {
	   if (--argc > 0) {
		  PortionSkip = atoi(*++argv);
		  printf("Portion skip is %d chunks\n", PortionSkip);
	   } else {
		  if (Pattern == PAT_LPS)
		    printf("\n\
* Note that PortionSkip for LPS should be P*x blocks,\n\
* where P=nprocs and\n\
* x is the smallest integer that is both\n\
*   greater than or equal to PortionLength, and\n\
*   relatively prime to the number of disks in use.\n\n");
		  printf("Portion Skip (chunks): ");
		  ReadNumber("Portion Skip", &PortionSkip);
	   }
    }

    if (Pattern == PAT_RND || Pattern == PAT_GRND 
	   || Pattern == PAT_LPR || Pattern == PAT_GPR)
	 if (--argc > 0) {
		filesize = atoi(*++argv);
		printf("File size is %u bytes\n", filesize);
	 } else {
		printf("Size of file in bytes: ");
		ReadNumber("File Size", &filesize);
	 }
    else
	 printf("Filesize will be computed...\n");
    head->filesize = filesize;

    if (--argc > 0) {
	   chunks = atoi(*++argv);
	   printf("Number of chunks to read is %u\n", chunks);
    } else {
	   printf("Number of chunks to read: ");
	   ReadNumber("Number of Chunks", &chunks);
    }

    if (--argc > 0) {
	   strncpy(line, *++argv, 10);
    } else {
	   printf("Do reads(r), or writes(w)? ");
	   ReadString(line, 10, "Read or Write");
    }
    switch (*line) {
	   case 'r':
	   case 'R': {
		  writes = FALSE;
		  break;
	   }
	   case 'w':
	   case 'W': {
		  writes = TRUE;
		  break;
	   }
	   default: {
		  printf("Bad read/write indication '%c'\n", *line);
		  exit(1);
	   }
    }
    head->writes = writes;

    if (--argc > 0) {
	   chunksize = atoi(*++argv);
	   printf("Chunk size is %u bytes\n", chunksize);
    } else {
	   printf("Size of chunk in bytes: ");
	   ReadNumber("Chunksize", &chunksize);
    }
    head->chunksize = chunksize;

    asked_last = FALSE;
    if (--argc > 0) {
	   computation = atoi(*++argv);
	   printf("Computation is %u msec per chunk\n", computation);
    } else {
	   printf("Amount of computation per chunk: ");
	   ReadNumber("Computation per Chunk, in msec", &computation);
	   asked_last = TRUE;
    }
    head->computation = computation;

    if (computation > 0) {
	   if (--argc > 0) {
		  strncpy(line, *++argv, 10);
	   } else {
		  printf("Is %u msec the average(a), or fixed amount(f)? ", 
			    computation);
		  ReadString(line, 10, "Computation type");
		  asked_last = TRUE;
	   }
	   switch (*line) {
		  case 'a':
		  case 'A': {
			 computation_fixed = FALSE;
			 break;
		  }
		  case 'f':
		  case 'F': {
			 computation_fixed = TRUE;
			 break;
		    }
		  default: {
			 printf("Bad computation type\n");
			 exit(1);
		  }
	   }
    } else
	 computation_fixed = TRUE;
    head->computation_fixed = computation_fixed;

    return(asked_last);
}

/* @SUBTITLE "ShareInfo: Share the pattern parameters" */
static void
ShareInfo()
{
    Share(&PortionSkip);
    Share(&PortionLength);
    Share(&filesize);
    Share(&nprocs);
    Share(&chunks);
    Share(&chunksize);
    Share(&writes);
    Share(&computation);
    Share(&computation_fixed);
}

