/* @TITLE "replace-toss.c - replacement algorithm - toss immediately" */
/* replace-toss.c:	Support for frame replacement algorithm. 
 *             TOSS IMMEDIATELY algorithm. 
 *			Includes the following: 
 * 
 *			RT_InitReplaceInode_toss() 
 *			RT_InitReplaceRPD_toss() 
 * and the local functions, called through pointers:
 *			FindFrame() 
 *			DoneReplaceInode() 
 *			DoneReplaceRPD() 
 *			NotifyReplaceA() 
 *			NotifyReplaceB() 
 *			Replaceable() 
 * 
 * The protocols used here are carefully designed to work together to 
 * provide exactly the right mutual exclusion properties and are very 
 * heavily based on the semantics of the problem. Be very careful 
 * if modifying any of these routines. 
 */

static char rcsid[] = "$Id: replace-toss.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

/* DEFINES */

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "replace.h"

/* From globalws.c */
extern void AddToGlobalWS();
extern void RemoveFromGlobalWS();
extern boolean FindFrame_ws();
extern FFDATA *InitGWS();
extern void ReplaceableGWS();
extern void DoneGWS();

struct tossdata {			/* for toss-immediately */
    int last;				/* last sector used (-1 if none) */
    FFDATA *ff;			/* findframe data */
};

static boolean FindFrame();
static void DoneReplaceInode();
static void DoneReplaceRPD();
static boolean NotifyReplaceA();
static void NotifyReplaceB();
static void Replaceable();
static void Dump();

/* @SUBTITLE "RT_InitReplaceInode_toss: initialize inode" */
void
RT_InitReplaceInode_toss(inode_ptr)
	RAPID_INODE *inode_ptr;
{
    /* make data structures needed by findframe */
    inode_ptr->replace.private = (ANYPTR) InitGWS(inode_ptr->num_frames);

    inode_ptr->replace.doneinode = DoneReplaceInode;
}

/* @SUBTITLE "RT_InitReplaceRPD_toss: initialize descriptor" */
/* ARGSUSED */
void
RT_InitReplaceRPD_toss(rpd, wss)
	RAPIDFILE *rpd;
	int wss;				/* unused */
{
    struct tossdata *mydata;

    mydata = (struct tossdata *)AllocLocal(sizeof(struct tossdata));
    mydata->last = -1;
    mydata->ff = (FFDATA *)(rpd->inode_ptr->replace.private);

    rpd->replace.private = (ANYPTR) mydata;

    rpd->replace.donerpd = DoneReplaceRPD;
    rpd->replace.notifyA = NotifyReplaceA;
    rpd->replace.notifyB = NotifyReplaceB;
    rpd->replace.findframe = FindFrame;
    rpd->replace.replaceable = Replaceable;
    rpd->replace.dump = Dump;
}

/* @SUBTITLE "FindFrame: find a frame for a sector" */
static boolean
FindFrame(rpd, sector, sme, prefetching)
	RAPIDFILE *rpd;
	int sector;			/* the sector we want */
	SECTOR_MAP_ENTRY *sme;	/* the entry for this sector */
	boolean prefetching;	/* are we doing this for prefetch */
{
    struct tossdata *mydata = (struct tossdata *) (rpd->replace.private);
    
    return(FindFrame_ws(rpd, sector, sme, mydata->ff, prefetching));
}

/* @SUBTITLE "Dump: dump debugging output for private data" */
static void
Dump(rpd)
	RAPIDFILE *rpd;
{
    struct tossdata *mydata = (struct tossdata *) (rpd->replace.private);

    printf("  Toss-immediate replacement private data:\n");
    if (mydata != NULL) {
	   printf("   last = %d\n", mydata->last);
	   DumpGWS(mydata->ff);
    } else 
	 printf("   Null tossdata\n");
}

/* @SUBTITLE "DoneReplaceInode: shut down replacement algorithm" */
static void
DoneReplaceInode(inode_ptr)
    RAPID_INODE *inode_ptr;
{
    DoneGWS(inode_ptr->replace.private);	/* findframe info */
    inode_ptr->replace.private = NULL;
}

/* @SUBTITLE "DoneReplaceRPD: shut down replacement algorithm" */
static void
DoneReplaceRPD(rpd)
	RAPIDFILE *rpd;
{
    struct tossdata *mydata = (struct tossdata *) (rpd->replace.private);

    if (mydata->last >= 0)
	 RemoveFromGlobalWS(rpd, mydata->last, mydata->ff);
    UsFree(mydata);
    rpd->replace.private = NULL;
}

/* @SUBTITLE "NotifyReplaceA: first part of replacement recording" */
static boolean
NotifyReplaceA(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct tossdata *mydata = (struct tossdata *) (rpd->replace.private);

    if (mydata->last != sector) {
	   if (mydata->last >= 0)
		RemoveFromGlobalWS(rpd, mydata->last, mydata->ff);
	   
	   mydata->last = sector;
	   
	   return(TRUE);
    } else 
	 /* rereference makes no change */
	 return(FALSE);
}

/* @SUBTITLE "NotifyReplaceB: second part of replacement recording" */
static void
NotifyReplaceB(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    AddToGlobalWS(rpd, sector);
}

/* @SUBTITLE "Replaceable: the given sector is now replaceable" */
/* as long as no one is using it */

static void
Replaceable(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct tossdata *mydata = (struct tossdata *)(rpd->replace.private);

    ReplaceableGWS(rpd, sector, mydata->ff);
}

