/* @TITLE "predict-RGAPS.c -- Random/Global-Access-Pattern" */
/* 
 * RGAPS: Random Global-Access-Pattern Sequentiality.
 * 
 * This is a variant of GAPS 1.20, with a different watch mode. 
 * Changes relevant to both that were made since the split were made to 
 * both. 
 * 
 * Define USEPARM to use the prefetch algorithm parameter for maxdist, 
 * otherwise maxjump is used.
 *
 * David Kotz April-May 1990
 */

static char rcsid[] = "$Id: predict-rgaps.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $";

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "stats.h"
#include "rapidelog.h"
#include "predict-debug.h"
#include "fifolock.h"
#include "wanted.h"
#include "own.h"
#include "queue.h"

/* @PAGE */

/* #define USEPARM */ /* define to use prefetch parm for maxdist */

/* #define GAPTRACE */

#ifdef GAPTRACE
# define TraceFlush() fflush(stdout)
#else
# define TraceFlush()
#endif

/* Elogs specific to GAPS/RGAPS */
#define GAPSELOG_NOTIFY (RTELOG_PFA+0) /* block */
#define GAPSELOG_START  (RTELOG_PFA+1) /* block */
#define GAPSELOG_END    (RTELOG_PFA+2) /* reason: 0=maxjump, 1=jumpback, 2=compfail*/
#define GAPSELOG_RANDOM (RTELOG_PFA+3) /* Random() time (tics) */
#define GAPSELOG_CONTIN (RTELOG_PFA+4) /* Contin() time (tics) */
#define GAPSELOG_TOSEQ  (RTELOG_PFA+7) /* ToSequential() time (tics) */


#define CONTEXTPERPROC 10	/* remember this many references per proc */
#define MINSAMPLE 4			/* must have a sample of at least this */
#define MINLEN 2			/* must have a group of at least this */
#define MAXSLOPE 3.0		/* to put a limit on maxjump */
#define RESTARTCOUNT 3		/* 3*nprocs is long enough to believe */

static int RunRep = 2;		/* number of runs in a row to assume regular */
static int SkipRep = 2;		/* number of skips in a row to assume regular */
static int ContextSize = 0;	/* total context size */

/* @PAGE */
/* These structures contain the entire state of the GAPS predictor. */
static struct rgapsdata {
    int parm;				/* parameter from command line */
    int nprocs;			/* == NPROCS */

    ALOCK contin_lock;		/* Continuation mode entry lock */
    short in_random;		/* count of procs in Random() */

    boolean contin;			/* is any process in Continuation mode */
    int **last;			/* [NPROCS] pointer to mylast for all procs  */
    boolean **sequential;	/* [NPROCS] pointer to mysequential for all */
    short num_sequential;	/* number of nodes thinking sequential */

    short num_ordered;		/* number of nodes that have order property */

    int maxjump;			/* maximum allowed jump past maxlast */

    ALOCK minlast_lock;		/* lock for minlast */
    FIFOLOCK *maxlast_fifo;	/* fifo lock for maxlast */
    int start;				/* start of portion */
    int minlast;			/* min over last[i] */
    int maxlast;			/* max over last[i] */
    boolean jumpgap;		/* TRUE when in JumpGap */

    boolean portion1;		/* true if first portion */
    int same_runlen;		/* number of runs with same len */
    int prev_runlen;		/* length of previous run */
    int same_skiplen;		/* number of skips with same len */
    int prev_skiplen;		/* length of previous skip */
    int prev_end;			/* end of previous portion */
    int exp_end;			/* expected end of next portion */

    int runlen;			/* run length if regular, else 0 */
    int skiplen;			/* skip length if regular, else 0 */

    FIFOLOCK *prefetch_fifo;	/* fifo lock for prefetch operations */
    FIFOLOCK *pchange_fifo;	/* fifo lock to change prefetch parameters */
    QH GiveBackQ;			/* queue for work given back */
    int nextpref;			/* last block prefetched or read */
    int startskip;			/* last block before a skip */
    int endskip;			/* first block after a skip */
    boolean prefetch_skip;	/* is there a skip? */
    int ignore;			/* ignore missing blocks below this point */
    int size;				/* size of the file in blocks */
    int limit;				/* current limit on prefetching */
    int hardlimit;			/* hard upper limit on prefetching */
    int maxdist;			/* max distance past maxlast that we prefetch */
};

/* @PAGE */
/* The local rgaps data structure, to replicate and localize constants */
struct rgapslocal {
    struct rgapsdata *common;	/* pointer to global common data above */

    int nprocs;			/* == NPROCS */

    int mylast;			/* last reference on this proc */
    int **last;			/* [NPROCS] pointer to mylast for all procs */
    boolean mysequential;	/* is this processor in sequential mode */
    boolean **sequential;	/* [NPROCS] pointer to mysequential for all */

    boolean ordered;		/* do we think we are ordered? */
    int ordercount;			/* number of blocks in order */
    int shortorder;			/* one of the below */
#define SHORTORDERA 4		/* number in order to consider ordered */
#define SHORTORDERB 3		/* " for large number of procs */

    ALOCK *minlast_lock;		/* pointer to lock for minlast */
    FIFOLOCK *maxlast_fifo;	/* fifo lock for maxlast */

    FIFOLOCK *prefetch_fifo;	/* fifo lock for prefetch */
    FIFOLOCK *pchange_fifo;	/* fifo lock for prefetch */
    QH GiveBackQ;			/* queue for work given back */

    int size;				/* size of the file in blocks */
};

/* @PAGE */
/* MACROS */
#define sqr(x) ((x)*(x))

/* This processes one mistake, as determined by other procedures. */
/* Run under minlast lock, or alone, from Finalize. */
#define OneMistake(rpd, block, sme) \
  if (SS_CLEAR((sme), SS_MARKED) & SS_MARKED) /* if was set, now clear */\
      RT_PrefetchMistake((rpd), (block))

/* FUNCTIONS - Entry points */
void InitPredict_RGAPS();
static void LateNotify();
static boolean Notify();
static void Done();
static void Dump();
static boolean CheckAvailable();
static boolean GetWork();
static void GiveBackWork();

/* FUNCTIONS - Internal Use Only */
static boolean Random();
static boolean Contin();
static void ToSequential();
static void FromSequential();
static void Finalize();
static void NewPortion();
static boolean ExtendPortion();
static void JumpGap();
static void EndPortion();

static void StartPrefetch();
static void StopPrefetch();
static void SoftLimitPrefetch();
static void LimitPrefetch();
static void Mistakes();
static int NextBlock();

extern double fabs();
extern double ceil();

/* @SUBTITLE "InitPredict_RGAPS - initialize data structures" */
/* This is an Entry Point */
/* This function allocates and initializes the GAPS data structures. Only 
 * one process will do most of the work (process 0). The others wait
 * until the gaps pointer is ready, and then do per-process initialization.
 */
void
InitPredict_RGAPS(rpd, parm)
	RAPIDFILE *rpd;
	int parm;				/* currently ignored */
{
    int nprocs = ProcsInUse();
    struct rgapsdata *rgaps;
    struct rgapslocal *lrgaps;

#if defined(WANTED) || defined(OWN_BUFFER)
    fprintf(stderr,
		  "\nERROR: RGAPS predictor used with incompatible options!\n");
    return;
#endif

    /* Everyone allocates the local gaps structure */
    lrgaps = (struct rgapslocal *)AllocLocal(sizeof(struct rgapslocal));
    /* and a place for local copies of arrays */
    lrgaps->last = (int **)
	 AllocLocal(nprocs * sizeof(int *));
    lrgaps->sequential = (boolean **)
	 AllocLocal(nprocs * sizeof(boolean *));

    /* and record the pointer for future access */
    rpd->prefetch.private = (ANYPTR)lrgaps;
    rpd->prefetch.available = FALSE;

    /* Set up function pointers */
    rpd->prefetch.donerpd = Done;
    rpd->prefetch.notify = Notify;
    rpd->prefetch.latenotify = LateNotify;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.getwork = GetWork;
    rpd->prefetch.giveback = GiveBackWork;
    rpd->prefetch.dump = Dump;

    ContextSize = CONTEXTPERPROC * nprocs;

    /* Only one process does the global initialization */
    if (Vnode == 0) {
	   rgaps = (struct rgapsdata *)AllocGlobal(sizeof(struct rgapsdata));

	   rgaps->parm = parm;
	   rgaps->nprocs = nprocs;
	   rgaps->contin = FALSE;

	   /* the values will be installed by the processors */
	   rgaps->last = (int **)
		AllocGlobal(nprocs * sizeof(int *));
	   rgaps->sequential = (boolean **)
		AllocGlobal(nprocs * sizeof(boolean *));

	   rgaps->in_random = 0;
	   rgaps->contin_lock = 0;

	   rgaps->num_sequential = nprocs; /* this changes to 0 below */
	   rgaps->num_ordered = 0; /* we start out unordered */

	   rgaps->maxjump = 0;

	   rgaps->minlast_lock = 0;
	   AllocFIFO(rgaps->maxlast_fifo);
	   rgaps->start = -1;
	   rgaps->minlast = -1;
	   rgaps->maxlast = 0;
	   rgaps->jumpgap = FALSE;

	   rgaps->portion1 = TRUE;
	   rgaps->same_runlen = 0;  
	   rgaps->prev_runlen = 0;  
	   rgaps->same_skiplen = 0; 
	   rgaps->prev_skiplen = 0; 
	   rgaps->prev_end = 0;     
	   rgaps->exp_end = 0;

	   rgaps->runlen = 0;
	   rgaps->skiplen = 0;

	   AllocFIFO(rgaps->prefetch_fifo);
	   AllocFIFO(rgaps->pchange_fifo);
	   rgaps->GiveBackQ = Make_queue(ContextSize); /* best size? */
	   rgaps->nextpref = 0;
	   rgaps->startskip = 0;
	   rgaps->endskip = 0;
	   rgaps->prefetch_skip = FALSE;
	   rgaps->size = RT_NumSectors(rpd, rpd->size);
	   rgaps->hardlimit = 0;
	   rgaps->limit = 0;
	   rgaps->ignore = 0;
	   rgaps->maxdist = 0;

	   rpd->inode_ptr->prefetch.available = FALSE; /* no prediction yet */

	   /* this will release the other processes from a barrier below */
	   rpd->inode_ptr->prefetch.private = (ANYPTR) rgaps;
    } else {
	   /* all procs except 0 */
	   while (rpd->inode_ptr->prefetch.private == NULL)
		UsWait(50);
    }

    /* @PAGE */
    /* Copy the constant stuff from global to local */
    rgaps = (struct rgapsdata *)(rpd->inode_ptr->prefetch.private);
    lrgaps->common = rgaps;

    lrgaps->nprocs = nprocs;
    lrgaps->mylast = -1;
    rgaps->last[Vnode] = &(lrgaps->mylast);
    lrgaps->mysequential = FALSE;
    rgaps->sequential[Vnode] = &(lrgaps->mysequential);
    lrgaps->ordered = FALSE;
    lrgaps->ordercount = 0;
    lrgaps->shortorder = (nprocs < 10 ? SHORTORDERA : SHORTORDERB);
    lrgaps->minlast_lock = &(rgaps->minlast_lock);
    lrgaps->maxlast_fifo = rgaps->maxlast_fifo;
    lrgaps->prefetch_fifo = rgaps->prefetch_fifo;
    lrgaps->pchange_fifo = rgaps->pchange_fifo;
    lrgaps->GiveBackQ = rgaps->GiveBackQ;
    lrgaps->size = rgaps->size;

    /* Indicate that we are done, and wait for the others */
    Atomic_add(&(rgaps->num_sequential), -1);
    while(rgaps->num_sequential > 0)
	 UsWait(10);
    /* now num_sequential is properly initialized to 0 */

    /* Now copy the arrays of pointers */
    btransfer(rgaps->last, lrgaps->last, nprocs * sizeof(int *));
    btransfer(rgaps->sequential, lrgaps->sequential, nprocs * sizeof(boolean *));

}

/* @SUBTITLE "LateNotify: Multiple notifications" */
static void
LateNotify(node, rpd, private, last, length, sector)
	int node;				/* processor number */
	RAPIDFILE *rpd;		/* may not be node's rpd */
	ANYPTR private;		/* private data */
	int last;				/* last block accessed in run */
	int length;			/* length of consecutive run */
	int sector;			/* most recent block accessed (-1 if none) */
{
    struct rgapslocal *lrgaps = (struct rgapslocal *)private;
    struct rgapsdata *rgaps = lrgaps->common;
    int orderlength = length;	/* could be zero */

#ifdef GAPTRACE
    printf("Proc %d (%d) in RGAPS LateNotify: last=%d, length=%d, sector=%d\n",
		 Vnode, node, last, length, sector);
    TraceFlush();
#endif /* GAPTRACE */

    if (sector >= 0)
	 if (length > 0 && sector > last) {
		orderlength = length+1;
	 } else {
		orderlength = 1;
	 }

    lrgaps->ordercount = orderlength;
    if (orderlength >= lrgaps->shortorder) {
	   /* count this proc as ordered */
	   Atomic_add(&(rgaps->num_ordered), 1);
	   lrgaps->ordered = TRUE;
    }

    lrgaps->mylast = last;
}

/* @SUBTITLE "Notify - update for a new reference" */
/* This is an Entry Point */
/* This is called with each sector that is accessed. Currently, this function 
 * is protected with a FIFO entry lock so that only one process will be 
 * in the code at a time. The whole predictor is thus serialized. 
 * The return value has to be a boolean for consistency with others,
 * but the return value is irrelevant unless WANTED is defined. This 
 * is not allowed (see above), so our return value never means anything.
 *  If this process is in sequential mode, Contin() is called. Otherwise
 * Watch() is called. Rereferences are ignored.
 */
static boolean				/* irrelevant */
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct rgapslocal *lrgaps = (struct rgapslocal *)(rpd->prefetch.private);
    struct rgapsdata *rgaps = lrgaps->common;
    int node = Vnode;
    TICS start = rtc;

    if (sector != lrgaps->mylast) { /* ignore rereferences by same node */
	   if (rgaps->contin && lrgaps->mysequential) {
		  /* check sequentiality */
		  ELOG_LOG(GAPSELOG_NOTIFY, sector);
		  if (Contin(rpd, lrgaps, (int)sector, node))
		    my_stats->correct++;
		  else
		    FromSequential(rpd, lrgaps, (int)sector, node);
		  ELOG_LOG(GAPSELOG_CONTIN, rtc-start); /* Contin() time */
		  TraceFlush();
	   } else {
		  ELOG_LOG(GAPSELOG_NOTIFY, sector);
		  if (!Random(rpd, lrgaps, (int)sector, node)) {
			 /* slipped into Continuation mode */
			 if (Contin(rpd, lrgaps, (int)sector, node))
			   my_stats->correct++;
			 else
			   FromSequential(rpd, lrgaps, (int)sector, node);
		    ELOG_LOG(GAPSELOG_CONTIN, rtc-start); /* Contin() time */
		  } else
		    ELOG_LOG(GAPSELOG_RANDOM, rtc-start); /* Random() time */

		  TraceFlush();
	   }
    }

    /* update the local copy of prefetch.available */
    rpd->prefetch.available = rpd->inode_ptr->prefetch.available;

    return(FALSE);			/* for WANTED, but is inconsistent */
}

/* @SUBTITLE "Random: handle random accesses" */
/* The purpose here is speedy processing of random-access patterns, 
 * with a crude-but-fast measure to detect a switch back from randomness
 * to a possibly sequential pattern. We look at orderedness: a sequential
 * pattern must have all processors doing ordered accesses; that is, 
 * each block number referenced is greater than or equal to the 
 * previous one. Jumpbacks occur when it is less. We come to random mode
 * because several processes experienced jumpbacks after very short 
 * ordered runs; this is a sign of randomness. We only go back to 
 * watching mode from here when a high percentage of processors 
 * have shown ordering. In a random pattern, it is unlikely that many
 * processors will simultaneously have longs strings of ordered references.
 */
static boolean				/* FALSE if we didn't handle it here */
Random(rpd, lrgaps, block, node)
	RAPIDFILE *rpd;
	struct rgapslocal *lrgaps;
	int block;
	int node;
{
    struct rgapsdata *rgaps = lrgaps->common;
    boolean ok;
    TICS start;

    /* rereferences have already been ignored */

    /* We must watch out for the lock that is used to shift into
	* continuation mode. Thus, we implement a shared/exclusive lock.
	* Here we obtain a shared lock; below, someone may be trying to 
	* upgrade to an exclusive lock.
	*/
    ok = FALSE;
    while (!ok) {
	   if (rgaps->contin_lock) {
		  Atomic_add(&(rgaps->in_random), 1);
		  if (rgaps->contin_lock)
		    Atomic_add(&(rgaps->in_random), -1);
		  else
		    ok = TRUE;
	   } else
		ok = TRUE;
    }

    /* Entered continuation mode? Exit and call Contin */
    if (rgaps->contin && lrgaps->mysequential) {
	   Atomic_add(&(rgaps->in_random), -1);
	   return(FALSE);
    }

    /* Process this request in Random mode */
    if (lrgaps->ordered) {
	   /* we were ordered */
	   if (block < lrgaps->mylast) {
		  /* jumpback! may not be ordered any more */
		  /* This code seems backward, but it is the original intent. See 
		   * research notes for 4/4/90. 
		   */
		  if (lrgaps->ordercount < lrgaps->shortorder) {
			 Atomic_add(&(rgaps->num_ordered), -1);
			 lrgaps->ordered = FALSE;
		  }				/* else maybe it was just a long portion */
		  lrgaps->ordercount = 1; /* this block */
	   } else {
		  /* in order; increment count */
		  lrgaps->ordercount++;
	   }
    } else {
	   /* we were not ordered */
	   if (block < lrgaps->mylast) {
		  /* jumpback! but doesn't change our status */
		  lrgaps->ordercount = 1; /* this block */
	   } else {
		  /* in order. Check count */
		  if (++(lrgaps->ordercount) >= lrgaps->shortorder) {
			 /* becoming ordered */
			 Atomic_add(&(rgaps->num_ordered), 1);
			 lrgaps->ordered = TRUE;
		  }
	   }
    }

    lrgaps->mylast = block;

#ifdef GAPTRACE
    printf("%2d %4d %d %c %d r\n", node, block, lrgaps->ordercount, 
		 lrgaps->ordered ? 'y' : 'n', rgaps->num_ordered);
#endif

    /* Now check to see if we should enter Continuation mode.
	* Hence, we either release the shared lock, or upgrade to an
	* exclusive lock, then release it. 
	*/
    if (!rgaps->contin && rgaps->num_ordered == lrgaps->nprocs) {
	   if (TryLock(&(rgaps->contin_lock))) {
		  Atomic_add(&(rgaps->in_random), -1); /* count us out */
		  if (!rgaps->contin && rgaps->num_ordered >= lrgaps->nprocs) {
			 while(rgaps->in_random > 0)	/* wait for all others to leave */
			   UsWait(5);
			 /* Now we have an exclusive lock */
			 start = rtc;
			 ToSequential(rpd, lrgaps);
			 ELOG_LOG(GAPSELOG_TOSEQ, rtc-start); /* Contin() time */
#ifdef GAPTRACE
			 printf("-> leaving random mode\n");
#endif GAPTRACE
		  }
		  /* release the exclusive lock */
		  UNLOCK(&(rgaps->contin_lock));
	   } else
		/* Someone else got the lock, and is doing what we would do. */
		/* release the shared lock */
		Atomic_add(&(rgaps->in_random), -1); /* count us out */
    } else
	 /* release the shared lock */
	 Atomic_add(&(rgaps->in_random), -1); /* count us out */
    
    return(TRUE);
}


/* @SUBTITLE "ToSequential: change into sequential mode" */
/* When the Watcher notices sequentiality, this function is called to 
 * send all processors into sequential mode. This also involves
 * breaking down the old data structure (the queue) and building the
 * new one (in the sector map table, conveniently). 
 */
static void
ToSequential(rpd, lrgaps)
	RAPIDFILE *rpd;
	struct rgapslocal *lrgaps;
{
    struct rgapsdata *rgaps = lrgaps->common;
    int i;
    int block;
    int start;
    int minlast;
    int maxlast;
    int last;
    SECTOR_MAP_ENTRY *sme;
    int nprocs = lrgaps->nprocs;

    /* Set the last counters in the table. The last count indicates
	* how many processes have that block as their most-recently-referenced
	* block. The count will be decremented as the processes move on.
	*/
    for (i = 0; i < nprocs; i++) {
	   last = *lrgaps->last[i];
	   sme = RT_smeForSector(rpd, last);
	   sme->misc.lastcount++;
	   SS_SET(sme, SS_USED);
	   if (last > maxlast || i == 0)
		maxlast = last;
	   if (last < minlast || i == 0)
		minlast = last;

	   *lrgaps->sequential[i] = TRUE;
    }
    rgaps->maxlast = maxlast;
    rgaps->minlast = minlast;
    rgaps->start = minlast;
    rgaps->maxjump = nprocs;
    rgaps->ignore = maxlast; /* ignore missing blocks in initial active zone */

    rgaps->num_sequential = nprocs;

    ELOG_LOG(GAPSELOG_START, rgaps->start);

    /* Record the new portion beginning now. */
    NewPortion(lrgaps, rgaps->start);

    /* And turn on prefetching, beginning after maxlast. We used to start 
	* at minlast, and prefetch in the active zone; however, when just
	* starting, there may be holes in the active zone that were really 
	* used; this avoids marking mistakes for blocks that were already used. 
	* Later, once the active zone extends past the current maxlast, we 
	* will be prefetching in the active zone.
	*/
    StartPrefetch(rpd, rgaps, maxlast);

    /* let people into continuation mode */
    rgaps->contin = TRUE;	
}


/* @SUBTITLE "Contin: watch the pattern as it continues to be sequential" */
/*   This is Continuation mode. Once sequentiality is noted, we shift modes and
 * track the sequentiality in a much more efficient manner. Thus, Notify
 * calls this function instead of Watch(). We return FALSE here if we 
 * fall out of sequentiality.
 *   This function, and its children, are executed concurrently
 * by many processors.
 *   The efficiency is possible since we know the pattern will be active
 * in a compact area, and we don't need the ordering information from the
 * queue so much as easy access to minlast/maxlast and completeness
 * information.
 *   The sector map table is used to store our information. We record
 * the lastcount, a count for each block of the number of processes that
 * have that block as their most-recently accessed, and a "used" bit for each 
 * block.
 *   We watch for jumpbacks, sure signs of sequentiality breaking, and 
 * forward jumps that are too far to be explained.  [Here we have hardwired 
 * maxjump to be equal to nprocs, somewhat of a hack.] Forward jumps longer
 * than maxjump past maxlast are also taken to be a loss in sequentiality
 * (this is modified when the portions are highly regular, and we can take
 * jumps over portion skip in stride; see ExtendPortion). 
 *   We must also maintain minlast and maxlast in order to check for
 * completeness failures (which arise due to small jumps and portion skips). 
 * Minlast is updated whenever the lastcount of a block goes to zero, and
 * that block was minlast. To find the new minlast, we sequentially scan
 * the table until another block with a nonzero lastcount is found. In the
 * process, we can look for blocks that were not used. Any found are
 * a completeness failure. We let JumpGap handle this case.
 */
static boolean
Contin(rpd, lrgaps, block, node)
	RAPIDFILE *rpd;
	struct rgapslocal *lrgaps;
	int block;
	int node;
{
    struct rgapsdata *rgaps = lrgaps->common;
    int oldlast;
    SECTOR_MAP_ENTRY *sme;
    boolean ok = TRUE;

    oldlast = lrgaps->mylast;

    /* rereferences have already been ignored */

    lrgaps->mylast = block;
    
    if (block < oldlast) {
#ifdef GAPTRACE
	   printf("-> %d jumpback from %d on node %d\n", 
			block, oldlast, node);
#endif
	   sme = RT_smeForSector(rpd, oldlast);
	   Atomic_add(&(sme->misc.lastcount), -1); /* take us out */
	   ELOG_LOG(GAPSELOG_END, 1);
	   return(FALSE);		/* a jump-back: cancel sequential */
    }

    /* now we know that (block > oldlast) */

    sme = RT_smeForSector(rpd, block);
    SS_SET(sme, SS_USED);	/* block.usedcount++ */
    
    if (block > rgaps->maxlast) {
	   /* update maxlast */
	   LockFIFO(lrgaps->maxlast_fifo, 10);
	   if (block > rgaps->maxlast) {
		  if (ExtendPortion(rgaps, oldlast, block)) {
			 rgaps->maxlast = block;
			 SoftLimitPrefetch(rgaps, block + rgaps->maxdist);
			 UnlockFIFO(lrgaps->maxlast_fifo);
		  } else {
			 UnlockFIFO(lrgaps->maxlast_fifo);
#ifdef GAPTRACE
			 printf("-> %d jump too large %d > %d\n",
				   block, block-rgaps->maxlast, rgaps->maxjump);
#endif
			 sme = RT_smeForSector(rpd, oldlast);
			 Atomic_add(&(sme->misc.lastcount), -1); /* take us out */

			 ELOG_LOG(GAPSELOG_END, 0);
			 return(FALSE);
		  }
	   } else
		UnlockFIFO(lrgaps->maxlast_fifo);
    }

    sme = RT_smeForSector(rpd, block);
    Atomic_add(&(sme->misc.lastcount), 1); /* block.lastcount++ */
    
    /* @PAGE */
    /* Here we try to update minlast. If our old last was the minlast, 
	* and no one else currently has that as their last, then 
	* the minlast must change. We have responsibility for updating it. 
	* There may be several processes at this point trying to do
	* this, so we need a lock. However, we do not wait for the
	* lock; it is sufficient for one process to get the lock
	* and do the job. Note also, that since minlast is incremented
	* as the job is performed, a race condition may lead us to
	* believe that we should update minlast, while another
	* process is in the inner loop. The lock helps to insure
	* that only one of us will do it.
	*  The prefetch code may be watching minlast change too. It avoids
	* prefetching during JumpGap, though it may sneak some through.
	* If they are not appropriate, they will be caught as mistakes.
	*/
    sme = RT_smeForSector(rpd, oldlast);
    Atomic_add(&(sme->misc.lastcount), -1); /* decrement oldlast count */
    if (rgaps->minlast == oldlast && sme->misc.lastcount == 0) {
	   /* update minlast */
	   while(rgaps->minlast == oldlast && ok)
		/* keep trying to update minlast */
		if (TryLock(lrgaps->minlast_lock)) {
		    /* we got a lock on the process; may already be done */
		    if (rgaps->minlast == oldlast)
			 /* nope; we loop to update minlast */
			 while (sme->misc.lastcount == 0 && ok) {
				if (SS_ISCLEAR(sme, SS_USED)) { /* i.usedcount==0 */
				    /* completeness failure */
				    if (rgaps->num_sequential < lrgaps->nprocs)
					 ok = FALSE; /* leave continuation mode */
				    else
					 JumpGap(rpd, lrgaps); /* changes minlast */
				} else {
				    SS_CLEAR(sme, SS_USED | SS_MARKED);
				    rgaps->minlast++; /* ok due to lock */
				}
				sme = RT_smeForSector(rpd, rgaps->minlast);
			 }
		    /* in any case, release lock */
		    UNLOCK(lrgaps->minlast_lock);
		    
		    /* For regular run length, check the expected end of portion */
		    if (rgaps->runlen > 0 && rgaps->minlast > rgaps->exp_end) {
			   /* we overshot the expected end of portion */
			   LockFIFO(lrgaps->maxlast_fifo, 5);
			   rgaps->runlen = 0;	/* cancel the regularity for now */
			   LimitPrefetch(rgaps, lrgaps->size-1, 
						  rgaps->maxlast + rgaps->maxdist);
			   UnlockFIFO(lrgaps->maxlast_fifo);
		    }
		}
    }
    
#ifdef GAPTRACE
    printf("%2d %4d %d %d  y\n",
		 node, block, rgaps->minlast, rgaps->maxlast);
#endif

    if (!ok)
	 ELOG_LOG(GAPSELOG_END, 2);

    return(ok);
}

/* @SUBTITLE "FromSequential: change out of sequential mode" */
/* This function handles any break from sequentiality. It is called for
 * each processor as it leaves the portion, either due to a jumpback or 
 * a too-long jump forward. [Of course, if not all processors leave the 
 * portion in this way, we're stuck.]  First thing we do is to limit the
 * prefetch to maxlast; prefetching to there is safe. Otherwise, 
 * we set our sequentiality flag to FALSE.
 *  Many processes may execute this concurrently.
 */
static void
FromSequential(rpd, lrgaps, block, node)
	RAPIDFILE *rpd;
	struct rgapslocal *lrgaps;
	int block;
	int node;
{
    struct rgapsdata *rgaps = lrgaps->common;
    int n;

    lrgaps->mysequential = FALSE;
#ifdef GAPTRACE
    printf("-> process %d ending sequential\n", node);
#endif

    rgaps->maxdist = 0;		/* to keep prefetching below maxlast */
    LockFIFO(lrgaps->maxlast_fifo, 5);
    SoftLimitPrefetch(rgaps, rgaps->maxlast);/* stop prefetching */
    UnlockFIFO(lrgaps->maxlast_fifo);

    lrgaps->ordered = FALSE;
    lrgaps->ordercount = 1;

    Atomic_add(&(rgaps->num_ordered), -1);

    /* If we are the last to leave, clean up */
    if (Atomic_add(&(rgaps->num_sequential), -1) == 1)
	 Finalize(rpd, rgaps);
}

/* @SUBTITLE "Finalize: Finish off this sequential portion" */
/* This function makes the final transition from Continuation mode to 
 * Watching mode, as the last processor leaves continuation mode.
 * We have to clear the table of all the used bits and last counts,
 * process all mistakes, stop prefetching, and end the portion. 
 * We also reset all state to be ready for Watching mode.
 *
 * Note that only one process is active here.
 */
static void
Finalize(rpd, rgaps)
	RAPIDFILE *rpd;
	struct rgapsdata *rgaps;
{
    int i;
    int end;
    SECTOR_MAP_ENTRY *sme;
    int maxlast = rgaps->maxlast;
    int oldstat;

    /* get that prefetching stopped! */
    StopPrefetch(rpd);

    /* clean out the array */
    end = rgaps->minlast;
    sme = RT_smeForSector(rpd, end);
    while (end <= maxlast && SS_ISSET(sme, SS_USED)) {
	   SS_CLEAR(sme, SS_USED | SS_MARKED); /* usedcount = 0 */
	   sme->misc.lastcount = 0;
	   end++;
	   sme = RT_smeForSector(rpd, end);
    }
    for (i = end; i <= maxlast; i++) {
	   sme = RT_smeForSector(rpd, i);
	   /* clear used bit, and if not used, process mistake */
	   oldstat = SS_CLEAR(sme, SS_USED | SS_MARKED);
	   if (!(oldstat & SS_USED) && (oldstat & SS_MARKED))
		RT_PrefetchMistake(rpd, i); /* it was not used, and was marked */
	   sme->misc.lastcount = 0;
    }
    end--;				/* first loop overshoots by one */

    Mistakes(rpd, rgaps, rgaps->maxlast+1);
    EndPortion(rgaps, end);

    rgaps->contin = FALSE;	/* let Watching mode begin */
}    

/* @SUBTITLE "NewPortion: Starting a new portion" */
/* This is the first of several functions for identifying and
 * tracking portions. We record the skip from the previous portion, 
 * if any, and look for regularity. Also, since this occurs at the 
 * start of sequentiality, we use any regularity in the portion
 * length or skip to set up limits for prefetching.
 *
 * This runs alone, from ToSequential, or alone, during minlast update.
 * Thus it is protected by either the watch_fifo lock or the minlast_lock.
 */
static void
NewPortion(lrgaps, start)
	struct rgapslocal *lrgaps;
	int start;			/* starting block number */
{
    struct rgapsdata *rgaps = lrgaps->common;
    int skip = start - rgaps->prev_end;
    int runlen = 0;			/* default: irregular run length */
    int skiplen = 0;		/* default: irregular skip length */
    int limit = lrgaps->size - 1; /* default hardlimit: EOF */
    int exp_end = 0;		/* expected end of portion */

    /* check skip */
    if (rgaps->same_skiplen == 0 || skip != rgaps->prev_skiplen) {
	   rgaps->same_skiplen = 1; 
	   rgaps->prev_skiplen = skip;
    } else
	 rgaps->same_skiplen++;

    /* now look for regularity */
    if (rgaps->same_runlen >= RunRep) {
	   /* run length is regular */
	   runlen = rgaps->prev_runlen;
	   exp_end = start + runlen - 1;
	   if (rgaps->same_skiplen >= SkipRep) {
		  /* skip length is regular */
		  skiplen = rgaps->prev_skiplen;
#ifdef GAPTRACE
		  printf("-> skip is regular %d\n", skip);
#endif
	   } 
	   /* With regular run length, but no positive regular skip, cut 
	    * off prefetching at end of portion.
	    */
	   if (skiplen <= 0)
		limit = exp_end;
    }

    /* Now compute the value of maxdist.
	*  The prefetch limit is determined from maxdist; maxdist is maxjump for 
	* irregular portions, or essentially infinite for regular portions.
	* It is also set to zero when we are shutting down continuation mode.
	*/
    if (rgaps->num_sequential == lrgaps->nprocs)
	 if (rgaps->portion1 || skiplen > 0)
	   rgaps->maxdist = rgaps->size; /* infinite */
	 else 
#ifdef USEPARM
	   rgaps->maxdist = rgaps->parm; /* limited lookahead */
#else
	   rgaps->maxdist = rgaps->maxjump; /* limited lookahead */
#endif /* USEPARM */
    else
	 rgaps->maxdist = 0;	/* active zone only */

    /* Now store the new values where the prefetchers can see them */
    LockFIFO(lrgaps->pchange_fifo, 5);
    rgaps->start = start;
    rgaps->exp_end = exp_end;
    rgaps->runlen = runlen;
    rgaps->skiplen = skiplen;
    UnlockFIFO(lrgaps->pchange_fifo);

    LockFIFO(lrgaps->maxlast_fifo, 5);
    LimitPrefetch(rgaps, limit, rgaps->maxlast + rgaps->maxdist);
    UnlockFIFO(lrgaps->maxlast_fifo);
}

/* @SUBTITLE "ExtendPortion: Record this sequential portion" */
/*  If the portion length and skip are regular, then we allow
 * jumps over the skip if they still meet the maxjump criteria. 
 * The skip must be positive and regular, and the length must be regular. 
 * The previous reference must be in the current portion, and the
 * current reference in the expected next portion. 
 *  Then, we allow a jump of maxjump plus the skip length.
 *  If the current reference is in the area we expect to skip, the 
 * regularity is assumed broken and we reset the flag. 
 * Otherwise we do a simple maxjump check on the jump.
 *  In irregular portions, we may allow the jump on faith that it will
 * remain sequential. This is risky, but usually beneficial. We record
 * the jump so the prefetches will avoid the jump.
 *  This function runs alone, under the maxlast_fifo lock.
 * Note there may be concurrent changes in skiplen and exp_end,
 * due to NewPortion activity. These should be ok.
 */

static boolean				/* is this a valid extension? */
ExtendPortion(rgaps, last, block)
	struct rgapsdata *rgaps;
	int last;				/* last block ref'd by this proc */
	int block;			/* the most recent reference */
{
    /* read these things once */
    int maxlast = rgaps->maxlast;
    int maxjump = rgaps->maxjump;
    int nprocs = rgaps->nprocs;
    int skiplen = rgaps->skiplen;

    if (skiplen > 0) {
	   /* regular, and positive skip */
	   int exp_end = rgaps->exp_end;
	   if (block > exp_end) {
		  if (last <= exp_end)
		    if (block >= exp_end + skiplen)
			 if (block - maxlast < maxjump + skiplen)
			   return(TRUE);
			 else
			   return(FALSE);
		    else
			 rgaps->skiplen = 0; /* regularity broken */
	   }
	   /* just a plain maxjump check */
	   if (block - maxlast <= maxjump)
		return(TRUE);
	   else
		return(FALSE);
    } else
	 /* Irregular skip length, anyway */
	 /* just a plain maxjump check */
	 if (block - maxlast <= maxjump)
	   return(TRUE);
	 else {
		/* The jump appears to be too large. We try to ride over it. */
		/* We won't ride gap if other processors have given up. */
		/* Test length of portion to determine our faith in sequentiality */
		if (rgaps->num_sequential < nprocs
		    || maxlast - rgaps->start < RESTARTCOUNT * nprocs) {
		    /* The portion was short; don't believe sequentiality */
		    return(FALSE);
		} else {
		    /* Fine, the portion was long enough to believe */
		    /* record this skip for the prefetchers */
		    LockFIFO(rgaps->pchange_fifo, 5);
		    if (rgaps->prefetch_skip)
			 rgaps->endskip = block;
		    else {
			   rgaps->startskip = maxlast;
			   rgaps->endskip = block;
			   rgaps->prefetch_skip = TRUE;
		    }
		    UnlockFIFO(rgaps->pchange_fifo);
#ifdef USEPARM
		    rgaps->maxdist = rgaps->parm;	/* change maxdist */
#else
		    rgaps->maxdist = rgaps->maxjump; /* change maxdist */
#endif /* USEPARM */
		    return(TRUE);	/* maxlast becomes block */
		}
	 }
}

/* @SUBTITLE "JumpGap: jump over an unused gap in the portion" */
/* This function is used to jump over non-used gaps in the table.
 * These gaps are either a completeness failure or the skip
 * between portions, depending on how you look at it. The gap must 
 * define a new portion, so one thing we do is call EndPortion and
 * NewPortion. We also scan the gap looking for mistakes and the
 * beginning of the next portion (the end of the gap).  It is
 * considered a completeness failure only if it was unexpected.
 *
 * This runs alone, under the minlast lock.
 */
static void
JumpGap(rpd, lrgaps)
	RAPIDFILE *rpd;
	struct rgapslocal *lrgaps;
{
    struct rgapsdata *rgaps = lrgaps->common;
    SECTOR_MAP_ENTRY *sme;
    int block = rgaps->minlast;
    int end;
    boolean ignored;

    /* This flag will stop prefetching temporarily */
    rgaps->jumpgap = TRUE;

    /* As we enter, minlast is a non-used block. Thus, the portion
	* ended with the previous block. However, we ignore skips that
	* are in the ignore region, ie < rgaps->ignore. This corresponds
	* to the initial active region. Any skip extending out of the
	* initial active region will be picked up below.
	*/
    end = block - 1;
    if (block >= rgaps->ignore) {
	   EndPortion(rgaps, end);
	   ignored = FALSE;
    } else
	 ignored = TRUE;

    /* it would be nice to find a way to optimize when we expect regularity */
    sme = RT_smeForSector(rpd, block);
    while (SS_ISCLEAR(sme, SS_USED)) {	/* minlast.usedcount==0 */
	   OneMistake(rpd, block, sme);
	   block = ++rgaps->minlast;		/* ok, due to lock */
	   sme = RT_smeForSector(rpd, block);
    }
    /* block (minlast) is now the next used block */

    if (ignored && block >= rgaps->ignore) {
	   EndPortion(rgaps, end);
	   ignored = FALSE;
    }
    if (!ignored)
	 NewPortion(lrgaps, block);

    rgaps->jumpgap = FALSE;

#ifdef GAPTRACE
    if (rgaps->skiplen == 0)
	 printf("-> completeness failure: skipping blocks %d-%d\n", 
		   end, rgaps->minlast-1);
#endif
}


/* @SUBTITLE "EndPortion: Record this sequential portion" */
/* Here we record the end of a portion, remembering its length
 * and comparing the length to previous portions.
 *  This runs alone, called either by Finalize, or under the minlast
 * lock. 
 */
static void
EndPortion(rgaps, end)
	struct rgapsdata *rgaps;
	int end;
{
    int length, skip;

    length = end - rgaps->start + 1;
    skip = rgaps->start - rgaps->prev_end;
    rgaps->prev_end = end;
    rgaps->portion1 = FALSE;

#ifdef GAPTRACE
    printf("-> finished portion: %d to %d length %d skip %d\n", 
		 rgaps->start, end, length, skip);
#endif

    if (rgaps->same_runlen == 0 || length != rgaps->prev_runlen) {
	   rgaps->same_runlen = 1;		/* reset length */
	   rgaps->prev_runlen = length;
	   rgaps->same_skiplen = 0;	/* reset skip */
	   rgaps->prev_skiplen = 0;
    } else {
	   rgaps->same_runlen++;		/* same length */
    }

#ifdef GAPTRACE
    if (rgaps->same_runlen >= RunRep) {
	   printf("-> portion length is regular %d\n", length);
    }
#endif

    /* we look at the skip at the start of the portion */
}

/* @SUBTITLE "Start/Stop/LimitPrefetch: turn on and off" */
/* This turns prefetching on, to begin after "block."  */
/* Always runs alone, from ToSequential */
static void
StartPrefetch(rpd, rgaps, block)
	RAPIDFILE *rpd;
	struct rgapsdata *rgaps;
	int block;
{
    QH giveback = rgaps->GiveBackQ;
    int dummy;

    rgaps->nextpref = block;	/* will start after block */
    rgaps->prefetch_skip = FALSE;
    rgaps->endskip = 0;
    /* we assume prefetch limits have already been set up */

    /* empty the give-back queue */
    while(Dequeue(giveback, &dummy))
	 ;

    rpd->inode_ptr->prefetch.available = TRUE;
}

/* This turns prefetching off. */
/* Run alone from Finalize */
static void
StopPrefetch(rpd)
	RAPIDFILE *rpd;
{
    rpd->inode_ptr->prefetch.available = FALSE;
}

/* This puts a hard and soft upper limit on prefetching. */
/* This done under maxlast lock */
static void
LimitPrefetch(rgaps, hardlimit, softlimit)
	struct rgapsdata *rgaps;
	int hardlimit;			/* block number */
	int softlimit;			/* block number */
{
    rgaps->hardlimit = hardlimit;
    if (softlimit <= hardlimit)
	 rgaps->limit = softlimit;
    else
	 rgaps->limit = hardlimit;
}

/* This puts an upper limit on prefetching. */
/* This done under maxlast lock. Too bad that it needs a lock. */
static void
SoftLimitPrefetch(rgaps, softlimit)
	struct rgapsdata *rgaps;
	int softlimit;			/* block number */
{
    if (softlimit <= rgaps->hardlimit)
	 rgaps->limit = softlimit;
}

/* @SUBTITLE "Mistakes: Handle mistakes" */
/* This scans the table, starting at "begin" and ending at the
 * prefetching high-water mark (the last block prefetched), to 
 * find and process mistakes. 
 *  Run alone, from Finalize, after prefetching has stopped.
 */
static void 
Mistakes(rpd, rgaps, begin)
	RAPIDFILE *rpd;
	struct rgapsdata *rgaps;
	int begin;			/* first block that may be a mistake */
{
    int i;
    int end = min(rgaps->nextpref, rgaps->size-1);
    SECTOR_MAP_ENTRY *sme;

    for (i = begin; i <= end; i++) {
	   sme = RT_smeForSector(rpd, i);
	   OneMistake(rpd, i, sme);
    }
}

/* @SUBTITLE "CheckAvailable: is there any prefetch work?" */
/* This is an Entry Point */
/* This tells whether prefetching is possible. */
static boolean
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
/* 
    struct rgapslocal *lrgaps = (struct rgapslocal *)(rpd->prefetch.private);
    struct rgapsdata *rgaps = lrgaps->common;
*/
    boolean available;

    /* note that rpd->prefetch.available is always TRUE on entry */
    available = rpd->inode_ptr->prefetch.available; /* just a hint, really */
    rpd->prefetch.available = available; /* propagate hint locally */

    return(available);
}

/* @SUBTITLE "GetWork: get prefetch work" */
/* This is an Entry Point */
/* This hands out prefetching work. It uses the prefetching state in the gaps
 * structure, information about the current portion, and state data
 * in the sector map (but only GAPS state), to find a block to prefetch.
 * We will not try to prefetch a block that has already been tried,
 * or that has been used. We also limit our prefetching according to the
 * limit in rgaps->limit;
 *  We also pay attention to the minlast lock, and avoid prefetching
 * while that is set. This avoids spurious mistakes and wasted time.
 *  This uses its own lock. The lock is also used in NewPortion.
 */
static boolean
GetWork(rpd, sector)
	RAPIDFILE *rpd;
	int *sector;	/* returned */
{
    struct rgapslocal *lrgaps = (struct rgapslocal *)(rpd->prefetch.private);
    struct rgapsdata *rgaps = lrgaps->common;
    int block;
    SECTOR_MAP_ENTRY *sme;
    boolean ok = FALSE;
    boolean overlimit = FALSE;
    int nextpref;

    do {
	   LockFIFO(lrgaps->prefetch_fifo,10);
	   LockFIFO(lrgaps->pchange_fifo, 1);
	   if (!rpd->inode_ptr->prefetch.available || rgaps->jumpgap) {
		  UnlockFIFO(lrgaps->pchange_fifo);
		  UnlockFIFO(lrgaps->prefetch_fifo);
		  return(FALSE);	/* give up NOW */
	   }

	   /* Check for any blocks to be retried */
	   if (Dequeue(lrgaps->GiveBackQ, &block)) {
		  /* Possibly retry this block number */
		  ok = (block > rgaps->minlast && block >= rgaps->endskip
			   && block <= rgaps->limit);
	   } else { 
		  /* get the next block in sequence */
		  nextpref = rgaps->nextpref; 

		  /* Should we jump an irregular portion gap? */
		  if (rgaps->prefetch_skip && nextpref >= rgaps->startskip)
		    /* we have gone past a skip */
		    if (nextpref < rgaps->endskip) {
			   /* jump the skip */
			   nextpref = rgaps->nextpref = rgaps->endskip;
			   rgaps->prefetch_skip = FALSE;
		    } else
			 rgaps->prefetch_skip = FALSE;

		  if (nextpref < rgaps->minlast)
		    /* prefetching fell behind */
		    block = NextBlock(rgaps->minlast, rgaps->start,
						  rgaps->runlen, rgaps->skiplen);
		  else
		    /* find next block in sequence */
		    block = NextBlock(nextpref, rgaps->start,
						  rgaps->runlen, rgaps->skiplen);

		  /* Check the prefetch limit */
		  if (block <= rgaps->limit) {
			 rgaps->nextpref = block; /* commit */
			 ok = TRUE;
		  } else
		    overlimit = TRUE;
	   }

	   UnlockFIFO(lrgaps->pchange_fifo);
	   UnlockFIFO(lrgaps->prefetch_fifo);

	   if (ok && !rgaps->jumpgap) {
		  /* Try to mark this block for prefetch */
		  sme = RT_smeForSector(rpd, block);
		  if (SS_ISCLEAR(sme, SS_USED | SS_MARKED)) {
			 /* This block is good to prefetch. */
			 /* mark it as work handed out */
			 SS_SET(sme, SS_MARKED);
			 /* check to see if minlast changed */
			 if (block <= rgaps->minlast) {
				/* oh well, minlast changed */
				SS_CLEAR(sme, SS_MARKED); 
				ok = FALSE;
			 }			/* else ok stays TRUE: prefetch it! */
		  } else
		    ok = FALSE;
	   } else
		ok = FALSE;
    } while (!ok && !overlimit);

    if (ok) {
	   *sector = block;

#ifdef GAPTRACE
	   printf("-> node %d prefetch work block %d\n", Vnode, block);
#endif
    }

    return(ok);
}

/* @SUBTITLE "NextBlock: compute the next block in sequence" */
/*   This computes the next block number in the sequence of blocks
 * in a regular-portioned pattern.
 *   This function is simplified by the assumption the skip is nonnegative
 * and any limits on the portion are handled elsewhere.
 * Thus all of our parameters (and return values) are nonnegative integers. 
 */
static int				/* next block number */
NextBlock(block, start, runlen, skiplen)
	int block;			/* current block number >= 0 */
	int start;			/* start of portion >= 0 */
	int runlen;			/* run length >= 0 */
	int skiplen;			/* skip length >= 0*/
{
    if (block < start)
	 return(start);

    if (runlen == 0 || skiplen == 0)
	 return(block + 1);
    else
	 if ((block - start) % (runlen+skiplen-1) < runlen - 1)
	   return(block + 1);
	 else
	   return(block + skiplen);
}

/* @SUBTITLE "GiveBackWork: give back some prefetch work" */
/* This is an Entry Point */
/* This function is used by the prefetchers to give back prefetch
 * work that could not be completed. These are unmarked, and
 * GAPS's prefetch pointer is reset so these may be tried again. */
static void
GiveBackWork(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct rgapslocal *lrgaps = (struct rgapslocal *)(rpd->prefetch.private);
    struct rgapsdata *rgaps = lrgaps->common;
    SECTOR_MAP_ENTRY *sme;

    sme = RT_smeForSector(rpd, sector);
    SS_CLEAR(sme, SS_MARKED | SS_MISTAKE);

    /* Put this block number on a queue, to be retried */
    /* (push out oldest entry on overflow) */
    if (rpd->inode_ptr->prefetch.available)
	 ForceEnqueue(lrgaps->GiveBackQ, sector, NULL);
}

/* @SUBTITLE "Done: Finished with reference pattern" */
/* This is an Entry Point */
/* This deallocates the rgaps data structures. Only once. */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    struct rgapslocal *lrgaps = (struct rgapslocal *)(rpd->prefetch.private);
    struct rgapsdata *rgaps = lrgaps->common;

    if (Vnode == 0) {
	   if (rgaps->num_sequential > 0)
		Finalize(rpd, rgaps);

	   FreeFIFO(rgaps->maxlast_fifo);
	   FreeFIFO(rgaps->prefetch_fifo);
	   FreeFIFO(rgaps->pchange_fifo);
	   UsFree(rgaps->GiveBackQ);
	   UsFree(rgaps->last);
	   UsFree(rgaps->sequential);
	   UsFree(rgaps);
	   rpd->inode_ptr->prefetch.private = NULL;
	   rpd->inode_ptr->prefetch.available = FALSE;
	   rpd->prefetch.available = FALSE;
    }

    UsFree(lrgaps->sequential);
    UsFree(lrgaps->last);
    UsFree(lrgaps);
}

/* @SUBTITLE "Dump: debugging dump" */
/* Internally, we can call Dump(NULL, lrgaps) instead. */
static void
Dump(rpd, lrgaps)
	RAPIDFILE *rpd;
	struct rgapslocal *lrgaps; /* may be supplied if rpd==NULL */
{
    struct rgapsdata *rgaps;

    if (rpd != NULL)		/* use rpd to find lrgaps */
	 lrgaps = (struct rgapslocal *)(rpd->prefetch.private);
    rgaps = lrgaps ? lrgaps->common : NULL;

    printf("  RGAPS Predictor: local\n");
    if (lrgaps != NULL) {
	   printf("   common = 0x%x\n", lrgaps->common);
	   printf("   nprocs = %d\n", lrgaps->nprocs);
	   printf("   mylast = %d\n", lrgaps->mylast);
	   printf("   last = 0x%x\n", lrgaps->last);
	   printf("   mysequential = %d\n", lrgaps->mysequential);
	   printf("   seqeuential = 0x%x\n", lrgaps->sequential);
	   printf("   ordered = %s\n", lrgaps->ordered ? "TRUE" : "FALSE");
	   printf("   ordercount = %d\n", lrgaps->ordercount);
	   printf("   shortorder = %d\n", lrgaps->shortorder);
	   printf("   minlast_lock = 0x%x\n", lrgaps->minlast_lock);
	   printf("   maxlast_fifo = 0x%x\n", lrgaps->maxlast_fifo);
	   printf("   prefetch_fifo = 0x%x\n", lrgaps->prefetch_fifo);
	   printf("   pchange_fifo = 0x%x\n", lrgaps->pchange_fifo);
	   printf("   GiveBackQ = 0x%x (%d)\n", lrgaps->GiveBackQ,
			lrgaps->GiveBackQ ? InQueue(lrgaps->GiveBackQ) : 0);
	   printf("   size = %d\n", lrgaps->size);
    } else
	 printf("   NULL rgapslocal data");

    printf("  RGAPS Predictor: global\n");
    if (rgaps != NULL) {
	   printf("   parm = %d\n", rgaps->parm);
	   printf("   nprocs = %d\n", rgaps->nprocs);
	   printf("   contin_lock = %s\n", rgaps->contin_lock ? "SET" : "CLEAR");
	   printf("   in_random = %d\n", rgaps->in_random);
	   printf("   contin = %s\n", rgaps->contin ? "TRUE" : "FALSE");
	   printf("   last = 0x%x\n", rgaps->last);
	   printf("   seqeuential = 0x%x\n", rgaps->sequential);
	   printf("   num_sequential = %d\n", rgaps->num_sequential);
	   printf("   num_ordered = %d\n", rgaps->num_ordered);

	   printf("   maxjump = %d\n", rgaps->maxjump);
	   printf("   minlast_lock = %s\n", rgaps->minlast_lock ? "SET" : "CLEAR");
	   printf("   maxlast_fifo = 0x%x\n", rgaps->maxlast_fifo);
	   printf("   start = %d\n", rgaps->start);
	   printf("   minlast = %d\n", rgaps->minlast);
	   printf("   maxlast = %d\n", rgaps->maxlast);
	   printf("   jumpgap = %s\n", rgaps->jumpgap ? "TRUE" : "FALSE");
	   printf("   portion1 = %s\n", rgaps->portion1 ? "TRUE" : "FALSE");
	   printf("   same_runlen = %d\n", rgaps->same_runlen);
	   printf("   prev_runlen = %d\n", rgaps->prev_runlen);
	   printf("   same_skiplen = %d\n", rgaps->same_skiplen);
	   printf("   prev_skiplen = %d\n", rgaps->prev_skiplen);
	   printf("   prev_end = %d\n", rgaps->prev_end);
	   printf("   runlen = %d\n", rgaps->runlen);
	   printf("   skiplen = %d\n", rgaps->skiplen);
	   printf("   prefetch_fifo = 0x%x\n", rgaps->prefetch_fifo);
	   printf("   pchange_fifo = 0x%x\n", rgaps->pchange_fifo);
	   printf("   GiveBackQ = 0x%x (%d)\n", lrgaps->GiveBackQ,
			lrgaps->GiveBackQ ? InQueue(lrgaps->GiveBackQ) : 0);
	   printf("   nextpref = %d\n", rgaps->nextpref);
	   printf("   startskip = %d\n", rgaps->startskip);
	   printf("   endskip = %d\n", rgaps->endskip);
	   printf("   prefetch_skip = %d\n", rgaps->prefetch_skip);
	   printf("   size = %d\n", rgaps->size);
	   printf("   limit = %d\n", rgaps->limit);
	   printf("   maxdist = %d\n", rgaps->maxdist);
    } else
	 printf("   NULL rgapsdata");

}
