/* @TITLE "predict-obl.c - OBL predict algorithm" */
/* predict-obl.c: One-block lookahead
 *
 * We use the 'private' element of the prefetch structure as an 
 * integer, recording the previous reference. We turn the 'available'
 * flag on and off; on when we get a reference, off when we hand out
 * last+1 as a prediction. We only predict one ahead.
 *
 * Immediate rereference counts as a correct prediction.
 *
 * FUNCTIONS:
 *      InitPredict_OBL
 *
 * 'Local' entry points:
 *      LateNotify
 *      Notify
 *      CheckAvailable
 *      GetWork
 *      GiveBackWork
 *      Done
 *      Dump
 *
 * David Kotz  November 1989
 */

static char rcsid[] = "$Id: predict-obl.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "stats.h"
#include "rapidelog.h"

static void Done();
static void Dump();
static boolean Notify();
static void LateNotify();
static boolean CheckAvailable();
static boolean GetWork();
static void GiveBackWork();

/* @SUBTITLE "InitPredict_OBL: Initialize for OBL algorithm" */
void
InitPredict_OBL(rpd, ignored)
	RAPIDFILE *rpd;
	int ignored;
{
    rpd->prefetch.available = FALSE; /* no prediction yet */
    rpd->prefetch.private = (ANYPTR) -10; /* last reference: impossible */

    rpd->prefetch.donerpd = Done;
    rpd->prefetch.notify = Notify;
    rpd->prefetch.latenotify = LateNotify;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.getwork = GetWork;
    rpd->prefetch.giveback = GiveBackWork;
    rpd->prefetch.dump = Dump;
}

/* @SUBTITLE "LateNotify: Multiple notifications" */
/* All we care about here is the most recent access. Very simple. */
static void
LateNotify(node, rpd, private, last, length, sector)
	int node;				/* processor number */
	RAPIDFILE *rpd;		/* can be trusted: this is local predictor */
	ANYPTR private;		/* private data */
	int last;				/* last block accessed in run */
	int length;			/* length of consecutive run */
	int sector;			/* most recent sector accessed (-1 if none) */
{
#ifdef TRACE
    printf("Proc %d (%d) in OBL LateNotify: last=%d, length=%d, sector=%d\n",
		 Vnode, node, last, length, sector);
    fflush(stdout);
#endif

    if (sector >= 0) {
	   /* record last reference */
	   rpd->prefetch.private = (ANYPTR) sector;
	   rpd->prefetch.available = TRUE; /* we have a prediction */
    } else if (length > 0) {
	   /* record last reference */
	   rpd->prefetch.private = (ANYPTR) last;
	   rpd->prefetch.available = TRUE; /* we have a prediction */
    }
}

/* @SUBTITLE "Notify: notification that a sector is accessed" */
static boolean
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    int last = (int) rpd->prefetch.private;
    boolean wanted;

    /* immediate re-reference is correct, but doesn't change prediction */
    if (last == sector) {
	   my_stats->correct++;
	   return(wanted = FALSE);
    }

    /* correct prediction? */
    if (last + 1 == sector) {
	   my_stats->correct++;
	   wanted = (rpd->prefetch.available == FALSE);
    } else if (rpd->prefetch.available == FALSE && last >= 0) {
	   /* we had prefetched last+1 */
	   RT_PrefetchMistake(rpd, last+1);
	   wanted = FALSE;
    }

    /* record last reference */
    rpd->prefetch.private = (ANYPTR)sector;
    rpd->prefetch.available = TRUE; /* we have a prediction */

    return(wanted);
}

/* @SUBTITLE "CheckAvailable: is there work available?" */
/* Are there any sectors to be prefetched? */

static boolean				/* TRUE if there's work */
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
    /* we have no further information */
    return(rpd->prefetch.available);
}

/* @SUBTITLE "GetWork: get a sector to prefetch" */
/* give last+1 as our prediction */

static boolean				/* TRUE if sector was found */
GetWork(rpd, sector)
	RAPIDFILE *rpd;
	int *sector;	/* returned */
{
    int last = (int) rpd->prefetch.private;
    boolean success;

    *sector = last+1;

    /* we won't make the same prediction twice */
    success = rpd->prefetch.available;
    rpd->prefetch.available = FALSE;

    return(success);
}

/* @SUBTITLE "GiveBackWork: Sector could not be prefetched" */

static void
GiveBackWork(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    /* allow us to hand it out again */
    rpd->prefetch.available = TRUE;
}

/* @SUBTITLE "Done: What happens at end of string" */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    int last = (int) rpd->prefetch.private;

    if (rpd->prefetch.available == FALSE && last >= 0)
	 /* we had prefetched last+1 */
	 RT_PrefetchMistake(rpd, last+1);
}

/* @SUBTITLE "Dump: debugging dump" */
static void
Dump(rpd)
	RAPIDFILE *rpd;
{
    int last = (int) rpd->prefetch.private;

    printf("  OBL Predictor\n");
    printf("   last = %d\n", last);
}

