/* @TITLE "predict-IPORT.c - IPORT predict algorithm" */
/* predict-IPORT.c: Portion-detection algorithm
 *
 * This is a variant of PORT that does not do the MaxDist cut when
 * in the first (presumably only) portion - thus this works more like
 * IBL when in the lw pattern, though still a little conservative at 
 * start-up, to avoid lots of mistakes in other patterns.
 *
 * This algorithm tries to notice regular sequential portions, and
 * identify regularity in the portion length and portion skip. This
 * then tries to do OBL as long as it thinks it's inside a portion.
 *
 * Immediate rereference counts as a correct prediction.
 *
 * FUNCTIONS:
 *      InitPredict_IPORT
 *
 * 'Local' entry points:
 *      LateNotify
 *      Notify
 *      CheckAvailable
 *      GetWork
 *      GiveBackWork
 *      Done
 *      Dump
 *
 * David Kotz  November 1989
 */

static char rcsid[] = "$Id: predict-iport.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $";

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "stats.h"
#include "rapidelog.h"
#include "predict-debug.h"

static void Done();
static void Dump();
static boolean Notify();
static void LateNotify();
static boolean CheckAvailable();
static boolean GetWork();
static void GiveBackWork();

static int predict();
#define NOPREDICT (-100)		/* value used for no prediction */

static void mistakes();

/* @SUBTITLE "Local Data" */
/* idea is to use 1,2,2 for these parms */
static int MinRun = 1;		/* number of blocks in a row to assume seq */
static int RunRep = 2;		/* number of runs in a row to assume regular */
static int SkipRep = 2;		/* number of skips in a row to assume regular */

/* use 1 for OBL-like behavior */
static int MaxDist = 5;		/* number of blocks ahead to predict */

struct iportdata {			/* private data for this algorithm */
    /* current run */
    int first;				/* start of current run */
    int last;				/* last block read */
    int next;				/* next block we think will be read */
    int prefetch;			/* the last block prefetched or read */
    int prevpref;			/* the previous block prefetched (for undo) */
    int nextpref;			/* the next block to prefetch */
    int distance;			/* number of blocks between last and prefetch */
    int runlen;			/* if regular, else 0 */
    int skiplen;			/* is regular, else 0 */
    boolean cutoff;			/* TRUE for MaxDist cutoff */
    /* previous runs */
    int same_runlen;		/* number of runs with same len */
    int prev_runlen;		/* length of previous run */
    int same_skiplen;		/* number of skips with same len */
    int prev_skiplen;		/* length of previous skip */
};


/* @SUBTITLE "InitPredict_IPORT: Initialize for IPORT algorithm" */
void
InitPredict_IPORT(rpd, parm)
	RAPIDFILE *rpd;
	int parm;
{
    struct iportdata *port;

    port = (struct iportdata *)AllocLocal(sizeof(struct iportdata));

    port->first = port->last = port->next = NOPREDICT;
    port->prefetch = port->last;
    port->nextpref = NOPREDICT;
    port->distance = 0;
    port->runlen = port->skiplen = 0;
    port->cutoff = FALSE;
    port->same_runlen = port->same_skiplen = 0;
    port->prev_runlen = port->prev_skiplen = 0;

    rpd->prefetch.available = FALSE; /* no prediction yet */
    rpd->prefetch.private = (ANYPTR) port;

    rpd->prefetch.donerpd = Done;
    rpd->prefetch.notify = Notify;
    rpd->prefetch.latenotify = LateNotify;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.getwork = GetWork;
    rpd->prefetch.giveback = GiveBackWork;
    rpd->prefetch.dump = Dump;

    if (parm > 0)
	 MaxDist = parm;
}

/* @SUBTITLE "LateNotify: Multiple notifications" */
static void
LateNotify(node, rpd, private, last, length, sector)
	int node;				/* processor number */
	RAPIDFILE *rpd;		/* can be trusted: this is local predictor */
	ANYPTR private;		/* private data */
	int last;				/* last block accessed in run */
	int length;			/* length of consecutive run */
	int sector;			/* most recent block accessed (-1 if none) */
{
    struct iportdata *port = (struct iportdata *)private;

#ifdef TRACE
    printf("Proc %d (%d) in IPORT LateNotify: last=%d, length=%d, sector=%d\n",
		 Vnode, node, last, length, sector);
    fflush(stdout);
#endif

    if (sector >= 0) {
	   if (length > 0)
		if (sector == last+1) {
		    /* just an extension to the same portion */
		    last = sector;
		    length++;
		    port->first = last-length+1;
		    port->last = last;
		    port->prefetch = last;
		} else {
		    /* actually two portions */
		    port->first = port->last = sector;
		    port->prefetch = last;
		    port->prev_runlen = length;
		    port->same_runlen = 1;
		    port->prev_skiplen = sector-last;
		    port->same_skiplen = 1;
		    port->cutoff = TRUE;
		}
	   else {
		  /* just one portion, containing only one sector */
		  port->first = port->last = sector;
		  port->prefetch = sector;
	   }
    } else {
	   if (length > 0) {
		  /* just one portion */
		  port->first = last-length+1;
		  port->last = last;
		  port->prefetch = last;
	   } /* else no portions */
    }

    /* predict the next block to be referenced */
    port->next = 
	 predict(port->first, 
		    port->runlen,
		    port->skiplen,
		    port->last,
		    port->last,	/* pretend none prefetched */
		    0,			/* pretend none prefetched */
		    port->cutoff);

    /* don't compute the next prefetched block yet */
    port->nextpref = NOPREDICT; /* none yet */
    rpd->prefetch.available = TRUE; /* to force one call to CheckAvailable */
}

/* @SUBTITLE "Notify: notification that a sector is accessed" */
static boolean
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct iportdata *port = (struct iportdata *)(rpd->prefetch.private);
    int runlen;
    int skiplen;
    boolean predicted;
    boolean wanted;
    boolean reref = FALSE;

    if (port->last < 0) {
	   /* first reference in this sequence */
	   port->first = port->last = sector;
	   predicted = FALSE;
	   wanted = FALSE;
    } else if (sector == port->last) {    /* immediate rereference */
	   /* port->next does not change */
	   predicted = TRUE;
	   reref = TRUE;
	   wanted = FALSE;
    } else {				/* not a rereference */
	   /* how did our prediction go? */
	   predicted = (sector == port->next);
	   wanted = (predicted && port->distance > 0);

	   if (!predicted)		/* clean up any mistakes now */
		mistakes(rpd, port);

	   /* out-of-order reference, update run/skip lengths */
	   if (sector != port->last + 1) {
		  runlen = port->last - port->first + 1;
		  skiplen = sector - port->last;
		  port->first = port->last = sector;
		  port->cutoff = TRUE; /* now be more conservative */
		  if (runlen == port->prev_runlen) {
			 /* same run length */
			 port->same_runlen++;
			 if (skiplen == port->prev_skiplen)
			   /* same run length, same skip length */
			   port->same_skiplen++;
			 else 
			   /* same run length, different skip length */
			   port->same_skiplen = 1;
		  } else {
			 /* different run length, any skip length */
			 port->same_runlen = 1;
			 port->same_skiplen = 1;
		  }
		  port->prev_runlen = runlen;
		  port->prev_skiplen = skiplen;
		  port->runlen 
		    = (port->same_runlen >= RunRep) ? port->prev_runlen : 0;
		  port->skiplen
		    = (port->runlen > 0 && port->same_skiplen >= SkipRep) ?
			 port->prev_skiplen : 0;
	   } else
		port->last = sector;
    }

    if (predicted) {
	   my_stats->correct++;
	   if (!reref && port->distance > 0)
		port->distance--;
    } 
    if (port->distance == 0)
	 port->prefetch = port->last;
	   
    /* predict the next block to be referenced */
    port->next = 
	 predict(port->first, 
		    port->runlen,
		    port->skiplen,
		    port->last,
		    port->last,	/* pretend none prefetched */
		    0,			/* pretend none prefetched */
		    port->cutoff);

    /* don't compute the next prefetched block yet */
    port->nextpref = NOPREDICT; /* none yet */
    rpd->prefetch.available = TRUE; /* to force one call to CheckAvailable */

    return(wanted);
}

/* @SUBTITLE "predict: predict next block for prefetch" */
/* parameters give the current state */
/* see notebook for 11/23/89 */
static int				/* next block to be prefetched */
predict(s, r, h, last, p, d, cutoff)
	int s;				/* start of current run */
	int r;				/* runlen if regular else 0 */
	int h;				/* skiplen if regular else 0 */
	int last;				/* last block referenced */
	int p;				/* last block prefetched */
	int d;				/* number of blocks between p and last */
	boolean cutoff;		/* TRUE for MaxDist cutoff */
{
    /* limit our prefetching to a certain distance */
    /* for irregular runs, we use the distance based on current runlen */
    /* for regular runs, we predict up to MaxDist ahead */
    if ((r == 0 && d >= distance(last-s+1, cutoff)) || (r > 0 && d >= MaxDist))
	 return(NOPREDICT);

    if (h == 0)
	 /* no regular skip known */
	 if (r == 0 || (p-s+1) < r)
	   return(p+1);		/* still in run */
	 else
	   return(NOPREDICT);	/* finished run */
    else 
	 /* regular run, skip known */
	 if ((d+last-s+1) % r > 0)
	   return(p+1);		/* still in run */
	 else
	   return(p+h);		/* time for a skip */

}

/* @SUBTITLE "distance: how far ahead do we predict?" */
/* This is the most tunable portion of the prediction algorithm */
static int 
distance(runlen, cutoff)
	int runlen;
	boolean cutoff;		/* TRUE for MaxDist cutoff */
{
    if (runlen < MinRun)		/* too short - none */
	 return(0);
    else if (cutoff && runlen >= MinRun + MaxDist) /* very long - cut off */
	 return(MaxDist);
    else					/* linear guess in between */
	 return (runlen - MinRun + 1);
}


/* @SUBTITLE "mistakes: process mistakes" */
/* assume all prefetches to be mistaken */
static void
mistakes(rpd, port)
	RAPIDFILE *rpd;		/* RPD to pass on */
	struct iportdata *port;	/* port data */
{
    int dist = port->distance; /* number of prefetches ahead of references */
    int s = port->first;		/* first in run */
    int r = port->runlen;	/* run length, if regular */
    int h = port->skiplen;	/* skip length, if regular */
    int last = port->last;	/* last reference */
    boolean cutoff = port->cutoff;
    int p;				/* block number for loop */
    int d;				/* loop distance variable */

    p = last;
    d = 0;

    while (d < dist) {
	   p = predict(s, r, h, last, p, d, cutoff);
	   RT_PrefetchMistake(rpd, p);
	   d++;
    }

    /* now reset prediction status */
    port->distance = 0;
    port->prefetch = port->last;
    port->nextpref = NOPREDICT;
}

/* @SUBTITLE "CheckAvailable: is there work available?" */
/* Are there any sectors to be prefetched? */

static boolean				/* TRUE if there's work */
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
    struct iportdata *port = (struct iportdata *)(rpd->prefetch.private);

    port->nextpref = 
	 predict(port->first, port->runlen, port->skiplen, 
		    port->last, port->prefetch, port->distance,
		    port->cutoff);

    rpd->prefetch.available = (port->nextpref != NOPREDICT);
    /* FALSE keeps us from being called until next Notify() */

    return(rpd->prefetch.available);
}

/* @SUBTITLE "GetWork: get a sector to prefetch" */
/* Here we assume CheckAvailable was called last
 * (ie, with no intervening call to Notify()). We use the prediction
 * that it remembers. nextpref should not be NOPREDICT because that
 * would have given a FALSE return from CheckAvailable and the caller
 * would not call GetWork.
 */

static boolean				/* TRUE if sector was found */
GetWork(rpd, sector)
	RAPIDFILE *rpd;
	int *sector;	/* returned */
{
    struct iportdata *port = (struct iportdata *)(rpd->prefetch.private);

    if (port->nextpref != NOPREDICT) {
	   port->prevpref = port->prefetch;
	   port->prefetch = port->nextpref;
	   port->distance++;
	   *sector = port->nextpref;
	   return(TRUE);
    } else
	 return(FALSE);
}

/* @SUBTITLE "GiveBackWork: Sector could not be prefetched" */

static void
GiveBackWork(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct iportdata *port = (struct iportdata *)(rpd->prefetch.private);

    /* undo the above */
    port->distance--;
    port->prefetch = port->prevpref;
    rpd->prefetch.available = TRUE;

    /* need not call CheckAvailable before calling GetWork again */
    /* unless Notify is called */
}

/* @SUBTITLE "Done: What happens at end of string" */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    struct iportdata *port = (struct iportdata *)(rpd->prefetch.private);

    mistakes(rpd, port);

    UsFree(port);
    rpd->prefetch.private = NULL;
}

/* @SUBTITLE "Dump: debugging dump" */
static void
Dump(rpd)
	RAPIDFILE *rpd;
{
    struct iportdata *iport = (struct iportdata *)(rpd->prefetch.private);

    printf("  IPORT Predictor (MaxDist = %d):\n", MaxDist);
    if (iport != NULL) {
	   printf("   first = %d\n", iport->first);
	   printf("   last = %d\n", iport->last);
	   printf("   next = %d\n", iport->next);
	   printf("   prefetch = %d\n", iport->prefetch);
	   printf("   prevpref = %d\n", iport->prevpref);
	   printf("   nextpref = %d\n", iport->nextpref);
	   printf("   distance = %d\n", iport->distance);
	   printf("   runlen = %d\n", iport->runlen);
	   printf("   skiplen = %d\n", iport->skiplen);
	   printf("   cutoff = %s\n", iport->cutoff ? "YES" : "NO");
	   printf("   same_runlen = %d\n", iport->same_runlen);
	   printf("   prev_runlen = %d\n", iport->prev_runlen);
	   printf("   same_skiplen = %d\n", iport->same_skiplen);
	   printf("   prev_skiplen = %d\n", iport->prev_skiplen);
    } else 
	 printf("   NULL iportdata");
}
