/* @TITLE "predict-GW.c -- Specialized to GW" */
/* 
 * GW predictor
 *   This predictor is specific to the GW pattern. It makes no attempt
 * to really understand the pattern - or even notice mistakes. This might
 * be useful with a user hint.
 *   We do not limit the prefetch distance. This is, in effect, IBL for global 
 * patterns. We do accept give-backs.
 *
 * David Kotz April 1990
 */

static char rcsid[] = "$Id: predict-gw.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $";

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "stats.h"
#include "rapidelog.h"
#include "predict-debug.h"
#include "queue.h"

/* This structure contains the entire state of the GW predictor. */
static struct gwdata {
    int parm;				/* parameter from command line */
    QH GiveBackQ;			/* queue for work given back */
    int maxlast;			/* maximum block number read so far */
    int nextpref;			/* next block to prefetch */
    ALOCK maxlast_lock;		/* maxlast update lock */
};

/* FUNCTIONS - Entry points */
void InitPredict_GW();
static void LateNotify();
static boolean Notify();
static void Done();
static void Dump();
static boolean CheckAvailable();
static boolean GetWork();
static void GiveBackWork();

/* @SUBTITLE "InitPredict_GW - initialize data structures" */
/* This is an Entry Point */
/* This function allocates and initializes the GW data structures. Only 
 * one process will do most of the work (process 0). The others wait
 * until the pointer is ready, and then do per-process initialization.
 */
void
InitPredict_GW(rpd, parm)
	RAPIDFILE *rpd;
	int parm;				/* currently ignored */
{
    struct gwdata *gw;
    RAPID_INODE *inode_ptr = rpd->inode_ptr;;

    /* no prefetching yet */
    rpd->prefetch.available = FALSE;

    /* Set up function pointers */
    rpd->prefetch.donerpd = Done;
    rpd->prefetch.notify = Notify;
    rpd->prefetch.latenotify = LateNotify;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.getwork = GetWork;
    rpd->prefetch.giveback = GiveBackWork;
    rpd->prefetch.dump = Dump;

    /* Only one process does the global initialization */
    if (Vnode == 0) {
	   gw = (struct gwdata *)AllocGlobal(sizeof(struct gwdata));

	   gw->parm = parm;
	   gw->nextpref = 0;
	   gw->maxlast = 0;
	   gw->maxlast_lock = 0;

	   gw->GiveBackQ = Make_queue(inode_ptr->prefetch.limit);

	   /* start prefetching now */
	   inode_ptr->prefetch.available = TRUE; 

	   /* this will release the other processes from a barrier below */
	   inode_ptr->prefetch.private = (ANYPTR) gw;
    } else {
	   /* all procs except 0 */
	   while (inode_ptr->prefetch.private == NULL)
		UsWait(50);
    }

    /* Copy the pointer to our local memory */
    gw = (struct gwdata *)(inode_ptr->prefetch.private);
    rpd->prefetch.private = (ANYPTR) gw;

    rpd->prefetch.available = TRUE;
}

/* @SUBTITLE "LateNotify: Multiple notifications" */
static void
LateNotify(node, rpd, private, last, length, sector)
	int node;				/* processor number */
	RAPIDFILE *rpd;		/* may not be node's rpd */
	ANYPTR private;		/* private data */
	int last;				/* last block accessed in run */
	int length;			/* length of consecutive run */
	int sector;			/* most recent block accessed (-1 if none) */
{
#ifdef TRACE
    printf("Proc %d (%d) in GW LateNotify: last=%d, length=%d, sector=%d\n",
		 Vnode, node, last, length, sector);
#endif /* TRACE */

    if (sector < 0 && length > 0)
	 sector = last;
    /* note sector may still be negative */

    /* Note that if node!=Vnode, rpd is not node's rpd.
	* But the rpd is good enough to allow Notify to get at the gwdata,
	* and for us to get at the inode's prefetch.available.
	*/
    if (sector > 0)
	 (void) Notify(rpd, sector);

    rpd->inode_ptr->prefetch.available = TRUE; /* tell everyone */
    rpd->prefetch.available = TRUE;
}

/* @SUBTITLE "Notify - update for a new reference" */
/* This is an Entry Point */
/* This is called with each sector that is accessed. 
 */
static boolean
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct gwdata *gw = (struct gwdata *)(rpd->prefetch.private);
    
    my_stats->correct++;

    /* update maxlast */
    while (sector > gw->maxlast)
	 if (TryLock(&(gw->maxlast_lock))) {
		if (sector > gw->maxlast)
		  gw->maxlast = sector;
		UNLOCK(&(gw->maxlast_lock));
	 }

    return(FALSE);			/* for WANTED */
}

/* @SUBTITLE "CheckAvailable: is there any prefetch work?" */
/* This is an Entry Point */
/* This tells whether prefetching is possible. */
static boolean
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
/* 
    struct gwdata *gw = (struct gwdata *)(rpd->prefetch.private);
*/
    rpd->prefetch.available = TRUE;

    return(TRUE);
}

/* @SUBTITLE "GetWork: get prefetch work" */
/* This is an Entry Point */
/* This hands out prefetching work. We pull something off the give-back
 * queue if possible, otherwise we increment the next pointer and 
 * give out that block number. We do not hand out any work that is less 
 * than maxlast. 
 */
static boolean
GetWork(rpd, sector)
	RAPIDFILE *rpd;
	int *sector;	/* returned */
{
    struct gwdata *gw = (struct gwdata *)(rpd->prefetch.private);
    int block;

    /* Check for any blocks to be retried */
    do {
	   if (!Dequeue(gw->GiveBackQ, &block))
		block = Atomic_add_long(&(gw->nextpref), 1);
    } while (block <= gw->maxlast);
    
    *sector = block;
    
    return(TRUE);
}

/* @SUBTITLE "GiveBackWork: give back some prefetch work" */
/* This is an Entry Point */
/* This function is used by the prefetchers to give back prefetch
 * work that could not be completed. They are put on the GiveBackQ.
 */
static void
GiveBackWork(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct gwdata *gw = (struct gwdata *)(rpd->prefetch.private);

    /* Put this block number on a queue, to be retried */
    /* (push out oldest entry on overflow) */
    ForceEnqueue(gw->GiveBackQ, sector, NULL);
}

/* @SUBTITLE "Done: Finished with reference pattern" */
/* This is an Entry Point */
/* This deallocates the gw data structures. Only once. */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    struct gwdata *gw = (struct gwdata *)(rpd->prefetch.private);

    if (Vnode == 0) {
	   UsFree(gw->GiveBackQ);
	   UsFree(gw);
	   rpd->inode_ptr->prefetch.private = NULL;
	   rpd->inode_ptr->prefetch.available = FALSE;
    }
    rpd->prefetch.available = FALSE;
}

/* @SUBTITLE "Dump: debugging dump" */
static void
Dump(rpd)
	RAPIDFILE *rpd;
{
    struct gwdata *gw = (struct gwdata *)(rpd->prefetch.private);

    printf("  GW Predictor:\n");
    if (gw != NULL) {
	   printf("   parm = %d\n", gw->parm);
	   printf("   GiveBackQ = 0x%x (%d)\n", gw->GiveBackQ,
			gw->GiveBackQ ? InQueue(gw->GiveBackQ) : 0);
	   printf("   maxlast = %d\n", gw->maxlast);
	   printf("   maxlast_lock = %s\n", gw->maxlast_lock ? "SET" : "CLEAR");
	   printf("   nextpref = %d\n", gw->nextpref);
    } else
	 printf("   NULL gwdata");
}
