/* @TITLE "predict-exact.c - EXACT predict algorithm" */
/* predict-exact.c: Predicts references based on knowing ref string.
 *   The reference string, which is in chunks is expanded into a block
 * reference string, indexed by chunks. This makes it much easier to track 
 * the progress of the pattern, and prefetch blocks that have not yet been
 * referenced. It allows for quick, concurrent Notifications for both local
 * and global patterns.
 *
 * FUNCTIONS:
 *      InitPredict_EXACT
 *
 * Local:
 *    BuildBlockList();
 *    Notify();
 *    LateNotify();
 *    CheckAvailable();
 *    GetWork();
 *    GiveBackWork();
 *    Done();
 *    Dump();
 *
 * David Kotz  November 1989
 */

static char rcsid[] = "$Id: predict-exact.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "refs.h"
#include "prefetch.h"
#include "stats.h"
#include "others.h"
#include "own.h"
#include "wanted.h"

/* PRIVATE DATA */
/* There may be one of these, if global pattern, or many, if local pattern */
struct exactdata {
    REFS *refs;			/* the reference pattern (we cheat) */
    boolean global;			/* TRUE if global pattern */
    lock_t lock;			/* for global and FOR_OTHERS */
    int *blocks;			/* list of blocks in pattern, in order */
    short *index;			/* chunk-index into block list */
    int nblocks;			/* length of blocks list */
    short nextblock;		/* index of next block to prefetch */
    /* Below is for FOR_OTHERS prefetching in local patterns */
    int nprocs;			/* number of procs involved */
    struct exactdata **others; /* access to other processes' data */
    short *rotation;		/* rotation counter, to pick a pattern */
};

/* FUNCTIONS */
static void BuildBlockList();
static boolean Notify();
static void LateNotify();
static boolean CheckAvailable();
static boolean GetWork();
static void GiveBackWork();
static void Done();
static void Dump();

#ifdef FOR_OTHERS
static void InitForOthers();
#endif FOR_OTHERS


/* @SUBTITLE "InitPredict_EXACT: Initialize for EXACT algorithm" */
void
InitPredict_EXACT(rpd, ignored)
	RAPIDFILE *rpd;
	int ignored;
{
    REFS *refs = (REFS *)rpd->prefetch.refs;
    struct exactdata *exact;	/* our exact data */

    rpd->prefetch.available = TRUE;

    rpd->prefetch.donerpd = Done;
    rpd->prefetch.notify = Notify;
    rpd->prefetch.latenotify = LateNotify;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.getwork = GetWork;
    rpd->prefetch.giveback = GiveBackWork;
    rpd->prefetch.dump = Dump;

    my_stats->mistake = 0;
    my_stats->wasted = 0;

    if (!refs->global || Vnode == 0) {
	   exact = (struct exactdata *)AllocLocal(sizeof(struct exactdata));
	   exact->refs = refs;
	   exact->global = refs->global;
	   exact->lock = 0;
	   /* Build the block list for tracking the pattern */
	   BuildBlockList(rpd, exact);
    }

    /* Distribute the exact pointer to all processes, if global */
    if (refs->global)
	 if (Vnode == 0)
	   rpd->inode_ptr->prefetch.private = (ANYPTR)exact;
	 else {
		/* Wait for Vnode 0 to allocate and initialize exact */
		exact = (struct exactdata *)rpd->inode_ptr->prefetch.private;
		while (exact == NULL) {
		    UsWait(20);
		    exact = (struct exactdata *)rpd->inode_ptr->prefetch.private;
		}
	 }
#ifdef FOR_OTHERS
    else
	 InitForOthers(rpd, exact);
#endif FOR_OTHERS
    
    /* Finally, store the pointer in rpd for future use */
    rpd->prefetch.private = (ANYPTR) exact;
}

/* @SUBTITLE "InitForOthers: initialization for FOR_OTHERS" */

/* local patterns, FOR_OTHERS only */
#ifdef FOR_OTHERS

static void
InitForOthers(rpd, exact)
	RAPIDFILE *rpd;
	struct exactdata *exact;
{
    struct exactdata *gexact;	/* a temporary pointer to Vnode0's exact */
    int i;

    if (Vnode == 0) {
	   /* As usual, node 0 is in charge of allocation */
	   exact->nprocs = ProcsInUse();
	   exact->rotation = (short *)AllocGlobal(sizeof(short));
	   *(exact->rotation) = exact->nprocs;
	   exact->others = (struct exactdata **)
		AllocLocal(exact->nprocs * sizeof(struct exactdata *));

	   /* Send this pointer to the others */
	   rpd->inode_ptr->prefetch.private = (ANYPTR)exact;

	   /* Now fill in our pointer */
	   exact->others[0] = exact;

	   /* Indicate that we are ready, and wait for others */
	   Atomic_add(exact->rotation, -1);
	   while(*(exact->rotation) > 0)
		UsWait(10);
    } else {
	   exact->nprocs = ProcsInUse();
	   exact->others = (struct exactdata **)
		AllocLocal(exact->nprocs * sizeof(struct exactdata *));

	   /* Wait for Vnode 0 to allocate and initialize exact */
	   gexact = (struct exactdata *)rpd->inode_ptr->prefetch.private;
	   while (gexact == NULL) {
		  UsWait(20);
		  gexact = (struct exactdata *)rpd->inode_ptr->prefetch.private;
	   }

	   /* Get a copy of rotation* and fill in our entry in others[] */
	   exact->rotation = gexact->rotation;
	   gexact->others[Vnode] = exact;

	   /* Indicate that we are ready, and wait for others */
	   Atomic_add(exact->rotation, -1);
	   while(*(exact->rotation) > 0)
		UsWait(10);

	   /* Now copy the others[] array */
	   btransfer(gexact->others, exact->others, 
			   exact->nprocs * sizeof(struct exactdata *));
    }
 
    /* And touch all the patterns */
    for (i = 0; i < exact->nprocs; i++)
	 TouchPattern(exact->others[i]->refs);
}
#endif /* FOR_OTHERS */


/* @SUBTITLE "BuildBlockList: build the list of blocks" */
static void
BuildBlockList(rpd, exact)
	RAPIDFILE *rpd;
	struct exactdata *exact;
{
    REFS *refs = exact->refs;
    int *blocks;			/* the block list */
    short *index;			/* the chunk-index to block list */
    int nblocks;
    int nchunks = refs->nchunks;
    int block;
    int chunk;
    int pos;
    int prevblock = -1;
    int lastblock;
    ONEREF *ref;
    extern double ceil();

    nblocks = nchunks * ceil((float)(refs->chunksize) / rpd->sector_size);
    if (exact->global) {
	   blocks = (int *) AllocGlobal(nblocks * sizeof(int));
	   index = (short *)AllocGlobal(nchunks * sizeof(short));
    } else {
	   blocks = (int *) AllocLocal(nblocks * sizeof(int));
	   index = (short *)AllocLocal(nchunks * sizeof(short));
    }

    pos = -1;
    for (chunk = 0, ref = refs->chunks; chunk < nchunks; chunk++, ref++) {
	   /* find first block number of this chunk */
	   block = RT_SectorNum(rpd, ref->offset);

	   /* add it to the list if different from previous block */
	   if (block != prevblock)
		blocks[++pos] = block;

	   /* record the position of the first block of this chunk in the index */
	   index[chunk] = (short) pos;

	   /* now do any other blocks in this chunk */
	   lastblock = RT_SectorNum(rpd, ref->offset + ref->length - 1);
	   for (block++; block <= lastblock; block++)
		blocks[++pos] = block;

	   /* remember the last block of this chunk */
	   prevblock = lastblock;
    }
    
    /* store away the information */
    exact->blocks = blocks;
    exact->index = index;
    exact->nblocks = pos+1;
    exact->nextblock = 0;
}


/* @SUBTITLE "LateNotify: Multiple notifications" */
static void
LateNotify(node, rpd, private, last, length, sector)
	int node;				/* processor number */
	RAPIDFILE *rpd;		/* may not be rpd for above node */
	ANYPTR private;		/* private data */
	int last;				/* last block accessed in run */
	int length;			/* length of consecutive run */
	int sector;			/* most recent sector accessed (-1 if none) */
{
    struct exactdata *exact = (struct exactdata *)private;
    REFS *refs = exact->refs;
    int chunk;
    int block;

#ifdef TRACE
    printf("Proc %d (%d) in EXACT LateNotify: last=%d, length=%d, sector=%d\n",
		 Vnode, node, last, length, sector);
    fflush(stdout);
#endif

    /* Unfortunately, this is the best we can do. For global patterns, 
	* we do not have the associated chunk number for each block number.
	* For local patterns, we cannot determine the current chunk number
	* of 'node' since we may not be that node, and we cannot read its
	* memory. So we cannot use the information passed to us. This is
	* a crude approximation of where to begin prefetching.
	*/

    if (exact->global) {
	   LOCK(&(exact->lock), 10);
	   chunk = refs->next;
	   exact->nextblock = exact->index[chunk];
	   UNLOCK(&(exact->lock));
    } else {
	   chunk = refs->next;
	   exact->nextblock = exact->index[chunk];
    }

    rpd->prefetch.available = TRUE;
}

/* @SUBTITLE "Notify: notification that a sector is accessed" */
static boolean
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct exactdata *exact = (struct exactdata *)(rpd->prefetch.private);
    REFS *refs = exact->refs;
    int chunk = *(refs->readingchunk); /* get current chunk number */
    boolean wanted = FALSE;
    int pos;
    int firstblock;

    /* we make all predictions correctly, and make no mistakes */
    my_stats->correct++;

    /* We know the chunk and block. We find it in the blocks array,
	* and mark it as used. We do this by making the block number
	* negative. 
	*/
    pos = exact->index[chunk];
    firstblock = exact->blocks[pos]; /* first block in chunk */
    pos += sector - abs(firstblock); /* correct block in chunk */
#ifdef WANTED
    /* This code does not work, first because Atomic_swap_long does not
	* exist, and second because if sector is 0 we cannot tell when it has
	* been referenced. See notebook 6/19/90. 
	*/
    firstblock = Atomic_swap_long(&(exact->blocks[pos]), -sector);
    wanted = (firstblock >= 0 && exact->nextblock > pos);
#else
    exact->blocks[pos] = -sector;
#endif WANTED

    return(wanted);
}

/* @SUBTITLE "CheckAvailable: is there work available?" */
/* Are there any sectors to be prefetched? */

static boolean				/* TRUE if there's work */
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
    struct exactdata *exact = (struct exactdata *)(rpd->prefetch.private);
    REFS *refs = exact->refs;
    boolean available;

#ifdef FOR_OTHERS
    if (!exact->global)
	 return(TRUE);
#endif /* FOR_OTHERS */

    if (rpd->prefetch.available) {
	   available = (exact->nextblock < exact->nblocks);
	   rpd->prefetch.available = available;
	   return(available);
    } else
	 return(FALSE);		/* we never re-evaluate */
}

/* @SUBTITLE "GetWork: get a sector to prefetch" */
/* Decide on a sector to be prefetched right now. 
 * This is determined by the info in exact* and refs*
 */
#define NORMAL_LEAD			/* ie, use normal prefetch-lead algorithm */

static boolean				/* TRUE if sector was found */
GetWork(rpd, sector)
	RAPIDFILE *rpd;
	int *sector;	/* returned */
{
    struct exactdata *exact = (struct exactdata *)(rpd->prefetch.private);
    REFS *refs = exact->refs;
    boolean success;
    int chunk, block;		/* the prefetch choice */
    int maybe;				/* a possible chunk number */
    int nchunks;			/* number of chunks in pattern */
    int *blocks;			/* the list of blocks */
    int limit;				/* highest pos is allowed to go */
    short pos;				/* position in block list */

#ifdef FOR_OTHERS
    int list;
    extern short Atomic_add_cyclic();
    /* Local patterns will prefetch for others, by rotating among
	* the reference lists. So here we choose a reference list,
	* if there is more than one.
	*/
    list = Atomic_add_cyclic(exact->rotation, (short)1,
					    (short) (exact->nprocs));
    exact = exact->others[list]; /* exact is now remote */
    refs = exact->refs;		/* refs is now remote */

    if (exact->nextblock >= exact->nblocks)
	 return(FALSE);
    LOCK(&(exact->lock), 5);
    if (exact->nextblock >= exact->nblocks) {
	   UNLOCK(&(exact->lock));
	   return(FALSE);
    }
#else
    if (exact->global)
	 LOCK(&(exact->lock), 10);
#endif FOR_OTHERS

    nchunks = (int) refs->nchunks;
    blocks = exact->blocks;
    limit = exact->nblocks;

    if (refs->portion_limit) {
	   /* chunk number of first disallowed chunk */
	   chunk = refs->cur_portion_end + 1;
	   if (chunk < nchunks)
		limit = exact->index[chunk];
    }
 
#ifdef NORMAL_LEAD
    /* this is the normal algorithm */
    /* Maybe lead a little beyond current pointer */
    if (rpd->prefetch.lead) {
	   maybe = refs->next + rpd->prefetch.lead;
	   if (maybe < nchunks &&
		  !(refs->portion_limit && maybe > refs->cur_portion_end)) {
		  pos = exact->index[maybe];
		  if (pos > exact->nextblock)
		    exact->nextblock = pos;
	   }
    }
#else
    /* a hack for some experiments */
    /* limit the distance we can prefetch */
    chunk = refs->next + rpd->prefetch.lead;
    if (chunk < nchunks) {
	   pos = exact->index[chunk];
	   if (pos < limit) 
		limit = pos;
    }
#endif NORMAL_LEAD

    /* Ok! now that we've handled special cases, loop until 
	* we find a block that has not been referenced.
	*/
    pos = exact->nextblock;
    success = FALSE;
    while (!success && pos < limit) {
	   /* pick a block to prefetch */
	   block = blocks[pos];
	   if (block >= 0) {
		  *sector = block;
		  success = TRUE;
	   }
	   pos++;				/* try next block */
    }

    /* possibilities for pos at this point:
	* a) pos initially >= limit: nextblock not changed
	* b) success==TRUE and *sector==pos-1: nextblock becomes pos
	* c) success==FALSE and pos==limit: nextblock becomes limit
	*/
    exact->nextblock = pos;

#ifdef FOR_OTHERS
    UNLOCK(&(exact->lock));
#else
    if (exact->global)
	 UNLOCK(&(exact->lock));
#endif

    return(success);
}

/* @SUBTITLE "GiveBackWork: Sector could not be prefetched" */
/*  Note that this function never gets called, because FindFrame never 
 * fails to find a ready frame, because EXACT never makes mistakes.
 * Thus, we leave it empty.
 */

static void
GiveBackWork(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
}

/* @SUBTITLE "Done: clean up" */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    struct exactdata *exact = (struct exactdata *)(rpd->prefetch.private);

#ifdef FOR_OTHERS
    if (!exact->global) {
	   UsFree(exact->others);
	   if (Vnode == 0)
		UsFree(exact->rotation);
    }
#endif

    if (!exact->global || Vnode == 0) {
	   UsFree(exact->blocks);
	   UsFree(exact->index);
	   UsFree(exact);
    }
    
    rpd->prefetch.available = FALSE;
}

/* @SUBTITLE "Dump: debugging dump" */
static void
Dump(rpd)
	RAPIDFILE *rpd;
{
    struct exactdata *exact = (struct exactdata *)(rpd->prefetch.private);
    REFS *refs;

    printf("  EXACT Predictor:\n");
    if (exact != NULL) {
	   refs = exact->refs;

	   if (refs != NULL ) {
		  printf("   refs.global = %s\n", refs->global ? "TRUE" : "FALSE");
		  printf("   refs.chunks = 0x%x\n", refs->chunks);
		  printf("   refs.nchunks = %d\n", refs->nchunks);
		  printf("   refs.chunksize = %d\n", refs->chunksize);
		  printf("   refs.portions = 0x%x\n", refs->portions);
		  printf("   refs.nportions = %d\n", refs->nportions);
		  printf("   refs.next = %d\n", refs->next);
		  printf("   refs.readingchunk = 0x%x->(%d)\n", refs->readingchunk, 
			    refs->readingchunk ? *(refs->readingchunk) : 0);
		  printf("   refs.cur_portion_end = %d\n", refs->cur_portion_end);
		  printf("   refs.portion_limit = %s\n", 
			    refs->portion_limit ? "TRUE" : "FALSE");
	   } else
		printf("   NULL refs");

	   printf("   global = %s\n",
			exact->global ? "TRUE" : "FALSE");
	   printf("   lock = %s\n",
			exact->lock ? "SET" : "CLEAR");
	   printf("   blocks = 0x%x\n", exact->blocks);
	   printf("   index = 0x%x\n", exact->index);
	   printf("   nblocks = %d\n", exact->nblocks);
	   printf("   nextblock = %d\n", exact->nextblock);
	   printf("   rotation = 0x%x->(%d)\n",
			exact->rotation, exact->rotation ? *(exact->rotation) : 0);
	   printf("   nprocs = %d\n", exact->nprocs);
	   printf("   others = 0x%x\n", exact->others);
    } else 
	 printf("   NULL exactdata");
    
}
