/* @TITLE "frame.c - manage frame buffers" */
/* frame.c:	Support for sector manipulation. 
 *			Includes the following: 
 * 
 *			UseSector() 
 *			DoneSector() 
 * 
 *		Additionally, the following support functions are included 
 *		in this file. 
 * 
 *             BufferHit() 
 *             DemandFetch() 
 *			SwapSectorOut() 
 *             WriteBack()
 * 
 * The protocols used here are carefully designed to work together to 
 * provide exactly the right mutual exclusion properties and are very 
 * heavily based on the semantics of the problem. Be very careful 
 * if modifying any of these routines. 
 */

static char rcsid[] = "$Id: frame.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

/* INCLUDES */

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "replace.h"
#include "prefetch.h"
#include "rapidelog.h"
#include "stats.h"
#include "error.h"
#include "vnode.h"
#include "own.h"
#include "wanted.h"
#include "queue.h"
#include "predict-debug.h"

/* DEFINES */
	
/* #define DEBUG_FLAG */		/* verbose output */
	
/* LOCAL FUNCTIONS */
	
static SECTOR NoCaching();
static boolean BufferHit();
static boolean DemandFetch();
static TICS WaitForIO();

/* @SUBTITLE "UseSector: before using a sector" */
SECTOR
UseSector(rpd, sector, writer, fullpagewrite)
	RAPIDFILE *rpd;
	int sector;
	boolean writer;
	boolean fullpagewrite;
{
    SECTOR_MAP_ENTRY *sme;
    boolean callB;
    boolean valid = FALSE;
    TICS t_notify;
    boolean wanted = FALSE;
    TICS delay;

    sme = RT_smeForSector(rpd, sector);
    
    /* Reading a zero page doesn't require any work here */
    if (!writer && SS_ISCLEAR(sme, SS_NONZERO))
	 return(0);			/* reader knows what to do with this */
    
#ifdef DEBUGPREDICT
    printf("%d using sector %d\n", UsProc_Node, sector);
#endif DEBUGPREDICT

    /* Direct technique for non-caching tests */
    if (!rpd->caching)
	 return(NoCaching(rpd, sector, writer, fullpagewrite, sme));
    
    if (!writer) {
	   /* Notify the prediction algorithm of this use */
	   t_notify = rtc;			/* measure predict notify time */
	   wanted = NotifyPredict(rpd, sector);
	   t_notify = rtc - t_notify;
	   ELOG_LOG(RTELOG_PREDICT, t_notify);
	   my_stats->notify += t_notify;
    }

    /* and notify the replacement algorithm */
    callB = RT_NotifyReplaceA(rpd, sector);
    
    while(!valid) {			/* Until we have a valid object */
	   /* Avoid any major activity in this sector */
	   while (IsChangingSector(sme))
		UsWait(100);
	   
	   /* Is there a valid sector object there? */
	   if (SS_ISSET(sme, SS_VALID)) {
		  valid = BufferHit(rpd, sector, sme, writer);
		  if (valid) {
			 delay = WaitForIO(rpd, sme, rpd->prefetch.bufhit);
			 if (delay > 0) {
				my_stats->pfbufhit_delay += delay;
				my_stats->prefetchbufhit++;
			 }
		  }
	   } else {
		  valid = DemandFetch(rpd, sector, sme, writer, fullpagewrite);
		  if (valid) {
			 delay = WaitForIO(rpd, sme, rpd->prefetch.demand);
			 if (delay > 0) {
				my_stats->pfdemand_delay += delay;
				my_stats->prefetchdemand++;
			 }
		  }
	   }
    }
    
    /* Notify the replacement algorithm of this use */
    if (callB)
	 RT_NotifyReplaceB(rpd, sector);
    
#ifdef WANTED
    if (wanted)
	 /* remove our interest in the sector, since we now have it */
	 Atomic_add(&(sme->misc.want_count), -1);
#endif    

    return(sme->sector_ptr);
}

/* @SUBTITLE "NoCaching: process a sector when not caching" */
static SECTOR
NoCaching(rpd, sector, writer, fullpagewrite, sme)
	RAPIDFILE *rpd;
	int sector;
	boolean writer;
	boolean fullpagewrite;
	SECTOR_MAP_ENTRY *sme;
{
    /* We have our own reserved buffer */
    SECTOR buf = rpd->buffer;

    /* We need to supply a buffer for them to use */
    if (!(writer && fullpagewrite) && SS_ISSET(sme, SS_NONZERO)) {
	   io_readwait(buf, sector * rpd->sector_size, rpd->sector_size);
	   my_stats->reads++;
    }
    return(buf);
}


/* @SUBTITLE "BufferHit: Process a buffer hit" */
static boolean
BufferHit(rpd, sector, sme, writer)
	RAPIDFILE *rpd;
	unsigned int sector;
	SECTOR_MAP_ENTRY *sme;
	boolean writer;
{
    long hitwait;
    short oldstat;

#ifdef DEBUG_FLAG
    printf("BufferHit: hit sector %d in frame %d\n",
		 sector, sme->sector_frame);
#endif DEBUG_FLAG
    
    Atomic_add(&(sme->use_count), 1); /* we're using it */
    
    /* Still valid and unlocked? */
    if (SS_ISCLEAR(sme, SS_VALID) || IsChangingSector(sme)) {
	   /* oops, it's gonna be swapped out. Undo the above. */
	   Atomic_add(&(sme->use_count), -1);
	   return(FALSE);
    }
    
    /* We got it! Now anyone who wants to swap it out must wait for us. */
    
#ifdef OWN_BUFFER
    /* If this is the first use of a prefetched page that we own */
    /* clear that status and decrement counter */
    if (SS_ISSET(sme, SS_PREFETCHED) && sme->misc.owner == Vnode) {
	   oldstat = SS_CLEAR(sme, SS_PREFETCHED);
	   my_stats->prefetch_used++;
	   ELOG_LOG(RTELOG_PREFETCH_USED, sector);
	   if (!(oldstat & SS_MISTAKE))
		Atomic_add(&(rpd->prefetch.unused), -1);
    } else
	 my_stats->bufferhit++;
#else
    /* If this is the first use of a prefetched page */
    /* clear that status and decrement counter */
    oldstat = SS_CLEAR(sme, SS_PREFETCHED);
    if ((oldstat & SS_PREFETCHED) == SS_PREFETCHED) {
	   my_stats->prefetch_used++;
	   ELOG_LOG(RTELOG_PREFETCH_USED, sector);
	   if (!(oldstat & SS_MISTAKE))
		Atomic_add(&(rpd->inode_ptr->prefetch.unused), -1);
    } else
	 my_stats->bufferhit++;
#endif OWN_BUFFER
    
    /* Is the sector ready to be used? */
    hitwait = sme->whenready - rtc;
    ELOG_LOG(RTELOG_HIT, hitwait);
    if (hitwait > 0) {
	   /* no - this is an unready hit */
	   my_stats->unready++;
	   my_stats->hitwait += hitwait;
    }

    return (TRUE);
}

/* @SUBTITLE "DemandFetch: Process a demand fetch (buffer miss)" */
static boolean
DemandFetch(rpd, sector, sme, writer, fullpagewrite)
	RAPIDFILE *rpd;
	unsigned int sector;
	SECTOR_MAP_ENTRY *sme;
	boolean writer;
	boolean fullpagewrite;
{
    boolean wasnew;

#ifdef DEBUG_FLAG
    printf("DemandFetch: miss sector %d\n", sector);
#endif DEBUG_FLAG
    
    /* If someone else already is fetching it then bag it. */
    if (!TryChangeLockSector(sme))
	 return(FALSE);
    
    /* Became valid in the interim? */
    if (SS_ISSET(sme, SS_VALID)) {
	   ChangeUnlockSector(sme);
	   return(FALSE);
    }
    
    /* Statistics for this fetch */
    ELOG_LOG(RTELOG_DEMAND_FETCH, sector);
    my_stats->demand++;
    
    /* We get to do the fetch. */
    /* First, find a frame, possibly emptying it */
    /* guaranteed to return with a frame */
    (void) RT_FindFrame(rpd, sector, sme, FALSE);
    
#ifdef DEBUG_FLAG
    printf("(%d) DemandFetch: fetching sector %d into frame %d (sme status 0x%x)\n",
		 UsProc_Node, sector, sme->sector_frame, sme->sector_status);
#endif DEBUG_FLAG
    
    /* How do we fill this frame? With data, zeroes, or nothing? */
    wasnew = SS_ISCLEAR(sme, SS_NONZERO);
    if (!fullpagewrite)
	 if (wasnew) {
		bzero(sme->sector_ptr, (int)rpd->sector_size);
	 } else {
		/* READ FROM DISK */
		sme->whenready = io_read(sme->sector_ptr,
							sector * rpd->sector_size,
							rpd->sector_size);
		my_stats->reads++;
	 }

    /* Use count indicates that we'll be using it. */
    sme->use_count = 1;
    
    /* Was this sector written in the past? this is a reread. */
    if (SS_CLEAR(sme, SS_WRITTEN) & SS_WRITTEN) 
	 my_stats->rereads++;

    /* Now set the status and unlock it */
    SS_SET(sme, SS_NONZERO | SS_VALID | (wasnew ? SS_NEW : 0));
    ChangeUnlockSector(sme); 
    
    return(TRUE);
}

/* @SUBTITLE "WaitForIO: Wait for I/O to complete on this sme */
/* We optionally do prefetching while we wait. */
/* But we only wait for reads. */
static TICS				/* amount of delay */
WaitForIO(rpd, sme, prefetch)
	RAPIDFILE *rpd;
	SECTOR_MAP_ENTRY *sme;
	boolean prefetch;		/* can we do prefetching? */
{
    TICS ready = sme->whenready;
    TICS last;
    
    /* Do nothing unless there is an outstanding read I/O */
    if (rtc < ready && SS_ISCLEAR(sme, SS_WRITTEN)) {
	   ELOG_LOG(RTELOG_IDLE, ready - rtc);
	   while (rtc < ready)
		if (MayBePrefetchWork(rpd) && prefetch 
		    && ready-rtc >= rpd->prefetch.mintime) {
		    last = rtc;
		    RT_Prefetch();
		    /*		    ELOG_LOG(RTELOG_PREFTIME, rtc-last); */
		}
	   
	   /* 
	   if (prefetch) {
		  ELOG_LOG(RTELOG_LASTLEN, rtc-last);
		  ELOG_LOG(RTELOG_LASTOK, ready-last);
	   }
	   */

	   return(rtc - ready);
    } else
	 return(0);
}

/* @SUBTITLE "DoneSector: Caller is done with sector" */
void
DoneSector(rpd, sector, nbytes_written)
	RAPIDFILE *rpd;
	int sector;
	int nbytes_written;
{
    SECTOR_MAP_ENTRY *sme = RT_smeForSector(rpd, sector);
    boolean isfull;			/* is the sector completely written? */
    int fullness;

    if (rpd->caching) {
	   if (nbytes_written > 0) {
		  SS_SET(sme, SS_DIRTY); /* something was written */
		  /* Check to see if it became full */
		  if (sme->misc.written < rpd->sector_size) {
			 fullness = Atomic_add(&(sme->misc.written), nbytes_written) 
			   + nbytes_written;
			 isfull = (fullness == rpd->sector_size);
		  } else
		    isfull = FALSE;
		  if (rpd->write.thru
			 || (isfull && rpd->write.full)
			 || (isfull && rpd->write.newfull && SS_ISSET(sme, SS_NEW))) 
		    WriteBack(rpd, sme, sector);
	   }
	   Atomic_add(&(sme->use_count), -1);
    } else {
	   /* we're not caching */
	   /* do I/O only if it was a write */
	   if (nbytes_written > 0) {
		  SS_SET(sme, SS_NONZERO);
		  io_writewait(rpd->buffer, 
					sector * rpd->sector_size, 
					rpd->sector_size);
		  my_stats->writes++;
	   }
    }
}

/* @SUBTITLE "SwapSectorOut: swap a sector out to disk" */
boolean					/* swap was successful? */
SwapSectorOut(rpd, frame, queue)
	RAPIDFILE *rpd;
	FRAME_ENTRY *frame;
	QH queue;				/* where to put it if it needs disk I/O */
{
    SECTOR_MAP_ENTRY *sme;
    int sector = frame->frame_sector;
    TICS ready;

    sme = RT_smeForSector(rpd, sector);
    
    ChangeLockSector(sme);
    
    /* Did someone sneak in and put it back in the WS? */
    /* Or, will they soon? */
    /* Give up, sigh. */
    if (sme->use_count > 0 || frame->ws_count > 0) {
	   ChangeUnlockSector(sme);
	   return(FALSE);
    }
    
#ifdef WANTED
    /* Is some process interested in this frame in the future? */
    /* Give up, sigh. */
    if (sme->misc.want_count > 0) {
	   frame->frame_status = FS_WANTED;
	   ChangeUnlockSector(sme);
	   return(FALSE);
    }
#endif

    if (SS_ISSET(sme, SS_VALID)) {
	   WriteBack(rpd, sme, sector); /* possible writeback of dirty frame */

	   /* there may be some outstanding disk I/O */
	   /* we won't wait for disk I/O on this sector to complete */
	   if (rtc < sme->whenready) {
		  if (queue && frame->frame_status != FS_OLDFRAME) {
			 LockFrame(frame);
			 if (rtc < sme->whenready 
				&& frame->frame_status != FS_OLDFRAME) {
				Enqueue(queue, sme->sector_frame, 0);
				frame->frame_status = FS_OLDFRAME;
			 }
			 UnlockFrame(frame);
		  }
		  ChangeUnlockSector(sme);
		  return(FALSE);
	   }
    }
	   
    ELOG_LOG(RTELOG_SWAPOUT, sector);
    
    /* if it was marked as a mistake */
    if (SS_ISSET(sme, SS_MISTAKE))
	 /* if never used, it was a waste */
	 if (SS_ISSET(sme, SS_PREFETCHED)) {
		my_stats->wasted++;
#ifdef DEBUGPREDICT
		printf("%d wasted %d\n", UsProc_Node, sector);
#endif DEBUGPREDICT
	 } else {
		/* if used, it was a bad move to mark as mistake */
		my_stats->usedmistake++;
#ifdef DEBUGPREDICT
		printf("%d used mistake  %d\n", UsProc_Node, sector);
#endif DEBUGPREDICT
	 }
    
    /* Note that it is no longer valid. */
    /* DIRTY is cleared in WriteBack */
    /* This also unlocks the sector, simultaneously. */
    SS_CLEAR(sme, SS_VALID | SS_PREFETCHED | SS_CHANGING 
		   | SS_NEW | SS_MISTAKE);
    /* ChangeUnlockSector(sme); */

    /* These are not strictly necessary but are nice 
	*    sme->sector_ptr = 0; 
	*    sme->sector_frame = -1; 
	*/
    
#ifdef DEBUG_FLAG
    printf("SwapSectorOut: Swapped sector %d out\n", sector);
#endif DEBUG_FLAG
    
    frame->frame_sector = -1;
    
    return(TRUE);
}


/* @SUBTITLE "WriteBack: write a sector back to disk" */
/* concurrency issues are the responsibility of the caller */
void
WriteBack(rpd, sme, sector)
	RAPIDFILE *rpd;
	SECTOR_MAP_ENTRY *sme;
	int sector;
{
    int bytes;

    if (SS_CLEAR(sme, SS_DIRTY) & SS_DIRTY) {
	   /* Was dirty, and now marked clean */

	   /* Writes into last sector might be partial writes */
	   if (sector+1 == RT_NumSectors(rpd, rpd->size))
		bytes = rpd->size % rpd->sector_size;
	   else
		bytes = rpd->sector_size;

	   if (SS_SET(sme, SS_WRITTEN) & SS_WRITTEN) 
		my_stats->rewrites++;

	   ELOG_LOG(RTELOG_WRITEBACK, sector);
	   sme->whenready = io_write(sme->sector_ptr,
					sector * rpd->sector_size,
					bytes);
	   my_stats->writes++;

	   /* we don't clear SS_NEW here; it will be cleared at swapout time */
    } 
}
