/* @TITLE "synch.c - synchronization" */
/*
 * Synchronize - A reusable synchronization point. Synchronizes n processes
 * and is reset to be used again.
 *
 * (For Uniform system) 
 *
 * David Kotz March 1988
 */

static char rcsid[] = "$Id: synch.c,v 7.1 91/05/09 19:34:39 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include <rapidelog.h>
#include "synch.h"

/* A pause between reads of a shared variable appropriate for n processors 
 * Should be more than enough for at least up to 64 processors 
 */
#define Pause(n) UsWaitSpin(((n)*5)/10)
  
/* Describes a synchronization variable */
struct Synch_struct {
    short entry;
    short exit;
    short nprocs;
    short filler;			/* for longword alignment */
    /* Statistics: */
    TICS lastdone;	/* time that last one finished */
    TICS firstdone;	/* time that first one finished */
    TICS maxdelay;	/* max synch time delay */
    TICS totaldelay;	/* total synch time delay */
    unsigned short count;	/* number of synchs */
    unsigned short countp;	/* number of proc-synchs */
};

/* @SUBTITLE "UsMakeSynch: Make a synchronization variable" */
SYNCH *
UsMakeSynch(n)		/* for n processors */
	int n;
{
    SYNCH *s = NULL;
    
    if (n > 0) {
	   s = (SYNCH *)AllocGlobal(sizeof(SYNCH));
	   if (s != NULL) {
		  SetSynch(s, n);
	   }
    }
    return(s);
}

/* @SUBTITLE "Set the synch variable for n processors" */
/* Set the synch s for n processors (sequential) */
void 
SetSynch(s, n)
	SYNCH *s;
	int n;
{
    if (n > 0)
	 s->entry = s->exit = s->nprocs = n;
    s->totaldelay = s->maxdelay = s->lastdone = 0;
    s->count = s->countp = 0;
}

/* @SUBTITLE "SynchronizeOut: Count me out" */
/* Count me out of future synchronization points. Don't make me wait now. */
void 
SynchronizeOut(s)
	SYNCH *s;
{
    /* If anyone is in SynchronizeFull they will be waiting on s->entry, */
    /* so we can safely adjust nprocs and exit without problems. */
    /* Once we change s->entry they might proceed into the exit loop, */
    /* so we must do that last. */
    Atomic_add(&(s->nprocs), -1);
    Atomic_add(&(s->exit), -1);
    Atomic_add(&(s->entry), -1);
}

/* @SUBTITLE "SynchronizeFull" */
/* Synchronize a number of processes. Call the worker function
 * if we are waiting for others to come in still.
 * Call the reset function ONCE during the sychronized period of time.
 */
void 
SynchronizeFull(s, worker, worker_arg, reset, reset_arg)
	SYNCH *s;
	void (*worker)();
	int worker_arg;
	void (*reset)();
	int reset_arg;
{
    int n = s->nprocs;
    TICS timeout;
    TICS timein;
    TICS last;
    int old;
    
    s->firstdone = 0;
    
    Atomic_add(&(s->entry), -1);
    
    timein = rtc;
    
    /* Wait for everybody to come into the procedure */
    if (worker) {
	   while (s->entry > 0) {
		  last = rtc;
		  (*worker)(worker_arg);
		  /*		ELOG_LOG(RTELOG_PREFTIME, rtc-last); */
	   }
    }
    else
	 while (s->entry > 0)
	   Pause(n);
    
    timeout = rtc;
    
    /* Wait for everybody to complete the above while loop */
    /* Last person to atomic_add will reset the variables */
    /* and everyone else must wait until this is done. */
    old = Atomic_add(&(s->exit), -1);
    if (old == s->nprocs)	/* first one done */
	 s->firstdone = timeout;
    
    ELOG_LOG(RTELOG_IDLE, timeout-timein);
/*
    if (worker) {
	   ELOG_LOG(RTELOG_LASTLEN, timeout-last);
	   ELOG_LOG(RTELOG_LASTOK, s->firstdone-last);
    }
*/
    
    if (old - 1 == 0) { /* last one done? */
	   timeout = rtc;		/* get a new number, sure to be highest */
	   if (reset) (*reset)(reset_arg);
	   while (s->firstdone == 0)
		Pause(n);
	   s->maxdelay += timeout - s->firstdone;
	   s->count++;
	   s->lastdone = timeout;
	   s->countp += s->nprocs;
	   s->exit = s->nprocs;
	   s->entry = s->nprocs;
    } else {
	   while(s->entry == 0)
		Pause(n);
	   Atomic_add_long(&(s->totaldelay), s->lastdone - timeout);
    }
}

/* @SUBTITLE "SynchTotal/SynchMax: statistics about synchronization" */
/* return the avg delay time taken and clear it */
float
SynchTotal(s)
	SYNCH *s;
{
    int count = s->countp;
    float time = (float) (s->totaldelay) / (float) count;
    
    s->totaldelay = 0;
    s->countp = 0;
    return (time);
}

/* return the avg max delay time and clear it */
float
SynchMax(s)
	SYNCH *s;
{
    int count = s->count;
    float time = (float) (s->maxdelay) / (float) count;
    
    s->maxdelay = 0;
    s->count = 0;
    return (time);
}

/******************* TEST TEST TEST ******************************* 
 * #define TEST 
 */

#ifdef TEST
SYNCH *sv;				/* synch var */
int foo();

main()
{
    InitializeUs();
    
    sv = UsMakeSynch(ProcsInUse());
    Share(&sv);
    
    GenTaskForEachProc(foo, 0);
    UsFree(sv);
}

foo()
{
    srandom(GetRtc() + Proc_Node);
    
    UsWait(random() % 1000000);
    printf("Synch in... %d\n", UsProc_Node);
    Synchronize(sv);
    printf("synched out %d\n", UsProc_Node);
    
    UsWait(random() % 1000000);
    printf("Synch in... %d\n", UsProc_Node);
    Synchronize(sv);
    printf("synched out %d\n", UsProc_Node);
}

#endif TEST
