/* @TITLE "driver.c - simulate file access" */
/* driver.c - Super-Testbed for experiments with Rapid System.
 * 
 *  This can drive lots of different access patterns on a RAPIDFILE, and
 * is instrumented for gathering statistics.
 *
 * Usage: 
 * EITHER
 *    driver [-elog] testfilename [startline]
 *   where:
 *     testfilename is a file containing lines of the form below.
 *     startline indicates the line of the testfile to start executing;
 *      if a number, then it is used directly; otherwise it is interpreted
 *      as the name of a file containing the line number. In the latter
 *      case the file is updated as each line is completed, so that it
 *      may be used to restart the driver at the same point.
 * OR
 *  driver [-elog] 
 *         datadirectoryname
 *         patternfilename
 *         { { nocaching(N) newfile(y/n)}
 *          |{prefetch(y | Y)
 *            synchpoint(y/n) PFBufhit(y/n) PFDemand(y/n) lead limit min
 *            alg(none/exact/obl/ibl/port/adapt) algparm}
 *          |{prefetch(n)}
 *          |{dowrites(w) newfile(y/n) cachesize}
 *          }
 *         wss
 *         blocksize
 *         disks
 *         synchronization(each/total/portion/none/neighbor)  count
 *         timetest-args Ntrials
 *
 * Where [-elog] is an optional argument specifying the number of trials
 * [on each line] that have elogs dumped; -1 is the default, to dump only
 * trial 1. Use -0 to dump no logs. Use -10 to dump the first 10 trials, etc.
 * A -elog specified on the command line overrides any in the test file.
 *
 * Prefetch y will do a prefetch n with the same reference string
 * Prefetch Y will do only the prefetch, not the prefetch n baseline
 *
 * Files: *.data and *.trial.elog   for each test
 *        * is defined as:
 * pref.sync.whenwait.overlap.lead.lim.min.alg.parm.wss.bsize.disks.sync.cnt.N
 *   (N=number of procs)
 * David Kotz  March 1988
 */

static char rcsid[] = "$Id: driver.c,v 7.1 91/05/09 19:34:39 dfk Tape2 $"; 

/* @SUBTITLE "Includes" */

#include <stdio.h>
#include <usdfk.h>
#include <ctype.h>

#include "rapid.h"
#include "rapidelog.h"
#include "refs.h"
#include "pattern.h"
#include "stats.h"
#include "synch.h"
#include "driver.h"
#include "disktime.h"
#include "touch.h"

/* @SUBTITLE "Definitions" */

static void DoTest();
static void DoBothTests();
static void Init();
static void InitOnceEach();
static void InitforProc();
void SynchPoint();			/* used by body */
void SynchOut();			/* used by body */
extern void Final();		/* in final.c */

/* GLOBAL VARIABLES - Testbed only */
#define NAME_SIZE 80

char Directory_name[NAME_SIZE]; /* name of the Data directory */
char ElogData_name[NAME_SIZE]; /* base name of the files */
char DataFile_name[NAME_SIZE]; /* name of the data file */
char ElogFile_name[NAME_SIZE]; /* name of the elog file */
int ElogDumpLimit = 1;		/* normally only first trial */
boolean ElogDumpIgnore = FALSE; /* ignore any -elog args in driver file */

RAPIDFILE	*my_rpd;			/* Private pointer to the (open) */
						/*  RAPIDFILE descriptor */
char *my_buf;				/* Private buffer for chunks */

FILE *DataFile;			/* data file */

boolean Caching;			/* are we caching? */
boolean Prefetch;			/* are we prefetching? */
boolean Baseline;			/* Should we do a baseline n-prefetch test? */
boolean PrefetchAtSynch;		/* prefetch at synch points? */
boolean PrefetchBufHit;		/* prefetch while waiting for bufhit? */
boolean PrefetchDemand;		/* prefetch during demand fetch? */
int PrefetchLead;			/* prefetch lead (in chunks) */
int PrefetchLimit;			/* prefetch limit */
int PrefetchMin;			/* minimum time for calling RT_Prefetch */
short PrefetchAlg;			/* Prefetch algorithm code (from PFA_*) */
int PrefetchParm;			/* Prefetch algorithm parameter */
int NumFrames;				/* number of frames to allocate */
boolean DoingWrites;		/* are we doing a writable-file test? */
int WriteAlg;				/* writeback algorithm */
boolean FileIsNew;			/* is the file newly created? */

char *Pattern;				/* pattern filename (not shared) */
	
int wss;					/* local working-set size for WS alg */

int blocksize;				/* size of disk blocks */
int mapsize;				/* the size of a sector map (bytes) */
unsigned int filesize;		/* the size of the simulated file (bytes) */

int ndisks;				/* number of disks */
int disktime;				/* disk-access time in msec */

int SynchStyle;			/* synchronization style */
int SynchCount;			/* count mentioned in style */

int Nprocs;				/* number of processors */
int Ntrials;				/* number of trials */
int trial;				/* trial number (not shared) */
	
SYNCH *SynchCounter;		/* synchronization variable */

/* @SUBTITLE "main" */
main(argc, argv)
	int argc;
	char **argv;
{
    char *testfilename;
    FILE *testfile;
    char *linefilename = NULL;
    FILE *linefile;
    int StartLine;
    char newlinefilename[NAME_SIZE+10];

    printf("\nRAPID DRIVER PROGRAM\n");
    printf("%s\n", rcsid);

    /* Initialize the Uniform system. */
    
    InitializeUs ();
    
    /* Arrange for signals to dump elogs and stdout, and exit */
    SignalInit();

    /* Initialize the Rapid system */
    
    InitializeRAPIDSystem();
    
    /* Allocate some global shared variables */
    SynchCounter = UsMakeSynch(1);
    Share(&SynchCounter);
    AllocateShare(SynchTime, TICS, 0);
    AllocateShare(Nsynchs, short, 0);
    
    /* Initialize the processors */
    GenTaskForEachProc(InitOnceEach, 0);
    
    /* Initialize Body */
    BodyInit();
    
    /* Now do the tests, depending on the arguments */
    if (argc == 2 || argc == 3 || argc == 4) { /* 1, 2, or 3 arguments */
	   /* handle any optional [-elog] argument */
	   if (OptionalArg(argc-1, argv+1)) {
		  /* don't allow any more changes to ElogDumpLimit */
		  ElogDumpIgnore = TRUE;
		  argc--;
		  argv++;
	   }

	   /* We assume the one argument is the name of a file
	    * which contains runs of this program. So we get our
	    * arguments from the file instead. The optional second
	    * argument is either the line number in the file to start with
	    * or the name of a file containing the line number. In the latter
	    * case we also update that file with the line number, as we
	    * complete the lines.
	    */
	   testfilename = *++argv;
	   if ((testfile = fopen(testfilename, "r")) == (FILE *)NULL) {
		  printf("Cannot open test file '%s'\n", testfilename);
		  exit(0);
	   }
	   
	   if (argc >= 3) {
		  ++argv;
		  if (isdigit(**argv)) {
			 StartLine = atoi(*argv);
			 linefilename = NULL;
		  } else {
			 StartLine = 1; /* the default if no file or empty file */
			 linefilename = *argv;
			 sprintf(newlinefilename, "%s.new", linefilename);
			 if ((linefile = fopen(linefilename, "r")) != (FILE *)NULL) {
				fscanf(linefile, "%d", &StartLine);
				fclose(linefile);
			 }
		  }
		  if (StartLine <= 0) {
			 printf("Start line should be > 0 (%s)\n", *argv);
			 exit(0);
		  }
		  if (StartLine > 1) {
			 printf("Skipping lines 1-%d in file %s\n",
				   StartLine-1, testfilename);
			 while (InputLine < StartLine-1)
			   if (!GetArgsFromFile(testfile, TRUE)) {
				  printf("Only %d lines in test file.\n", InputLine);
				  exit(1);
			   }
		  }
	   }
	   
	   while (GetArgsFromFile(testfile, FALSE))
		if (ParseArgs(0, 0)) {
		    printf("Test on line %d starting.\n", InputLine);
		    DoBothTests();
		    printf("Test on line %d complete.\n", InputLine);
		    if (linefilename)
			 if ((linefile = fopen(newlinefilename, "w")) != (FILE *)NULL) {
				fprintf(linefile, "%d\n", InputLine+1);
				fclose(linefile);
				rename(newlinefilename, linefilename);
				sync();
			 }
		} else {
		    printf("\nSTOPPING PREMATURELY due to error.\n");
		    break;
		}
	   
	   fclose(testfile);
	   printf("Test file %s complete.\n", testfilename);
    } else {
	   /* The arguments are on the command line. We extract them */
	   /* from there and do one test. */
	   if (ParseArgs(argc, argv))
		DoBothTests();
    }
}

/* @SUBTITLE "DoBothTests: a matched pair of tests" */
static void
DoBothTests()
{
    /* get the pattern loaded into memory */
    if (!LoadPattern(Pattern, &filesize)) {
	   printf("Cannot load pattern file '%s'\n", Pattern);
	   return;
    }
    GenTaskForEachProc(TouchPattern, 0);
    
    if (Caching && !DoingWrites)
	 /* Compute the total number of frames, once  */
	 NumFrames = wss*ProcsInUse() + PrefetchLimit;

    /* Do the version asked for by the user */
    DoTest();
    
    /* If it was a prefetch test, now run a corresponding non-prefetch */
    if (Prefetch && Baseline) {
	   printf("\nBASE CASE test for line %d\n", InputLine);
	   
	   Prefetch = PrefetchAtSynch = PrefetchBufHit = PrefetchDemand = FALSE;
	   PrefetchLead = PrefetchLimit = PrefetchMin = 0;
	   PrefetchAlg = PFA_NONE;
	   PrefetchParm = 0;

	   Share(&Prefetch);
	   Share(&PrefetchAtSynch);
	   Share(&PrefetchBufHit);
	   Share(&PrefetchDemand);
	   Share(&PrefetchLead);
	   Share(&PrefetchLimit);
	   Share(&PrefetchMin);
	   Share(&PrefetchAlg);
	   Share(&PrefetchParm);
	   
	   DoTest();
    }
}

/* @SUBTITLE "DoTest: Perform one test" */
char PrefetchAlgCodes[] = "neoipaOPHgrws";
char WriteAlgCodes[] = "btlfN";
char SynchCodes[] = "xetpa";

static void
DoTest()
{
    char front[50];

    if (Nprocs != ProcsInUse()) {
	   printf("\nNot enough processors available for this test.\n");
	   return;
    }
    
    /* Open data file */
    if (Caching && !DoingWrites)
	 sprintf(front, "%c.%c.%c.%c.%02d.%03d.%03d.%c.%02d", 
		  Prefetch ? 'y' : 'n', PrefetchAtSynch ? 'y' : 'n',
		  PrefetchBufHit ? 'y' : 'n', PrefetchDemand ? 'y' : 'n',
		  PrefetchLead, PrefetchLimit, PrefetchMin,
		  PrefetchAlgCodes[PrefetchAlg],
		  PrefetchParm);
    else if (DoingWrites)
	 sprintf(front, "w.%c.%03d.%c", 
		    FileIsNew ? 'y' : 'n', NumFrames, WriteAlgCodes[WriteAlg]);
    else /* non-caching */
	 sprintf(front, "N.%c", FileIsNew ? 'y' : 'n');

    sprintf(ElogData_name,
		  "%s/%s.%1d.%04d.%02d.%c.%03d.%02d",
		  Directory_name, front,
		  wss,
		  blocksize,
#ifdef DISKTIME_HACK
		  disktime,
#else
		  ndisks,
#endif
		  SynchCodes[SynchStyle],
		  SynchCount,
		  ProcsInUse());
    sprintf(DataFile_name, "%s.data", ElogData_name);
    printf("Data file name is %s\n", DataFile_name);
    DataFile = fopen(DataFile_name, "w");
    if (DataFile == NULL) {
	   printf("Could not open data file. Exiting...\n");
	   exit(0);
    }

    /* Proceed with the test. */
    for (trial = 1; trial <= Ntrials; trial++) {
	   /* each of these may generate parallel tasks */
	   Init();			/* init for test */
	   TotalTime = Body();	/* run the timed test */
	   Final();			/* cleanup, and write results */
	   fflush(stdout);		/* sync screen output */
    }

    fclose(DataFile);
}

/* @SUBTITLE "InitOnceEach: Initialize each processor ONCE per run." */
static void
InitOnceEach()
{
    TouchTextData();		/* make sure whole program is paged in */
}

/* @SUBTITLE "Init: initialize for THIS test" */
static void
Init()
{
    int replace_style;

    InitVnodes();			/* initialize a set of virtual node numbers */

    InitElogs();			/* initialize event logging */
    
    ResetPattern();			/* reset reference strings */

    *SynchTime = 0;
    *Nsynchs = 0;
    
    SetSynch(SynchCounter, ProcsInUse());
    
    /* set up I/O simulation */
    io_init_global(ndisks, blocksize, disktime, disktime);

    mapsize = 1024;			/* for now, this seems good */
    if (Caching)
	 replace_style = (wss == 1 ? RT_REPLACE_TOSS : RT_REPLACE_WS);
    else
	 replace_style = RT_REPLACE_NONE;

    /* "open" the file */
    RT_open(&my_rpd, blocksize, mapsize, filesize, NumFrames,
		  Caching,
		  PrefetchLimit, PrefetchLead, PrefetchMin, 
		  PrefetchDemand, PrefetchBufHit,
		  WriteAlg, FileIsNew,
		  replace_style, wss, &my_refs);
    /* RT_dump(my_rpd); */

    /* (Re)initialize statistics */
    RT_InitStats();
    
    GenTaskForEachProc(InitforProc);

    RT_WhatPrefetch(my_rpd);
/*    RT_WhatWrite(my_rpd); */

    /* io_clearstat(); => moved to the body of the tests for better timing */

    fflush(stdout);
}

/* @SUBTITLE "InitForProc: worker for Init" */
/* (parallel) per-process initialization */
static void
InitforProc ()
{
    /* this fixes a bug in ELOG_SETUP: free the elog data space first */
    if (eptr_base != NULL)
	 cfree(eptr_base);
    ELOG_SETUP(RTELOG,Vnode,15000);
    ELOG_TOUCH();
    ELOG_LOG(E_PROCS, ProcsInUse()); /* ensures touching other structs */
    
    /* Allocate buffer for reading chunks */
    my_buf = AllocPrivate (my_refs->chunksize);
    TouchBlock(my_buf, my_refs->chunksize, TRUE);
    
    /* Start prefetching in this file */
    if (Prefetch)
	 RT_StartPrefetch(my_rpd, PrefetchAlg, PrefetchParm);

    io_touch();

    fflush(stdout);
}

/* @SUBTITLE "SynchPoint: synchronize with other processors" */
void
SynchPoint (reset)
	void (*reset)();
{
    TICS start = GetRtc();
    
    ELOG_LOG(E_SYNCHSTART, 0);
    
    if (PrefetchAtSynch)
	 SynchronizeFull(SynchCounter, RT_Prefetch, 0, reset, 0);
    else
	 SynchronizeReset(SynchCounter, reset, 0);
    
    ELOG_LOG(E_SYNCHEND, 0);
    
    Atomic_add_long(SynchTime, GetRtc() - start);
    Atomic_add(Nsynchs, 1);
}

/* SynchOut */
void
SynchOut ()
{
    SynchronizeOut(SynchCounter);
}

/* @SUBTITLE "SynchPointNeighbor: synchronize for SYNCH_NEIGHBOR" */
void
SynchPointNeighbor()
{
    TICS start = GetRtc();
    
    ELOG_LOG(E_SYNCHSTART, 0);
    
    if (PrefetchAtSynch)
	 SynchronizeNeighbor(RT_Prefetch, 0);
    else
	 SynchronizeNeighbor(NULL, 0);
    
    ELOG_LOG(E_SYNCHEND, 0);
    
    Atomic_add_long(SynchTime, GetRtc() - start);
    Atomic_add(Nsynchs, 1);
}
