/* @TITLE "args.c: Parse command arguments for driver" */
/* args.c: parse the arguments for the driver 
 *
 * Contains: 
 *      GetArgsFromFile
 *      ParseArgs
 *
 * Split off from driver.c 5/4/89
 * David Kotz 
 */

static char rcsid[] = "$Id: args.c,v 7.2 91/05/09 19:34:39 dfk Tape2 $"; 

/* INCLUDES */

#include <stdio.h>
#include <usdfk.h>

#include "rapid.h"
#include "synch.h"
#include "driver.h"
#include "disktime.h"
#include "write.h"

static boolean BadArg();
static boolean MissingArg();
static boolean YesNoArg();

#define NARGS 25			/* we expect NARGS arguments */

#define COMMENT_CHAR '#'		/* comments to the end-of-line */

int InputLine = 0;			/* line in input test file */

/* assigned by GetArgsFromFile */
static int argc_line;		/* # of words on line */
static char *argv_line[NARGS]; /* the words on the line */

/* @SUBTITLE "GetArgsFromFile: Extract arguments from a file" */
/* blank lines will be skipped */
boolean
GetArgsFromFile(file, ignore)
	FILE *file;
	boolean ignore;		/* ignore this line? */
{
    static char buf[BUFSIZ];	/* holds part of the file */
    static char *nextline = buf; /* start of next line in buf */
    static int nbytes = 0;	/* number of valid bytes in buf */
    register char *p;		/* pointer into buf */
    boolean newword;		/* are we between words? */
    boolean comment;		/* are we scanning a comment? */

    do {					/* repeat until a non-empty line is read */
	   /* Move to front of buffer any previously obtained stuff */
	   nbytes -= nextline-buf;	/* subtract off length of previous line */
	   strncpy(buf, nextline, nbytes);
	   
	   /* Fill the buffer again */
	   nbytes += fread(buf + nbytes, 1, BUFSIZ-nbytes, file);
	   
	   if (nbytes == 0)
		return(FALSE);
	   
	   InputLine++;			/* working on a new line */
	   
	   if (ignore) {			/* scan off the line */
		  for (p = buf; p-buf < nbytes && *p != '\n'; p++)
		    ;
	   } else {
		  /* Scan the buffer, replacing white space by nulls */
		  /* and recording the start of each argument in argv */
		  argc_line = 0;
		  newword = TRUE;
		  comment = FALSE;
		  for (p = buf; p-buf < nbytes && *p != '\n'; p++) {
			 if (!comment) {			   
				if (*p == COMMENT_CHAR) {
				    /* comment: ignore rest of line */
				    comment = TRUE;
				    continue;
				}
				if (*p == ' ' || *p == '\t') {
				    *p = '\0';
				    newword = TRUE;
				} else if (newword) {
				    if (argc_line >= NARGS) {
					   printf("Too many arguments in testfile line %d\n", 
							InputLine);
					   return(FALSE);
				    }
				    argv_line[argc_line++] = p;
				    newword = FALSE;
				}
			 }
		  }
	   }
	   
	   /* Shouldn't overflow buffer or get to end of file */
	   if (*p != '\n') {
		  printf("Line %d too long in test file\n", InputLine);
		  return(FALSE);
	   } else *p = '\0';
	   
	   /* Remember the beginning of the next line */
	   nextline = ++p;
    } while (!ignore && argc_line == 0);

    return(TRUE);
}

/* @SUBTITLE "ParseArgs: parse arguments on command line" */
boolean
ParseArgs(argc, argv)
	int argc;
	char **argv;
{
    /* pass null arguments to use the last line from GetArgsFromFile */
    if (argv == NULL) {
	   argc = argc_line;
	   argv = argv_line;
    }

    if (InputLine > 0)
	 printf("\n\nArguments for line %d:\n", InputLine);
    
    if (OptionalArg(argc-1, argv+1)) { /* handle [-elog] */
	   argc--;
	   argv++;
    }
    printf("ElogDumpLimit: %d\n", ElogDumpLimit);
    Share(&ElogDumpLimit);

    /* Directory name */
    if (--argc > 0) {
	   strcpy(Directory_name, *++argv);
	   printf("Directory name: %s\n", Directory_name);
    } else return(MissingArg("Directory name"));
    
    /* Pattern name */
    if (--argc > 0) {
	   Pattern = *++argv;
	   printf("Pattern file name: %s\n", Pattern);
    } else return(MissingArg("Pattern file name"));
    
    /* Prefetch? */
    if (--argc > 0) {
	   ++argv;
	   if (strlen(*argv) != 1) 
		return(BadArg("Prefetch/write switch should be one of [yYnwN]", 
				    *argv));
	   switch(**argv) {
		  case 'n': {
			 Caching = TRUE;
			 Prefetch = FALSE;
			 Baseline = FALSE;
			 DoingWrites = FALSE;
			 WriteAlg = WBACK;
			 break;
		  }
		  case 'y': {
			 Caching = TRUE;
			 Prefetch = TRUE;
			 Baseline = TRUE;
			 DoingWrites = FALSE;
			 WriteAlg = WBACK;
			 break;
		  }
		  case 'Y': {
			 Caching = TRUE;
			 Prefetch = TRUE;
			 Baseline = FALSE;
			 DoingWrites = FALSE;
			 WriteAlg = WBACK;
			 break;
		  }
		  case 'w': {
			 Caching = TRUE;
			 Prefetch = FALSE;
			 Baseline = FALSE;
			 DoingWrites = TRUE;
			 WriteAlg = WBACK; /* for now */
			 break;
		  }
		  case 'N': {
			 Caching = FALSE;
			 Prefetch = FALSE;
			 Baseline = FALSE;
			 DoingWrites = FALSE;
			 WriteAlg = WBACK;
			 break;
		  }
		  default: {
			 return(BadArg("Prefetch/write switch should be in [yYnwN]", 
						*argv));
		  }
	   }
	   printf("Caching?  %s\n", Caching ? "YES" : "NO");
	   printf("DoingWrites?  %s\n", DoingWrites ? "YES" : "NO");
	   printf("Prefetch?  %s\n", Prefetch ? "YES" : "NO");
	   printf("Prefetch Baseline?  %s\n", Baseline ? "YES" : "NO");
	   /* no need to Share(&Baseline) */
    } else return(MissingArg("Prefetch/write flag\n"));

    /* Prefetch when waiting for sector, advise, and limit */
    if (Prefetch) {
	   /* Prefetch at synch points? */
	   if (--argc > 0) {
		  if (!YesNoArg("Prefetch at sync point", *++argv, &PrefetchAtSynch))
		    return(FALSE);
		  printf("Prefetch at synch points?  %s\n",
			    PrefetchAtSynch ? "YES" : "NO");
	   } else return(MissingArg("Prefetch at synch points?"));
	   /* Prefetch when waiting? */
	   if (--argc > 0) {
		  if (!YesNoArg("Prefetch during buffer hit", *++argv, 
					 &PrefetchBufHit))
		    return(FALSE);
		  printf("Prefetch during buffer hit?  %s\n",
			    PrefetchBufHit ? "YES" : "NO");
	   } else return(MissingArg("Prefetch during buffer hit?"));
	   /* Prefetch overlapping cpu? */
	   if (--argc > 0) {
		  if (!YesNoArg("Prefetch while demand", *++argv, &PrefetchDemand))
		    return(FALSE);
		  printf("Prefetch while demand?  %s\n",
			    PrefetchDemand ? "YES" : "NO");
	   } else return(MissingArg("Prefetch while demand fetch?"));
	   /* Prefetch lead */
	   if (--argc > 0) {
		  PrefetchLead = atoi(*++argv);
		  if (PrefetchLead < 0)
		    return(BadArg("Prefetch lead should be >= 0", *argv));
		  printf("Prefetch lead: %d\n", PrefetchLead);
	   } else return(MissingArg("Prefetch lead"));
	   /* Prefetch limit */
	   if (--argc > 0) {
		  PrefetchLimit = atoi(*++argv);
		  if (PrefetchLimit < 0)
		    return(BadArg("Prefetch limit should be >= 0", *argv));
		  printf("Prefetch limit: %d\n", PrefetchLimit);
	   } else return(MissingArg("Prefetch limit"));
	   /* Prefetch minimum */
	   if (--argc > 0) {
		  PrefetchMin = atoi(*++argv);
		  if (PrefetchMin < 0)
		    return(BadArg("Prefetch minimum time should be >= 0", *argv));
		  printf("Prefetch minimum time: %d\n", PrefetchMin);
	   } else return(MissingArg("Prefetch minimum time"));
	   /* Prefetch algorithm */
	   if (--argc > 0) {
		  ++argv;
		  if (strcmp(*argv, "none") == 0) PrefetchAlg = PFA_NONE;
		  else if (strcmp(*argv, "exact") == 0) PrefetchAlg = PFA_EXACT;
		  else if (strcmp(*argv, "obl") == 0) PrefetchAlg = PFA_OBL;
		  else if (strcmp(*argv, "ibl") == 0) PrefetchAlg = PFA_IBL;
		  else if (strcmp(*argv, "port") == 0) PrefetchAlg = PFA_PORT;
		  else if (strcmp(*argv, "adapt") == 0) PrefetchAlg = PFA_ADAPT;
		  else if (strcmp(*argv, "iobl") == 0) PrefetchAlg = PFA_IOBL;
		  else if (strcmp(*argv, "iport") == 0) PrefetchAlg = PFA_IPORT;
		  else if (strcmp(*argv, "ioport") == 0) PrefetchAlg = PFA_IOPORT;
		  else if (strcmp(*argv, "gaps") == 0) PrefetchAlg = PFA_GAPS;
		  else if (strcmp(*argv, "rgaps") == 0) PrefetchAlg = PFA_RGAPS;
		  else if (strcmp(*argv, "gw") == 0) PrefetchAlg = PFA_GW;
		  else if (strcmp(*argv, "switch") == 0) PrefetchAlg = PFA_SWITCH;
		  else return(BadArg("Prefetch Algorithm", *argv));
		  printf("Prefetch Algorithm: %s\n", *argv);
	   } else return(MissingArg("Prefetch Algorithm"));
	   /* Prefetch algorithm parameter */
	   if (--argc > 0) {
		  PrefetchParm = atoi(*++argv);
		  printf("Prefetch algorithm parameter: %d\n", PrefetchParm);
	   } else return(MissingArg("Prefetch algorithm parameter"));
    } else {
	   PrefetchAtSynch = FALSE;
	   PrefetchBufHit = FALSE;
	   PrefetchDemand = FALSE;
	   PrefetchLead = 0;
	   PrefetchLimit = 0;
	   PrefetchMin = 0;
	   PrefetchAlg = PFA_NONE;
	   PrefetchParm = 0;
    }
    Share(&Prefetch);
    Share(&PrefetchAtSynch);
    Share(&PrefetchDemand);
    Share(&PrefetchBufHit);
    Share(&PrefetchLead);
    Share(&PrefetchLimit);
    Share(&PrefetchMin);
    Share(&PrefetchAlg);
    Share(&PrefetchParm);
    
    FileIsNew = FALSE;

    if (DoingWrites) {
	   /* is the file new? */
	   if (--argc > 0) {
		  if (!YesNoArg("File is new", *++argv, &FileIsNew))
		    return(FALSE);
		  printf("File is new?  %s\n", FileIsNew ? "YES" : "NO");
	   } else return(MissingArg("File is new?"));
	   /* Cache size */
	   if (--argc > 0) {
		  NumFrames = atoi(*++argv);
		  if (NumFrames <= 0)
		    return(BadArg("Cache size should be > 0 blocks", *argv));
		  printf("Cache size (blocks): %d\n", NumFrames);
	   } else return(MissingArg("Cache size"));
	   /* Write algorithm */
	   if (--argc > 0) {
		  ++argv;
		  if (strcmp(*argv, "wback") == 0) WriteAlg = WBACK;
		  else if (strcmp(*argv, "wthru") == 0) WriteAlg = WTHRU;
		  else if (strcmp(*argv, "wfull") == 0) WriteAlg = WFULL;
		  else if (strcmp(*argv, "wnfull") == 0) WriteAlg = WNFULL;
		  else if (strcmp(*argv, "wleave") == 0) WriteAlg = WLEAVE;
		  else return(BadArg("Write Style", *argv));
		  printf("Write Style: %s\n", *argv);
	   } else return(MissingArg("Write Style"));
    }
    /* no need to share FileIsNew, NumFrames, WriteAlg */

    if (!Caching) {
	   /* is the file new? */
	   if (--argc > 0) {
		  if (!YesNoArg("File is new", *++argv, &FileIsNew))
		    return(FALSE);
		  printf("File is new?  %s\n", FileIsNew ? "YES" : "NO");
	   } else return(MissingArg("File is new?"));

	   NumFrames = 0;
    }

    /* Working set size */
    if (--argc > 0) {
	   wss = atoi(*++argv);
	   if (wss < 0 || (wss == 0 && Caching))
		return(BadArg("Working set size should be > 0", *argv));
	   printf("Working set size: %d\n", wss);
	   Share(&wss);
    } else return(MissingArg("Working set size"));
    
    if (!Caching)
	 wss = 0;

    /* Block size */
    if (--argc > 0) {
	   blocksize = atoi(*++argv);
	   if (blocksize <= 0)
		return(BadArg("Block size should be > 0", *argv));
	   printf("Block size: %d\n", blocksize);
	   Share(&blocksize);
    } else return(MissingArg("Block size"));
    
#ifdef DISKTIME_HACK
    /* Where the number of disks usually goes, we read the disk-access time */
    /* units are msec */
    if (--argc > 0) {
	   disktime = atoi(*++argv);
	   if (disktime <= 0)
		return(BadArg("Disk-access time should be > 0", *argv));
	   printf("Disk-access time: %d msec\n", disktime);
    } else return(MissingArg("Disk-access time"));
    /* ndisks set below */
#else
    /* Number of disks */
    if (--argc > 0) {
	   ndisks = atoi(*++argv);
	   if (ndisks <= 0)
		return(BadArg("Number of disks should be > 0", *argv));
	   printf("Number of disks: %d\n", ndisks);
    } else return(MissingArg("Number of disks"));
    disktime = 0;			/* ie, use the default access time */
#endif DISKTIME_HACK
    
    /* Synchronization style */
    if (--argc > 0) {
	   ++argv;
	   if (strcmp(*argv, "none") == 0) SynchStyle = SYNCH_NONE;
	   else if (strcmp(*argv, "each") == 0) SynchStyle = SYNCH_EACH;
	   else if (strcmp(*argv, "total") == 0) SynchStyle = SYNCH_TOTAL;
	   else if (strcmp(*argv, "portion") == 0) SynchStyle = SYNCH_PORTION;
	   else if (strcmp(*argv, "nbor") == 0) SynchStyle = SYNCH_NEIGHBOR;
	   else if (strcmp(*argv, "neighbor") == 0) SynchStyle = SYNCH_NEIGHBOR;
	   else return(BadArg("Synchronization style", *argv));
	   printf("Synch style: %s\n", *argv);
	   Share(&SynchStyle);
    } else return(MissingArg("Synch Style"));
    
    /* Synchronization count */
    if (--argc > 0) {
	   SynchCount = atoi(*++argv);
	   if (SynchCount <= 0)
		return(BadArg("SynchCount should be > 0", *argv));
	   printf("Synch Count: %d\n", SynchCount);
	   Share(&SynchCount);
    } else return(MissingArg("SynchCount"));
    
    /* Number of processors */
    if (--argc > 0) {
	   Nprocs = atoi(*++argv);
	   if (Nprocs < 0)
		return(BadArg("Number of procs should be > 0", *argv));
    } else return(MissingArg("Number of procs"));
    Share(&Nprocs);
    printf("Number of processors: %d\n", Nprocs);

#ifdef DISKTIME_HACK
    ndisks = Nprocs;		/* number of disks = number of procs */
    printf("Number of disks: %d\n", ndisks);
#endif DISKTIME_HACK

    /* Number of trials */
    if (--argc > 0) {
	   Ntrials = atoi(*++argv);
	   if (Ntrials <= 0)
		return(BadArg("Number of trials should be >= 0", *argv));
	   printf("Number of trials: %d\n", Ntrials);
    } else return(MissingArg("Number of trials"));
    
    if (--argc > 0) {
	   printf("\nIgnoring extraneous arguments: first is %s\n", *argv);
    }
    
    if (DoingWrites && NumFrames < wss * Nprocs) {
	   printf("Cache size %d too small for WS %d\n", NumFrames, wss*Nprocs);
	   return(FALSE);
    }
    
    printf("\n");
    
    return(TRUE);
}

/* @SUBTITLE "OptionalArg: process [-elog] argument, if there" */
boolean					/* TRUE if optional arg was parsed */
OptionalArg(argc, argv)
	int argc;
	char **argv;	
{
    char *opt = argv[0];
    int dummy;

    if (argc > 0 && *opt == '-') {
	   /* We currently allow "-#" where # is the highest trial
	    * number to get its elog dumped 
	    */
	   if (sscanf(opt, "-%d", &dummy) != 1)
		return(BadArg("ElogDumpLimit", opt));
	   if (!ElogDumpIgnore)
		ElogDumpLimit = dummy;
	   return(TRUE);
    } else 
	 return(FALSE);
}


/* @SUBTITLE "YesNoArg: Parse a yes/no argument" */
static boolean
YesNoArg(name, string, value)
	char *name;			/* name of parameter */
	char *string;			/* input string */
	boolean *value;
{
    switch(*string) {
	   case 'y':
	   case 'Y': {
		  *value = TRUE;
		  return(TRUE);
	   }
	   case 'n':
	   case 'N': {
		  *value = FALSE;
		  return(TRUE);
	   }
	   default: {
		  return(BadArg(name, string));
	   }
    }
}

/* @SUBTITLE "MissingArg, BadArg: print error and die" */
static boolean
MissingArg(name)
	char *name;
{
    if (InputLine > 0)
	 printf("Missing argument: %s on line %d\n", name, InputLine);
    else
	 printf("Missing argument: %s\n", name);
    return(FALSE);
}

static boolean
BadArg(name, value)
	char *name;
	char *value;
{
    if (InputLine > 0)
	 printf("Bad argument (%s): %s, on line %d\n", value, name, InputLine);
    else
	 printf("Bad argument (%s): %s\n", value, name);
    return(FALSE);
}

