/* @TITLE "best: OPRA - Offline PRefetch Analyzer" */
/* 
 * best - best-possible real-time algorithm
 * 
 * David Kotz 1989
 *
 * This algorithm is included for comparison. It reflects the best we 
 * possibly do in real-time with no user hints. Thus, the first block is 
 * a miss, and there's a mistake and a miss at each portion change, unless
 * the portion length/skip is regular, in which case this happens only
 * on the first portion break. If the length is regular and skip is not,
 * then we get a miss at each portion, but a mistake only at the end of
 * the first. 
 * 
 * With mistakes, however, it is always possible in every situation to
 * imagine a predictor that will choose not to prefetch in that situation. 
 * So the mistake rate of BEST is always zero.
 *
 * Basically, as each block comes in, we ask ourselves, "could we have
 * predicted that?"  If the answer is yes, say we did.
 *
 * Immediate rereference counts as a correct prediction.
 * 
 */

#include <stdio.h>
#include "dfk.h"
#include "opra.h"

static struct {
    /* current run */
    int first;				/* start of current run */
    int last;				/* last block read */
    /* previous runs */
    int same_runlen;		/* number of runs with same len */
    int prev_runlen;		/* length of previous run */
    int same_skiplen;		/* number of skips with same len */
    int prev_skiplen;		/* length of previous skip */
} best;					/* data for this algorithm */

void
ResetBEST()
{
    best.first = best.last = -100;
    best.same_runlen = best.same_skiplen = 0;
    best.prev_runlen = best.prev_skiplen = 0;
}

boolean
RefBEST(block, mistake)
	int block;
	int *mistake;
{
    int runlen;
    int skiplen;
    boolean predicted;		/* did we correctly predict? */

    if (best.last < 0) {
	   /* first reference in this sequence */
	   best.first = best.last = block;
	   *mistake = 0;
	   return(predicted = FALSE);
    }

    /* immediate rereference or sequential reference */
    if (block == best.last || block == best.last + 1) {
	   best.last = block;
	   *mistake = 0;
	   predicted = TRUE;
    } else {
	   /* out-of-order reference */
	   runlen = best.last - best.first + 1;
	   skiplen = block - best.last;
#ifdef DEBUG
	   printf("run length %d, skip %d\n", runlen, skiplen);
#endif DEBUG
	   best.first = best.last = block;
	   if (runlen == best.prev_runlen && skiplen == best.prev_skiplen) {
		  /* lps-like pattern is predictable into next portion */
		  predicted = TRUE;
		  *mistake = 0;
		  best.same_runlen++;
		  best.same_skiplen++;
	   } else if (runlen == best.prev_runlen) {
		  /* same length, different skip: miss at start of portion */
		  /* but no mistake after last portion */
		  predicted = FALSE;
		  *mistake = 0;
		  best.same_runlen++;
		  best.same_skiplen = 1;
	   } else {
		  /* different run, different skip: miss at next */
		  /* this is the only place we might get a mistake */
		  /* but some algorithms may have chosen not to prefetch */
		  predicted = FALSE;
		  *mistake = 0;
		  best.same_runlen = 1;
		  best.same_skiplen = 1;
	   }
	   best.prev_runlen = runlen;
	   best.prev_skiplen = skiplen;
    }
    return(predicted);
}

/* @SUBTITLE "LastBEST: What happens at end of string" */
void
LastBEST(mistake)
	int *mistake;
{
    int runlen;

    *mistake = 0;

    runlen = best.last - best.first + 1;
    
    if (runlen == best.prev_runlen) {
	   best.same_runlen++;
    } else {
	   best.same_runlen = 1;
    }
    best.prev_runlen = runlen;

#ifdef DEBUG
    printf("final state was run %d x %d, skip %d x %d mistake %d\n",
		 best.prev_runlen, best.same_runlen, 
		 best.prev_skiplen, best.same_skiplen,
		 *mistake);
#endif DEBUG
}
