/* @TITLE "verify2: verify a file" */
/*
 *  verify2 - take a set of DISK files and check the data.
 *  This program expects much more complex data than the other verify;
 *  we expect to see generated by the Data_* functions of cp.ca.
 *
 * usage:
 *    verify { 1 | 2 }
 * if 1, it's a one-dimensional matrix (ie, wxn, wxb, wxc)
 * if 2, it's a two-dimensional matrix
 *
 *  The disk layout must have been CONTIGUOUS.
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "dmcache.h"
#include "aux.h"
#include "file.h"
#include "diskdriver.h"
#include "diskmodel.param"

#ifndef LAYOUT_CONTIGUOUS
hey! Should be LAYOUT-CONTIGUOUS !!!
#endif

void CheckRecord(char *buf, ulong row, ulong col);

/* Factor n, n>0, into two factors f1 and f2, so that f1 is the largest factor 
 * less than sqrt(n).  Basically, so that f1 and f2 are as close together as 
 * possible, and f1 <= f2.  
 */
void Factor(int n, int *f1, int *f2);

int
main(int argc, char **argv)
{
    int disk;
    int block;
    char diskname[10];
    FILE *diskfp[NO_OF_DISKS];
    char *matrix;	      /* the entire matrix */
    char *buf;		      /* some place in the matrix */
    int nRrows, nRcols;	      /* mapping of records to 2-d matrix */
    int row,col;
    int dims;

    if (argc != 2) {
	fprintf(stderr, "usage: %s {1 | 2}\n", argv[0]);
	exit(1);
    }

    dims = atoi(argv[1]);
    if (dims == 1) {
	nRrows = NUM_RECORDS;
	nRcols = 1;
    } else if (dims == 2) {
	/* map records into a 2-d matrix */
	Factor(NUM_RECORDS, &nRcols, &nRrows);
    } else {
	fprintf(stderr, "usage: %s {1 | 2}\n", argv[0]);
	exit(1);
    }

    printf("Expecting %d rows by %d cols\n", nRrows, nRcols);
    matrix = (char *)malloc (nRrows * nRcols * RECORD_SIZE);

    if (matrix == NULL) {
	fprintf(stderr, "%s: not enough memory for matrix\n", argv[0]);
	exit(1);
    }

    /* open all the files */
    for (disk = 0; disk < NO_OF_DISKS; disk++) {
	sprintf(diskname, "DISK%02d", disk);
	diskfp[disk] = fopen(diskname, "r");
	if (diskfp[disk] == (FILE *)NULL) {
	    fprintf(stderr, "Cannot open file '%s'\n", diskname);
	    exit(1);
	}
    }

    /* read each file block into the matrix */
    buf = matrix;
    for (block = 0, disk = 0; block < FILE_BLOCKS; 
	 block++, disk = (disk+1) % NO_OF_DISKS) {
	if (fread(buf, BLOCK_SIZE, 1, diskfp[disk]) != 1) {
	    fprintf(stderr, "Error reading block %d from disk %d\n", 
		    block, disk);
	    exit(1);
	}

	buf += BLOCK_SIZE;
    }

    buf = matrix;
    for (row = 0; row < nRrows; row++)
      for (col = 0; col < nRcols; col++) {
	  CheckRecord(buf, row, col);
	  buf += RECORD_SIZE;
      }

    return(0);
}

/* @SUBTITLE "CheckRecord: check buffer for a test pattern" */

#define TESTVALUE(r,c) ((r) + 1.) + (((c) + 1.) * 10000.);

void
CheckRecord(char *buf, ulong row, ulong col)
{
    float *p = (float *)buf;
    float *end = (float *)(buf + RECORD_SIZE);
    float val = TESTVALUE(row, col);
	
    for (; p < end; p++) 
      if (*p != val) {
	  printf("CheckRecord failed (%g != %g) at row %lu, col %lu\n", 
		 *p, val, row, col);
	  return;
      }
}


/* @SUBTITLE "Factor - factor a number" */
/* Factor n, n>0, into two factors f1 and f2, so that f1 is the largest factor 
 * less than sqrt(n).  Basically, so that f1 and f2 are as close together as 
 * possible, and f1 <= f2.  Very simplistic algorithm!
 */
void
Factor(int n, int *f1, int *f2)
{
    int x;		      /* potential factor */

    x = (int) sqrt((double) n);

    /* Keep trying factors that are <= sqrt(n) until we find one. */
    /* At worst, we'll stop when x=1. */
    while(n % x != 0)
      x--;

    *f1 = x;		      /* the smaller factor */
    *f2 = n / x;	      /* the larger factor */
}

