/* @TITLE "cpfs.h: interface to cpfs*.ca"*/
/* 
 * This provides a simple file system interface. 
 * functions:
 *   FSinit: initialize CP file system
 *   FSdone: shut down the CP file system
 *   FSread: arbitrary read from file
 *   FSwrite: arbitrary write into file
 *
 * Part of 
 *           The STARFISH Parallel file-system simulator
 *      (Simulation Tool for Advanced Research in File Systems)
 *
 *                              David Kotz
 *                          Dartmouth College
 *                             Version 3.0
 *                             October 1996
 *                         dfk@cs.dartmouth.edu
 */

/* $Id: cpfs.h,v 3.0 1996/10/18 06:05:51 dfk RELEASE3 dfk $ */
/* based on file.h,v 1.3 92/12/13 */

#ifndef CPFS_H
#define CPFS_H

#include "dmcache.h"
#include "pattern.h"
#include "userdata.h"

extern void FSinit(int CPnum);
extern void FSflush();
extern void FSdone();
extern void FSread(UserData userbuf, ulong offset, ulong bytes);
extern void FSwrite(UserData userbuf, ulong offset, ulong bytes);
extern void FSdebug();

/* Some are supported only by CP_DIRECT, and IOP_GENERAL */
/* For others, there are stubs in cpfs-direct-stubs */
extern void FSread_lw(UserData userbuf);
extern void FSwrite_lw1(UserData userbuf);
extern void FSread_seg(UserData userbuf, ulong my_records);
extern void FSwrite_seg(UserData userbuf, ulong my_records);
extern int FSread_lu(float *array, int first_col, int ncols, 
		     int leftmost, int slab_cols, int nrows);
extern int FSwrite_lu(float *array, int first_col, int ncols, 
		      int leftmost, int slab_cols, int nrows);
extern void FSread_gen(int CPnum, boolean write, UserData userbuf, 
		       PATTERN pattern, 
		       ulong my_records, ulong nRecords, 
		       ulong nRcols, ulong nRrows, 
		       int nPcols, int nProws);
extern void FSwrite_gen(int CPnum, boolean write, UserData userbuf, 
			PATTERN pattern, 
			ulong my_records, ulong nRecords, 
			ulong nRcols, ulong nRrows, 
			int nPcols, int nProws);

/* Add system call overhead, defined by TRAP_CYCLES.
 * Do it in small increments so Proteus can grab control in middle.
 * Unfortunately, the loop overhead is 10 cycles, so we take that 
 * off before calling AddTime. This loop overhead applies to the last 
 * (unused) loop iteration as well, so we have to take that off the 
 * final AddTime statement as well.
 */
#if TRAP_CYCLES <= 0
#define SYSCALL
#else
#define SYSCALL { \
   int i; \
   for (i = 0; i+20 <= (TRAP_CYCLES-10); i += 20) \
	AddTime(20-10); \
   AddTime((TRAP_CYCLES-10) % 20); \
}
#endif

#endif /* CPFS_H */
