/****************************************************************************
 *
 *                     Parallel Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module:  Statistics and Events Module
 *
 * Description:  Defines types for statistics and interface macros.  Also
 *     defines metrics.
 *
 * Last Modified:  8-20-91
 *
 * Data Structures:   Stat, ArrayStat, Metric, ArrayMetric
 *
 * Constants:  Event fields, stat and metric macros
 *    stats_           set of individual statistics
 *    astats_          set of array statistics
 *    metrics_         set of individual metrics
 *    ametrics_        set of array metrics
 *    event_file_      FILE * for event/stat file
 *
 ***************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ***************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/simex/RCS/event.h,v 1.3 1996/07/04 03:30:05 dfk YAY dfk $
 * $Log: event.h,v $
 * Revision 1.3  1996/07/04 03:30:05  dfk
 * this is a nice working version.
 *
 * Revision 1.2  1996/07/02 02:58:17  dfk
 * completely new command-line interface
 * detailed control over what gets printed
 * ability to use abbreviated names for metrics
 *
 * Revision 1.1  1996/06/30 21:33:10  dfk
 * Initial revision
 *
 * Revision 1.11  1993/02/17  17:57:15  brewer
 * Changed metrics to use doubles instead of floats
 *
 * Revision 1.10  92/10/13  13:52:26  brewer
 * Updated TraceFileComment args
 * 
 * Revision 1.9  92/10/09  16:00:50  brewer
 * Added TraceFileComment
 * 
 * Revision 1.8  92/10/08  13:44:28  brewer
 * Changed ATEVENT to use AtEvent procedure
 * Removed ARRAYEVENT macro
 * 
 * Revision 1.7  92/10/07  16:47:49  brewer
 * system now does its own buffering, so I removed all references to
 * event_file_ and putw(), including macros ATEVENT and ARRAYEVENT.
 * 
 * Revision 1.6  92/10/05  00:05:11  brewer
 * Updated trace file format, set up for libevt.a
 * 
 * Revision 1.5  92/08/11  13:27:36  brewer
 * Updated to support libevt library
 * 
 * Revision 1.4  92/04/28  11:23:21  brewer
 * Added ifndef _EVENT_H_ to allow multiple inclusion.
 * 
 * Revision 1.3  92/03/31  14:06:38  brewer
 * Changed event field macros to use (x) instead of x
 * 
 * Revision 1.2  92/02/13  16:53:57  brewer
 * fixed EVENT_TO_INT and MAKE_EVENT_FIELD macros with unsigned long cast
 * 
 * Revision 1.1  92/02/11  14:19:37  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef _EVENT_H_
#define _EVENT_H_

/* mark this version of event.h as supporting buffered output */
#define BUFFERED_EVENT_H

#ifndef __STDC__  /* if not ANSI C, remove const qualifier */
#ifndef const
#define const
#endif
#endif

#include <stdio.h>

/* fields */
#define EVENT_FIELD(x)          ((x) & 0xff000000)
#define EVENT_TO_INT(x)         ((unsigned long)(x) >> 24)
#define SHORT_INDEX(x)          (((x) >> 14) & 0x3ff)
#define MAKE_SHORT_INDEX(x)     (((unsigned long)(x) & 0x3ff) << 14)
#define LONG_INDEX(x)           ((x) & 0xffffff)
#define SHORT_VALUE(x)          ((x) & 0x3fff)
#define IS_SHORT_FORMAT(x)      ((x) & 0x80000000)
#define MAKE_EVENT_FIELD(x)     ((unsigned long)(x) << 24)
#define SHORT_FORMAT            0x80000000

#include "events.h"

#define STAT_CACHE      EVENT_TO_INT((unsigned long)EV_CACHE)
#define STAT_BUSCONT    EVENT_TO_INT((unsigned long)EV_BUSCONT)
#define STAT_NETCONT    EVENT_TO_INT((unsigned long)EV_NETCONT)

/* the title of the simulation should be stored here */
extern const char *simulation_title_;


/***************************************************************************/
/* statistics types */

typedef struct {
    long value;
    short count, update;
    const char *name;
} Stat;

typedef struct {
    long *value;
    short *count;
    short update, size;
    const char *name;
} ArrayStat;

extern Stat stats_[];
extern ArrayStat astats_[];

/***************************************************************************/
/* metrics types */

typedef struct {
    double value, tmp1, tmp2, tmp3;
    const char *name;
} Metric;

typedef struct {
    double *value, *tmp1, *tmp2, *tmp3;
    const char *name;
    int size;
} ArrayMetric;

extern Metric metrics_[];
extern ArrayMetric ametrics_[];
extern double sqrt(double);

/***************************************************************************/
/* event and statistics macros */

/* provided to support older versions of Proteus.  The recommended
   event-generation procedures are Event() and IndexEvent() */
#define ATEVENT(kind, index_and_value, time) \
  AtEvent(kind, index_and_value, time)

#define SUM_STAT(stat, val, time) \
    { stats_[stat].value += val; \
      if (++(stats_[stat].count) >= stats_[stat].update) {\
	write_stat(stat, time); \
        stats_[stat].value = 0; } \
}

#define SUM_ARRAY_STAT(stat, index, val, time) \
    { astats_[stat].value[index] += val; \
      if (++(astats_[stat].count[index]) >= astats_[stat].update) {\
	  write_array_stat(stat, index, time); \
	  astats_[stat].value[index] = 0; } \
}


/***************************************************************************/
/* metric macros */

#define GET_METRIC(metric) (metrics_[metric].value)
#define GET_ARRAY_METRIC(metric, index) (ametrics_[metric].value[index])

#define SET_METRIC(metric, val) (metrics_[metric].value = (double)(val))
#define SET_ARRAY_METRIC(m, i, val) (ametrics_[m].value[i] = (double)(val))

#define SUM_METRIC(metric, val) sum_metric_(metric, (double)(val))

#define SUM_ARRAY_METRIC(metric, index, val) \
    sum_array_metric_(metric, index, (double)(val))

/* metric for running average */
/* tmp1 is used to keep the count of values, tmp2 tracks their sum */
#define AVG_METRIC(m, val) \
    avg_metric_(m, (double)(val))

#define AVG_ARRAY_METRIC(m, i, val) \
    avg_array_metric_(m, i, (double)(val))

/* metric for running standard deviation, requires an average metric */
/* tracks the standard deviation for the average metric `avg_metric'.
   which should be updated just before this metric.  tmp1 tracks the
   running sum of the squares.  The average, the count, and the running sum
   are read from the average metric */
#define STDDEV_METRIC(metric, avg_metric, val) \
    stddev_metric_(metric, avg_metric, (double)(val))
	
#define STDDEV_ARRAY_METRIC(metric, avg_met, i, val) \
    stddev_array_metric_(metric, avg_met, i, (double)(val))

/* reset like set, but also zeroes all tmp values */
#define RESET_METRIC(metric, val) \
  (metrics_[metric].tmp1 = metrics_[metric].tmp2 = metrics_[metric].tmp3 = 0,\
   SET_METRIC(metric, val))

#define RESET_ARRAY_METRIC(m, i, val) \
  (ametrics_[m].tmp1[i] = ametrics_[m].tmp2[i] = ametrics_[m].tmp3[i] = 0,\
   SET_ARRAY_METRIC(m, i, val))

/***************************************************************************/
/* procedures */

void init_statistics_(const char *parameters, const char *filename);
void exit_statistics_(unsigned long endtime);
void init_stat(short stat, long value, short rate);
void init_array_stat(short stat, short size, long value, short rate);
void write_stat(short stat, unsigned long t);
void write_array_stat(short stat, short index, unsigned long t);
void flush_event_file_(void);

int new_metric(const char *name, double init_value);
int new_array_metric(const char *name, int size, double init_value);
double sum_metric_(int metric, double val);
double sum_array_metric_(int metric, int index, double val);
double max_metric_(int metric, double val);
double max_array_metric_(int metric, int index, double val);
double min_metric_(int metric, double val);
double min_array_metric_(int metric, int index, double val);
double avg_metric_(int metric, double val);
double avg_array_metric_(int metric, int index, double val);
double stddev_metric_(int metric, int avg_metric, double val);
double stddev_array_metric_(int metric, int avg_metric, int index, double val);

void Event(unsigned long evt, unsigned long value, unsigned long time);
void IndexEvent(unsigned long evt, unsigned long index,
		unsigned long value, unsigned long time);
void AtEvent(unsigned long evt, unsigned long index_and_value,
	     unsigned long time);
void TraceFileComment(const char *type, int index, unsigned long time,
		      const char *buf, int length);

#endif /* _EVENT_H_ */
