 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: thread.h -- Header file for thread management module
 *
 * Description: Constants, Structures and Variables used to manage threads
 *              (File thread.c)
 *
 * Last Modified: $Date: 92/11/12 13:02:35 $ (eab)
 *
 * Data Structures:
 *     ThreadInfo: Structure with control information for a thread
 *
 * Constants:
 *     Thread states, registers, MAX_ARGS
 *
 * Referenced parameters:
 *     MAX_THREADS
 *
 * Macros: T_TIME, T_KILLED, T_NAME, T_OSBLOCK, Invalidtid
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ***************************************************************************
 * $Header: /psg/proteus/RCS/thread.h,v 1.5 92/11/12 13:02:35 brewer Exp $
 * $Log:	thread.h,v $
 * Revision 1.5  92/11/12  13:02:35  brewer
 * Changed T_INTERRUPTED from state to flag -- see rt_thread.ca log for
 * details.
 * 
 * Revision 1.4  92/11/09  10:58:26  brewer
 * Added include of rt_thread_def.h
 * 
 * Revision 1.3  92/10/05  12:26:11  brewer
 * Added ifndef _THREAD_H_
 * 
 * Revision 1.2  92/04/27  13:06:35  brewer
 * Changed REGISTERS from 65 to 67 to reflect the addition of the
 * multiplication registers needed for full context switch.
 * 
 * Revision 1.1  92/02/11  14:20:02  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef _THREAD_H_
#define _THREAD_H_

#ifndef SIMREQ
#include "simreq.h"
#endif

#include "rt_thread_def.h"

/* Runnable states */
#define T_CURRENT        0101          /* current thread on a processor */
#define T_READY          0102          /* thread is in ready list */
#define T_SPINNING       0103          /* thread is spinning on a lock */
#define T_INTERRUPTED   04000          /* ihandler is running on curr.proc. */

/* Suspended states */
#define T_SUSPENDED      0110          /* thread is temp. suspended */
#define T_SLEEPING       0120          /* waiting for interrupt */

/* Inactive states */
#define T_FREE          00000          /* free thread slot */
#define T_FORWARDER     01000          
#define T_RESERVED      02000          /* simulator tid reserved for use */

/* Generic masks */
#define T_ACTIVE         0100          /* not FREE or FORWARDER thread */
#define T_RUNNABLE       0007          /* CURRENT, READY, SPINNING
					    (perhaps INTERRUPTED) */ 


#define T_NAMELEN          20
#define REGISTERS          67    /* 32 + 32 + pc + mult_lo + mult_hi */

#define ARG1                4
#define SP                 29
#define FP                 30
#define RIP                31

#define MAX_ARGS           10  /* max no of args to any thread */


#define Invalidstid(t)     (((t)<0)||((t)>=MAX_THREADS))


#define T_NAME(t)              (Invalidstid(t) ? ERROR : \
				thread_table_[(t)].t_name)
#define T_TIME(t)              (Invalidstid(t) ? ERROR : \
				thread_table_[(t)].t_time)
#define T_OSBLOCK(t)           ((Thread *)(thread_table_[(t)].t_OS_block))

typedef struct {
  Word    t_regs[REGISTERS];
  int     t_state;
  char    t_name[T_NAMELEN];
  int     t_stkblk;
  char    *t_stkbase;
  FuncPtr t_func;
  Word    t_a[MAX_ARGS];
  Time    t_time;
  char    *t_OS_block;
} ThreadInfo;


extern GLOBAL ThreadInfo thread_table_[MAX_THREADS];
extern GLOBAL int numthreads;           /* No of active threads */
extern GLOBAL int currtid;              /* Id of current thread */
extern GLOBAL ThreadInfo *currtptr;     /* Address of active ThreadInfo */
extern GLOBAL Thread *currosptr;        /* address of active OS thread block */

 
#endif /* ifndef _THREAD_H_ */
