 
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator                
 *                Eric A. Brewer  and  Chris N. Dellarocas                
 *                     Laboratory for Computer Science                    
 *                  Massachusetts Institute of Technology                 
 *
 * Module: Thread management routines                                     
 *
 * Description:                                                           
 *         Routines to handle simulator threads                    
 *
 * Last Modified: $Date: 1996/07/21 14:44:41 $ (eab)
 * 
 * Global Functions:                                                      
 *     int create_thread(ulong sim_tid, FuncPtr func, unsigned stacksize,
 *                       char *name, Time timestamp,
 *                       void *OS_block, unsigned argc, Word args)        
 *         Creates a new thread: Allocates stack, fills in thread table      
 *         entry, returns thread error code or sim_tid.
 *                                                                        
 *     int resume_thread(ulong sim_tid, Time timestamp)
 *         Switches an active thread on.                                  
 *
 *     void init_threads(void)                                                
 *         Initializes thread table.                                      
 *
 *     ulong simulator_tid(void)
 *         Returns the simulator tid of a free thread.
 *
 *     void release_simulator_tid(ulong sim_tid)
 *
 *     void thread_top_level(void)
 *         Base routine for a stack frame
 *
 * Global Variables:                                                      
 *     ThreadInfo thread_table_[]   : Table with info for all threads         
 *     int numthreads           : Number of currently active threads      
 *     int currtid              : id of currently executing thread        
 *     ThreadInfo *currtptr     : address of current thread's structur    
 *     OS_ThreadInfo *currosptr : address of current thread's OS structure
 *                                                                        
 ****************************************************************************
 *   Copyright 1991                                                       
 *   Eric A. Brewer  and  Chris N. Dellarocas                             
 *   Massachusetts Institute of Technology                                
 *                                                                        
 *   Permission to use, copy, modify, and distribute this program         
 *   for any purpose and without fee is hereby granted, provided          
 *   that this copyright and permission notice appear on all copies       
 *   and supporting documentation, the name of M.I.T. not be used         
 *   in advertising or publicity pertaining to distribution of the        
 *   program without specific prior permission, and notice be given       
 *   in supporting documentation that copying and distribution is         
 *   by permission of M.I.T.  M.I.T. makes no representations about       
 *   the suitability of this software for any purpose.  It is pro-        
 *   vided "as is" without express or implied warranty.		          
 ****************************************************************************
 * $Header: /usr/zealand1/dfk/research/parallel/proteus/current/engine/RCS/thread.c,v 1.2 1996/07/21 14:44:41 dfk Exp $
 * $Log: thread.c,v $
 * Revision 1.2  1996/07/21 14:44:41  dfk
 * I also modified sched_initialize in rt_*_sched.ca to turn off cycle
 * counting, because it's called from here in init_threads, when we are
 * not prepared for a context switch.   But that procedure's prologue and
 * epilogue still count a few cycles (9 for FCFS), and call SimQuantum
 * if that causes end of quantum.
 *
 * So here I changed init_threads so that the call to sched_initialize
 * cannot possibly trigger a context switch, by subtracting and then
 * adding back 100 cycles to the clock.
 *
 * Revision 1.1  1996/07/21 14:40:48  dfk
 * Initial revision
 *
 * Revision 1.4  1993/01/22  20:03:38  wchsieh
 * fixed case where simulator_tid returns 0 as a tid
 * 0 no longer can be returned
 *
 * Revision 1.3  1992/10/05  12:24:39  brewer
 * Added arg type void for procedures with no args.
 *
 * Revision 1.2  92/04/02  14:18:34  brewer
 * changed code that copies thread name to remove fence-post bug
 * 
 * Revision 1.1  92/02/11  13:56:32  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "user.h"
#include "simreq.h"
#include "processor.h"
#include "intreq.h"
#include "mem.h"   /* for stack routines */
#include "q.h"     /* for newqueue() */
#include "event.h"

static int nextthread = MAX_THREADS-1; /* index of next free thread */

GLOBAL ThreadInfo thread_table_[MAX_THREADS];

GLOBAL int numthreads;       /* No of currently active threads */
GLOBAL int currtid;          /* Id of current thread */
GLOBAL ThreadInfo *currtptr; /* Addr. of ThreadInfo block for current thread */
GLOBAL Thread *currosptr;

/**************************************************************************/
/* create_thread -- Create a new thread: allocate stack, fill thread table
 * entry, return its id 
 */

GLOBAL ulong create_thread(ulong tid, FuncPtr func, unsigned stacksize,
			   const char *name, Time timestamp,
			   void *OS_block, unsigned argc, Word *argv) 
{
    int i, n;
    ThreadInfo *tptr;
    Word *a;

#ifdef THREADDEBUG
    printf("Entering create_thread(%d,0x%x,%d,%s,time=%lu,OS=0x%x,argc=%d)\n",
	   tid, func, stacksize, name, timestamp, OS_block, argc);
#endif

    /* Check arguments, get a free tid and allocate memory for thread stack */
    if ( argc > MAX_ARGS ) {
	fatal("create_thread: too many arguments (max = %d)", MAX_ARGS);
    }

    if ( (n = allocstk(stacksize)) == ERROR) {
#ifdef FATAL
	fatal("create_thread: cannot allocate stack for thread %d", tid);
#endif
	return(ERROR);
    }

    tptr = &thread_table_[tid];

    /* Fill ThreadInfo structure */
    for(i=0; i<T_NAMELEN && (tptr->t_name[i] = name[i]); i++);
    tptr->t_name[i-1]=0;
    tptr->t_stkblk = n;
    tptr->t_stkbase = stkbase(n);
    tptr->t_regs[SP] = (Word) stktop(n);
    tptr->t_regs[RIP] = (Word) thread_top_level;
    tptr->t_regs[ARG1] = (Word) tptr->t_regs;
    tptr->t_time = timestamp;

    tptr->t_OS_block = (char *)OS_block;

    /* Copy thread function and arguments into appropriate ThreadInfo fields */
    tptr->t_func = func;
    a = argv;                      /* a points to first argument    */ 
    for( i=0; i<argc; i++, a++)    /* copy arguments into t_a table */
      tptr->t_a[i] = *a;

#ifdef THREADDEBUG
    printf("Arguments passed:\n");
    for(i=0; i<argc; i++)
      printf("%ld,",tptr->t_a[i]);
    printf("\n");
#endif

    return(tid);
}


/**************************************************************************/
/* init_threads -- initialize thread table */

static void init_runtime_threads(void)
{
    int i;
    
    for(i=0; i<NO_OF_PROCESSORS; i++) {
	threadPrivate *p = &processor_private[i];
	p->nextthread = MAX_PROC_THREADS-1;
	p->sleep_nonempty = FALSE;
	p->clockq = newqueue();
	p->quantum = OS_QUANTUM;
	define_timer_handler(i, (FuncPtr) _timer_handler);
    }

    define_local((char **)&PP, (char *)processor_private,
		 sizeof(threadPrivate));
}

GLOBAL void init_threads(void) 
{
    init_runtime_threads();
    AddTime(-100); /* so sched_initialize won't trigger context switch */
    sched_initialize();
    AddTime(100);
}

/**************************************************************************/
/* newtid -- allocate an unused thread table slot */

GLOBAL ulong simulator_tid(void) 
{
    ulong tid;
    int i;

    for(i=0; i < MAX_THREADS; i++) {
	if ( (tid = nextthread--) <= 0)
	  nextthread = MAX_THREADS-1;
	if (thread_table_[tid].t_state == T_FREE) {
	  thread_table_[tid].t_state = T_RESERVED;
	  return(tid);
	}
    }

    fatal(
      "cannot create new thread -- increase size of simulator thread table");

    /* unreachable, return statement just prevents warnings */
    return((ulong) 0);  
}


GLOBAL void release_simulator_tid(tid) 
ulong tid;
{
    ThreadInfo *tptr;
    (tptr = &thread_table_[tid])->t_state = T_FREE;
    tptr->t_OS_block = NULL;
}


/**************************************************************************/
/* thread_top_level -- all threads start execution from this function */

GLOBAL void thread_top_level(void) 
{
    ThreadInfo *tp = currtptr;

    CheckForInterrupts();

    (*tp->t_func)(tp->t_a[0], tp->t_a[1], tp->t_a[2], tp->t_a[3], 
		  tp->t_a[4], tp->t_a[5], tp->t_a[6],
		  tp->t_a[7], tp->t_a[8], tp->t_a[9]);

    thread_destroy(MY_TID);
}
