 
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: sim.h -- Global Simulator Header File
 *
 * Description:
 *     Constants and data type definitions used throughout the simulator
 *
 * Last Modified: 6-5-91  (eab)
 *
 * Types:
 *     ulong, Time, Word, FuncPtr
 *
 * Data Structures:
 *     ThreadInfo: Structure with control information for a thread
 *
 * Constants: see below
 *
 * Referenced parameters: none
 *
 * Macros:
 *     MAKE_REQUEST_<k>, MAKE_RESUME_REQUEST
 *     GLOBALTIME, CURRENTTIME
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ***************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/sim.h,v 1.3 94/01/24 00:39:37 dfk Time64bit Locker: dfk $
 * $Log:	sim.h,v $
 * Revision 1.3  94/01/24  00:39:37  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.2  93/02/26  01:04:48  dfk
 * wrapped FALSE and TRUE to avoid warnings
 * 
 * Revision 1.1  93/02/26  01:04:15  dfk
 * Initial revision
 * 
 * Revision 1.9  1993/01/14  16:59:13  brewer
 * Updated MININT to be ANSI compatible.
 *
 * Revision 1.8  1992/12/13  20:19:39  brewer
 * Changed cycles_ to be volatile for augmented files (_CA_)
 *
 * Revision 1.7  92/11/19  14:17:46  brewer
 * Added PLURAL macro.
 * 
 * Revision 1.6  92/11/13  17:40:16  brewer
 * Updated definitions of MAXINT, MININT
 * 
 * Revision 1.5  92/07/20  12:43:02  brewer
 * Fixed MAKE_REQUEST_2 to set argc to 2 instead of 1
 * Added MAKE_REQUEST_3
 * 
 * Revision 1.4  92/07/06  11:24:50  brewer
 * Added MAKE_REQUEST_2
 * 
 * Revision 1.3  92/04/28  11:21:16  brewer
 * Added PROFILE_CALLS macro.
 * 
 * Revision 1.2  92/04/02  14:43:21  brewer
 * Changed declaration of cycles_ from Time to int
 * 
 * Revision 1.1  92/02/11  14:19:59  brewer
 * Initial revision
 * 
 \**************************************************************************/

#ifndef SIM_H
#define SIM_H

#include <stdio.h>
#include "common.h"

/* Constants used throughout the program */

#define EMPTY             -1
#define USER               0
#define ENGINE             1
#define MININT             (int)(-2147483647)  /* ANSI C */
#define MAXINT             (int)(2147483647)   /* ANSI C */
#define PREEMPT_ON         TRUE
#define PREEMPT_OFF        FALSE
#define INTR_ON            TRUE
#define INTR_OFF           FALSE
#define TAG_MISMATCH       -654370321
#define YES                TRUE
#define NO                 FALSE
#define FULL               TRUE
#define NOATTACH          -1
#define ON                 TRUE
#define OFF                FALSE
#define FLUSH              2
#define KILLED             -760287105
#define I_DEFAULT         -1

/* warning: min and max evaluate x, y multiple times */
#define MIN(x,y)           ( ((x)<(y)) ? (x) : (y))
#define MAX(x,y)           ( ((x)>(y)) ? (x) : (y))

#include <assert.h>


/***************************************************************************\
*   Type definitions
\***************************************************************************/

typedef unsigned long      Word;
typedef Word (*FuncPtr)();

/* data structure used by augmented code to store profiling data */
typedef struct {
    ulong total_time;   /* must be first field */
    ulong numcalls;     /* must be second field */
} profileInfo;


/***************************************************************************\
*   Global variables for communication between engine and current thread
\***************************************************************************/

int  processor_;   /* processor where current thread is executing     */

/* cycles_  contains number of cycles left for current thread */
#ifdef _CA_
volatile signed long cycles_; /* volatile for augemented files */
#else
signed long cycles_;      
#endif

Time base_time_;   /* starting (resume) time for current thread       */
Time sim_quantum_; /* standard value of simulator quantum             */
Time quantum_;     /* min of sim_quantum and OS_quantum_left          */
char *stackmin_;   /* bottom of current stack, overflow if stack
			     pointer goes below this value */
BOOL expired_;     /* was the last simulator call an "expire" ?       */
int  why_expired_; /* reason of expiration (SIM_EXPIRED or OS_EXPIRED)*/

Word result_;      /* result of simulator call                        */
int  mode_;        /* USER or ENGINE mode                             */
BOOL profiling_;   /* is the user program doing profilling?           */

BOOL sim_running_;  /* Distinguishes between initialize. and running phase */


/****************************************************************************/
/* global macros */

#define GLOBALTIME  (hash_time_ + hash_index_)
#define CURRENTTIME (base_time_ - cycles_)

#define MAKE_REQUEST_1(type, the_tid, time, a1) \
{ SimRequest *req = new_request_(); req->h.reqtype = (type); \
  req->h.tid = (the_tid); req->h.timestamp = (time); req->h.argc = 1; \
  req->argv[0] = (Word)(a1); enqueue_request_(req); }

#define MAKE_REQUEST_2(type, the_tid, time, a1, a2) \
{ SimRequest *req = new_request_(); req->h.reqtype = (type); \
  req->h.tid = (the_tid); req->h.timestamp = (time); req->h.argc = 2; \
  req->argv[0]=(Word)(a1); req->argv[1]=(Word)(a2); enqueue_request_(req); }

#define MAKE_REQUEST_3(type, the_tid, time, a1, a2, a3) \
{ Word *a; SimRequest *req = new_request_(); req->h.reqtype = (type); \
   req->h.tid = (the_tid); req->h.timestamp = (time); req->h.argc = 3; \
   a = req->argv; *a++ = (Word)(a1); *a++ = (Word)(a2); *a = (Word)(a3); \
   enqueue_request_(req); }

#define MAKE_REQUEST_4(type, the_tid, time, a1, a2, a3, a4) \
{ Word *a; SimRequest *req = new_request_(); req->h.reqtype = (type); \
  req->h.tid = (the_tid); req->h.timestamp = (time); req->h.argc = 4; \
  a = req->argv; *a++ = (Word)(a1); *a++ = (Word)(a2); *a++ = (Word)(a3); \
  *a = (Word)(a4); enqueue_request_(req); }

#define MAKE_REQUEST_5(type, the_tid, time, a1, a2, a3, a4, a5) \
{ Word *a; SimRequest *req = new_request_(); req->h.reqtype = (type); \
  req->h.tid = (the_tid); req->h.timestamp = (time); req->h.argc = 5; \
  a = req->argv; *a++ = (Word)(a1); *a++ = (Word)(a2); *a++ = (Word)(a3); \
  *a++ = (Word)(a4); *a++ = (Word)(a5); enqueue_request_(req); }

#define MAKE_REQUEST_6(type, the_tid, time, a1, a2, a3, a4, a5, a6) \
{ Word *a; SimRequest *req = new_request_(); req->h.reqtype = (type); \
  req->h.tid = (the_tid); req->h.timestamp = (time); req->h.argc = 6; \
  a = req->argv; *a++ = (Word)(a1); *a++ = (Word)(a2); *a++ = (Word)(a3); \
  *a++ = (Word)(a4); *a++ = (Word)(a5); *a = (Word)(a6); \
  enqueue_request_(req); }

#define MAKE_REQUEST_7(type, the_tid, time, a1, a2, a3, a4, a5, a6, a7) \
{ Word *a; SimRequest *req = new_request_(); req->h.reqtype = (type); \
  req->h.tid = (the_tid); req->h.timestamp = (time); req->h.argc = 7; \
  a = req->argv; *a++ = (Word)(a1); *a++ = (Word)(a2); *a++ = (Word)(a3); \
  *a++ = (Word)(a4); *a++ = (Word)(a5); *a++ = (Word)(a6); *a = (Word)(a7); \
  enqueue_request_(req); }

#define MAKE_RESUME_REQUEST(the_tid, time, result) \
{ SimRequest *req = new_request_(); req->h.reqtype = SC_RESUME; \
  req->h.tid = (the_tid); req->h.timestamp = (time); req->h.argc = 1; \
  req->argv[0] = (Word)(result); enqueue_request_(req); }

#define CancelPendingTimerInterrupt(pp) \
   { \
       if ((pp)->p_timerreq != NULL) { \
	  ((pp)->p_timerreq)->h.reqtype = SC_NOP; \
	  (pp)->p_timerreq = NULL; \
	  } \
   }


/* PLURAL(x) returns "s" if x != 1, "" otherwise */
#define PLURAL(x) (((x) != 1)?"s":"")

#define IPRIO_TRAP        1000


/**************************************************************************\
* Declare all GLOBAL procedures
\**************************************************************************/

#include "prototypes.h"

/* PROFILE_CALLS macro generates number of calls info for non .ca files */
#ifndef _CA_
#define PROFILE_CALLS \
{ static profileInfo prof; \
  if (prof.numcalls++==0) set_prof_(GetAddress_(), &prof); }
#endif



#endif  /* ifndef SIM_H */
