
/**************************************************************************\
 *                                                                          *
 *                 Proteus Parallel-Architecture Simulator                  *
 *                Eric A. Brewer  and  Chris N. Dellarocas                  *
 *                     Laboratory for Computer Science                      *
 *                  Massachusetts Institute of Technology                   *
 *                                                                          *
 * Module: shmem.common.c
 *                                                                          *
 * Description: Routines common to all shared-memory modules
 *                                                                          *
 * Last Modified:  6-4-91  (eab)                                            *
 *                                                                          *
 * Global Functions:                                                        *
 *                                                                          *
 * Global Variables: none                                                   *
 *                                                                          *
 ****************************************************************************
 *   Copyright 1991                                                         *
 *   Eric A. Brewer  and  Chris N. Dellarocas                               *
 *   Massachusetts Institute of Technology                                  *
 *                                                                          *
 *   Permission to use, copy, modify, and distribute this program           *
 *   for any purpose and without fee is hereby granted, provided            *
 *   that this copyright and permission notice appear on all copies         *
 *   and supporting documentation, the name of M.I.T. not be used           *
 *   in advertising or publicity pertaining to distribution of the          *
 *   program without specific prior permission, and notice be given         *
 *   in supporting documentation that copying and distribution is           *
 *   by permission of M.I.T.  M.I.T. makes no representations about         *
 *   the suitability of this software for any purpose.  It is pro-          *
 *   vided "as is" without express or implied warranty.		            *
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/shmem.common.c,v 1.2 94/01/24 00:39:36 dfk Time64bit Locker: dfk $
 * $Log:	shmem.common.c,v $
 * Revision 1.2  94/01/24  00:39:36  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  16:10:16  dfk
 * Initial revision
 * 
 * Revision 1.2  92/04/01  11:24:01  brewer
 * cleanup: fixed gcc warnings, ANSI prototypes
 * 
 * Revision 1.1  92/02/11  13:56:25  brewer
 * Initial revision
 * 
 \**************************************************************************/


#ifdef BUS
#ifdef WATCH_BUS
#define EVENT_BUSIDLE(time)   ATEVENT(EV_BUSIDLE, 0, time)
#define EVENT_BUSBUSY(time)   ATEVENT(EV_BUSBUSY, 0, time)
#else
#define EVENT_BUSIDLE(time)
#define EVENT_BUSBUSY(time)
#endif

#ifdef WATCH_BUS
#define BUS_UTIL(t, b) \
  /* if ((t) > (b) || (b) == 0) { \
     if ((b) > 0) EVENT_BUSIDLE(b); \
     EVENT_BUSBUSY(t); \
     } */
#define BUS_CONTENTION(t, b) \
{ \
    Time d = ((b) > (t)) ? ((b) - (t)) : 0; \
      SUM_STAT(STAT_BUSCONT, (d), (t)); \
    }
#else
#define BUS_UTIL(t, b)
#define BUS_CONTENTION(t, b)
#endif


Time BusAccess(Time t) 
{
    extern Time bus_free_again;
    
    BUS_UTIL(t, bus_free_again);
    BUS_CONTENTION(t, bus_free_again);
    t = MAX(t, bus_free_again) + MEM_ACCESS_LATENCY;
    bus_free_again = t; 
    return t;
}
#endif

#define IndexToSid(a)     ((a)/LINE_SIZE)

int AddressToProcessor(ulong address) 
{
    Word sid     = IndexToSid(address);
    
    if (sid < MAX_SEMAPHORES)
      return(semaphore[sid].processor);
    else
      return(((unsigned)(address-MAX_SEMAPHORES*LINE_SIZE))
	     >> MODULE_BITS);
}


/*
 * Choose processor according to low order bits (provides interleaving)
 *
 AddressToProcessor(address) 
 Word address;
 {
 Word sid     = IndexToSid(address);
 
 if (sid < MAX_SEMAPHORES)
 return(semaphore[sid].processor);
 else
 return( (address-MAX_SEMAPHORES*LINE_SIZE) % NO_OF_PROCESSORS);
 }
 */



Word shared_memory(void *address, Word mode) 
{
    Word val;
    
    mode &= ACCESS_MASK;
    
    switch(mode)
      {
	case BYTE:
	  val = *(char *)address;
	  break;
	case HALFWORD:
	  val = *(short *)address;
	  break;
	case WORD:
	  val = *(Word *)address;
	  break;
	default:
	  printf("illegal access mode %lu in shared memory read", mode);
	  val = *(Word *)address;
      }
    
#ifdef MEMDEBUG
    printf("shared_memory(%x, 0%o) returns %d\n",address, mode, val);
#endif
    return(val);
}


void write_shared_memory(void *address, Word value, Word mode) 
{
    Word atomic = mode & ATOMIC_MASK;
    Word access = mode & ACCESS_MASK;
    
    switch(atomic)
      {
	case FADD:
	  value = shared_memory(address, mode) + value;
	  break;
	case FAND:
	  value = shared_memory(address, mode) & value;
	  break;
	case FAOR:
	  value = shared_memory(address, mode) | value;
	  break;
	case INCREMENT:
	  value = shared_memory(address, mode)+1;
	  break;
	case DECREMENT:
	  value = shared_memory(address, mode)-1;
	  break;
      }
    
    switch(access)
      {
	case BYTE:
	  *(char *)address = (char)value;
	  break;
	case HALFWORD:
	  *(short *)address = (short)value;
	  break;
	case WORD:
	  *(Word *)address = (Word)value;
	  break;
	default:
	  printf("illegal access mode %lu in shared memory write", mode);
	  *(Word *)address = (Word)value;
      }
#ifdef MEMDEBUG
    printf("write_shared_memory(%x, %d, 0%o) writes %d\n",address,value,mode,value);
#endif
}


Word shmem_result(Word previous, Word value, Word mode) 
{    
    if (mode & PREVIOUS_MASK)
      return previous;
    else
      return value;
}



double shared_memory_fl(void *address, Word mode) 
{
    double val;
    
    mode &= ACCESS_MASK;
    
    switch(mode)
      {
	case DOUBLEWORD:
	  val = *(double *)address;
	  break;
	case WORD:
	  val = (double)*(float *)address;
	  break;
	default:
	  printf("illegal access mode %lu in shared memory fl_read", mode);
	  val = *(double *)address;
      }
    
#ifdef MEMDEBUG
    printf("Shared_Memory_Fl(%d,%d) = %lf\n", address, mode, val);
#endif
    
    return(val);
}


void write_shared_memory_fl(void *address, double value, Word mode) 
{
    Word atomic = mode & ATOMIC_MASK;
    Word access = mode & ACCESS_MASK;
    
    switch(atomic)
      {
	case INCREMENT:
	  value = shared_memory_fl(address, mode)+1;
	  break;
	case DECREMENT:
	  value = shared_memory_fl(address, mode)-1;
	  break;
      }
    
    switch(access)
      {
	case DOUBLEWORD:
	  *(double *)address = value;
	  break;
	case WORD:
	  *(float *)address = (float)value;
	  break;
	default:
	  printf("illegal access mode %lu in shared memory write_fl", mode);
	  *(double *)address = (double)value;
      }
}


double shmem_result_fl(double previous, double value, Word mode)
{    
    if (mode & PREVIOUS_MASK) {
	return previous;
    } else {
	switch(mode & ATOMIC_MASK) {
	  case INCREMENT:
	    return previous+1;
	  case DECREMENT:
	    return previous-1;
	  default:
	    return value;
	}
    }
}
