 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *      Eric A. Brewer  and  Chris N. Dellarocas  and  Anthony D. Joseph
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: profile.c
 *
 * Description: Loads and Outputs profile data, symbol table routines
 *
 * Last Modified:  12-Oct-92 (adj)
 *
 * Global Functions:
 *    void set_prof_(pc, profileInfo *)
 *        Record address of profileInfo for procedure at pc.
 *        Called only from InitProfile_
 *    void init_profile(char *executable_name)
 *        Store the executable name should profiling be used.
 *        up the profile data structures.
 *    void OutputFinalEvents(FILE *)
 *        Outputs the profiling data to the event file.
 *    char *sym_name_(long address)
 *        Returns the name of the global symbol preceeding `address'
2 *
 *    void user_event_(evt, value)
 *    void user_index_event_(evt, index, value)
 *    void user_time_event_(evt, value, time)
 *    void user_index_time_event_(evt, index, value, time)
 *
 *    void trace_stack_(FILE *out, ulong pc, ulong *tregs, ulong tid)
 *    void trace_thread_(FILE *out, int tid)
 *    void Disassemble(long address, long pc)
 *    void DisassembleSymbol(char *symbol)
 *
 *    char* time_print(Time)
 *
 * Global Variables: none
 *
 ****************************************************************************
 *   Copyright 1991, 1992                                                      
 *   Eric A. Brewer, Chris N. Dellarocas, and Anthony D. Joseph
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /usr/wildcat/dfk/research/parallel/proteus/proteus-V3.01/engine/RCS/profile.c,v 1.3 94/01/27 01:20:19 dfk Time64bit $
 * $Log:	profile.c,v $
 * Revision 1.3  94/01/27  01:20:19  dfk
 * Needed to change floats to doubles
 * to workaround a gcc bug where you can't do arithmetic
 * that mixes floats with long longs
 * 
 * Revision 1.2  94/01/24  00:39:18  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  11:35:03  dfk
 * Initial revision
 * 
 * Revision 1.11  92/12/09  15:16:24  brewer
 * Updated stack trace routines to take FILE *
 * 
 * Revision 1.10  92/12/07  13:52:50  brewer
 * (adj) Changes to Emerald and Fixup block support
 * 
 * Revision 1.9  92/11/16  13:57:03  brewer
 * Removed cc compiler warning from macro StorageSymbolType
 * 
 * Revision 1.8  92/11/13  17:03:06  brewer
 * Removed compiler warnings.
 * 
 * Revision 1.7  92/11/05  11:34:33  brewer
 * adj fixed minor bug in LookupThreadPC; now gets the right line number
 * 
 * Revision 1.6  92/11/03  13:29:36  brewer
 * adj added LookupThreadPC and related procedures
 * improved stack_trace to handle empty threads
 * 
 * Revision 1.5  92/10/07  16:43:16  brewer
 * changed OutputProfData to OutputFinalEvents and made event_file_ an arg
 * changed user event procs to use Event and IndexEvent
 * 
 * Revision 1.4  92/09/23  15:49:32  brewer
 * Included <stdlib.h>
 * added prototype for disassemble(...)
 * 
 * Revision 1.3  92/09/23  12:19:02  brewer
 * Added extensive symbol table procedures (added by adj)
 * 
 * Revision 1.2  92/04/28  11:18:29  brewer
 * Changed time_cmp to sort by numcalls if total_time is equal.
 * 
 * Revision 1.1  92/02/11  13:56:06  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include <filehdr.h>
#include <syms.h>
#include <ldfcn.h>
#include "sim.h"
#include "conf.param"
#include "rt_thread_def.h"
#include "event.h"
#include "thread.h"
#include "profile.h"

/* declare here instead of including <disassembler.h> because GNU
   C preprocessor can't handle the Pascal code */

LIBRARY int disassembler(ulong address, int regstyle,
                         const char * (*get_symname)(),
                         int (*get_regvalue)(),
                         long (*get_bytes)(),
                         void (*print_header)());

static void LoadSymbols();
static int thread_top_index;
extern char etext, _ftext; 	/* text segment boundaries; see "man ld" */

static LDFILE *ldptr; 		/* runtime data pointer */
static PDR theProc;   		/* procedure symbol work area */
static SYMR theSymbol;  	/* symbol work area */
static pLINER lines;		/* line numbers for file */
static pEXTR pext;		/* externals for this cu */

/* executable name: should be argv[0] */
#ifdef MAIN
static char *exec_name_ = "profile";
#else
static const char *exec_name_ = "proteus";
#endif

/* symbol table states */
#define UNINITIALIZED  -1
#define ABSENT         0

static int num_procs = UNINITIALIZED; 	/* number of procedures */
static int num_files = 0; 		/* number of file descriptors */


typedef struct {
    ulong address;			/* procedure's address */
    long procindex;			/* procedure's proc index */
    long index;				/* procedure's symbol table index */
    char *name;				/* procedure's name */
    profileInfo *prof;			/* profiling info */
    long startline, lowline, highline;	/* procedure's line number table
					   offset, low and high line numbers */
} profData;
static profData *prof_data_;	/* profile data array, allocated at runtime */

typedef struct {
  long index;				/* file's symbol table index */
  char *name;				/* filename */
  pFDR entry;				/* pointer to FDR (file descriptor
					   record) */
} files;
static files *file_data_;      	/* file data array, allocated at runtime */


/**************************************************************************\
* Symbol-table procedures
\**************************************************************************/

LIBRARY void ldreadst(LDFILE *, int);
LIBRARY int ldclose(LDFILE *);
LIBRARY char *ldgetname(LDFILE *, pSYMR);
LIBRARY long ldgetpd(LDFILE *, long, pPDR);
LIBRARY int ldtbread(LDFILE *, long, pSYMR);
LIBRARY pAUXU ldgetaux(LDFILE *, long);
LIBRARY void *calloc(size_t nelem, size_t size);

/**************************************************************************\
* Definitions for symbol table entries
\**************************************************************************/

/*
 * Storage Classes (as defined in /usr/include/symconst.h)
 */

static char *Sclass[] = {
  "scNil",
  "Text",	/* 1 -- text symbol */
  "Data",	/* 2 -- initialized data symbol */
  "Bss",	/* 3 -- un-initialized data symbol */
  "Register",	/* 4 -- value of symbol is register number */
  "Abs",	/* 5 -- value of symbol is absolute */
  "Undefined",	/* 6 -- who knows? */
  "CdbLocal",	/* 7 -- variable's value is IN se->va.?? */
  "Bits",	/* 8 -- this is a bit field */
  "CdbSystem/Dbx",/* 9 -- variable's value is IN CDB's address space,
		          overlap dbx internal use */
  "RegImage",	/* 10 -- register value saved on stack */
  "Info",	/* 11 -- symbol contains debugger information */
  "UserStruct",	/* 12 -- address in struct user for current process */
  "SData",	/* 13 -- load time only small data */
  "SBss",	/* 14 -- load time only small common */
  "RData",	/* 15 -- load time only read only data */
  "Var",	/* 16 -- Var parameter (fortran,pascal) */
  "Common",	/* 17 -- common variable */
  "SCommon",	/* 18 -- small common */
  "VarRegister",/* 19 -- Var parameter in a register */
  "Variant",	/* 20 -- Variant record */
  "SUndefined",	/* 21 -- small undefined(external) data */
  "Init",	/* 22 -- .init section symbol */
  "BasedVar"};	/* 23 -- Fortran or PL/1 ptr based var */ 


/*
 *   Symbol Types (as defined in /usr/include/symconst.h)
 */

static char *Symtype[] = {
  "stNil",	/* 0 -- Nuthin' special */
  "Global",	/* 1 -- external symbol */
  "Static",	/* 2 -- static */
  "Param",	/* 3 -- procedure argument */
  "Local",	/* 4 -- local variable */
  "Label",	/* 5 -- label */
  "Proc",	/* 6 --     "      "	 Procedure */
  "Block",	/* 7 -- beginnning of block */
  "End",	/* 8 -- end (of anything) */
  "Member",	/* 9 -- member (of anything	- struct/union/enum */
  "Typedef",	/* 10 -- type definition */
  "File",	/* 11 -- file name */
  "RegReloc",	/* 12 -- register relocation */
  "Forward",	/* 13 -- forwarding address */
  "StaticProc",	/* 14 -- load time only static procs */
  "Constant",	/* 15 -- const */
  "StaParam"};	/* 16 -- Fortran static parameters */


/* definitions for fields in TIR (Type Information Record) 
 *    (as defined in /usr/include/symconst.h)
 *    type qualifiers for ti.tq0 -> ti.(itqMax-1)
 *
 *    These are modifiers for the basic types listed below
 */

static char *Tq[] = {
  "",		/* 0 -- (Nil) bt is what you see */
  "*",		/* 1 -- pointer */
  "()",	/* 2 -- procedure */
  "Array",	/* 3 -- duh */
  "Far",	/* 4 -- longer addressing - 8086/8 land */
  "Vol",	/* 5 -- volatile */
  "",		/* 6 */
  "",		/* 7 */
  ""};		/* 8 */

/* 
 * Basic types as seen in ti.bt (as defined in /usr/include/symconst.h)
 */

static char *Bt[] = {
  "btNil ",		/* 0 -- undefined */
  "address ",		/* 1 -- address - integer same size as pointer */
  "char ",		/* 2 -- character */
  "unsigned char ",	/* 3 -- unsigned character */
  "short ",		/* 4 -- short */
  "unsigned short ",	/* 5 -- unsigned short */
  "int ",		/* 6 -- int */
  "unsigned int ",	/* 7 -- unsigned int */
  "long ",		/* 8 -- long */
  "unsigned long ",	/* 9 -- unsigned long */
  "float ",		/* 10 -- float (real) */
  "double ",		/* 11 -- Double (real) */
  "struct ",		/* 12 -- Structure (Record) */
  "union ",		/* 13 -- Union (variant) */
  "enum ",		/* 14 -- Enumerated */
  "typedef ",		/* 15 -- defined via a typedef, isymRef points */
  "range ",		/* 16 -- subrange of int */
  "set ",		/* 17 -- pascal sets */
  "complex ",		/* 18 -- fortran complex */
  "double complex ",	/* 19 -- fortran double complex */
  "indirect ",		/* 20 -- forward or unnamed typedef */
  "fixed decimal ",	/* 21 -- Fixed Decimal */
  "float decimal ",	/* 22 -- Float Decimal */
  "string ",		/* 23 -- Varying Length Character String */
  "bit ",		/* 24 -- Aligned Bit String */
  "picture ",		/* 25 -- Picture */
  "void "};		/* 26 -- void */


/**************************************************************************\
* Macros for accessing symbol table entries
\**************************************************************************/

#define GET_SYMBOL(IDX,SYM)   (ldtbread(ldptr, IDX, &SYM) ? 1 :               \
			       (fatal("symbol table lookup failed %ld\n",     \
				      IDX), 0))

#define GET_NAME(SYM) 	      ((char *)ldgetname(ldptr, &SYM))

#define GET_PROC(IDX,PROC)    (ldgetpd(ldptr, IDX, &PROC) ? 1 : 	      \
			       (fatal("Unable to load procedure %d\n",	      \
				      IDX), 0))

#define SYMBOLS_AVAIL() 	((num_procs != UNINITIALIZED) ? 1 :	      \
				 (LoadSymbols(),			      \
				  ((num_procs != ABSENT) ? 1 :		      \
				   (fprintf(stderr,			      \
					    "Symbol table not available.\n"), \
				    0))))

#define SymTabIndexToFileName(SI)	(find_file(SI)->name)

/* 
 * Compute the line number of a procedure address
 *    The line number table stores entries for each word in a proceedure.
 *    To compute the line number of an address, determine the offset in
 *    words from the start of the procedure, then add that to the
 *    procedure's offset into the line number table
 */
#define ProcAddressToLineNumber(ProcStartLine,ProcStartAddr,Addr) \
  ((long) lines[ProcStartLine +	((Addr - ProcStartAddr)/4)])

static long get_prologue_size(int procStart);

/**************************************************************************\
* Macros for determining the type of a symbol table entry
\**************************************************************************/

#define IsTextProc(SYM)	 ((SYM.sc == scText) && \
			  ((SYM.st == stProc) || (SYM.st == stStaticProc)))

#define IsTextBlock(SYM) ((SYM.sc == scText) && (SYM.st == stBlock))

#define IsTextEnd(SYM)   ((SYM.sc == scText) && (SYM.st == stEnd))

#define IsVariable(SYM)	 ((SYM.st == stStatic) || (SYM.st == stParam) || \
			  (SYM.st == stLocal))

#define IsParam(SYM)	 (SYM.st == stParam)

#define IsLocal(SYM)	 (SYM.st == stLocal)

#define IsStatic(SYM)	 (SYM.st == stStatic)

#define IsAbsolute(SYM)  (SYM.sc == scAbs)

#define IsRegister(SYM)  (SYM.sc == scRegister)

#define IsBss(SYM)  	 (SYM.sc == scBss)

#define IsExternal(IDX)	 (IDX >= SYMHEADER(ldptr).isymMax)

#define NumSymbols()	 (SYMHEADER(ldptr).isymMax + SYMHEADER(ldptr).iextMax)

#define NumIntSymbols()	 (SYMHEADER(ldptr).isymMax)

#define NumExtSymbols()	 (SYMHEADER(ldptr).iextMax)

#define StorageSymbolType(SYM) \
  ((IsAbsolute(SYM) && (IsLocal(SYM) || IsParam(SYM))) ? FRAME_ABS : \
   ((IsRegister(SYM) && (IsLocal(SYM) || IsParam(SYM))) ? FRAME_REG : \
    ((IsBss(SYM) && IsStatic(SYM)) ? FRAME_BSS \
     : (fatal("StorageSymbolType: Unknown storage class (%s) or type (%s)\n", \
	      Sclass[SYM.sc], Symtype[SYM.st]), 0))))

/****************************************************************************/

/* find the nearest procedure, and return its profData index */
static int find_proc(ulong address)
{
    int top, bottom, mid;

    /* load symbol table if necessary */
    if (!SYMBOLS_AVAIL()) return(0);

    /* Check for address after end of text or before beginning of text */
    if ((address > (ulong)&etext) || (address < (ulong)&_ftext))
      return(-1);

    top = 0;
    bottom = num_procs - 1;
    while (top < bottom) {
	mid = (top + bottom + 1) / 2;
	if (address < prof_data_[mid].address) {
	    bottom = mid - 1;
	} else {
	    top = mid;
	}
    }
    return(top);
}

/****************************************************************************/

/* find the nearest file, and return its files entry */
static files *find_file(long index)
{
    int top, bottom, mid;

    /* load symbol table if necessary */
    if (!SYMBOLS_AVAIL()) return(0);

    top = 0;
    bottom = num_files - 1;
    while (top < bottom) {
	mid = (top + bottom + 1) / 2;
	if (index < file_data_[mid].index) {
	    bottom = mid - 1;
	} else {
	    top = mid;
	}
    }
    return(&(file_data_[top]));
}


/****************************************************************************/

/* comparison function to sort by address in ascending order */
static int address_cmp(const profData *p1, const profData *p2)
{
    if (p1->address < p2->address) return(-1);
    if (p1->address > p2->address) return(1);
    return(0);
}


/* load the symbol table */
static void LoadSymbols()
{
    long i;
    char *name;
    pFDR pfd;

    printf("reading symbolic information ...");
    ldptr = ldopen(exec_name_, NULL);

    if (ldptr == NULL) {
	fprintf(stderr,
		"Warning: Unable to load symbol table from executable `%s'\n",
		exec_name_);
	fprintf(stderr, "    Continuing without symbol table.\n");
	num_procs = ABSENT;
	prof_data_ = (profData *) malloc(sizeof(profData));
	if (prof_data_ == NULL) fatal("Out of memory");
	bzero((char *)prof_data_, sizeof(profData));
	prof_data_[0].name = "???";
	return;
    }

    /* load symbol table */
    if (SYMTAB(ldptr) == NULL) {
	fprintf(stderr, "Warning: Executable does not have a symbol table.\n");
	num_procs = ABSENT;
	prof_data_ = (profData *) malloc(sizeof(profData));
	if (prof_data_ == NULL) fatal("Out of memory");
	bzero((char *)prof_data_, sizeof(profData));
	prof_data_[0].name = "???";
	return;
    }

    num_procs = SYMHEADER(ldptr).ipdMax;

    prof_data_ = (profData *) malloc(num_procs * sizeof(profData));
    if (prof_data_ == NULL)
      fatal("Unable to allocate memory for profiling data.\n");
    bzero((char *)prof_data_, num_procs*sizeof(profData));

    for (i=0; i<num_procs; i++) {
	GET_PROC(i, theProc);
	prof_data_[i].procindex = i;
	prof_data_[i].address = theProc.adr;
	prof_data_[i].index = theProc.isym;
	prof_data_[i].startline = theProc.iline;
	prof_data_[i].lowline = theProc.lnHigh;
	prof_data_[i].lowline = theProc.lnLow;

	GET_SYMBOL(theProc.isym, theSymbol);
	name = GET_NAME(theSymbol);
	prof_data_[i].name = malloc(strlen(name)+1);
	strcpy(prof_data_[i].name, name);
    }

    /* sort symbols by address in ascending order */
    qsort((char *)prof_data_, num_procs, sizeof(profData),
	  (int (*)())address_cmp);

    thread_top_index = find_proc((ulong)thread_top_level);

    profiling_ = TRUE;

    /* Get the file table associated with the symbol table */
    pfd = PFD(ldptr);			/* The file descriptors (this is the
					   in-memory copy loaded by SYMTAB) */
    num_files = PSYMTAB(ldptr)->cfd;	/* The count */
    file_data_ = (files *) malloc(num_files * sizeof(files));
    if (file_data_ == NULL)
      fatal("LoadSymbols: Unable to allocate memory for fd table.\n");
    bzero((char *)file_data_, num_files*sizeof(files));

    /* Iterate through the file descriptors, saving usefule information */
    for (i = 0; i < num_files; i++) {

      /* Get the symbol table entry of the file for this descriptor index */
      GET_SYMBOL(pfd[i].isymBase, theSymbol);

      /* Now, get the name */
      name = GET_NAME(theSymbol);
      /* Save the symbol table index, the entry itself and the name */
      file_data_[i].index = pfd[i].isymBase;
      file_data_[i].entry = &(pfd[i]);
      file_data_[i].name = malloc(strlen(name)+1);
      strcpy(file_data_[i].name, name);

/*    Uncomment this if you want warnings about files compiled without
      debugging information

      if (pfd[i].glevel == GLEVEL_0)
	fprintf(stderr, 
		"Warning: %s compiled without -g, symbol table incomplete\n",
		name);
*/
    }
    
    /* Initialize line number area pointer to point to the in-memory copy
       of line number information */
    lines = PSYMTAB(ldptr)->pline;	/* The line numbers */
    /* Initialize external symbol table pointer to point to the in-memory
       copy of external symbol information. We can't use the information
       returned by ldtbread because it doesn't provide the file offset
       (although the man page lies otherwise) */
    pext = PSYMTAB(ldptr)->pext;	/* The external symbols */

    printf("\n");
}


/* comparison function to sort by time in descending order */
static int time_cmp(profData *p1, profData *p2)
{
    unsigned long time1, time2, calls1, calls2;

    if (p1->prof == NULL) return 1;
    if (p2->prof == NULL) return -1;
    time1 = p1->prof->total_time;
    time2 = p2->prof->total_time;
    if (time1 < time2) return(1);
    if (time1 > time2) return(-1);
    /* times equal, judge by calls */
    calls1 = p1->prof->numcalls;
    calls2 = p2->prof->numcalls;
    if (calls1 < calls2) return(1);
    if (calls1 > calls2) return(-1);
    return(0);
}


/* find the nearest symbol, and return its profData address */
/* called by assembly routine InitProfile_ */
GLOBAL void set_prof_(ulong pc, profileInfo *data)
{
    int top, bottom, mid;

    /* initialize profiling if necessary */
    if (num_procs == UNINITIALIZED) LoadSymbols();

    top = 0;
    bottom = num_procs - 1;
    while (top < bottom) {
	mid = (top + bottom + 1) / 2;
	if (pc < prof_data_[mid].address) {
	    bottom = mid - 1;
	} else {
	    top = mid;
	}
    }
    prof_data_[top].prof = data;
    /* fprintf(stderr, "***** set prof_data_(%s) to 0x%lx\n",
	    prof_data_[top].name, data); */
}


GLOBAL const char *sym_name_(address)
ulong address;
{
    return(prof_data_[find_proc(address)].name);
}

/***************************************************************************/


GLOBAL void init_profile(char *executable)
{
    exec_name_ = executable;
}


/* output profiling information to the events file */

GLOBAL void OutputFinalEvents(FILE *event_file_)

{
    long i, count;
    const char *ptr;

    if (profiling_) {
	/* sort symbols by total_time in descending order */
	qsort((char *)prof_data_, num_procs, sizeof(profData),
	      (int (*)())time_cmp);
	
	/* count symbols */
	for (count=0; count<num_procs; count++) {
	    if (prof_data_[count].prof == NULL) break;
	    if (prof_data_[count].prof->numcalls == 0) break;
	}
	
	putw(EV_INIT_PROF | count, event_file_);
	
	for (i=0; i<count; i++) {
	    putw(EV_PROF | i, event_file_);
	    for (ptr = prof_data_[i].name; *ptr != 0; ptr++)
	      putc(*ptr, event_file_);
	    putc(0, event_file_);
	    putw(prof_data_[i].prof->numcalls, event_file_);
	    putw(prof_data_[i].prof->total_time, event_file_);
	}
    }

/*     ldclose(ldptr); */
}



/*****************************************************************************
   Proteus-specific event-generation routines
*****************************************************************************/

/* these are provided only for backward compatibility */

GLOBAL void user_event_(ulong evt, ulong value)
{
    Event(evt, value, base_time_ - cycles_);
}


GLOBAL void user_index_event_(ulong evt, ulong index, ulong value)
{
    IndexEvent(evt, index, value, base_time_ - cycles_);
}


GLOBAL void user_time_event_(ulong evt, ulong value, Time time)
{
    Event(evt, value, time);
}


GLOBAL void user_index_time_event_(ulong evt, ulong index,
				   ulong value, Time time)
{
    IndexEvent(evt, index, value, time);
}



/***************************************************************************\
*   Tracing routines
\***************************************************************************/

/*
 * Search the linked-list of frames to see if the register is
 * saved somewhere
 *
 * Heuristically try to find a value in a register in a frame. Since child
 * frames may have used the register, the current value of the register may
 * not be the correct value. Any child that uses the register, however, must
 * save it (if it is a register that is preseved by procedure cals). Since
 * stack frames are traced from child to parent, we have a list of each
 * child's frame, with register usage and saved register location on
 * the stack. Locating a register starts by tracing through the list of
 * frames looking for an indication that it has been saved. If it is
 * not saved, the current value of the register is valid and the appropriate
 * current register is returned. Otherwise, if the register is found
 * in a child frame, the offset value from that frame is returned.
 */

static ulong get_reg_offset(int regno, ulong *tregs, FrameInfo *info)
{
  FrameInfo *ptr;

  if (info == NULL) {
     /* Register never touched, return current value */
    return((ulong) &(tregs[regno]));
  } else {
    ptr = info->child;	/* Start with the child frame */
    while (ptr) {
      if (ptr->regs[regno]) 		/* Check for the frame here */
	return(ptr->regs[regno]);	/* Got it */
      else
	ptr = ptr->child;		/* Not found, try this frame's child */
    }
    /* Register never touched, return current value */
    return((ulong) &(tregs[regno]));
  }
}


/* 
 * Given a thread's pc (may be the same as $31) and its registers
 * (at a minimum, the pc ($31), stack pointer ($29), fp ($30)),
 * heuristically, determine the procedure parameters for the top
 * frame on the stack. 
 *
 * The symbol table provides the same information, however it may not
 * always be correct due to insufficient debugging information (e.g.,
 * not compiled with -g), alloca's, or changes made by augment (e.g., 
 * calls to SimQuantum from the prologue, before the frame is
 * completely configured). The core of this procedure is based
 * upon code from gdb-4.6's mips routines for tracing stacks, 
 * specifically, heuristic_proc_desc in mips-tdep.c .
 * 
 * The procedure works by looking in the actual assembly language for the
 * procedure on the stack. It looks for stack pointer, frame pointer and
 * other register saves on the stack. It also computes several other
 * useful procedure paramters.
 *
 * FUTURE MODIFICATIONS --- This really should be done: Currently, we use
 * get_prologue_size, but it's not accurate for procedures compiled without 
 * debugging. As a sanity check, scanning of the assembly code should end
 * with the first jump encountered.
 */
static FrameInfo *heuristic_get_frame_info(ulong pc, ulong *tregs)

{
  PDR proc;			/* Procedure descriptor */
  FrameInfo *info;		/* Frame information */
  ulong start_pc, limit_pc;	/* Proc. start and estimate of prologue end */
  ulong cur_pc;			/* Current program counter */
  int reg30 = -1;       	/* Value of $r30. Used by gcc for frame ptr */
  int frame_size, i;		/* Size of frame and temporary variable */
  ulong sp = tregs[SP];		/* Stack pointer */
  ulong fp = tregs[FP];		/* Frame pointer */
  ulong ilimit;

  i = find_proc(pc);		/* Get the containing procedure */

  /* Stop when we hit the top of the stack or get an invalid symbol */
  if (i == thread_top_index  || (i <= 0)) return (NULL);
    
  start_pc = prof_data_[i].address;	/* Get procedure start */

  /* Get the end of the prologue */
  ilimit = limit_pc = start_pc + get_prologue_size(prof_data_[i].index);
  /* Check if the limit is after the pc. If it is, it means that either:
     1. The value returned by get_prologue_size is too big (quite likely)
     2. The procedure is in the prologue (as in we've called SimQuantum)
     For case 2, by stopping at the pc, we get the location that
     augment has stashed the return pc */
/* We need to find where the arguments get stashed, so keep looking up to pc
   if it's outside the prologue
  if ((pc != start_pc) && (limit_pc > pc))
    limit_pc = pc; */

  if ((pc != start_pc) && (ilimit > pc))
    ilimit = pc;

#ifdef DEBUG_PROF
  printf("Start_pc 0x%x limit_pc 0x%x ilimit 0x%x pc 0x%x\n", start_pc,
	 limit_pc, ilimit, pc);
#endif

  if (pc != start_pc)
      limit_pc = pc;

  /* Allocate a new frame descriptor and save some information */
  info = (FrameInfo *) calloc(1, sizeof(FrameInfo));
  info->fp = fp;
  info->sp = sp;
  info->cur_sym = i;

  /* Sanity check, since get_prologue_size may not be too large */
/*  if (start_pc + 200 < limit_pc) limit_pc = start_pc + 200; */

 restart:
  frame_size = 0;			/* Start w/ zero size frame */
  /* Loop through the frame, a word at a time */
  for (cur_pc = start_pc; cur_pc < limit_pc; cur_pc += 4) {
    unsigned long word;

    word =  *((ulong *) cur_pc);	/* Get the instruction */
/*    printf("0x%lx: 0x%lx  fs:%d\n", cur_pc, word, frame_size); */
    if ((word & 0xFFFF0000) == 0x27bd0000) 	       /* addiu $sp,$sp,-i */
      frame_size += (-word) & 0xFFFF;
    else if ((word & 0xFFFF0000) == 0x23bd0000)    /* addu $sp,$sp,-i */
      frame_size += (-word) & 0xFFFF;
    else if ((word & 0xFFE00000) == 0xafa00000) {  /* sw reg,offset($sp) */
      int reg = (word & 0x001F0000) >> 16;
#ifdef DEBUG_PROF
      printf("Found register %d [0x%x] word: %d ", reg, sp + (short)word,
	     (short) word);
#endif
      if (((reg == 31) && ((short) word == -4) && 
	   (info->regs[reg] || ((pc <= cur_pc) && ((pc - 16) > cur_pc))))
	  || ((cur_pc > ilimit) && ((reg < 4) || (reg > 7)))) {
#ifdef DEBUG_PROF
	printf("unused\n");
#endif
	continue;
      } else {
#ifdef DEBUG_PROF
	printf("USED\n");
#endif
	info->reg_mask |= 1 << reg;
	info->regs[reg] = sp + (short)word;
      }
    } else if ((word & 0xFFFF0000) == 0x27be0000) {/* addiu $30,$sp,size */
      if ((unsigned short)word != frame_size)
	reg30 = sp + (unsigned short)word;
      else if (!info->has_frame_reg) {
	int alloca_adjust;
	info->has_frame_reg = 1;
	
	reg30 = fp;
	alloca_adjust = reg30 - (sp + (unsigned short)word);
	if (alloca_adjust > 0) {
	  /* FP > SP + frame_size. This may be because
	   * of an alloca or somethings similar.
	   * Fix sp to "pre-alloca" value, and try again.
	   */
#ifdef DEBUG_PROF
	  printf("Found alloca frame: %d\n", alloca_adjust);
#endif
	  sp += alloca_adjust;
	  for (i = 0; i < 32; info->regs[i++] = 0);
	  goto restart;
	}
      }
    } else if ((word & 0xFFE00000) == 0xafc00000) {/* sw reg,offset($30) */
      int reg = (word & 0x001F0000) >> 16;
/*
      if (reg30 == -1)
	fprintf(stderr,
		"profile.c:%d: warning: $30 used but not initialized\n",
		__LINE__);
*/
#ifdef DEBUG_PROF
      printf("Found register %d [0x%x]\n", reg, reg30 + (short)word);
#endif
      if ((cur_pc > ilimit) && ((reg < 4) || (reg > 7))) {
#ifdef DEBUG_PROF
	printf("unused\n");
#endif
	continue;
      } else {
#ifdef DEBUG_PROF
	printf("USED\n");
#endif
	info->reg_mask |= 1 << reg;
	info->regs[reg] = reg30 + (short)word;
      }
    }
  }
  if (info->has_frame_reg) {
    info->frame_reg = FP;
    info->frame_offset = 0;
#ifdef DEBUG_PROF
    if (fp != (sp + frame_size))
      printf("FP [0x%x] != (SP [0x%x] + Frame Size [0x%x])\n", fp, sp,
	     frame_size); 
#endif
    info->sp = sp;
    info->new_sp = fp;
    info->new_fp = *((ulong *) info->regs[FP]);
  } else {
#ifdef DEBUG_PROF
    printf("sp reg\n"); 
#endif
    info->frame_reg = SP;
    info->frame_offset = frame_size;
    info->new_sp = sp + frame_size;
    info->new_fp = fp;
    info->fp = info->new_sp;
#ifdef DEBUG_PROF
    if (info->new_sp == sp)
	  printf("0 frame size\n");
#endif
  }
  info->pc_reg = RIP;

  GET_PROC(prof_data_[info->cur_sym].procindex, proc);

  /* printf("mask = 0x%lx\n", proc.regmask);
     printf("framesize = %d\n", proc.frameoffset);
     printf("pcreg = %d = 0x%x\n", proc.pcreg, proc.pcreg);
     printf("regoffset = 0x%x\n", proc.regoffset);
     printf("stack = 0x%x\n", sp); */


  /* These are sanity checks for the code. They can be commented out,
     although they may prove useful in detecting/debugging any errors */
  if (proc.frameoffset != info->frame_offset) {
    GET_SYMBOL(proc.isym, theSymbol);

#ifdef FIXUP_BLOCKS
    if ((strcmp(GET_NAME(theSymbol), "EmeraldFixup")) &&
	(strcmp(GET_NAME(theSymbol), "Fixup")))
      printf("Frame offset for %s computed is %d vs. %d (bad prologue size estimate?)\n",
	     GET_NAME(theSymbol), info->frame_offset, proc.frameoffset);
#endif

    if (info->frame_offset == 65536) {
#ifdef DEBUG_PROF
      printf("Resetting offset to %d\n", proc.frameoffset);
#endif
      info->frame_offset = proc.frameoffset;
      info->new_sp = sp + proc.frameoffset;
    }
  }
  if (info->pc_reg != proc.pcreg)
    printf("PC register computed is %d vs. %d\n", info->pc_reg, proc.pcreg);
/*
  if (proc.framereg != info->frame_reg)
    printf("Frame register computed is %d vs. %d\n", info->frame_reg,
	   proc.framereg);
*/
  if (proc.pcreg != 31) {
    /* The code should properly handle non-$31 frames, but just in case,
       let's print a warning and let the user deal with it */
    fprintf(stderr, "\tReturn PC in $%d; not $31!\n", proc.pcreg);
  }
      
  if (proc.framereg != 0) {
    /* 
      Usually, the algorithm is that the return PC is at:
      sp + frame size + save register offset (negative)
      or:  return_pc = sp + proc.frameoffset + proc.regoffset;
      However, that's not the case if the frame has an alloca,
      so we use the following: */
    if (info->regs[info->pc_reg])
      info->return_pc_addr = info->regs[info->pc_reg];
    else {
      info->return_pc_addr = get_reg_offset(RIP, tregs, info);
    }
  } else
    /* If frame pointer is in register $0, then the return PC is saved
       four bytes from the virtual frame pointer */
    /* return_pc = sp + proc.frameoffset - 4; */
    info->return_pc_addr = fp - 4;


#ifdef DEBUG_PROF
  printf("alt rpc 0x%x\n",  sp + proc.frameoffset + proc.regoffset);
  printf("sp = 0x%x return_pc = 0x%lx, contents = 0x%lx\n", sp,
	 info->return_pc_addr, *((ulong *) info->return_pc_addr));
#endif

  return(info);
}


/* 
 * Trace and print a thread's stack given its registers (at a minimum, the
 * pc($31), stack pointer ($29), and fp ($30)), the pc (may be the same as
 * $31), and the thread's tid (used to trace around fixup blocks).
 *
 * In addition to tracing the stack, this routine also prints the argument
 * parameters (including those in registers) and determines register offsets
 * on the stack. For best operation, the current registers of the thread
 * should be provided.
 *
 */
GLOBAL void trace_stack_(FILE *out, ulong pc, ulong *tregs, ulong tid)
{
  static void print_arguments(FILE *out, ulong procStart, ulong fp, ulong sp,
			      ulong *tregs, FrameInfo *regs);
  int i, frame;			/* Temp var and frame count */
  FrameInfo *info;		/* Current frame information */
  FrameInfo *child_info = NULL;	/* Child frame information */
  ulong sp = tregs[SP];		/* Stack pointer */
  ulong fp = tregs[FP];		/* Frame pointer */
  ulong wregs[32];		/* Working registers */

  for (i = 0; i < 32; i++)	/* Copy registers to working set */
    wregs[i] = tregs[i]; 	

  if (!SYMBOLS_AVAIL()) return;	/* Can't work without symbol table */

  for (frame = 0;;frame++) {

    /* Get the frame descriptor */
    info = heuristic_get_frame_info(pc, wregs);
    if (!info)			/* No info means no more */
      break;
    
    fp = info->fp;		/* copy out possibly changed information */
    sp = info->sp;

    /* Pretty print the information for this frame */
    fprintf(out, "    %d %s(", frame, prof_data_[info->cur_sym].name);
    print_arguments(out, prof_data_[info->cur_sym].index, fp, sp, wregs, info);
    fprintf(out, ") [\"%s\":%ld, 0x%lx]\n",
	   SymTabIndexToFileName(prof_data_[info->cur_sym].index),
	   ProcAddressToLineNumber(prof_data_[info->cur_sym].startline,
				   prof_data_[info->cur_sym].address, pc),
	   pc);


#ifdef FIXUP_BLOCKS    
/*    fprintf(out, "pc 0x%x EF 0x%x New pc 0x%x\n", pc, (ulong) EmeraldFixup,
	   *((ulong *)(info->return_pc_addr))); */
    /* Check if we've hit a Fixup block */
    if (((pc == (ulong) EmeraldFixup) && tid) ||
	((pc == (ulong) Fixup) && tid)) {
      fprintf(out, "Found Fixup block, ignoring it. New RIP is: 0x%x\n", 
	     T_OSBLOCK(tid)->t_rip);
      /* Reset program counter and ignore fixup block */
      pc = wregs[RIP] = T_OSBLOCK(tid)->t_rip;
      free(info);	/* Ignore incorrect information */
    } else {
#endif

      if (!info->return_pc_addr) {
	fprintf(stderr, "Couldn't find return PC!\n");
	break;
      }
      /* To compute the new PC, take the value at the return PC location add
	 subtract out 8 bytes to account for the jump (4) and delay slot (4).
	 This way we see where the call came from (instead of where we'll
	 return to afterwards) */
      wregs[RIP] = pc = *((ulong *)(info->return_pc_addr)); /* - 8; */

      /* reset for the next loop iteration */
      /* Set the child and parent pointers */
      info->child = child_info;
      child_info = info;
      wregs[SP] = sp = info->new_sp;
      wregs[FP] = fp = info->new_fp;

#ifdef FIXUP_BLOCKS
    }
#endif


#ifdef DEBUG_PROF
    fprintf(out, "New sp 0x%x New fp 0x%x\n", sp, fp);
#endif
  }
  if ((frame == 0) && (find_proc(pc) == thread_top_index)) {
    fprintf(out, "    <New thread>\n");
  }
  fprintf(out, "\n");

  /* Free stack frame information */
  while (info) {
    child_info = info->child;
    free(info);
    info = child_info;
  }
}


GLOBAL void trace_thread_(FILE *fp, ulong tid)
{
    ThreadInfo *thread;

    thread = &thread_table_[tid];
    trace_stack_(fp, thread->t_regs[31], thread->t_regs, tid);
}


/* Given a program counter (PC), the procedure place the procedure name
   in proc_name, the source file name in file_name, and returns the line
   number. Execeptions: Returns -1 if pc is invalid, -2 if no symbol
   information is available. The strings returned by this procedure
   should NOT be modified.
 */

int LookupPC_(ulong pc, char **proc_name, char **file_name)
{
  int cur_sym = find_proc(pc);

  if (!SYMBOLS_AVAIL()) return(-2);

  if (cur_sym == 0)
    return(-2);
  else if (cur_sym == -1)
    return(-1);
  else {
    *proc_name = prof_data_[cur_sym].name;
    *file_name = SymTabIndexToFileName(prof_data_[cur_sym].index);
    return(ProcAddressToLineNumber(prof_data_[cur_sym].startline,
				   prof_data_[cur_sym].address, pc));
  }

}

/* Given a thread id (tid) a list of procedures to avoid and a count
   of the number to avoid, the procedure place the procedure name
   in proc_name, the source file name in file_name, and returns the line
   number. Execeptions: Returns -1 if pc is invalid, -2 if no symbol
   information is available, -3 if the thread's stack contains only
   those procedures in the avoidance list, and -4 if the thread is newly
   created (i.e., empty stack). The strings returned by this
   procedure should NOT be modified.
 */
int LookupThreadPC_(ulong tid, char **proc_name, char **file_name,
		    ulong *avoidList, ulong count)
{
  FrameInfo *frinfo, *parent, *stack_trace();
  int i, cur_sym, symlist[20];

  if (!SYMBOLS_AVAIL()) return(-2);

  if (count > 20)
    fatal("Avoid list too large, increase size in LookupThreadPC_\n");

  frinfo = stack_trace((ulong *) thread_table_[tid].t_regs, tid);
  if (frinfo == NULL) {
    i = find_proc(thread_table_[tid].t_regs[31]); /* Get containing proc */
    if (i == thread_top_index)
      return(-4);
    else if (i <= 0)
      return(-1);
  }
  
  for (i = 0; i < count; i++) {
    symlist[i] = find_proc(avoidList[i]);
    if (symlist[i] == 0)
      fatal("Entry %d in avoid list is not a procedure address 0x%x\n",
	    i, avoidList[i]);
  }

  parent = NULL;
  while (frinfo != NULL) {
#ifdef DEBUG_PROF
    printf("Tracing %s [cs %d]\n", prof_data_[frinfo->cur_sym].name,
	   frinfo->cur_sym);
#endif
    for (i = 0; i < count; i++) {
#ifdef DEBUG_PROF
      printf("CS for %s is %d\n", prof_data_[symlist[i]].name,symlist[i]);
#endif
      if (frinfo->cur_sym == symlist[i]) {
	if (parent == NULL)
	  return(-3);
	else {
	  cur_sym = parent->cur_sym;
#ifdef DEBUG_PROF
	  printf("Parent CS is %d\n", cur_sym);
#endif
	  *proc_name = prof_data_[cur_sym].name;
	  *file_name = SymTabIndexToFileName(prof_data_[cur_sym].index);
#ifdef DEBUG_PROF
	  printf("Parent proc name %s Parent file name %s\n", *proc_name,
		 *file_name);
	  printf("Parent pc 0x%x Parent line number %d\n",
		 *((ulong *) parent->regs[31]), 
		 ProcAddressToLineNumber(prof_data_[cur_sym].startline,
					 prof_data_[cur_sym].address, 
					 *((ulong *) frinfo->return_pc_addr)));
#endif
	  i = ProcAddressToLineNumber(prof_data_[cur_sym].startline,
				      prof_data_[cur_sym].address, 
				      *((ulong *) frinfo->return_pc_addr));
	  free(parent);
	  while (frinfo) {
	    parent = frinfo; 
	    frinfo = frinfo->child; 
	    free(parent);
	  }
	  return(i);
	}
      }
    }
    if (parent != NULL)
      free(parent);
    parent = frinfo;
    frinfo = frinfo->child;
  }
  return(-3);
}
  

/***************************************************************************\
*   Disassembly routines
\***************************************************************************/

static int dis_sym, dis_sym_len;
static long end_address;
static long start_address;

static long *cur_adr, cur_pc = 0;

static long get_bytes()
{
    register long tmp;
    tmp = *cur_adr;
    cur_adr++;
    return(tmp);
}

static void print_header(address, bytes)
ulong address, bytes;
{
    int i;
    
    if (address >= end_address) {
	dis_sym++;
	dis_sym_len = strlen(prof_data_[dis_sym].name) / 8;
	end_address = (dis_sym + 1 == num_procs) ?
	  (ulong)&etext : prof_data_[dis_sym+1].address;
	start_address = prof_data_[dis_sym].address;
    }
    printf("0x%lx%s  %s [\"%s\":%ld]", address,
	   (address==cur_pc)?"<-PC":"    ",
	   prof_data_[dis_sym].name,
	   SymTabIndexToFileName(prof_data_[dis_sym].index),
	   ProcAddressToLineNumber(prof_data_[dis_sym].startline,
				   start_address, address));

    for (i = dis_sym_len; i<3; i++) putchar('\t');
}

/* disassemble starting from address;  if (pc > 0) then mark the address
   contained in pc as the current location of the program counter */
GLOBAL void Disassemble(address, pc)
ulong address, pc;
{
    long i;
    char *cur_byte, ch;

    if ((char *)address >= &etext || (char *)address < &_ftext) {
	/* printf("etext = 0x%lx\n", &etext); */
	printf("address out of range\n");
	return;
    }

    if (pc > 0) cur_pc = pc;

    dis_sym = find_proc(address);
    dis_sym_len = strlen(prof_data_[dis_sym].name) / 8;
    end_address = (dis_sym + 1 == num_procs) ?
      (long)&etext : prof_data_[dis_sym+1].address;
    start_address = prof_data_[dis_sym].address;

    cur_byte = (char *) address;

    for (;;) {
	for (i=0; i<20; i++) {
	    cur_adr = (long *) cur_byte;
	    cur_byte += disassembler((ulong) cur_byte, 1, sym_name_,
				     (int (*)()) NULL, get_bytes,
				     print_header);
	    if (cur_byte >= &etext - 4) {
		printf("\nEnd of code segment.\n\n");
		cur_pc = 0;
		return;
	    }
	}
	printf("Return for more, a or q to quit: ");
	for (;;) {
	    ch = tolower(getchar());
	    if (ch == 'q' || ch == 'a' || ch=='\n') break;
	}
	if (ch == 'q' || ch == 'a') break;
    }

    /* remove rest of line from standard input */
    while ((ch = getchar()) != '\n');

    putchar('\n');
    cur_pc = 0;
}



/* disassemble code starting at symbol named by 'symbol' */

GLOBAL void DisassembleSymbol(symbol)
const char *symbol;
{
    int sym;
    
    /* initialize profiling if necessary */
    if (!SYMBOLS_AVAIL()) return;

    printf("DisassembleSymbol(%s)\n", symbol);
    for (sym = 0; sym<num_procs; sym++)
      if (strcmp(symbol, prof_data_[sym].name)==0) break;

    if (sym == num_procs) {
	printf("Unknown symbol.\n");
	return;
    }

    Disassemble(prof_data_[sym].address, 0);
}


/***************************************************************************\
*   Symbol Table Manipulation routines
\***************************************************************************/

/* Get the AUX table entry for index */
static pAUXU get_aux(long index)
{
  pAUXU aux = auxNil;
  
  if (!(aux = ldgetaux(ldptr, index)))
    fatal("get_iaux: Couldn't retrieve auxilliary entry for %ld\n", index);
  return(aux);
}

/* Get the name associated with the relative symbol record for index and
 *  symindex
 */

static char *get_rndx(long index, long symindex)
{
  RNDXR rsym;
  ulong file;
  
  rsym = get_aux(index)->rndx;		/* Get AUX record/coerce it to RNDXR */
  if ((rsym.rfd == 0) && (rsym.index == 0))
    return("");				/* Null value for null record */
  if (rsym.rfd != 4095)			/* 4095 means it's in the global
					   symbol table; otherwise, ???? */
    fatal("get_rndx: Don't know how to handle rfd != 4095 (%u) idx (%u)\n",
	  rsym.rfd, rsym.index);

  /* Relative offsets are symbol table offsets that are relative to the
     file containing the symbol. To compute the actual offset, we
     locate the nearest file and add its offset to the relative
     offset.  External symbols, however, represent a major pain becuase
     we can't get the file offset by looking in the symbol table. So,
     we look them up directly in the external symbol table */
  if (IsExternal(symindex)) {
    /* Get the ifd from the external symbol record. symindx's are formed
       by concating the internal and external symbols, so the real
       external index is offset from the symindex */
    file = (ulong) pext[symindex - NumIntSymbols()].ifd;
    GET_SYMBOL((long) rsym.index + file_data_[file].index, theSymbol);
  } else
    GET_SYMBOL((long) (rsym.index + find_file(symindex)->index), theSymbol);
  /* and get the name from it */
  return(GET_NAME(theSymbol));
}

/* Given the Storage Class, Symbol Type, index field, and symbol table index,
 *  decode the Auxillary Information (if any).
 *
 * This procedure is experimental at best
 *    Basically, the auxillary information is organized as a series of
 *    entries. The first entry is either an isyMac or a TIR. Depending
 *    on the symbol's type and class there may be additional fields.
 *    The table in symbolTypeToChar lists the meanings of the AUX
 *    record (it is, unfortunately, by no means complete).
 * 
 */
static char *decode_iaux(int st, int sc, long index, long symindex)
{
/*
 *        (from /usr/include/sym.h)
 * Auxillary information occurs only if needed.
 * It ALWAYS occurs in this order when present.

	isymMac		used by stProc only
	TIR		type info
	TIR		additional TQ info (if first TIR was not enough)
	rndx		if (bt == btStruct,btUnion,btEnum,btSet,btRange,
			    btTypedef):
			  rsym.index == iaux for btSet or btRange
			else rsym.index == isym
	dimLow		btRange, btSet
	dimMac		btRange, btSet
	rndx0		As many as there are tq arrays
	dimLow0
	dimHigh0
	...
	rndxMax-1
	dimLowMax-1
	dimHighMax-1
	width in bits	if (bit field), width in bits.
 *
 */

  TIR ref;			/* A Type Information Record */
  int bit = 0;			/* Set if TIR specifies a bit width */
  int continued = 0;		/* Set if additional TQ info in next AUX */
  int bt = 0;			/* The basic type */
  int tq[12], tqi;		/* The Type Qualifiers and an index */
  int tqimax = 6;		/* The number of TQs */
  static char buffer[128];	/* The buffer for the finished product */
  char tbuf[20];		/* Temporary buffer */
  long width;			/* A bit width */
  long dimsl[12];		/* Array of low bounds for array dimensions */
  long dimsh[12];		/* Array of high bounds for array dimensions */
  int dcount = 0;		/* No. array dimensions == no. of array TQs */ 
  int i;			/* Temporary variable */

  if (index == indexNil)
    return("");			/* No index info */
  ref = get_aux(index)->ti;	/* Get TIR from AUX record */
  for (tqi = 0; tqi < 6; tq[tqi++] = 0);	/* Zero first 6 TQs */
  bit = ref.fBitfield;				/* Save bits from TIR */
  continued = ref.continued;	/* Get continuation flag */
  bt = ref.bt;			/* Get base type */
  tq[0] = ref.tq0;		/* Copy out the TQs */    
  tq[1] = ref.tq1;		/* Not all the TQs may be set (most are nil) */
  tq[2] = ref.tq2;
  tq[3] = ref.tq3;
  tq[4] = ref.tq4;
  tq[5] = ref.tq5;
  sprintf(buffer, "%s", Bt[bt]);	/* Translate the base type */
  index++;				/* Skip to the next AUX record */
  if (continued) {			/* If the continuation bit is set,
					   the next record is another TIR */
    ref = get_aux(index)->ti;		/* Get the optional TIR */
    for (tqi = 6; tqi < 12; tq[tqi++] = 0);	/* Zero last 6 TQs */
    tq[6] = ref.tq0;		/* Copy out the TQs */
    tq[7] = ref.tq1;
    tq[8] = ref.tq2;
    tq[9] = ref.tq3;
    tq[10] = ref.tq4;
    tq[11] = ref.tq5;
    tqimax = 12;		/* Mark the number of TQs */
    index++;			/* Skip to the next AUX record */
  }

  /* If the basic type is: btStruct, btUnion, btEnum, btSet, btRange, 
     or btTypedef, then there is an additional field that indicates the
     name of the constructor. */
  if ((bt == btStruct) || (bt == btUnion) || (bt == btEnum)||
      (bt == btTypedef)) {

    /* Add the name to the buffer */
    strcat(buffer, get_rndx(index, symindex));

    /* The following should be one, but the information after the rndx is
       not clearly specified. Using one interferes with any array dimension
       usage */
    index += 2;		/* Not sure why, but increment by 2 NOT 1 */

  } else if ((bt == btSet) || (bt == btRange))
    /* Can't handle btSet or btRange, but they're pascal anyway */
    fatal("decode_iaux: Don't know how to handle btSet or btRange\n");

  if (sc == scBits) {	/* Check if this is a bit field */
    if (!bit)		/* Sanity check */
      fatal("decode_iaux: fBitfield is 0, while looking for bit width\n");
    width = get_aux(index)->width;	/* Get the width */
    sprintf(tbuf, "  : %ld", width);	/* Add it to the output */
    strcat(buffer, tbuf);
    index++;		/* Skip to the next AUX record */		
  }

  /* Process the TQs */
  for (tqi = 0; tqi < tqimax; tqi++) {
    if (tq[tqi] == tqArray) {
      /* Array is the most difficult (and of course, least documented TQ) */
      index += 2;	/* Skip rndxi & junk (what does it contain anyway?) */
      dimsl[dcount] = get_aux(index)->dnLow;	/* Low value */
      index++;
      dimsh[dcount++] = get_aux(index)->dnHigh; /* High value */
      index += 2;		/* Not sure why, but increment by 2; NOT 1 */
    } else { /* We assume that array TQs are listed before other TQs.
		If not, who knows what will happen */
      if (dcount) {	/* Output the dimensions in reverse order (C-style) */
	for (i = dcount - 1; i >= 0; i--) {
#ifdef ShowArrayLowDimen	  
	  sprintf(tbuf, "[%ld:%ld]", dimsl[i], dimsh[i]);
#else
	  sprintf(tbuf, "[%ld]", dimsh[i] + 1);
#endif
	  strcat(buffer, tbuf);
	}
	dcount = 0;
      }
      strcat(buffer, Tq[tq[tqi]]);	/* For normal TQs, just add 'em */
    }
  }
  return(buffer);
}

/* Given the Storage Class, Symbol Type, index field, and symbol table index,
 *  return a character string for the entry
 * 
 *    NOTE: Not all of the entries in this procedure have been tested
 */
static char *symbolTypeToChar(int st, int sc, long index, long symindex)
{
/*	(from /usr/include/sym.h)
 * The following table defines the meaning of each SYM field as
 * a function of the "st". (scD/B == scData OR scBss)
 *
 * Note: the value "isymMac" is used by symbols that have the concept
 * of enclosing a block of related information.	 This value is the
 * isym of the first symbol AFTER the end associated with the primary
 * symbol. For example if a procedure was at isym==90 and had an
 * isymMac==155, the associated end would be at isym==154, and the
 * symbol at 155 would probably (although not necessarily) be the
 * symbol for the next procedure.  This allows rapid skipping over
 * internal information of various sorts. "stEnd"s ALWAYS have the
 * isym of the primary symbol that started the block.
 * 

ST		SC	VALUE		INDEX
--------	------	--------	------
stFile		scText	address		isymMac
stLabel		scText	address		---
stGlobal	scD/B	address		iaux
stStatic	scD/B	address		iaux
stParam		scAbs	offset		iaux
stLocal		scAbs	offset		iaux
stProc		scText	address		iaux	(isymMac is first AUX)
stStaticProc	scText	address		iaux	(isymMac is first AUX)

stMember	scNil	ordinal		---	(if member of enum)
stMember	scNil	byte offset	iaux	(if member of struct/union)
stMember	scBits	bit offset	iaux	(bit field spec)

stBlock		scText	address		isymMac (text block)
stBlock		scNil	cb		isymMac (struct/union member define)
stBlock		scNil	cMembers	isymMac (enum member define)

stEnd		scText	address		isymStart
stEnd		scNil	-------		isymStart (struct/union/enum)

stTypedef	scNil	-------		iaux
stRegReloc	sc???	value		old register number
stForward	sc???	new address	isym to original symbol

stConstant	scInfo	value		--- (scalar)
stConstant	scInfo	iss		--- (complex, e.g. string)

 */

  static char field[40];

  field[0] ='\0';
  switch(st) {
  case stNil:
    break;
  case stGlobal:
    return(decode_iaux(st, sc, index, symindex));
    break;
  case stStatic:
    return(decode_iaux(st, sc, index, symindex));
    break;
  case stParam:
    return(decode_iaux(st, sc, index, symindex));
    break;
  case stLocal:
    return(decode_iaux(st, sc, index, symindex));
    break;
  case stLabel:
    if ((sc == scText) || (index == indexNil))
      strcpy(field, "");
    else
      strcpy(field, "<Unknown>");
    break;
  case stProc:
    if (sc == scText) {		/* Refer to the end of the procedure */
      if (IsExternal(symindex))
	sprintf(field, "ref=%ld ", index);	/* Refer to proc location */
      else {
	sprintf(field, "end=%ld ", 		/* Refer to proc end */
		get_aux(index)->isym + find_file(symindex)->index);
	/* The AUX info has the return type of the procedure */
	strcat(field, decode_iaux(st, sc, ++index, symindex));
      }
    } else
      sprintf(field, "<Unknown>");
    break;
  case stBlock:
    if ((sc == scText) || (sc == scNil) || (sc == scInfo))
      sprintf(field, "ref=%ld", index);	/* Refer to the end of the block */
    else 
      sprintf(field, "<Unknown>");
    break;
  case stEnd:
    if ((sc == scText) || (sc == scInfo))
      sprintf(field, "ref=%ld", index);	/* Refer procedure beginning */
    else if (sc == scNil)
      strcpy(field, "");
    else 
      strcpy(field, "<Unknown>");
    break;
  case stMember:
    return(decode_iaux(st, sc, index, symindex));
    break;
  case stTypedef:
    return(decode_iaux(st, sc, index, symindex));
    break;
  case stFile:
    if (sc == scText) {
      sprintf(field, "ref=%ld", index);	/* Refer to end of file */
    } else
      sprintf(field, "<Unknown>");
    break;
  case stRegReloc:
    sprintf(field, "New reg=%ld\n", index);
    break;
  case stForward:
    sprintf(field, "ref=%ld ", get_aux(index)->isym);
    break;
  case stStaticProc:
    if (sc == scText)
      sprintf(field, "end=%ld ", 		/* Refer to proc end */
	      get_aux(index)->isym + find_file(symindex)->index);
    else
      sprintf(field, "<Unknown>");
    strcat(field, decode_iaux(st, sc, ++index, symindex));
    break;
  case stConstant:
    break;
  case stStaParam:
    return(decode_iaux(st, sc, index, symindex));
    break;
  default:
    fatal("symbolTypeToChar: Unknown storage class encountered: %d\n", st);
  }
  return(field);
}

/* Given the symbol table index of a procedure, return the best guess at the 
 * size of its prologue
 */
static long get_prologue_size(int procStart)
{
  long symindex = procStart;	/* Save the start in case we need it */

  GET_SYMBOL(procStart, theSymbol);
  
  /* Sanity check */
  if (!IsTextProc(theSymbol))
    fatal("get_prologue_size: Symbol at index isn't a procedure.\n");

  /* Basically, iterate through the table looking for the first text
     block. Stop if we hit the end of the procedure */
  while(TRUE) {
    GET_SYMBOL(symindex, theSymbol);
    if (IsTextBlock(theSymbol))	{	/* Found first text block */
      return(theSymbol.value);		/* Number of prologue instructions */
    } else if (IsTextEnd(theSymbol)) {
      /* Found end of code, make sure it's ours */
      if (procStart == theSymbol.index) {
	/* Couldn't find the Text Block, so this is probably a file
	   compiled w/o -g (debugging) or an assembly language file.
	   As a hack, use the total size of the procedure found in the
	   info for the Text End. As a further hack, subtract off 3
	   words (one for the jump, one for the stack adjust, and one
	   for the delay slot, in case the compiler doesn't fill it) so
	   that we skip the stack re-adjustment info left by the
	   compiler (which will confuse trace_stack_). A better solution
	   is too look for the first jump instruction in trace_stack_.
	   */
	return(theSymbol.value - 12);
      } else
	fatal("Found text end for different procedure!\n");
    } else
      symindex++;			/* Keep looking */
  }
}

/* 
 * Print out simple (non-pointer) values
 */

GLOBAL void print_simple_value(FILE *out, ulong address, ulong index)
{
  TIR ref;			/* A Type Information Record */
  int bt = 0;			/* The basic type */
  int tq[12], tqi;		/* The Type Qualifiers and an index */
  int tqimax = 6;		/* The number of TQs */
  int i, ptr;			/* Temporary variable */

  ref = get_aux(index)->ti;			/* Get TIR from AUX record */
  for (tqi = 0; tqi < 6; tq[tqi++] = 0);	/* Zero first 6 TQs */
  bt = ref.bt;			/* Get base type */
  tq[0] = ref.tq0;		/* Copy out the TQs */    
  tq[1] = ref.tq1;		/* Not all the TQs may be set (most are nil) */
  tq[2] = ref.tq2;
  tq[3] = ref.tq3;
  tq[4] = ref.tq4;
  tq[5] = ref.tq5;
  index++;				/* Skip to the next AUX record */
  if (ref.continued) {			/* If the continuation bit is set,
					   the next record is another TIR */
    ref = get_aux(index)->ti;		/* Get the optional TIR */
    for (tqi = 6; tqi < 12; tq[tqi++] = 0);	/* Zero last 6 TQs */
    tq[6] = ref.tq0;		/* Copy out the TQs */
    tq[7] = ref.tq1;
    tq[8] = ref.tq2;
    tq[9] = ref.tq3;
    tq[10] = ref.tq4;
    tq[11] = ref.tq5;
    tqimax = 12;		/* Mark the number of TQs */
    index++;			/* Skip to the next AUX record */
  }

  /* Check the tq's for any non btNil modifiers */
  for (i = 0, ptr = 0; i < tqimax; i++)
    if (tq[i] != 0)
      ptr = 1;

  if (ptr) {			/* Print pointers as hex values */
    if (*((ulong *) address) == NULL)
      fprintf(out, "(nil)");
    else
      fprintf(out, "%s0x%lx", (LegalPointer(*((ulong *) address)) ? 
			"" : "(bad address) "), *((ulong *) address));
  } else {
    switch(bt) {		/* Switch off the base type */
    case btNil:
      fprintf(out, "nil");
      break;
    case btAdr:
    case btStruct:
    case btUnion:
    case btEnum:
      if (*((ulong *) address) == NULL)
	fprintf(out, "(nil)");
      else
      fprintf(out, "%s0x%lx", (LegalPointer(*((ulong *) address)) ? 
			"" : "(bad address) "), *((ulong *) address));
      break;
    case btChar:
    case btUChar:
      fprintf(out, "'%c'", *((char *) address));
      break;
    case btShort:
      fprintf(out, "%d", ((int) *((short *) address) & 0xFFFF));
      break;
    case btInt:
    case btLong:
      fprintf(out, "%d", *((int *) address));
      break;
    case btUShort:
      fprintf(out, "%u", ((unsigned int) *((short *) address) & 0xFFFF));
      break;
    case btUInt:
    case btULong:
      fprintf(out, "%u", *((unsigned int *) address));
      break;
    case btFloat:
      fprintf(out, "%f", *((float *) address));
      break;
    case btDouble:
      fprintf(out, "%f", *((double *) address));
      break;
    case btTypedef:		/* Can't really handle any of these */
    case btRange:
    case btSet:
    case btComplex:
    case btDComplex:
    case btIndirect:
    case btFixedDec:
    case btFloatDec:
    case btString:
    case btBit:
    case btPicture:
    case btVoid:
    default:
      fprintf(out, "???");
      break;
    }
  }
}

/*
 * Pretty print the arguments of a procedure
 */
static void print_arguments(FILE *out, ulong procStart,
			    ulong fp, ulong sp, ulong *tregs,
			    FrameInfo *regs)
{
  long procEnd;		/* Proc end in symbol table */
  long i;
  int first = 1;
  ulong location;
  short regno;

    /* hack: if argument regs are saved, guess these contain args */
/*    if ((reg_mask & 0xF0) == 0) num_args = -1;
    else if ((reg_mask & 0x80) == 0) num_args = 4;
    else if ((reg_mask & 0x40) == 0) num_args = 3;
    else if ((reg_mask & 0x20) == 0) num_args = 2;
    else if ((reg_mask & 0x10) == 0) num_args = 1; */
  GET_SYMBOL(procStart, theSymbol);
  if (!IsTextProc(theSymbol))
    fatal("print_arguments: Symbol at index isn't procedure\n");
  
  /* Compute the location of the end of the procedure's entry in the 
     symbol table */
  procEnd = get_aux(theSymbol.index)->isym + find_file(procStart)->index;

  for (i = procStart + 1; i < procEnd; i++) {
    GET_SYMBOL(i, theSymbol);
    if (!IsParam(theSymbol))		/* Only looking for parameter info */
      continue;
    if (first)				/* Pretty printer stuff */
      first = 0;
    else
      fprintf(out, ", ");

    fprintf(out, "%s = ", GET_NAME(theSymbol));

    /* Now, find the parameter */
    if (IsAbsolute(theSymbol)) {
      location = theSymbol.value + fp;
      print_simple_value(out, location, theSymbol.index);
    } else if (IsRegister(theSymbol)) {
      regno = theSymbol.value;
      /* Have to be clever. Search the frames to see if the register is
	 saved somewhere (May not get argument reg values correct) */
      location = get_reg_offset(regno, tregs, regs);
      if (location)
	print_simple_value(out, location, theSymbol.index);
      else
	fprintf(out, "<Reg Unavail>");	/* This will never happen */
    } else
      fatal("print_arguments: Got unknown storage class: %s\n",
	    Sclass[theSymbol.sc]);
  }
}
  

/* 
 * Determine if index refers to a non-pointer value
 */
static int isPointer(ulong index)
{
  TIR ref;			/* A Type Information Record */

  ref = get_aux(index)->ti;			/* Get TIR from AUX record */
  if (ref.tq0 || ref.tq0 || ref.tq1 || ref.tq2 || ref.tq3 || ref.tq4 ||
      ref.tq5)
    return(TRUE);
  else if (ref.continued) {		/* If the continuation bit is set,
					   the next record is another TIR */
    index++;				/* Skip to the next AUX record */
    ref = get_aux(index)->ti;		/* Get the optional TIR */
    if (ref.tq0 || ref.tq0 || ref.tq1 || ref.tq2 || ref.tq3 || ref.tq4 ||
	ref.tq5)
      return(TRUE);
  }
  return(FALSE);
}

GLOBAL void PrintStorageSymbol(FILE *out, ulong value, ulong type, ulong fp,
			       FrameInfo *info, ulong *tregs)
{
  ulong ptr;

  switch (type) {
  case FRAME_ABS:
    ptr = value + fp;
    fprintf(out, "0x%lx ", ptr);
    if (LegalPointer(ptr))
      fprintf(out, "[0x%lx] (abs)\n",  *((ulong *) ptr));
    else
      fprintf(out, "[Invalid] (abs)\n");
    break;
  case FRAME_REG:
    ptr = (ulong) get_reg_offset(value, tregs, info);
    fprintf(out, "0x%lx ", ptr);
    if (LegalPointer(ptr))
      fprintf(out, "[0x%lx] (reg $%lu)\n", *((ulong *) ptr), value);
    else 
      fprintf(out, "[Invalid] (reg $%lu)\n", value);
    break;
  case FRAME_BSS:
    ptr = value;
    fprintf(out, "0x%lx ", value);
    if (LegalPointer(ptr))
      fprintf(out, "[0x%lx] (bss)\n", *((ulong *) value));
    else
      fprintf(out, "[Invalid] (bss)\n");
    break;
  default:
    fprintf(out, "PrintStorageSymbol: Got unknown storage class\n");
    break;
  }
}

GLOBAL ulong StorageSymbolAddress(ulong value, ulong type, ulong fp,
				  FrameInfo *info, ulong *tregs)
{
  switch (type) {
  case FRAME_ABS:
    return((ulong) value + fp);
    break;
  case FRAME_REG:
    return((ulong) get_reg_offset(value, tregs, info));
    break;
  case FRAME_BSS:
    return((ulong) value);
    break;
  default:
    fprintf(stderr,
	    "StorageSymbolAddress: Got unknown storage class (%lu)\n", type);
    return(NULL);
    break;
  }
}

/* Given frame information, a symbol name, and a thread's registers (at a 
 * minimum, the pc($31), stack pointer ($29), fp ($30)), return the
 * value of the symbol in the procedure's argument list, if it exists.
 * For best operation, the current registers of the thread should be provided.
 */
GLOBAL RetValue *FrameInfoToValue(FrameInfo *info, char *name, ulong *tregs)
{

  RetValue *rv;		/* Return value */
  long procStart;	/* Proc start in symbol table */
  long procEnd;		/* Proc end in symbol table */
  long i;
  int has_selfaddr = 0;
  int size = strlen(name);
  ulong pc = tregs[RIP];
  ulong fp = tregs[FP];
  SYMR Symbol;  	/* symbol work area */

  if (!info) {
    i = find_proc(pc);	/* Get the containing procedure */
    /* Stop when we hit the top of the stack or get an invalid symbol */
    if ((pc == NULL) || (i == thread_top_index)) {
      return(NULL);
    } else
      fatal("SymbolToValue: Failed to find frame info for %s [pc = 0x%x]\n",
	    prof_data_[i].name, pc);
  }

  rv = (RetValue *) calloc(1, sizeof(RetValue));
  rv->info = info;
  rv->proc_start = prof_data_[rv->info->cur_sym].address;
  fp = rv->info->fp;				/* Save some info */
  procStart = prof_data_[rv->info->cur_sym].index; /* Get symbol table idx */
  GET_SYMBOL(procStart, Symbol);
  if (!IsTextProc(Symbol))
    fatal("SymbolToValue: Symbol at index isn't procedure\n");
  procEnd = get_aux(Symbol.index)->isym + find_file(procStart)->index;
  rv->ripAddr = rv->info->return_pc_addr;
  rv->name = prof_data_[rv->info->cur_sym].name; /* Get procedure name */
  rv->pointer = isPointer(Symbol.index + 1);
/*  printf("proc \"%s\" runs from %ld to %ld\n",
	 prof_data_[rv->info->cur_sym].name, procStart, procEnd - 1); */

  for (i = procStart + 1; i < procEnd; i++) {
    GET_SYMBOL(i, Symbol);
    if (!IsVariable(Symbol))
      continue;
    if (IsLocal(Symbol) && !has_selfaddr) /* Stop at first non-argument */
      break;
    if (IsParam(Symbol) && !strcmp(GET_NAME(Symbol), "selfAddress")) {
#ifdef DEBUG
      printf("1. Found %s in procedure %s ", GET_NAME(Symbol), rv->name);
#endif
      has_selfaddr = 1;
      rv->valueAddr = Symbol.value;
      rv->type = StorageSymbolType(Symbol);
#ifdef DEBUG
      PrintStorageSymbol(rv->valueAddr, rv->type, fp, info, tregs);
#endif
    } else if (!strncmp(name, GET_NAME(Symbol), size)) {
#ifdef DEBUG
      printf("2. Found %s in procedure %s ", GET_NAME(Symbol), rv->name);
#endif
      rv->valueAddr = Symbol.value;
      rv->type = StorageSymbolType(Symbol);
#ifdef DEBUG
      PrintStorageSymbol(rv->valueAddr, rv->type, fp, info, tregs);
#endif
      break;
    }
  }
  return(rv);
}	  


/* Given a symbol name and a thread's registers (at a minimum, the pc($31),
 * stack pointer ($29), fp ($30)), return the value of the symbol in the
 * procedure's argument list, if it exists. For best operation, the current
 * registers of the thread should be provided.
 */
GLOBAL RetValue *SymbolToValue(char *name, ulong *tregs)
{
  RetValue *rv;		/* Return value */
  FrameInfo *info;	/* Frame information */
  ulong pc = tregs[RIP];

  info = heuristic_get_frame_info(pc, tregs);
  rv = FrameInfoToValue(info, name, tregs);
  return(rv);
}	  

/* Free RetValue information */
GLOBAL void RetValueFree(RetValue *info)
{
  free(info->info);
  free(info);
}

/* Given a thread's registers (at a minimum, the pc($31), stack pointer ($29),
 * fp ($30)), return the number of varables (argument/parameter, local,
 * and static) and their locations. For best operation, the current
 * registers of the thread should be accessible.
 */

GLOBAL FrameVariables *GetFramePtrVariables(ulong *tregs, RetValue *frameInfo)
{

  FrameVariables *rv;
  long procStart;	/* Proc start in symbol table */
  long procEnd;		/* Proc end in symbol table */
  long i, j, idx, fsize;
  ulong sp = tregs[SP];
  ulong fp = tregs[FP];
  short type;
  SYMR Symbol;  	/* symbol work area */

  rv = (FrameVariables *) calloc(1, sizeof(FrameVariables));
  rv->count = 0;

  fp = frameInfo->info->fp;
  sp = frameInfo->info->sp;
  fsize = frameInfo->info->frame_offset;
  if ((sp + fsize) != fp) {
    printf("fp [0x%lx] invalid, using 0x%lx\n", fp, sp + fsize);
    fp = sp + fsize;
  }
#ifdef DEBUG
  printf("Looking for variables in frame for %s\n",
	 prof_data_[frameInfo->info->cur_sym].name);
#endif
  /* Get symbol table index */
  procStart = prof_data_[frameInfo->info->cur_sym].index;
  GET_SYMBOL(procStart, Symbol);
  if (!IsTextProc(Symbol))
    fatal("SymbolToValue: Symbol at index isn't procedure\n");
  procEnd = get_aux(Symbol.index)->isym + find_file(procStart)->index;

  /* Inefficient, but make two passes */
  for (i = procStart + 1; i < procEnd; i++) {
    GET_SYMBOL(i, Symbol);
    if (IsVariable(Symbol) && isPointer(Symbol.index))
      rv->count++;
  }
  rv->vars = (ulong *) calloc(rv->count, sizeof(ulong));
  rv->types = (short *) calloc(rv->count, sizeof(short));
  for (i = procStart + 1, idx = 0; i < procEnd; i++) {
    GET_SYMBOL(i, Symbol);
    if (!IsVariable(Symbol) || !isPointer(Symbol.index))
      continue;
#ifdef DEBUG
    printf("Found pointer object %s: ", GET_NAME(Symbol));
#endif
    if (IsVariable(Symbol)) {
      type = StorageSymbolType(Symbol);
#ifdef DEBUG
      PrintStorageSymbol(Symbol.value, type, fp, frameinfo->info, tregs);
#endif
      for (j = 0; j < idx; j++)
	if ((rv->types[j] == type) &&
	    (rv->vars[j] == Symbol.value))
	  break;
      if (j != idx)
	continue;
      rv->types[idx] = type;
      rv->vars[idx++] = Symbol.value;
    }
  }
  rv->count = idx;
  return(rv);
}	  


GLOBAL void dump_symbols()
{
  int all_symbols = 0;
  long symindex;
  char *name, *reftype;
  char idx[32];

  /* initialize symbol table, if necessary */
  if (!SYMBOLS_AVAIL()) return;

  all_symbols = NumSymbols();

  printf("\n\n                        ***SYMBOL TABLE INFORMATION***\n");
  printf("%-8s %-10s%-10s   %-9s %-9s %s\n%s:\n", "[Index]", "Name", "Value",
	 "Sclass", "Symtype","Ref/Type", exec_name_);

  for (symindex = 0; symindex < all_symbols; symindex++) {
    GET_SYMBOL(symindex, theSymbol);
    name = GET_NAME(theSymbol);
    sprintf(idx, "[%ld]", symindex);
    printf("%-8s %-10s0x%08lx   %-9s %-9s ", idx, name, theSymbol.value,
	   Sclass[theSymbol.sc], Symtype[theSymbol.st]);
    reftype = symbolTypeToChar(theSymbol.st, theSymbol.sc, theSymbol.index,
			       symindex);
    printf("%s\n",reftype);
  }
}


/* 
 * Trace a thread's stack given its registers (at a minimum, the
 * pc($31), stack pointer ($29), and fp ($30)), the pc (may be the same as
 * $31), and the thread's tid (used to trace around fixup blocks).
 *
 */
GLOBAL FrameInfo *stack_trace(ulong *tregs, ulong tid)
{
  int i, frame;			/* Temp var and frame count */
  FrameInfo *info = NULL;	/* Frame information */
  FrameInfo *newinfo;		/* Current frame information */
  FrameInfo *child_info = NULL;	/* Child frame information */
  ulong pc = tregs[RIP];		/* Program Counter */
  ulong wregs[32];		/* Working registers */

  for (i = 0; i < 32; i++)	/* Copy registers to working set */
    wregs[i] = tregs[i]; 	

  if (!SYMBOLS_AVAIL()) return(NULL);	/* Can't work without symbol table */

  for (frame = 0;;frame++) {

    /* Get the frame descriptor */
    newinfo = heuristic_get_frame_info(pc, wregs);
    if (!newinfo)			/* No info means no more */
      break;
    info = newinfo;
    info->tid = tid;
    
#ifdef FIXUP_BLOCKS    
    /* Check if we've hit a Fixup block */
    if (((pc == (ulong) EmeraldFixup) && tid) ||
	((pc == (ulong) Fixup) && tid)) {
      /* Reset program counter and ignore fixup block */
      pc = wregs[RIP] = T_OSBLOCK(tid)->t_rip;
      info = (FrameInfo *) calloc(1, sizeof(FrameInfo));
      info->pc_reg = -1;
      info->tid = tid;
      /* Set the child and parent pointers */
      info->child = child_info;
      child_info = info;
    } else {
#endif

      if (!info->return_pc_addr) {
	fprintf(stderr, "Couldn't find return PC!\n");
	break;
      }
      /* To compute the new PC, take the value at the return PC location add
	 subtract out 8 bytes to account for the jump (4) and delay slot (4).
	 This way we see where the call came from (instead of where we'll
	 return to afterwards) */
      wregs[RIP] = pc = *((ulong *)(info->return_pc_addr)); /* - 8; */

      /* reset for the next loop iteration */
      /* Set the child and parent pointers */
      info->child = child_info;
      child_info = info;
      wregs[SP] = info->new_sp;
      wregs[FP] = info->new_fp;

#ifdef FIXUP_BLOCKS
    }
#endif


#ifdef DEBUG_PROF
    printf("New sp 0x%x New fp 0x%x\n", wregs[SP], wregs[FP]);
#endif
  }
  return(info);
}

/* This is code to handle saved registers? Not really sure why it is
   needed
	  } else if (IsRegister(Symbol)) {
	    if (!isSavedReg(Symbol.value) &&
		rv->info->regs[Symbol.value]) {
	      printf("Saved register 0x%x vs. 0x%x\n", 
		     *((ulong *) rv->info->regs[Symbol.value]),
		     (ulong) tregs[Symbol.value]);
#ifdef DEBUG
	      printf("0x%x [0x%x] (SAVED reg $%d) Offset 0x%x\n",
		     (ulong) rv->info->regs[Symbol.value], 
		     *((ulong *) rv->info->regs[Symbol.value]),
		     Symbol.value, rv->info->regs[Symbol.value] - fp);
#endif
	      rv->type = FRAME_ABS;
	      rv->valueAddr = rv->info->regs[Symbol.value] - fp;
	    } else {
	      rv->type = FRAME_REG;
	      rv->valueAddr = (ulong) Symbol.value;
#ifdef DEBUG
	      printf("0x%x [0x%x] (reg $%d)\n", 
		     (ulong) rv->valueAddr, 
		     ((rv->type == FRAME_ABS) ? *((ulong *)(rv->valueAddr+fp)):
		      ((rv->type == FRAME_REG) ? (tregs[rv->valueAddr]):
		       rv->valueAddr)),
		     Symbol.value);
#endif
	    }
*/

/* time_print(Time t)
 * requires	GNU "unsigned long long int" is a 64 bit value
 * effects	Returns ascii representation for Time value.
 *		The returned string is a static string that should not
 *		be modified by the caller, freed, or stashed away
 *		for long periods of time.
 * See also the debugging routines time_now() and time_rptr()
 */
char* time_print(Time t) {
    /* List of static buffers */
    static char buffers[100][21]; /* 20 digits holds a 64-bit number */
    static int next_buffer = 0;
    char* buffer;

    buffer = buffers[next_buffer];
    next_buffer = (next_buffer + 1) % 100;

#ifdef LONG_LONG_TIME
    /* Print out seven digits at a time */
    {
	unsigned long part1, part2, part3;

	part3 = (unsigned long) (t % 10000000);
	t = t / 10000000;

	part2 = (unsigned long) (t % 10000000);
	t = t / 10000000;

	part1 = (unsigned long) t;

	if (part1 > 0) {
	    sprintf(buffer, "%lu%07lu%07lu", part1, part2, part3);
	}
	else if (part2 > 0) {
	    sprintf(buffer, "%lu%07lu", part2, part3);
	}
	else {
	    sprintf(buffer, "%lu", part3);
	}
    }
#else
    sprintf(buffer, "%lu" t);
#endif

    return buffer;
}

/* timep_print(Time *t)
 *    gdb can't pass unsigned long longs correctly, but it can pass 
 * *pointers* correctly.  So this is a front end to time_print()
 * for debugging.
 */
char* timep_print(Time *t) {  return(time_print(*t)); }

