 
 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module:  Main
 *
 * Description:  Initilization and main simulator loop
 *
 * Last Modified:  $Date: 1996/07/20 02:07:37 $ ($Author)
 *
 * Global Functions:
 *    main(int argc, char *argv[])
 *
 * Global Variables:
 *    float version_number_
 *        float that specifies the Proteus version number
 *    char* date_string_
 *        string that specifies the version-specific text
 *    BOOL abort_on_fatal_
 *
 ***************************************************************************
 *   Copyright 1991, 1992, 1993
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /usr/wildcat/dfk.keep/parallel/proteus/current/engine/RCS/main.c,v 1.5 1996/07/20 02:07:37 dfk Exp $
 * $Log: main.c,v $
 * Revision 1.5  1996/07/20 02:07:37  dfk
 * added proteusSeed global variable
 *
 * Revision 1.4  1996/06/07 14:29:08  dfk
 * added "-snap end" feature
 *
 * Revision 1.3  1994/05/22 03:02:18  dfk
 * added seeding of the rand() generator
 * removed seeding of the vfast_random() because it makes no sense
 *
 * Revision 1.2  94/01/24  00:39:13  dfk
 * Added Sanjay's support for 64-bit timers.
 * Added thread_sleep_until.
 * Fixed a few bugs in thread sleep/wakeup code (my bugs).
 * Fixed printf formats in many places.
 * 
 * Revision 1.1  94/01/23  11:34:59  dfk
 * Initial revision
 * 
 * Revision 1.14  1993/02/18  22:05:36  brewer
 * Updated for version 3.00 (non beta)
 *
 * Revision 1.13  1992/12/13  18:52:04  brewer
 * changed long_to_string to ulong_to_string (thanks to David Kotz)
 *
 * Revision 1.12  92/12/04  17:02:08  brewer
 * Added call to vfast_random for -seed argument.
 * 
 * Revision 1.11  92/12/02  11:21:38  brewer
 * Updated version number to 3.0 beta r1
 * 
 * Revision 1.10  92/11/19  14:21:00  brewer
 * Added support for SEND_RECEIVE model; can now start usermain on all
 * processors at time 0.
 * 
 * Revision 1.9  92/10/20  11:23:59  brewer
 * Added Wall Clock Time metric.
 * 
 * Revision 1.8  92/10/07  17:31:02  brewer
 * removed OutputProfData call from cleanup(); it's now called from
 * exit_statistics_
 * 
 * Revision 1.7  92/09/23  15:53:59  brewer
 * Added cleanup procedure to ensure that profile info gets dumped.
 * 
 * Revision 1.6  92/08/11  13:50:22  brewer
 * Added command line arg -o filename to specify trace-file file name.
 * 
 * Revision 1.5  92/08/11  13:26:25  brewer
 * Changed args to init_statistics
 * 
 * Revision 1.4  92/06/25  13:26:32  brewer
 * Update PrintHelp to reflect -abort and proteus instead of gensim
 * 
 * Revision 1.3  92/06/25  13:08:38  brewer
 * Added -abort flag
 * 
 * Revision 1.2  92/04/01  17:04:10  brewer
 * updated version number, added prototypes for init_packet_, init_network_
 * 
 * Revision 1.1  92/02/11  13:55:57  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "sim.h"
#include "conf.h"
#include "rt_thread_def.h"
#include "simreq.h"
#include "processor.h"
#include "simcalls.h"
#include "thread.h"
#include "event.h"
#include "intreq.h"
#include "packet.h"

GLOBAL float version_number_ = 3.0;
GLOBAL const char *date_string_ = "February 18, 1993";
GLOBAL BOOL abort_on_fatal_ = FALSE;
GLOBAL int proteusSeed = 1;
static char *events_filename = NULL;

/* To count exact no of cycles simulated */
GLOBAL Time total_cycles, previous_time, curr_busy_procs;

BOOL snapAtEnd = FALSE;

extern init_network_();

/***************************************************************************/

static void cleanup()
{
    new_metric("Execution Time", (double)CURRENTTIME);

    exit_statistics_(CURRENTTIME);
}

/***************************************************************************/

static char *Time_to_string(Time time, char *buffer)
{
    char *b = buffer, *t;
    int len, i;
    
    t = time_print(time);
    len = strlen(t);
    
    for (i=0; i< (len % 3); i++)
      *b++ = *t++;
    
    if (i > 0 && strlen(t) > 0) *b++ = ',';
    
    while (*t != 0) {
	assert(strlen(t) % 3 == 0);  /* loop invariant */
	*b++ = *t++;
	*b++ = *t++;
	*b++ = *t++;
	if (strlen(t) > 0) *b++ = ',';
    }
    
    *b = 0;
    
    return(buffer);
}


/***************************************************************************/
/* Handle command line arguments */

static Word userarg[2];

/* interpret command line arguments */

static void PrintHelp(arg)
char *arg;
{
    if (arg != NULL) {
	fprintf(stderr, "Invalid command line argument: `%s'.\n", arg);
    }

    fprintf(stderr,
   "Usage: proteus [-seed seed] [-t title] [-abort] [-snap time]\n");
    fprintf(stderr, "\t[-help] [-o events] [-v args]\n\n");
    fprintf(stderr,
	  "   -seed seed    Set the random number seed.\n");
    fprintf(stderr,
	  "   -snap time    Schedule a snapshot.\n");
    fprintf(stderr,
	  "   -t title      Set the simulation title.\n");
    fprintf(stderr,
	  "   -abort        Abort on fatal error (otherwise enter snapshot).\n");
    fprintf(stderr,
	  "   -help         Print this description.\n");
    fprintf(stderr,
	  "   -o events     Use `events' for trace-file output.\n");
    fprintf(stderr,
	  "   -v args       Pass the rest of the arguments to usermain.\n\n");
}

static void HandleArgs(argc, argv) 
int argc;
char *argv[];
{
    int i;
    Time time;
    
    for (i=1; i<argc; i++) {
	if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
	      case 's':
		if (strcmp(argv[i], "-seed") == 0) {
		    proteusSeed = atoi( argv[++i] );
		    srandom(proteusSeed);
		    srand(proteusSeed);
		    /* this has no seeding effect */
		    /* vfast_random(random()); */ 
		} else if (strcmp(argv[i], "-snap") == 0) {
		   if (strcmp(argv[++i], "end") == 0) {
		     snapAtEnd = TRUE;
		   } else {
		     time = atoi(argv[++i]);
		     if (time > 0)
		       enqueue_snapshot_(time);
		     else
		       fatal("Invalid snapshot time `%s' on command line.",
			     argv[i]);
		   }
		} else {
		  PrintHelp(argv[i]);
		  exit(1);
		}
		break;
	      case 'a':
		if (strcmp(argv[i], "-abort") != 0) {
		    PrintHelp(argv[i]);
		    exit(1);
		} else {
		    abort_on_fatal_ = TRUE;
		}
		break;
	      case 't':
		if (strcmp(argv[i], "-t") != 0) {
		    PrintHelp(argv[i]);
		    exit(1);
		} else {
		    simulation_title_ = argv[++i];
		}
		break;
	      case 'h':
		if (strcmp(argv[i], "-help") != 0) {
		    PrintHelp(argv[i]);
		    exit(1);
		} else {
		    PrintHelp(NULL);
		    exit(0);
		}
		break;
	      case 'o':
		if (strcmp(argv[i], "-o") != 0) {
		    PrintHelp(argv[i]);
		    exit(1);
		} else {
		    events_filename = argv[++i];
		}
		break;
	      case 'v':
		if (strcmp(argv[i], "-v") != 0) {
		    PrintHelp(argv[i]);
		    exit(1);
		} else {
		    userarg[0] = (Word)argc - i -1;
		    userarg[1] = (Word)&argv[i+1];
		}
		return;
	      default:
		PrintHelp(argv[i]);
		exit(1);
	    }
	}
    }
}

/***************************************************************************/

static void start_usermain(int proc)
{
    Thread *osptr;
    int tid;

    osptr = &processor_private[proc].thread_table[MAX_PROC_THREADS-1];
    tid = create_thread(simulator_tid(), (FuncPtr)usermain, USERMAIN_STACK, 
			 "usermain", (Time) 0, (void *)osptr,
			 2, (Word *)userarg);
    osptr->t_tid = MAX_PROC_THREADS-1;
    osptr->t_stid = tid;
    osptr->t_state = T_CURRENT;
    osptr->t_processor = proc;
    osptr->priority = 1 /* USERMAIN_PRIO */;
    osptr->t_preempt  = PREEMPT_ON;
    osptr->t_intrflag = INTR_ON;

    set_local_pointers(proc);
    processor_ = proc;
    sched_insert(osptr);

    proc_table_[proc].p_tid = tid;
    MAKE_RESUME_REQUEST(tid, 0, OK);

    EVENT_START_THREAD((Time) 0, tid);
    EVENT_BUSY((Time) 0, proc);

    numthreads++;
#ifdef MULT_RL
    proc_table_[proc].p_numthreads++;
#endif
}

/***************************************************************************/

int main(argc, argv) 
int argc;
char *argv[];
{
    SimRequest *rptr;
    int i;
    BOOL pending_intrs_found;
    extern BOOL snap;
    ulong rt;

    printf("Proteus\n");
    printf("Version %.2f  %s\n", version_number_, date_string_);

    /* initialize request queue before handling arguments, in case a
       snapshot request is scheduled */
    init_request_queue_();

    HandleArgs(argc, argv);

    sim_running_ = FALSE;

    base_time_ = 0;
    cycles_ = sim_quantum_ = 0;

    init_profile(argv[0]);

    makepool(MAX_STKBLKS);
    init_Q();
    init_semaphores();
    init_shmem();
    init_simcalls_();
    init_statistics_(NULL, events_filename);
    init_intr_request_queue();
    init_processors_();
#ifdef NET
    init_packet_();
    init_network_();
#endif

    cycles_ = sim_quantum_ = 1000000; /* for cycle-counted stuff */
    init_threads();

    init_stat(STAT_BUSCONT, 0, 100);
    init_array_stat(STAT_NETCONT, NO_OF_PROCESSORS, 0, 100);

    sim_running_ = TRUE;

    start_usermain(0);

#ifdef SEND_RECEIVE
    for(i=1; i<NO_OF_PROCESSORS; i++)
      start_usermain(i);
    set_local_pointers(0);
    processor_ = 0;
#endif

    cycles_ = sim_quantum_ = 0;

    snap = FALSE;
    signal(SIGINT, control_C_);
    signal(SIGBUS, bus_error_);
    signal(SIGSEGV, bus_error_);
    signal(SIGILL, bus_error_);
    
    if (atexit(cleanup) != 0)
      fatal("Unable to register cleanup routine!\n");

    printf("Starting simulation...\n");

    sim_quantum_ = SIM_QUANTUM;

    clock();

    /* Primary simulator loop: 1) get request
                               2) execute it
                               3) free up request block */

    do {
	while( (rptr = next_request_()) != (SimRequest *)NULL )
	  {
	      if (rptr->h.reqtype != SC_NOP)
		(*request_handler_[rptr->h.reqtype])(rptr);
	      
	      free_request_(rptr);
	      
	      if (snap) snapshot();
	  }

	/* check for pending interrupts, otherwise done */
	pending_intrs_found = FALSE;
	
	for(i=0; i<NO_OF_PROCESSORS; i++) {
	    if (intr_pending[i] > 0) {
		ProcBlk *pptr = &proc_table_[i];
		pptr->p_time = MAX(pptr->p_time,
				   first_intr_request(i)->itime);
		if ( Idle(i) ) EVENT_BUSY(pptr->p_time, i);
		pptr->p_tid = pptr->p_ihandler;
		pptr->p_interrupted = ERROR;
		T_OSBLOCK(pptr->p_ihandler)->t_state = T_CURRENT;
		idle_processors--;
		MAKE_RESUME_REQUEST(pptr->p_ihandler, pptr->p_time, NULL);
		pending_intrs_found = TRUE;
	    }
	}
    } while (pending_intrs_found);

    /****  simulation done  ****/

    rt = clock();

    new_metric("Wall Clock Time", (double) rt/CLOCKS_PER_SEC);

    printf("End of Simulation.\n");

    {
	Time mt=0;
	char tmp[27];	      /* 2^64 is 20 digits, plus 6 commas, plus \0 */
	Time cyc_per_sec;

	for(i=0; i<NO_OF_PROCESSORS; i++)
	  if (mt < proc_table_[i].p_time) mt = proc_table_[i].p_time;

	printf("A. Program took %s cycles on %d processors.\n",
	       Time_to_string(mt, tmp), NO_OF_PROCESSORS);

	printf("B. Program took %s microseconds of real time.\n",
	       Time_to_string((Time)rt, tmp));

	cyc_per_sec = (mt * NO_OF_PROCESSORS) / ((double)rt / CLOCKS_PER_SEC);
	printf("C. Simulated cycles per second = %s.\n",
	       Time_to_string(cyc_per_sec, tmp));
    }

    if (snapAtEnd)
      snapshot();

    return(0);
}


