
/****************************************************************************
 *                                                                          *
 *                     Parallel Architecture Simulator                      *
 *                Eric A. Brewer  and  Chris N. Dellarocas                  *
 *                     Laboratory for Computer Science                      *
 *                  Massachusetts Institute of Technology                   *
 *                                                                          *
 * Module: cache.bus.c
 *                                                                          *
 * Description: Goodman cache-coherence protocol
 *                                                                          *
 * Last Modified: $Date: 92/04/01 17:00:20 $
 *                                                                          *
 * Global Functions:                                                        *
 *                                                                          *
 * Global Variables:                                                        *
 *                                                                          *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.
 ****************************************************************************
 * $Header: /psg/proteus/RCS/cache.bus.c,v 1.2 92/04/01 17:00:20 brewer Exp $
 * $Log:	cache.bus.c,v $
 * Revision 1.2  92/04/01  17:00:20  brewer
 * cleaned up gcc warnings, added ANSI prototypes
 * 
 * Revision 1.1  92/02/11  13:55:03  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "shmem.h"
#include "net.h"
#include "simcalls.h"

/* packet routines not used in BUS version */
void CtoM_Pkt() {
    fatal("CtoM_Pkt called in BUS version; file out of date?");
}
void MtoC_Pkt() {
    fatal("MtoC_Pkt called in BUS version; file out of date?");
}

/***************************************************************************/
/* Goodman snoopy cache-coherence protocol */

#define C_INVALID         000
#define C_VALID           001
#define C_RESERVED        002
#define C_DIRTY           004
#define C_SOFT_VALID      101
#define C_SOFT_RESERVED   102
#define C_SOFT_DIRTY      104

#define DIRTY_BIT         004

static char *printstate(s) 
     int s;
{
    static char *states[] = {"INVALID", "VALID", "RESERVED", "DIRTY"};
    return(states[s]);
}


int processor_request_pre(Time *time, int reqtype, int processor,
			  int tid, ulong address) 
{
    int set  = GetSetFromAddr(address);
    int tag  = GetTagFromAddr(address);
    int line = GetLineFromAddr(processor, address);
    short *state;
    int i;
    
    *time += CACHE_ACCESS_LATENCY;
    
#ifdef CACHEDEBUG
    printf("line = %d", line);
#endif
    
    if (line < 0) {
	line = ChooseLineToEvict(processor, address);
	if (CacheState(processor, set, line) & DIRTY_BIT) {
	    *time = BusAccess(*time);
	    /* Copy data from cache to memory */
	    WriteCacheLine(&memory[CachedAddr(processor, set, line)], 
			   CacheData(processor, set, line));
	    WriteCacheLine(&memtag[CachedAddr(processor, set, line)], 
			   CacheFebits(processor, set, line));
	}
	CacheTag(processor, set, line) = tag;
	CacheState(processor, set, line) = C_INVALID;
    }
    
    
#ifdef CACHEDEBUG
    printf("PRE:t=%d p=%d req=%d addr=%x set=%d line=%d data=%d state=%s\n",
	   *time, processor, reqtype, address, set, line,
	   &cache[processor][set].data[line],
	   printstate(cache[processor][set].state[line]));
    printf("line %d AddrInCache(%d,%d) = %d\n",
	   GetLineFromAddr(processor, address), processor, address,
	   AddrInCache(processor, address));
#endif
    
    /* See how this request affects other caches that snoop the bus */
    for(i=0; i<NO_OF_PROCESSORS; i++)
      if (i != processor) {
	  int line2 = GetLineFromAddr(i, address);
	  if (line2 >= 0) {
	      int set2 = GetSetFromAddr(address);
	      short *state2 = &CacheState(i, set2, line2);
	      
	      switch(reqtype) {
		case MR_READ:
		case MR_READSOFT:
		  switch(*state2) {
		    case C_RESERVED:
		      *state2 = C_VALID;
		      break;
		    case C_DIRTY:
		      *state2 = C_VALID;
		      /* Copy data from cache to memory */
		      /* BusAccess is not counted here since in reality,
		       * data is supplied to demanding cache at the same time
		       * that it is being written to memory
		       */
		      WriteCacheLine(&memory[address],
				     CacheData(i, set2, line2));
		      WriteCacheLine(&memtag[address],
				     CacheFebits(i, set2, line2));
		      break;
		  }
		  break;
		case MR_WRITE:
		case MR_WRITESOFT:
		  switch(*state2) {
		    case C_DIRTY:
		      WriteCacheLine(&memory[address],
				     CacheData(i, set2, line2));
		      WriteCacheLine(&memtag[address],
				     CacheFebits(i, set2, line2));
		    case C_VALID:
		    case C_RESERVED:
		      *state2 = C_INVALID;
		      break;
		  }
		  break;
	      }
	  }
      }
    
    /* See how this request affects us */
    switch(*(state = &CacheState(processor, set, line))) {
      case C_INVALID:
	switch(reqtype) {
	  case MR_READ:
	  case MR_READSOFT:
	    CACHE_MISS;
	    *time = BusAccess(*time);
	    /* Copy data from memory to cache */
	    WriteCacheLine(CacheData(processor, set, line),
			   &memory[address]);
	    WriteCacheLine(CacheFebits(processor, set, line),
			   &memtag[address]);
	    CacheTag(processor, set, line) = tag;
	    *state = reqtype == MR_READ ? C_VALID : C_SOFT_VALID;
	    break;
	  case MR_WRITE:
	  case MR_WRITESOFT:
	    CACHE_MISS;
	    *time = BusAccess(*time);
	    /* Copy data from memory to cache */
	    WriteCacheLine(CacheData(processor, set, line),
			   &memory[address]);
	    WriteCacheLine(CacheFebits(processor, set, line),
			   &memtag[address]);
	    CacheTag(processor, set, line) = tag;
	    *state = reqtype == MR_WRITE ? C_RESERVED : C_SOFT_RESERVED;
	    /* Must Copy data from cache to memory after writing */
	    break;
	  case MR_FLUSHSOFT:
	    break;
	}
	break;
      case C_VALID:
	switch(reqtype) {
	  case MR_READ:
	    CACHE_HIT;
	    break;
	  case MR_WRITE:
	    CACHE_MISS;
	    *time = BusAccess(*time);
	    /* Must Copy data from cache to memory after writing */
	    *state = C_RESERVED;
	    break;
	  case MR_READSOFT:
	    CACHE_HIT;
	    *state = C_SOFT_VALID;
	    break;
	  case MR_WRITESOFT:
	    CACHE_MISS;
	    *time = BusAccess(*time);
	    /* Must Copy data from cache to memory after writing */
	    *state = C_SOFT_RESERVED;
	    break;
	  case MR_FLUSHSOFT:
	    *state = C_INVALID;
	    break;
	}
	break;
      case C_RESERVED:
	switch(reqtype) {
	  case MR_READ:
	    CACHE_HIT;
	    break;
	  case MR_WRITE:
	    CACHE_HIT;
	    *state = C_DIRTY;
	    break;
	  case MR_READSOFT:
	    CACHE_HIT;
	    *state = C_SOFT_RESERVED;
	    break;
	  case MR_WRITESOFT:
	    CACHE_HIT;
	    *state = C_SOFT_DIRTY;
	    break;
	  case MR_FLUSHSOFT:
	    *state = C_INVALID;
	    break;
	}
	break;
      case C_DIRTY:
	switch(reqtype) {
	  case MR_READ:
	    CACHE_HIT;
	    break;
	  case MR_WRITE:
	    CACHE_HIT;
	    break;
	  case MR_READSOFT:
	    CACHE_HIT;
	    *state = C_SOFT_DIRTY;
	    break;
	  case MR_WRITESOFT:
	    CACHE_HIT;
	    *state = C_SOFT_DIRTY;
	    break;
	  case MR_FLUSHSOFT:
	    *time = BusAccess(*time);
	    /* Write data from cache to memory */
	    WriteCacheLine(&memory[address], CacheData(processor, set, line));
	    WriteCacheLine(&memtag[address], CacheFebits(processor,set,line));
	    *state = C_INVALID;
	    break;
	}
	break;
      case C_SOFT_VALID:
	switch(reqtype) {
	  case MR_READ:
	  case MR_WRITE:
	    return(MEM_ERROR);
	  case MR_READSOFT:
	    CACHE_HIT;
	    break;
	  case MR_WRITESOFT:
	    CACHE_MISS;
	    *time = BusAccess(*time);
	    /* Must Copy data from cache to memory after writing */
	    *state = C_SOFT_RESERVED;
	    break;
	  case MR_FLUSHSOFT:
	    *state = C_INVALID;
	    break;
	}
	break;
      case C_SOFT_RESERVED:
	switch(reqtype) {
	  case MR_READ:
	  case MR_WRITE:
	    return(MEM_ERROR);
	  case MR_READSOFT:
	    CACHE_HIT;
	    break;
	  case MR_WRITESOFT:
	    CACHE_HIT;
	    *state = C_SOFT_DIRTY;
	    break;
	  case MR_FLUSHSOFT:
	    *state = C_INVALID;
	    break;
	}
	break;
      case C_SOFT_DIRTY:
	switch(reqtype) {
	  case MR_READ:
	  case MR_WRITE:
	    return(MEM_ERROR);
	  case MR_READSOFT:
	    CACHE_HIT;
	    break;
	  case MR_WRITESOFT:
	    CACHE_HIT;
	    break;
	  case MR_FLUSHSOFT:
	    *time = BusAccess(*time);
	    /* Write data from cache to memory */
	    WriteCacheLine(&memory[address], CacheData(processor, set, line));
	    WriteCacheLine(&memtag[address], CacheFebits(processor,set,line));
	    *state = C_INVALID;
	    break;
	}
	break;
    }
    
#ifdef CACHEDEBUG
    printf("line %d AddrINCache(%d,%d) = %d\n",
	   GetLineFromAddr(processor, address), processor, address,
	   AddrInCache(processor, address));
    
#endif
    
    return(MEM_READY);
}


int processor_request_post(Time *time, int reqtype, int processor,
			   int tid, ulong address) 
{
    int set = GetSetFromAddr(address);
    int line = GetLineFromAddr(processor, address);
    short *state = &CacheState(processor, set, line);
    if ((reqtype == MR_WRITE && *state == C_RESERVED) ||
	(reqtype == MR_WRITESOFT && *state == C_SOFT_RESERVED)) {
	*time = BusAccess(*time);
	WriteCacheLine(&memory[address], CacheData(processor, set, line));
	WriteCacheLine(&memtag[address], CacheFebits(processor, set, line));
    }
    return(MEM_READY);
}


void protocol_fence() 
{}
