

 /**************************************************************************\
 *
 *                 Proteus Parallel-Architecture Simulator
 *                Eric A. Brewer  and  Chris N. Dellarocas
 *                     Laboratory for Computer Science
 *                  Massachusetts Institute of Technology
 *
 * Module: alewife.net_exact.c
 *
 * Description:
 *         Hop-by-hop Alewife Network module.
 *
 * Last Modified:  $Date: 92/11/13 16:34:48 $
 *
 * Global Functions:
 *     int channel_busy(int target)
 *     void send_packet_handler_(SimRequest *rptr) 
 *     void receive_packet_handler_(SimRequest *rptr) 
 *     void route_packet_handler_(SimRequest *rptr) 
 *     void init_network_() 
 *     FuncPtr define_router(FuncPtr r) 
 *     int default_router_(int from, int to, int prev, int curr, int *line) 
 *
 * Global Variables: none
 *
 ****************************************************************************
 *   Copyright 1991
 *   Eric A. Brewer  and  Chris N. Dellarocas
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.		
 ****************************************************************************
 * $Header: /psg/proteus/RCS/alewife.exact_net.c,v 1.4 92/11/13 16:34:48 brewer Exp $
 * $Log:	alewife.exact_net.c,v $
 * Revision 1.4  92/11/13  16:34:48  brewer
 * Now calls dispatch_packet_ to handle packets.
 * 
 * Revision 1.3  92/05/01  17:21:08  brewer
 * Removed  rptr->h.timestamp -= SWITCH_DELAY from send_request_handler_
 * It sometimes caused CC messages to arrive out of order.
 * 
 * Revision 1.2  92/03/31  16:50:40  brewer
 * cleaned up default_router_
 * removed unused variables req and ra from handlers
 * 
 * Revision 1.1  92/02/11  13:54:49  brewer
 * Initial revision
 * 
 \**************************************************************************/

#include "simcalls.h"
#include "thread.h"
#include "processor.h"
#include "cache.h"
#include "shmem.h"
#include "sema.h"
#include "event.h"
#include "packet.h"

#ifdef WATCH_NET
#define EVENT_NETIDLE(p, time)   ATEVENT(EV_NETIDLE, p << 16, time)
#define EVENT_NETBUSY(p, time)   ATEVENT(EV_NETBUSY, p << 16, time)
#else
#define EVENT_NETIDLE(p, time)
#define EVENT_NETBUSY(p, time)
#endif

#ifdef WATCH_NET
#define NET_UTIL(p, t, b) \
  /* if ( (t) > (b) || (b) == 0) { \
     if ( (b) > 0 ) EVENT_NETIDLE(p, b); \
     EVENT_NETBUSY(p, t); \
     } */
#define NET_CONTENTION(p, t, b) \
{ \
    Time d = ((b) > (t)) ? ((b) - (t)) : 0; \
      SUM_ARRAY_STAT(STAT_NETCONT, (p), (d), (t)); \
    }
#else
#define NET_UTIL(p, t, b)
#define NET_CONTENTION(p, t, b)
#endif


static Time in_free_again[NO_OF_PROCESSORS][NO_OF_LINES];
static Time out_free_again[NO_OF_PROCESSORS][NO_OF_LINES];

#define DEFAULT_ROUTER default_router_
static FuncPtr router_;

/* These lines will be moved somewhere more appropriate */
#undef  SC_RECV_PACKET
#define SC_RECV_PACKET       SC_ROUTE_PACKET
#define IPRIO_LOCAL          200      

#define LAUNCH_COST   1  /* delay to caller of 'launch' instruction */


 /**************************************************************************\
 *  Route a packet one hop
 \**************************************************************************/

int default_router_(from, to, prev, curr, line) 
int from;
int to;
int prev;
int curr;
int *line;
{
    static int routetab[NO_OF_PROCESSORS][NO_OF_PROCESSORS];
    static int linetab[NO_OF_PROCESSORS][NO_OF_PROCESSORS];
    
    int dcurr, dto, diff, incr = 1;
    int i;
    int *rp;
    int ccurr = curr, cto = to;
    unsigned adiff;
    
    /* if line and next node already known, look them up */
    if (*(rp = &routetab[ccurr][cto])) {
	*line = linetab[ccurr][cto];
	return( *rp - 1 );
    }
    
    /* line, next node unknown, so compute them */

    for(i=0; i<Ndim; i++) {
	dcurr = curr % Nk;
	dto   = to   % Nk;
	curr  = (curr - dcurr) / Nk;
	to    = (to   - dto  ) / Nk;
	if (adiff = abs(diff = dto - dcurr))
#ifdef N_BIDIR
	  if ( (diff > 0 && adiff <= Nk-adiff) ||
	      (diff < 0 && adiff >= Nk-adiff) ) {
	      *line = linetab[ccurr][cto] = i;
	      *rp   = (dcurr == Nk-1 ? ccurr - dcurr*incr : ccurr + incr) + 1;
	      return( *rp - 1 );
	  } else {
	      *line = linetab[ccurr][cto] = i + Ndim;
	      *rp   = (dcurr == 0   ? ccurr + (Nk-1)*incr : ccurr - incr) + 1;
	      return( *rp - 1 );
	  }
#else
	{
	    *line = linetab[ccurr][cto] = i;
	    *rp   = (dcurr == Nk-1 ? ccurr - dcurr*incr : ccurr + incr) + 1;
	    return( *rp - 1 );
	}
#endif
	incr *= Nk;
    }
    fatal("ERROR in default_router_!!\n");
    return(0);
}      


FuncPtr define_router(r) 
FuncPtr r;
{
    FuncPtr previous = router_;
    
    if ( r == (FuncPtr)I_DEFAULT)
      router_ = (FuncPtr)DEFAULT_ROUTER;
    else if ( r != NULL )
      router_ = r;
    
    return( previous );
}


int channel_busy(target)
int target;
{
    int line, next;
    
    if (target == processor_)
      fatal("channel_busy called with target == processor_\n");
    
    next = (*(int (*)())router_)(processor_, target, -1, processor_, &line);
    return(out_free_again[processor_][line] > CURRENTTIME);
}



typedef struct {
    SimHeader  h;
    Word       from;        /* argv[0] */
    Word       to;          /* argv[1] */
    Packet     *packet;     /* argv[2] */
    Word       flags;       /* argv[3] */
    Word       previous;    /* argv[4] */
    Word       current;     /* argv[5] */
    Word       line;        /* argv[6] */
} SendRequest;

 /**************************************************************************\
 *  Handler to send a packet
 \**************************************************************************/

void send_packet_handler_(rptr) 
SendRequest *rptr;
{
    Time receivetime, start_sending, *outbuf;
    int from, to, next, line;
    
#ifdef NETDEBUG
    printf("#%d Net will send packet from %d to %d T%d\n",
	   rptr->h.tid, rptr->from, rptr->to,
	   rptr->h.timestamp);
    DisplayPacket(stdout, rptr->packet);
#endif
    
    from = rptr->from;
    to   = rptr->to;
    
    if ( InvalidProc(to) )
      fatal("Attempt to send packet to nonexistent processor %d", to);
    
    if (rptr->flags & RESUME_CALLER) {
	MAKE_RESUME_REQUEST(rptr->h.tid, rptr->h.timestamp + LAUNCH_COST, OK);
    }

    /* Special case -- if dest == source then packet is immediately dispatched 
     * This feature is used in the cache coherence protocol to deliver memory
     * requests to a processor's own shared memory module
     */
    if (from == to) {
	rptr->current = from;               /* Make sure Route will dispatch */
	route_packet_handler_(rptr);
	return;
    }
    
    next = (*(int (*)())router_)(from, to, -1, from, &line);
    outbuf = &out_free_again[from][line];
    
    /*
      if (*outbuf > rptr->h.timestamp) printf("Delay %d cycles\n", *outbuf > rptr->h.timestamp ? *outbuf - rptr->h.timestamp : 0);
      */
    
    NET_UTIL(from, rptr->h.timestamp, *outbuf);
    NET_CONTENTION(from, rptr->h.timestamp, *outbuf);
    start_sending =  MAX(rptr->h.timestamp, *outbuf);
    receivetime = start_sending + WIRE_DELAY + SWITCH_DELAY;
    *outbuf = start_sending + SWITCH_DELAY * rptr->packet->length;
    
    MAKE_REQUEST_7(SC_RECV_PACKET, rptr->h.tid, receivetime, from, to,
		   rptr->packet, FALSE, from, next, line);

    /* Schedule completion interrupt, if required */
    if (rptr->flags & COMPLETE)
      MAKE_REQUEST_5(SC_SCHEDULE_INTR, rptr->h.tid,
		     start_sending
		     + SWITCH_DELAY * rptr->packet->length,
		     from, INTR_SEND_COMPLETE, IPRIO_LOCAL, 0, 0); 
}


 /**************************************************************************\
 *  Handler for receiving from network: receive_packet_handler_
 \**************************************************************************/

void receive_packet_handler_(rptr) 
SendRequest *rptr;
{
    int line;
    Time receivetime, *inbuf;
    
    line = (rptr->line + Ndim ) % NO_OF_LINES;
    inbuf = &in_free_again[rptr->current][line];
    
    /*
    if (*inbuf > rptr->h.timestamp)
      printf("Delay %d cycles\n",
             *inbuf > rptr->h.timestamp ? *inbuf - rptr->h.timestamp : 0);
    */
    
    receivetime = MAX(rptr->h.timestamp, *inbuf);
    *inbuf = receivetime + SWITCH_DELAY * rptr->packet->length;
    
    MAKE_REQUEST_6(SC_ROUTE_PACKET, rptr->h.tid, receivetime,
		   rptr->from, rptr->to, rptr->packet, FALSE,
		   rptr->previous, rptr->current);
}


 /**************************************************************************\
 *  Handler for routing: route_packet_handler_
 \**************************************************************************/

void route_packet_handler_(rptr) 
SendRequest *rptr;
{
    
    Time receivetime, start_sending, *outbuf;
    int line, next;
    
    receivetime = rptr->h.timestamp;
    
    if ( rptr->current == rptr->to ) {
	
#ifdef NEWNETDEBUG
	printf("%d. (%d --> %d ) Packet received.\n", receivetime, rptr->from, rptr->to);
#endif
	
	dispatch_packet_(rptr->h.tid, receivetime + SWITCH_DELAY, 
			 rptr->from, rptr->to, rptr->packet);
    } else {
	if (rptr->current > NO_OF_PROCESSORS)
	  fatal("Routine to invalid processor: %d\n", rptr->current);
	
	next = (*(int (*)())router_)
	  (rptr->from, rptr->to, rptr->previous, rptr->current, &line);
	
	outbuf = &out_free_again[rptr->current][line];
	
	NET_UTIL(rptr->current, rptr->h.timestamp, *outbuf);
	NET_CONTENTION(rptr->current, rptr->h.timestamp, *outbuf);
	start_sending = MAX(rptr->h.timestamp, *outbuf);
	receivetime = start_sending + WIRE_DELAY + SWITCH_DELAY;
	*outbuf = start_sending + SWITCH_DELAY * rptr->packet->length;
	
#ifdef NEWNETDEBUG
	printf("%d. (%d --> %d) @ %d next = %d line = %d\n",
	       receivetime, rptr->from, rptr->to, rptr->current, next, line);
#endif
	
	MAKE_REQUEST_7(SC_RECV_PACKET, rptr->h.tid, receivetime, 
		       rptr->from, rptr->to, rptr->packet, FALSE,
		       rptr->current, next, line);
    }
}


void init_network_() 
{
    (void)define_router((FuncPtr)I_DEFAULT);
    printf("(Alewife hop-by-hop network simulation)\n");
}
