/******************************************************************************
*
*  Copyright (C) 1995 A. Bode, J. Pruyne and G. Stellner
*
*  This file is part of CoCheck
*
*  CoCheck is free software; you can redistribute it and/or
*  modify it under the terms of the GNU Library General Public
*  License as published by the Free Software Foundation; either
*  version 2 of the License, or (at your option) any later version.
*
*  CoCheck is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  Library General Public License for more details.
*
*  You should have received a copy of the GNU Library General Public
*  License along with this library; if not, write to the Free
*  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*  Contact to the authors:
*
*  electronic mail: {bode,stellner}@informatik.tu-muenchen.de
*
*  paper mail:      Prof. Dr. A. Bode
*                   Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                   Institut f"ur Informatik
*                   Technische Universit"at M"unchen
*                   80290 M"unchen
*                   Germany
*
******************************************************************************/
/******************************************************************************

  signal.c,v
  1995/11/07 13:51:46
  1.8
  Exp
  stellner

  Authors: G. Stellner

  Description: Signal handler installation routines

******************************************************************************/
static char rcs_id[] = "signal.c,v 1.8 1995/11/07 13:51:46 stellner Exp";

#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <signal.h>

#include "pvm3.h"
#include "condor_sys.h"

#include "kojakmalloc.h"
#include "types.h"
#include "error.h"
#include "CoCheckExport.h"

static int disable_count = 0;
static int in_sig_hand = 0;
static int olderrno;

#ifdef BSD_SIG
static int oldmask;
#endif

static int osbuf;
static int orbuf;
static int scm;
static int TidListMid = -1;
static int CaughtSig = 0;

void _EnterSigHand(oerrno)
int oerrno;
{
  if (in_sig_hand++ > 1 || disable_count++ > 1)
    _Error(FAIL, "_EnterSigHand", _LOC_,
           "in_sig_hand=%d, disable_count=%d", in_sig_hand, disable_count);

  scm = SetSyscalls(SYS_LOCAL | SYS_UNMAPPED);

  osbuf = PVM_setsbuf(PVM_mkbuf(PvmDataDefault));
  orbuf = PVM_setrbuf(PVM_mkbuf(PvmDataDefault));

  olderrno = oerrno;
}

int _LeaveSigHand()
{
  if (in_sig_hand-- < 0 || disable_count-- < 0)
    _Error(FAIL, "_LeaveSigHand", _LOC_,
           "in_sig_hand=%d, disable_count=%d", in_sig_hand, disable_count);

  PVM_freebuf(PVM_setsbuf(osbuf));
  PVM_freebuf(PVM_setrbuf(orbuf));

  SetSyscalls(scm);
#if 0
{
  int one=1;
  while(one);
}
#endif

  return(olderrno);
}

void _SigDis()
{
#ifndef BSD_SIG
  sigset_t set;
#endif

  if (disable_count == 0 && in_sig_hand == 0)
  {
#ifdef BSD_SIG
    oldmask = sigblock(sigmask(CO_CHECK_SIG));
#else
    sigemptyset(&set);
    sigaddset(&set, CO_CHECK_SIG);
    sigprocmask(SIG_BLOCK, &set, (sigset_t *)0);
#endif
  }

  disable_count++;

  _Db(9, "_SigDis()", _LOC_, "count: %d", disable_count);

  return;
}

void _SigEn()
{
#ifndef BSD_SIG
  sigset_t set;
#endif

  if (disable_count == 0)
  {
    _Error(WARN,  "_SigEn", _LOC_, "no matching _SigDis()");
    return;
  }

  _Db(9, "_SigEn()", _LOC_, "count: %d", disable_count);

  disable_count--;              /* Decrement while signals are blocked. */

  if (disable_count == 0 && in_sig_hand == 0)
  {
#ifdef BSD_SIG
    sigsetmask(oldmask);
#else
    sigemptyset(&set);
    sigaddset(&set, CO_CHECK_SIG);
    sigprocmask(SIG_UNBLOCK, &set, (sigset_t *)0);
#endif
  }

  return;
}

int _CoCheckSigHand(sig, cod, scp)
int sig;
int cod;
struct sigcontext *scp;
{
  _EnterSigHand(errno);
  _Db(0, "_CoCheckSigHand", _LOC_, "sig %d, TidListMid %d", sig, TidListMid);
  _LeavePVM(TidListMid);        /* gracefully disconnect from PVM */
  if ( ReallyDoCkpt() ) {
	  _DoCkpt();                /* create checkpoint file of this process */
  }

#if defined(STOPAFTERRST)
{
  int one=1;
  while(one);
}
#endif

  _ReJoinPVM();                 /* return from checkpoint, reconnect to PVM */
  CaughtSig = sig;              /* set flag for CatchSignal */
  TidListMid = -1;
  errno = _LeaveSigHand();
}

int _InSigHand(va_alist)
va_dcl
{
  va_list ap;
  int i;
  int sig;
  HANDLER hand;
  int bsigc;
  int *bsigv;
#ifdef BSD_SIG
  struct sigvec vec;
#else
  struct sigaction sigact;
#endif

  va_start(ap);
  sig = va_arg(ap, int);
  hand = (HANDLER)va_arg(ap, HANDLER);
  bsigc = 0;

  if ((bsigv = (int *)TMALLOC(1, int, "SIGNAL")) == (int *)0)
  {

    _Error(WARN, "_InSigHand", _LOC_, "no memory for sig vector");
    return(-1);
  }

  while ((i = va_arg(ap, int)) != 0)
  {
    if ((bsigv = (int *)TREALLOC(bsigv, bsigc+1, int)) == (int *)0)
    {
       _Error(WARN, "_InSigHand", _LOC_,
              "no memory for additional sig vector: %d", bsigc);
       return(-1);
     }

    bsigv[bsigc++] = i;
  }

  va_end(ap);

#ifdef BSD_SIG
  vec.sv_handler = hand;
  vec.sv_flags = 0;
  vec.sv_mask = 0;

  for (i=0; i<bsigc; i++)
    vec.sv_mask |= sigmask(bsigv[i]);

  if (sigvec(sig, &vec, (struct sigvec*)0) != 0)
  {
    _Error(WARN, "_InSigHand", _LOC_, "sig hand *0x%x with 0x%x not installed",
           vec.sv_handler, vec.sv_mask);
    return(-1);
  }
  _Db(9, "_InSigHand", _LOC_, "sig handler *0x%x with 0x%x installed",
      vec.sv_handler, vec.sv_mask);
#else
  sigact.sa_handler = hand;
  sigact.sa_flags = 0;

  if (sigemptyset(&sigact.sa_mask) != 0)
  {
    _Error(WARN, "_InSigHand", _LOC_, "empty set not created");
    return(-1);
  }


  for (i=0; i<bsigc; i++)
    if (sigaddset(&sigact.sa_mask, bsigv[i]) != 0)
    {
      _Error(WARN, "_InSigHand", _LOC_, "sig %d not added", bsigv[i]);
      return(-1);
    }

  if (sigaction(sig, &sigact, (struct sigaction *)0) < 0)
  {
    _Error(WARN, "_InSigHand", _LOC_, "sig hand *0x%x with 0x%x not installed",
           sigact.sa_handler, sigact.sa_mask);
    return(-1);
  }

  _Db(9, "_InSigHand", _LOC_, "sig handler *0x%x with 0x%x installed",
         sigact.sa_handler, sigact.sa_mask);
#endif

  FREE(bsigv);

  return(0);
}

void _CatchSig(sig, bufid)
int sig;
int bufid;
{
  pid_t pid;
#ifndef BSD_SIG
  sigset_t set;
#endif

  _Db(0, "_CatchSig", _LOC_, "waiting for signal %d", sig);

  TidListMid = bufid;           /* tell sig hand: sync mes already arrived */
  CaughtSig = 0;                /* make sure we get right signal */

#ifndef BSD_SIG
  sigemptyset(&set);
#endif

  while (CaughtSig != sig)      /* as long as we don't get right signal */
#ifdef BSD_SIG
    sigpause(0);
#else
    sigsuspend(&set);
#endif
}
