/******************************************************************************
*
*  Copyright (C) 1995 A. Bode, J. Pruyne and G. Stellner
*
*  This file is part of CoCheck
*
*  CoCheck is free software; you can redistribute it and/or
*  modify it under the terms of the GNU Library General Public
*  License as published by the Free Software Foundation; either
*  version 2 of the License, or (at your option) any later version.
*
*  CoCheck is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  Library General Public License for more details.
*
*  You should have received a copy of the GNU Library General Public
*  License along with this library; if not, write to the Free
*  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*  Contact to the authors:
*
*  electronic mail: {bode,stellner}@informatik.tu-muenchen.de
*
*  paper mail:      Prof. Dr. A. Bode
*                   Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                   Institut f"ur Informatik
*                   Technische Universit"at M"unchen
*                   80290 M"unchen
*                   Germany
*
******************************************************************************/
/******************************************************************************

  pvm.c,v
  1995/11/07 13:51:44
  1.6
  Exp
  stellner

  Authors: G. Stellner

  Description: Provide the PVM stub functions

******************************************************************************/
static char rcs_id[] = "pvm.c,v 1.6 1995/11/07 13:51:44 stellner Exp";

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <signal.h>

#ifdef  __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "pvm3.h"
#include "pvmtev.h"

#include "kojakmalloc.h"
#include "types.h"
#include "error.h"
#include "CoCheckExport.h"

extern int (*PVM_recvf())();
extern int _CoCheckSigHand();
extern char *condor_get_executable_name();

static int MyTid = 0;
static int ParentTid = 0;
static int (*MatchMesFunc)() = NULL;
static int CoCheckSyn = 0;

static int MatchMes(mid, tid, code)
int mid;
int tid;
int code;
{
  int len;
  int ctid;
  int ccode;
  int ret;

  CoCheckSyn = 0;              /* default: no sync message arrived */

  _Db(1, "MatchMes", _LOC_, "want mid %d 0x%x, %d", mid, tid, code);
  ret = PVM_bufinfo(mid, &len, &ccode, &ctid);
  _Db(1, "MatchMes", _LOC_, "have %d 0x%x, %d", mid, ctid, ccode);

  if (ret < 0)
    return(ret);

  if (ccode == CO_CHECK_SYN)    /* is it the sync message */
  {
    CoCheckSyn = 1;             /* yes, then set the flag for pvm_recv */
    return(1);                  /* tell PVM to pick this message */
  }

  if (MatchMesFunc == NULL)     /* is there a user defined function */
  {                             /* no, then perform standard match */
    if (((tid == -1) || (tid == ctid)) &&
        ((code == -1) || (code == ccode)))
      return(1);
    else
      return(0);
  }
  else                          /* yes, there is one, so let it do the work */
    return((*MatchMesFunc)(mid, code, tid));
}

int pvm_addhosts(names, count, svp)
char **names;
int count;
int *svp;
{
  int ret;

  _SigDis();

  ret = PVM_addhosts(names, count, svp);

  _SigEn();

  return(ret);
}

int pvm_archcode(arch)
char *arch;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_archcode()", _LOC_, "--> PVM_archcode(%s)", arch);

  ret = PVM_archcode(arch);

  _SigEn();

  return(ret);
}

int pvm_bufinfo(mid, len, code, tid)
int mid;
int *len;
int *code;
int *tid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_bufinfo()", _LOC_, "--> PVM_bufinfo(%d, *0x%x, *0x%x, *0x%x)",
      mid, len, code, tid);

  ret = PVM_bufinfo(mid, len, code, tid);

  if (ret == PvmOk)
    *tid = _Rarp(*tid);         /* convert to original tid */

  _SigEn();

  return(ret);
}

int pvm_catchout(ff)
FILE *ff;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_catchout()", _LOC_, "--> PVM_catchout(*0x%x)", ff);

  ret = PVM_catchout(ff);

  _SigEn();

  return(ret);
}

int pvm_config(nhostp, narchp, hostp)
int *nhostp;
int *narchp;
struct pvmhostinfo **hostp;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_config()", _LOC_, "--> PVM_config(*0x%x, *0x%x, **0x%x)",
      nhostp, narchp, hostp);

  ret = PVM_config(nhostp, narchp, hostp);

  _SigEn();

  return(ret);
}

int pvm_delete(name, req)
char *name;
int req;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_delete()", _LOC_, "--> PVM_delete(%s, %d)", name, req);

  ret = PVM_delete(name, req);

  _SigEn();

  return(ret);
}

int pvm_delhosts(names, count, svp)
char **names;
int count;
int *svp;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_delhosts()", _LOC_, "--> PVM_delete(**0x%x, %d, *0x%x)",
      names, count, svp);

  ret = PVM_delhosts(names, count, svp);

  _SigEn();

  return(ret);
}

int pvm_exit()
{
  int ret;

  _SigDis();
  _Db(8, "pvm_exit()", _LOC_, "--> PVM_exit()");

  ret = PVM_exit();

  _SigEn();

  return(ret);
}

int pvm_freebuf(mid)
int mid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_freebuf()", _LOC_, "--> PVM_freebuf(%d)", mid);

  ret = PVM_freebuf(mid);

  _SigEn();

  return(ret);
}

int pvm_getfds(fds)
int **fds;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_getfds()", _LOC_, "--> PVM_getfds(**0x%x)", fds);

  ret = PVM_getfds(fds);

  _SigEn();

  return(ret);
}

int pvm_getmwid(mid)
int mid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_getmwid()", _LOC_, "--> PVM_getmwid(%d)", mid);

  ret = PVM_getmwid(mid);

  _SigEn();

  return(ret);
}

int pvm_getopt(what)
int what;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_getopt()", _LOC_, "--> PVM_getopt(%d)", what);

  ret = PVM_getopt(what);

  if ((ret != PvmBadParam) && ((what == PvmOutputTid) ||
                               (what == PvmTraceTid) ||
                               (what == PvmSelfOutputTid) ||
                               (what == PvmSelfTraceTid)))
    ret = _Rarp(ret);           /* convert to original tid */

  _SigEn();

  return(ret);
}

int pvm_getrbuf()
{
  int ret;

  _SigDis();
  _Db(8, "pvm_getrbuf()", _LOC_, "--> PVM_getrbuf()");

  ret = PVM_getrbuf();

  _SigEn();

  return(ret);
}

int pvm_getsbuf()
{
  int ret;

  _SigDis();
  _Db(8, "pvm_getsbuf()", _LOC_, "--> PVM_getsbuf()");

  ret = PVM_getsbuf();

  _SigEn();

  return(ret);
}

int pvm_gettmask(who, tmask)
int who;
Pvmtmask tmask;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_gettmask()", _LOC_, "--> PVM_gettmask(%d, %s)", who, tmask);

  ret = PVM_gettmask(who, tmask);

  _SigEn();

  return(ret);
}

int pvm_halt()
{
  int ret;

  _SigDis();
  _Db(8, "pvm_halt()", _LOC_, "--> PVM_halt()");

  ret = PVM_halt();

  _SigEn();

  return(ret);
}

int pvm_hostsync(host, clk, delta)
int host;
struct timeval *clk;
struct timeval *delta;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_hostsync()", _LOC_, "--> PVM_hostsync(0x%x, *0x%x, *0x%x)",
      host, clk, delta);

  ret = PVM_hostsync(host, clk, delta);

  _SigEn();

  return(ret);
}

int pvm_initsend(enc)
int enc;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_initsend()", _LOC_, "--> PVM_initsend(%d)", enc);

  ret = PVM_initsend(enc);

  _SigEn();

  return(ret);
}

int pvm_insert(name, req, data)
char *name;
int req;
int data;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_insert()", _LOC_, "--> PVM_insert(%s, %d, %d)", name, req, data);

  ret = PVM_insert(name, req, data);

  _SigEn();

  return(ret);
}

int pvm_kill(tid)
int tid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_kill()", _LOC_, "--> PVM_kill(0x%x)", tid);

  ret = PVM_kill(_Arp(tid));

  _SigEn();

  return(ret);
}

int pvm_lookup(name, req, datap)
char *name;
int req;
int *datap;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_lookup()", _LOC_, "--> PVM_lookup(%s, %d, 0x%x)",
      name, req, datap);

  ret = PVM_lookup(name, req, datap);

  _SigEn();

  return(ret);
}

int pvm_mcast(tids, count, code)
int *tids;
int count;
int code;
{
  int ret;
  int i;

  _SigDis();
  _Db(8, "pvm_mcast()", _LOC_, "--> PVM_mcast(*0x%x, %d, %d)",
      tids, count, code);

  for (i=0; i<count; i++)       /* take all tids and*/
    tids[i] = _Arp(tids[i]);    /* convert them to current tids */

  ret = PVM_mcast(tids, count, code);

  for (i=0; i<count; i++)       /* take them all again and */
    tids[i] = _Rarp(tids[i]);   /* convert them back; user sees no change */

  _SigEn();

  return(ret);
}

int pvm_mkbuf(enc)
int enc;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_mkbuf()", _LOC_, "--> PVM_mkbuf(%d)", enc);

  ret = PVM_mkbuf(enc);

  _SigEn();

  return(ret);
}

int pvm_mstat(host)
char *host;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_mstat()", _LOC_, "--> PVM_mstat(%s)", host);

  ret = PVM_mstat(host);

  _SigEn();

  return(ret);
}

int pvm_mytid()
{
  int ret;
  extern int pvmschedtid;
  char	*binary_name;

  _SigDis();
  _Db(8, "pvm_mytid()", _LOC_, "--> PVM_mytid()");

  if (MyTid == 0)
  {
    MyTid = PVM_mytid();        /* get my original tid */
    ParentTid = PVM_parent();   /* get parent tid */
	binary_name = condor_get_executable_name();
	PVM_initsend(PvmDataDefault);
	PVM_pkstr(binary_name);
	PVM_send(pvmschedtid, CO_CHECK_EXEC_NAME);

    MatchMesFunc = PVM_recvf(MatchMes); /* new match function for messages */
    _InSigHand(CO_CHECK_SIG, _CoCheckSigHand, CO_CHECK_SIG, 0);
  }

/*  ret = _Arp(MyTid);            /* convert to current tid */
  ret = MyTid;

  _SigEn();

  return(ret);                  /* make sure we return the tid of the task */
}

int pvm_notify(what, code, count, tids)
int what;
int code;
int count;
int *tids;
{
  int i;
  int ret;
  int ntid;

  _SigDis();
  _Db(8, "pvm_notify()", _LOC_, "--> PVM_notify(%d, %d, %d, 0x%x)",
      what, code, count, tids);

  for (i=0; i<count; i++) {      /* take all tids and*/
	  ntid = _Arp(tids[i]);    /* convert them to current tids */
	  fprintf(stderr, "pvm_notify() mapped from t%x to t%x\n",
			  tids[i], ntid);
	  tids[i] = ntid;
  }

  ret = PVM_notify(what, code, count, tids);

  for (i=0; i<count; i++)       /* take them all again and */
    tids[i] = _Rarp(tids[i]);   /* convert them back; user sees no change */

#if 0
  if (ret == PvmOk)
    _LogNotification(what, code, count, tids);
#endif

  _SigEn();

  return(ret);
}

int pvm_nrecv(tid, code)
int tid;
int code;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_nrecv()", _LOC_, "--> PVM_nrecv(0x%x, %d)", tid, code);

  ret = PVM_nrecv(_Arp(tid), code);

  _SigEn();

  return(ret);
}

int
#ifdef __STDC__
pvm_packf(const char *fmt, ...)
#else
pvm_packf(va_alist)
va_dcl
#endif
{
	va_list ap;
	int ret;

#ifdef __STDC__
  _SigDis();
	va_start(ap, fmt);
#else
	char    *fmt;

  _SigDis();
	va_start(ap);
	fmt = va_arg(ap, char *);
#endif
  _Db(8, "pvm_packf()", _LOC_, "--> PVM_packf()");

  ret = PVM_packf(fmt, ap);

  va_end(ap);
  _SigEn();

  return(ret);
}

int pvm_parent()
{
  int ret;

  _SigDis();
  _Db(8, "pvm_parent()", _LOC_, "--> PVM_parent()");

  ret = _Arp(ParentTid);        /* prepare to return current parent tid */

  if (ret == 0)                 /* was pvm_mytid already called? */
    ret = PVM_parent();         /* no, then let PVM produce an error */

  _SigEn();

  return(ret);
}

int pvm_perror(s)
char *s;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_perror()", _LOC_, "--> PVM_perror(%s)", s);

  ret = PVM_perror(s);

  _SigEn();

  return(ret);
}

pvm_pkbyte(cp, cnt, std)
char *cp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkbyte()", _LOC_, "--> PVM_pkbyte(*0x%x, %d, %d)", cp, cnt, std);

  ret = PVM_pkbyte(cp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkcplx(xp, cnt, std)
float *xp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkcplx()", _LOC_, "--> PVM_pkcplx(*0x%x, %d, %d)", xp, cnt, std);

  ret = PVM_pkcplx(xp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkdcplx(zp, cnt, std)
double *zp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkdcplx()", _LOC_, "--> PVM_pkdcplx(*0x%x, %d, %d)",
      zp, cnt, std);

  ret = PVM_pkdcplx(zp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkdouble(dp, cnt, std)
double *dp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkdouble()", _LOC_, "--> PVM_pkdouble(%lf, %d, %d)",
      *dp, cnt, std);

  ret = PVM_pkdouble(dp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkfloat(fp, cnt, std)
float*fp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkfloat()", _LOC_, "--> PVM_pkfloat(%f, %d, %d)",
      *fp, cnt, std);

  ret = PVM_pkfloat(fp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkint(ip, cnt, std)
int *ip;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkint()", _LOC_, "--> PVM_pkint(%d, %d, %d)",
      *ip, cnt, std);

  ret = PVM_pkint(ip, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pklong(lp, cnt, std)
long *lp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pklong()", _LOC_, "--> PVM_pklong(%ld, %d, %d)",
      *lp, cnt, std);

  ret = PVM_pklong(lp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkshort(sp, cnt, std)
short *sp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkshort()", _LOC_, "--> PVM_pkshort(%d, %d, %d)",
      *sp, cnt, std);

  ret = PVM_pkshort(sp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkstr(cp)
char *cp;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkstr()", _LOC_, "--> PVM_pkstr(%s)", cp);

  ret = PVM_pkstr(cp);

  _SigEn();

  return(ret);
}

pvm_pkuint(ip, cnt, std)
unsigned int *ip;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkuint()", _LOC_, "--> PVM_pkint(%d, %d, %d)",
      *ip, cnt, std);

  ret = PVM_pkuint(ip, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkulong(lp, cnt, std)
unsigned long *lp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkulong()", _LOC_, "--> PVM_pkulong(%ld, %d, %d)",
      *lp, cnt, std);

  ret = PVM_pkulong(lp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_pkushort(sp, cnt, std)
unsigned short *sp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pkushort()", _LOC_, "--> PVM_pkushort(%d, %d, %d)",
      *sp, cnt, std);

  ret = PVM_pkushort(sp, cnt, std);

  _SigEn();

  return(ret);
}

int pvm_precv(tid, tag, cp, len, dt, rtid, rtag, rlen)
int tid;
int tag;
void *cp;
int len;
int dt;
int *rtid;
int *rtag;
int *rlen;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_precv()", _LOC_,
      "--> PVM_precv(%d, %d, *0x%x, %d, %d, *0x%x, *0x%x, *0x%x)",
      tid, tag, cp, len, dt, rtid, rtag, rlen);

  ret = PVM_precv(_Arp(tid), tag, cp, len, dt, rtid, rtag, rlen);

  if (ret == PvmOk)
    *rtid = _Rarp(*rtid);       /* convert to current tid */

  _SigEn();

  return(ret);
}

int pvm_probe(tid, code)
int tid;
int code;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_probe()", _LOC_, "--> PVM_probe(0x%x, %d)", tid, code);

  ret = PVM_probe(_Arp(tid), code);

  _SigEn();

  return(ret);
}

int pvm_psend(tid, tag, cp, len, dt)
int tid;
int tag;
void *cp;
int len;
int dt;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_psend()", _LOC_,
      "--> PVM_psend(%d, %d, *0x%x, %d, %d)", tid, tag, cp, len, dt);

  ret = PVM_psend(_Arp(tid), tag, cp, len, dt);

  _SigEn();

  return(ret);
}

int pvm_pstat(tid)
int tid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_pstat()", _LOC_, "--> PVM_pstat(%d)", tid);

  ret = PVM_pstat(_Arp(tid));

  _SigEn();

  return(ret);
}

int pvm_recv(tid, code)
int tid;
int code;
{
  int ret;
  int lop;

  _SigDis();
  _Db(8, "pvm_recv()", _LOC_, "--> PVM_recv(0x%x, %d)", tid, code);

  do
  {
    lop = 0;                    /* default: no signal during recv */
    ret = _RecFromBuf(tid, code); /* try to extract from buffers first */

    if (ret < 0)                /* was there something matching extracted */
    {                           /* no, nothing matching was buffered */
      ret = PVM_recv(_Arp(tid), code); /* so, try to receive something */

      if (! (ret < 0))          /* did we really get a message */
      {

        if (CoCheckSyn == 1)   /* check for sync message; set in MatchMes() */
        {                       /* signal waiting for us */
          lop = 1;              /* remember to loop  */
          _Db(0, "pvm_recv()", _LOC_, "CO_CHECK_SYN mid %d", ret);
          _CatchSig(CO_CHECK_SIG, ret); /* get signal from RM */
          _Db(0, "pvm_recv()", _LOC_, "was ckpt'ed, looping: %d", lop);
        }
      }
    }
    else
      PVM_setrbuf(ret);

  } while (lop == 1);

  _SigEn();

  return(ret);
}

int (*pvm_recvf(new))()
#ifdef	IMA_SCO
	int (*new)(int,int,int);
#else
	int (*new)();
#endif
{
#ifdef	IMA_SCO
	int (*ret)(int,int,int);
#else
	int (*ret)();
#endif

  _SigDis();
  _Db(8, "pvm_recvf()", _LOC_, "--> PVM_recvf(*0x%x)", new);

  ret = PVM_recvf(new);

  _SigEn();

  return(ret);
}

int pvm_reg_hoster()
{
  int ret;

  _SigDis();
  _Db(8, "pvm_reg_hoster()", _LOC_, "--> PVM_reg_hoster()");

  ret = PVM_reg_hoster();

  _SigEn();

  return(ret);
}

int pvm_reg_rm(hip)
struct pvmhostinfo **hip;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_reg_rm()", _LOC_, "--> PVM_reg_rm(**0x%x)", hip);

  ret = PVM_reg_rm(hip);

  _SigEn();

  return(ret);
}

int pvm_reg_tasker()
{
  int ret;

  _SigDis();
  _Db(8, "pvm_reg_tasker()", _LOC_, "--> PVM_reg_tasker()");

  ret = PVM_reg_tasker();

  _SigEn();

  return(ret);
}

int pvm_send(tid, code)
int tid;
int code;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_send()", _LOC_, "--> PVM_send(0x%x, %d)", tid, code);

  ret = PVM_send(_Arp(tid), code);

  _SigEn();

  return(ret);
}

int pvm_sendsig(tid, signum)
int tid;
int signum;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_sendsig()", _LOC_, "--> PVM_sendsig(0x%x, %d)", tid, signum);

  ret = PVM_sendsig(_Arp(tid), signum);

  _SigEn();

  return(ret);
}

int pvm_setmwid(mid, wid)
int mid;
int wid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_setmwid()", _LOC_, "--> PVM_setmwid(%d, %d)", mid, wid);

  ret = PVM_setmwid(mid, wid);

  _SigEn();

  return(ret);
}

int pvm_setopt(what, val)
int what;
int val;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_setopt()", _LOC_, "--> PVM_setopt(%d, %d)", what, val);


  if ((what == PvmOutputTid) || (what == PvmTraceTid) ||
      (what == PvmSelfOutputTid) || (what == PvmSelfTraceTid))
    val = _Arp(val);            /* convert to current tid */

  ret = PVM_setopt(what, val);

  _SigEn();

  return(ret);
}

int pvm_setrbuf(mid)
int mid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_setrbuf()", _LOC_, "--> PVM_setrbuf(%d)", mid);

  ret = PVM_setrbuf(mid);

  _SigEn();

  return(ret);
}

int pvm_setsbuf(mid)
int mid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_setsbuf()", _LOC_, "--> PVM_setsbuf(%d)", mid);

  ret = PVM_setsbuf(mid);

  _SigEn();

  return(ret);
}

int pvm_spawn(file, argv, flags, where, count, tids)
char *file;
char **argv;
int flags;
char *where;
int count;
int *tids;
{
  int ret;

  _SigDis();
  _Db(0, "pvm_spawn()",
      _LOC_, "--> PVM_spawn(%s, **0x%x, %d, %s, %d, *0x%x)",
      file, argv, flags, where, count, tids);

  ret = PVM_spawn(file, argv, flags, where, count, tids);

  _SigEn();

  return(ret);
}

int pvm_start_pvmd(argc, argv, block)
int argc;
char **argv;
int block;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_start_pvmd()",
      _LOC_, "--> PVM_start_pvmd(%d, **0x%x, %d)", argc, argv, block);

  ret = PVM_start_pvmd(argc, argv, block);

  _SigEn();

  return(ret);
}

int pvm_tasks(where, ntaskp, taskp)
int where;
int *ntaskp;
struct pvmtaskinfo **taskp;
{
  int i;
  int ret;
  struct pvmtaskinfo *_taskp;

  _SigDis();
  _Db(8, "pvm_tasks()",
      _LOC_, "--> PVM_tasks(%d, *0x%x, **0x%x)", where, ntaskp, taskp);

  /* strange, it doesn't work if I use taskp directly XXX */
  ret = PVM_tasks(where, ntaskp, &_taskp);

  if (ret == PvmOk)
    for (i=0; i<*ntaskp; i++)
    {
      _taskp[i].ti_tid = _Rarp(_taskp[i].ti_tid);
      _taskp[i].ti_ptid = _Rarp(_taskp[i].ti_ptid);
    }

  *taskp = _taskp;              /* write the result in taskp XXX */

  _SigEn();

  return(ret);
}

int pvm_tickle(narg, argp, nresp, resp)
int narg;
int *argp;
int *nresp;
int *resp;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_tickle()",
      _LOC_, "--> PVM_tickle(%d, *0x%x, *0x%x, *0x%x)",
      narg, argp, nresp, resp);

  ret = PVM_tickle(narg, argp, nresp, resp);

  _SigEn();

  return(ret);
}

int pvm_tidtohost(tid)
int tid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_tidtohost()", _LOC_, "--> PVM_tidtohost(0x%x", tid);

  /* to fully support this call we would need to set up a mapping table for
   * daemon tids. This would afford an additional request to the RM and some
   * more work during the restart. XXX
   */
  ret = PVM_tidtohost(_Arp(tid));

  _SigEn();

  return(ret);
}

int pvm_trecv(tid, code, tmout)
int tid;
int code;
struct timeval *tmout;
{
  int ret;
  struct timeval placed;
  struct timezone tz;

  _SigDis();
  _Db(8, "pvm_trecv()", _LOC_,
      "--> PVM_trecv(0x%x, %d, *0x%x)", tid, code, tmout);
  gettimeofday(&placed, &tz);

  /* record the time when the call was placed. The difference to the current
   * time when a checkpoint is created can be used as the new time for the call
   * after restart. XXX
   */
  ret = PVM_trecv(_Arp(tid), code, tmout);

  _SigEn();

  return(ret);
}

int
#ifdef __STDC__
pvm_unpackf(const char *fmt, ...)
#else
pvm_unpackf(va_alist)
va_dcl
#endif
{
	va_list ap;
	int ret;

#ifdef __STDC__
  _SigDis();
	va_start(ap, fmt);
#else
	char    *fmt;

  _SigDis();
	va_start(ap);
	fmt = va_arg(ap, char *);
#endif
  _Db(8, "pvm_unpackf()", _LOC_, "--> PVM_unpackf()");

  ret = PVM_unpackf(fmt, ap);

  va_end(ap);
  _SigEn();

  return(ret);
}

pvm_upkbyte(cp, cnt, std)
char *cp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkbyte()", _LOC_,
      "--> PVM_upkbyte(*0x%x, %d, %d)", cp, cnt, std);

  ret = PVM_upkbyte(cp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkcplx(xp, cnt, std)
float *xp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkcplx()", _LOC_,
      "--> PVM_upkcplx(*0x%x, %d, %d)", xp, cnt, std);

  ret = PVM_upkcplx(xp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkdcplx(zp, cnt, std)
double *zp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkdcplx()", _LOC_, "--> PVM_upkdcplx(*0x%x, %d, %d)",
      zp, cnt, std);

  ret = PVM_upkdcplx(zp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkdouble(dp, cnt, std)
double *dp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkdouble()", _LOC_, "--> PVM_upkdouble(%lf, %d, %d)",
      *dp, cnt, std);

  ret = PVM_upkdouble(dp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkfloat(fp, cnt, std)
float*fp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkfloat()", _LOC_, "--> PVM_upkfloat(%f, %d, %d)",
      *fp, cnt, std);

  ret = PVM_upkfloat(fp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkint(ip, cnt, std)
int *ip;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkint()", _LOC_, "--> PVM_upkint(%d, %d, %d)",
      *ip, cnt, std);

  ret = PVM_upkint(ip, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upklong(lp, cnt, std)
long *lp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upklong()", _LOC_, "--> PVM_upklong(%ld, %d, %d)",
      *lp, cnt, std);

  ret = PVM_upklong(lp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkshort(sp, cnt, std)
short *sp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkshort()", _LOC_, "--> PVM_upkshort(%d, %d, %d)",
      *sp, cnt, std);

  ret = PVM_upkshort(sp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkstr(cp)
char *cp;
{
  int ret;

  _SigDis();

  _Db(8, "pvm_upkstr()", _LOC_, "--> PVM_upkstr(%s)", cp);

  ret = PVM_upkstr(cp);

  _SigEn();

  return(ret);
}

pvm_upkuint(ip, cnt, std)
unsigned int *ip;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkuint()", _LOC_, "--> PVM_upkint(%d, %d, %d)",
      *ip, cnt, std);

  ret = PVM_upkuint(ip, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkulong(lp, cnt, std)
unsigned long *lp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkulong()", _LOC_, "--> PVM_upkulong(%ld, %d, %d)",
      *lp, cnt, std);

  ret = PVM_upkulong(lp, cnt, std);

  _SigEn();

  return(ret);
}

pvm_upkushort(sp, cnt, std)
unsigned short *sp;
int cnt;
int std;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_upkushort()", _LOC_, "--> PVM_upkushort(%d, %d, %d)",
      *sp, cnt, std);

  ret = PVM_upkushort(sp, cnt, std);

  _SigEn();

  return(ret);
}


/* group related functions */

int pvm_barrier(group, cnt)
char *group;
int cnt;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_barrier()", _LOC_, "--> PVM_barrier(%s, %d)", group, cnt);

  ret = PVM_barrier(group, cnt);

  _SigEn();

  return(ret);
}

int pvm_bcast(group, msgtag)
char *group;
int msgtag;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_bcast()", _LOC_, "--> PVM_bcast(%s, %d)", group, msgtag);

  ret = PVM_bcast(group, msgtag);

  _SigEn();

  return(ret);
}

int pvm_gather(result, data, count, datatype, msgtag, gname, rootinst)
void *result;
void *data;
int count;
int datatype;
int msgtag;
int rootinst;
char *gname;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_gather()", _LOC_,
      "--> PVM_gather(*0x%x, *0x%x, %d, %d, %d, %s, %d)",
      result, data, count, datatype, msgtag, gname, rootinst);

  ret = PVM_gather(result, data, count, datatype, msgtag, gname, rootinst);

  _SigEn();

  return(ret);
}

int pvm_getinst(group, tid)
char *group;
int tid;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_getinst()", _LOC_, "--> PVM_getinst(%s, %d)", group, tid);

  ret = PVM_getinst(group, _Arp(tid));

  _SigEn();

  return(ret);
}

int pvm_gettid(group, inst)
char *group;
int inst;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_gettid()", _LOC_, "--> PVM_gettid(%s, %d)", group, inst);

  ret = _Rarp(PVM_gettid(group, inst));

  _SigEn();

  return(ret);
}

int pvm_gsize(group)
char *group;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_gsize()", _LOC_, "--> PVM_gsize(%s)", group);

  ret = PVM_gsize(group);

  _SigEn();

  return(ret);
}

int pvm_joingroup(group)
char *group;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_joingroup()", _LOC_, "--> PVM_joingroup(%s)", group);

  ret = PVM_joingroup(group);

  _SigEn();

  return(ret);
}

int pvm_lvgroup(group)
char *group;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_lvgroup()", _LOC_, "--> PVM_lvgroup(%s)", group);

  ret = PVM_lvgroup(group);

  _SigEn();

  return(ret);
}

int  pvm_reduce(func, data, count, datatype, msgtag, gname, rootinst)
#ifdef  IMA_SCO
void (*func)(int*, void*, void*, int*, int*);
#else
void (*func)();
#endif
void *data;
int count;
int datatype;
int msgtag;
int rootinst;
char *gname;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_reduce()", _LOC_,
      "--> PVM_reduce(*0x%x, *0x%x, %d, %d, %d, %s, %d)",
      func, data, count, datatype, msgtag, gname, rootinst);

  ret = PVM_reduce(func, data, count, datatype, msgtag, gname, rootinst);

  _SigEn();

  return(ret);
}

int pvm_scatter(result, data, count, datatype, msgtag, gname, rootinst)
void *result;
void *data;
int count;
int datatype;
int msgtag;
int rootinst;
char *gname;
{
  int ret;

  _SigDis();
  _Db(8, "pvm_scatter()", _LOC_,
      "--> PVM_scatter(*0x%x, *0x%x, %d, %d, %d, %s, %d)",
      result, data, count, datatype, msgtag, gname, rootinst);

  ret = PVM_scatter(result, data, count, datatype, msgtag, gname, rootinst);

  _SigEn();

  return(ret);
}

