/******************************************************************************
*
*  Copyright (C) 1995 A. Bode, J. Pruyne and G. Stellner
*
*  This file is part of CoCheck
*
*  CoCheck is free software; you can redistribute it and/or
*  modify it under the terms of the GNU Library General Public
*  License as published by the Free Software Foundation; either
*  version 2 of the License, or (at your option) any later version.
*
*  CoCheck is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  Library General Public License for more details.
*
*  You should have received a copy of the GNU Library General Public
*  License along with this library; if not, write to the Free
*  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*  Contact to the authors:
*
*  electronic mail: {bode,stellner}@informatik.tu-muenchen.de
*
*  paper mail:      Prof. Dr. A. Bode
*                   Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                   Institut f"ur Informatik
*                   Technische Universit"at M"unchen
*                   80290 M"unchen
*                   Germany
*
******************************************************************************/
/******************************************************************************

  kojak.c,v
  1995/11/07 13:51:19
  1.4
  Exp
  stellner

  Authors: G. Stellner

  Description: Tool to checkpoint/restart a PVM application

******************************************************************************/
static char rcs_id[] = "kojak.c,v 1.4 1995/11/07 13:51:19 stellner Exp";

#include <stdio.h>
#include <pvm3.h>
#include "CoCheckExport.h"

#define NOPVMRUNNING "No PVM system is running\n"
#define NORMRUNNING "No RM ``pvm_rm'' is running\n"

#define RST 0
#define CKPT 1
#define MIG 2

extern char *optarg;
extern int optind;

static int CkptMethod;
static int CreateCkpt;
static int CkptMigrateN;
static int CkptVacateN;
static int *CkptMigrate;
static int *CkptVacate;

void usage(argc, argv)
int argc;
char **argv;
{
  printf("Usage: %s [-c [-f|-p|-m tid]|-r|-m tid] file\n", argv[0]);
  printf("\t-c: create checkpoint (this is the default)\n");
  printf("\t-m tid: migrate task tid (muliple occurences possible)\n");
  printf("\t-r: restart from checkpoint\n");
  printf("\t-f: write checkpoint directly to files\n");
  printf("\t-p: write checkpoint over the network (this is the default)\n");
  printf("file: name of the master checkpoint file\n");
  exit(1);
}

void kojak(file, mode)
char *file;
int mode;
{
  int tid;
  int i;
  int n;
  int *dest;

  if ((tid = pvm_mytid()) < 0)
  {
    printf(NOPVMRUNNING);
    exit(1);
  }

  pvm_initsend(PvmDataDefault);
  pvm_pkstr(file);

  switch(mode)
  {
  case CKPT:                    /* just create a ckpt */
    printf("0x%x(%d): Creating checkpoint >%s<\n", tid, tid, file);
    CoCheckCkpt(file, 0, CkptMethod);
    break;
  case RST:                     /* just restart from ckpt */
    printf("0x%x(%d):Restarting from checkpoint >%s<\n", tid, tid, file);
    CoCheckRst(file);
    break;
  case MIG:                     /* do migration */
    printf("0x%x(%d):Migrating some processes >%s<\n\t", tid, tid, file);

    for(i=0; i<CkptMigrateN; i++)
      printf("0x%x(%d) ", CkptMigrate[i], CkptMigrate[i]);

    printf("\n");
    CoCheckMig(CkptMigrateN, CkptMigrate, n, dest, 0);
    break;
  }

  pvm_exit();
}

int main(argc,argv)
int argc;
char **argv;
{
  char opt;
                                /* set default values for ... */
  CkptMethod = -1;              /* ... no method */
  CreateCkpt = -1;              /* ... no selection */
  CkptMigrateN = 0;             /* up to now no processes to migrate */
  CkptVacateN = 0;

  if ((CkptMigrate = (int *)malloc(sizeof(int))) == NULL ||
      (CkptVacate = (int *)malloc(sizeof(int))) == NULL)
  {
    printf("%s: no more memory\n", argv[0]);
    exit(1);
  }

  while ((opt = getopt(argc, argv, "cfm:rp")) != -1)
    switch(opt)
    {
    case 'c':
      if (CreateCkpt != -1)     /* action already selected */
        usage(argc, argv);      /* error */

      CreateCkpt = CKPT;        /* action is ckpt */
      break;
    case 'm':
      if (CreateCkpt != -1   && /* action already selected */
          CreateCkpt != CKPT &&
          CreateCkpt != MIG)
        usage(argc, argv);      /* error */

      if (CreateCkpt == CKPT && /* method to create a ckpt already selected */
          CkptMethod != -1)
        usage(argc, argv);      /* error */

      if (CreateCkpt == -1)     /* nothing specified up to now */
      {
        CreateCkpt = MIG;       /* action is then migration */
        CkptMethod = CO_CHECK_METHOD_MIGR; /* with the migration method */
      }

      CkptMigrate = (int *)realloc(CkptMigrate, (CkptMigrateN+1), sizeof(int));
      CkptMigrate[CkptMigrateN++] = atoi(optarg);
      break;
    case 'r':
      if (CreateCkpt != -1)     /* action already selected */
        usage(argc, argv);      /* error */

      CreateCkpt = RST;         /* action is restart */
      break;
    case 'f':
      if (CreateCkpt != CKPT)   /* action is not create a ckpt */
        usage(argc, argv);      /* error */

      if (CkptMethod != -1)     /* method already set */
        usage(argc, argv);      /* error */

      CkptMethod = CO_CHECK_METHOD_FILE; /* set the method to ckpt to a file */
      break;
    case 'p':
      if (CreateCkpt != CKPT)   /* action is not create a ckpt */
        usage(argc, argv);      /* error */

      if (CkptMethod != -1)     /* method already set */
        usage(argc, argv);      /* error */

      CkptMethod = CO_CHECK_METHOD_PORT; /* set the method to ckpt to a port */
      break;
    default:
      usage(argc, argv);        /* an error has occured */
      break;
    }

  if ((argc - optind) != 1)     /* only 1 argument should be left: filename */
    usage(argc, argv);

  if (CreateCkpt == -1)         /* no action option specified */
    CreateCkpt = 1;             /* set default value: create ckpt */

  if (CkptMethod == -1)         /* no method option specified */
    CkptMethod = CO_CHECK_METHOD_PORT; /* set default value: port method */

  kojak(argv[optind], CreateCkpt);


  exit(0);
}
