/******************************************************************************
*
*  Copyright (C) 1995 A. Bode, J. Pruyne and G. Stellner
*
*  This file is part of CoCheck
*
*  CoCheck is free software; you can redistribute it and/or
*  modify it under the terms of the GNU Library General Public
*  License as published by the Free Software Foundation; either
*  version 2 of the License, or (at your option) any later version.
*
*  CoCheck is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*  Library General Public License for more details.
*
*  You should have received a copy of the GNU Library General Public
*  License along with this library; if not, write to the Free
*  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*  Contact to the authors:
*
*  electronic mail: {bode,stellner}@informatik.tu-muenchen.de
*
*  paper mail:      Prof. Dr. A. Bode
*                   Lehrstuhl f"ur Rechnertechnik und Rechnerorganisation
*                   Institut f"ur Informatik
*                   Technische Universit"at M"unchen
*                   80290 M"unchen
*                   Germany
*
******************************************************************************/
/******************************************************************************

  pvm_rm_support.c,v
  1995/11/07 13:51:10
  1.5
  Exp
  stellner

  Authors: J. Pruyne, G. Stellner

  Description: Additional routines used in the vanilla RM of CoCheck

******************************************************************************/
static char rcs_id[] = "pvm_rm_support.c,v 1.5 1995/11/07 13:51:10 stellner Exp";

/* THIS MUST BE THE SAME AS MAGIC DEFINED IN image.h OF THE CONDOR CKPT
   LIBRARY!!!!!!!!!!!! */
#define CONDOR_CKPT_MAGIC 0xfeafea

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "pvm3.h"
#include "../src/global.h"
#include "pvmsdpro.h"
#include "task.h"

struct pvmhostinfo	*hostlist = 0;
int		num_hosts = 0;
int		num_arches = 0;
int		host_list_size = 0;

typedef struct {
struct pvmtaskinfo pvm_tinfo;
int		otid;
char	*ckpt_file;
} TaskListEntry;

TaskListEntry	*tasklist = 0;
int		num_tasks = 0;
int		task_list_size = 0;

struct notification {
	int		kind;
	int		for_whom;
	int		msg_tag;
	int		on_tid;
} *notifylist;
int		num_notifys = 0;
int		notify_list_size = 0;

struct pvmhostinfo *find_host();
extern struct pvmhostinfo *local_host;

char *gen_ckpt_name();

new_notification(kind, for_whom, msg_tag, on_tid)
int kind, for_whom, msg_tag, on_tid;
{
	int		i;

	if (notify_list_size == 0) {
		notify_list_size = 10;
		notifylist = (struct notification *) malloc(notify_list_size *
												 sizeof(struct notification));
	}
	if (num_notifys + 1 >= notify_list_size) {
		notify_list_size *= 2;
		notifylist = (struct notification *) realloc(notifylist,
													 notify_list_size *
												  sizeof(struct notification));
	}

	notifylist[num_notifys].kind = kind;
	notifylist[num_notifys].for_whom = for_whom;
	notifylist[num_notifys].msg_tag = msg_tag;
	notifylist[num_notifys].on_tid = on_tid;

	num_notifys++;
	return 0;
}


send_notification(kind, on_tid)
int		kind, on_tid;
{
	int		sbuf;
	struct  pvmhostinfo	*notify_host;
	int		i;

	for (i = 0; i < num_notifys; i++) {
		if (notifylist[i].kind == kind && ((kind == PvmHostAdd) ||
										   (notifylist[i].on_tid == on_tid))) {
			sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
			if (kind == PvmTaskExit) {
				pvm_pkint(&on_tid, 1, 1);
			} else {
				notify_host = find_host(on_tid);
				pack_host(notify_host);
			}
			pvm_send(notifylist[i].for_whom, notifylist[i].msg_tag);
			pvm_freebuf(pvm_setsbuf(sbuf));
			if (kind != PvmHostAdd) {
				del_notification(notifylist[i].for_whom, on_tid);
			} else {
				notifylist[i].on_tid--;
				if (notifylist[i].on_tid <= 0) {
					del_notification(notifylist[i].for_whom,
									 notifylist[i].on_tid);
				}
			}
		}
	}
}


del_notification(for_whom, on_tid)
int		for_whom, on_tid;
{
	int		i;

	for (i = 0; i < num_notifys; i++) {
		if (notifylist[i].for_whom == for_whom &&
			notifylist[i].on_tid == on_tid) {
			break;
		}
	}

	if (i >= num_notifys) {
		fprintf(stderr, "del_notification: Unable to find notify struct\n");
		return -1;
	}

	notifylist[i] = notifylist[num_notifys - 1];
	num_notifys--;
	return 0;
}


new_host(host)
struct pvmhostinfo *host;
{
	int		i;
	int		old_arch;

	if (host_list_size == 0) {
		host_list_size = 10;
		hostlist = (struct pvmhostinfo *) malloc(host_list_size *
												 sizeof(struct pvmhostinfo));
	}
	if (num_hosts + 1 >= host_list_size) {
		host_list_size *= 2;
		hostlist = (struct pvmhostinfo *) realloc(hostlist, host_list_size *
												  sizeof(struct pvmhostinfo));
	}

	/* See if this host is already in the list */
	if (find_host(host->hi_tid) != 0) {
		return 0;
	}

	hostlist[num_hosts].hi_tid = host->hi_tid;
	hostlist[num_hosts].hi_name = strdup(host->hi_name);
	hostlist[num_hosts].hi_arch = strdup(host->hi_arch);
	hostlist[num_hosts].hi_speed = 0;

	old_arch = 0;
	for (i = 0; i < num_hosts; i++) {
		if (! strcmp(hostlist[i].hi_arch, host->hi_arch)) {
			old_arch = 1;
			break;
		}
	}

	if (!old_arch) {
		num_arches++;
	}
	num_hosts++;

	send_notification(PvmHostAdd, host->hi_tid);

	return 0;
}


del_host(host_tid)
int		host_tid;
{
	int		i;
	int		host_to_del;
	int		old_arch;
	struct pvmhostinfo *host;

	for (i = 0; i < num_hosts; i++) {
		if (hostlist[i].hi_tid == host_tid) {
			break;
		}
	}
	if (i >= num_hosts) {
		fprintf(stderr, "del_host: Unable to find host t%x\n", host->hi_tid);
		return -1;
	}

	send_notification(PvmHostDelete, host_tid);

	host_to_del = i;
	old_arch = 0;
	for (i = 0; i < num_hosts; i++) {
		if (i != host_to_del &&
			!strcmp(hostlist[i].hi_arch, hostlist[host_to_del].hi_arch)) {
			old_arch = 1;
		}
	}

	if (old_arch == 0) {
		num_arches--;
	}
	free(hostlist[host_to_del].hi_name);
	free(hostlist[host_to_del].hi_arch);
	hostlist[host_to_del] = hostlist[num_hosts - 1];
	num_hosts--;
	return 0;
}


pack_host(host)
struct pvmhostinfo	*host;
{
	pvm_pkint(&(host->hi_tid), 1, 1);
	pvm_pkstr(host->hi_name);
	pvm_pkstr(host->hi_arch);
	pvm_pkint(&(host->hi_speed), 1, 1);
}


pack_host_list()
{
	int		i;

	pvm_pkint(&num_hosts, 1, 1);
	pvm_pkint(&num_arches, 1, 1);
	for (i = 0; i < num_hosts; i++) {
		pack_host(&hostlist[i]);
	}
}


ckpt_host(host, ckpt_file)
struct pvmhostinfo	*host;
FILE	*ckpt_file;
{
	fprintf(ckpt_file, "%d\n", host->hi_tid);
	fprintf(ckpt_file, "%s\n", host->hi_name);
	fprintf(ckpt_file, "%s\n", host->hi_arch);
	fprintf(ckpt_file, "%d\n", host->hi_speed);
}


ckpt_host_list(ckpt_file)
FILE	*ckpt_file;
{
	int		i;

	write_ckpt_comment(ckpt_file, "Host List");
	fprintf(ckpt_file, "%d %d\n", num_hosts, num_arches);
	for (i = 0; i < num_hosts; i++) {
		ckpt_host(&hostlist[i], ckpt_file);
	}
}


struct pvmhostinfo *
find_host(tid)
int		tid;
{
	int		i;

	for (i = 0; i < num_hosts; i++) {
		if( hostlist[i].hi_tid == tid ) {
			return &(hostlist[i]);
		}
	}
	return 0;
}


select_host(flag, where)
int		flag;
char	*where;
{
	int		i;
	int		low_load = 99999;
	struct pvmhostinfo *best_host, *this_host;
	int		flag_mask;

	flag_mask = PvmTaskDebug | PvmTaskTrace | PvmMppFront | PvmHostCompl;
	best_host = 0;
	for (i = 0; i < num_hosts; i++) {
		this_host = 0;
		switch(flag & ~flag_mask) {
		    case PvmTaskDefault:
			    this_host = &(hostlist[i]);
			    break;
			case PvmTaskHost:
				if (!strcmp(where, hostlist[i].hi_name)) {
					this_host = &(hostlist[i]);
				}
				break;
			case PvmTaskArch:
				if (!strcmp(where, hostlist[i].hi_arch)) {
					this_host = &(hostlist[i]);
				}
				break;
		}
		if (flag & PvmHostCompl) {
			if (this_host) {
				this_host = 0;
			} else {
				this_host = &(hostlist[i]);
			}
		}
		if (this_host && this_host->hi_speed < low_load) {
			best_host = this_host;
			low_load = this_host->hi_speed;
		}
	}
	if (best_host) {
		if (best_host->hi_tid == local_host->hi_tid) {
			return TIDPVMD;
		} else {
			return best_host->hi_tid;
		}
	} else {
		fprintf(stderr, "Unable to find suitable host for where %s\n", where);
		return 0;
	}
}



new_task(task, ckpt_file)
struct pvmtaskinfo *task;
char	*ckpt_file;
{
	int		i;
	int		old_arch;
	struct pvmhostinfo	*host;

	if (task_list_size == 0) {
		task_list_size = 10;
		tasklist = (TaskListEntry *) malloc(task_list_size *
												 sizeof(TaskListEntry));
	}
	if (num_tasks + 1 >= task_list_size) {
		task_list_size *= 2;
		tasklist = (TaskListEntry *) realloc(tasklist, task_list_size *
												  sizeof(TaskListEntry));
	}
	tasklist[num_tasks].pvm_tinfo.ti_tid = task->ti_tid;
	tasklist[num_tasks].pvm_tinfo.ti_ptid = task->ti_ptid;
	tasklist[num_tasks].pvm_tinfo.ti_host = task->ti_host;
	tasklist[num_tasks].pvm_tinfo.ti_flag = task->ti_flag;
	tasklist[num_tasks].pvm_tinfo.ti_a_out = strdup(task->ti_a_out);
	tasklist[num_tasks].pvm_tinfo.ti_pid = task->ti_pid;
	tasklist[num_tasks].otid = task->ti_tid;
	tasklist[num_tasks].ckpt_file = strdup(ckpt_file);
	host = find_host(task->ti_host);
	if (host) {
		host->hi_speed++;	/* speed is really the load */
	}
	num_tasks++;
	return 0;
}


del_task(tid)
int tid;
{
	int		i;
	int		task_to_del;
	int		old_arch;
	struct pvmhostinfo *host;

	for (i = 0; i < num_tasks; i++) {
		if (tasklist[i].pvm_tinfo.ti_tid == tid) {
			break;
		}
	}
	if (i >= num_tasks) {
/*		fprintf(stderr, "del_task: Unable to find task t%x\n", tid); */
		return -1;
	}

	send_notification(PvmTaskExit, tid);

	task_to_del = i;
	free(tasklist[task_to_del].pvm_tinfo.ti_a_out);
	free(tasklist[task_to_del].ckpt_file);
	host = find_host(tasklist[task_to_del].pvm_tinfo.ti_host);
	if (host) {
		host->hi_speed--;	/* speed is really the load */
	}
	tasklist[task_to_del] = tasklist[num_tasks - 1];
	num_tasks--;
	return num_tasks;
}


pack_task(task_num)
int		task_num;
{
	pvm_pkint(&(tasklist[task_num].pvm_tinfo.ti_tid), 1, 1);
	pvm_pkint(&(tasklist[task_num].pvm_tinfo.ti_ptid), 1, 1);
	pvm_pkint(&(tasklist[task_num].pvm_tinfo.ti_host), 1, 1);
	pvm_pkint(&(tasklist[task_num].pvm_tinfo.ti_flag), 1, 1);
	pvm_pkstr(tasklist[task_num].pvm_tinfo.ti_a_out);
	pvm_pkint(&(tasklist[task_num].pvm_tinfo.ti_pid), 1, 1);
}


pack_task_list(where)
int		where;
{
	int		i;
	struct pvmhostinfo	*host;

	host = find_host(where);
	for (i = 0; i < num_tasks; i++) {
		if (where == 0 ||
			(host != 0 && tasklist[i].pvm_tinfo.ti_host == host->hi_tid) ||
			tasklist[i].pvm_tinfo.ti_tid == where) {
			pack_task(i);
		}
	}
}


int
get_task_tid_list(tid_list, exclude)
int	**tid_list;
int exclude;
{
	int	i;
  int cnt;

	*tid_list = (int *) malloc(num_tasks * sizeof(int));

	for (cnt = 0, i = 0; i < num_tasks; i++) {
		if ((tasklist[i].pvm_tinfo.ti_tid != exclude) &&
			((tasklist[i].pvm_tinfo.ti_flag == T_RUNNING) ||
			 (tasklist[i].pvm_tinfo.ti_flag == T_STAYING))) {
			(*tid_list)[cnt++] = tasklist[i].pvm_tinfo.ti_tid;
		}
	}

	return(cnt);
}


int
get_orig_tid_list(tid_list)
int	**tid_list;
{
	int i;
  int cnt;

	*tid_list = (int *) malloc(num_tasks * sizeof(int));

	for (cnt = 0, i = 0; i < num_tasks; i++) {
		if (((tasklist[i].pvm_tinfo.ti_flag & T_RESTARTING) == T_RESTARTING) ||
			((tasklist[i].pvm_tinfo.ti_flag & T_STAYING) == T_STAYING)) {
			(*tid_list)[cnt++] = tasklist[i].otid;
		}
    }

	return(cnt);
}

char	*
get_task_ckpt_file(tid)
int		tid;
{
	int		i;

	for(i = 0; i < num_tasks; i++) {
		if (tasklist[i].pvm_tinfo.ti_tid == tid) {
			return tasklist[i].ckpt_file;
		}
	}
	return 0;
}


set_task_ckpt_file(tid, ckpt_file)
int		tid;
char	*ckpt_file;
{
	int		i;

	for(i = 0; i < num_tasks; i++) {
		if (tasklist[i].pvm_tinfo.ti_tid == tid) {
			if (tasklist[i].ckpt_file) {
				free(tasklist[i].ckpt_file);
			}
			tasklist[i].ckpt_file = strdup(ckpt_file);
		}
	}
	return 0;
}


int	get_task_state(tid)
int		tid;
{
	int		i;

	for (i=0; i<num_tasks; i++)
		if (tasklist[i].pvm_tinfo.ti_tid == tid)
			return(tasklist[i].pvm_tinfo.ti_flag);

	return(T_RUNNING);
}

int set_task_state(tid, state)
int tid;
int state;
{
  int i;
  int ostate = T_RUNNING;

	for (i=0; i<num_tasks; i++) {
		if (tasklist[i].pvm_tinfo.ti_tid == tid) {
			ostate = tasklist[i].pvm_tinfo.ti_flag;
			tasklist[i].pvm_tinfo.ti_flag = state;
		}
	}

  return(ostate);
}

void set_task_tid(ctid, otid)
int ctid;
int otid;
{
  int i;

  for (i=0; i<num_tasks; i++)
    if ((tasklist[i].otid == otid) &&
        (((tasklist[i].pvm_tinfo.ti_flag & T_RESTARTING) == T_RESTARTING) ||
        ((tasklist[i].pvm_tinfo.ti_flag & T_STAYING) == T_STAYING)))
    {
      tasklist[i].pvm_tinfo.ti_tid = ctid;
      tasklist[i].pvm_tinfo.ti_flag = T_RUNNING;
    }
}


int get_task_ctid(otid)
int otid;
{
	int i;

	for (i=0; i<num_tasks; i++) {
		if ((tasklist[i].otid == otid)) {
			return tasklist[i].pvm_tinfo.ti_tid;
		}
	}
	return otid;	/* If it isn't in the table, assume no mapping */
}


int get_task_otid(ctid)
int ctid;
{
	int i;

	for (i=0; i<num_tasks; i++) {
		if ((tasklist[i].pvm_tinfo.ti_tid == ctid)) {
			return tasklist[i].otid;
		}
	}
	return ctid;	/* If it isn't in the table, assume no mapping */
}


set_task_binary_name(tid, exec_name)
int		tid;
char	*exec_name;
{
	int		i;
  char *ckpt_name;

	for (i = 0; i < num_tasks; i++) {
		if (tasklist[i].pvm_tinfo.ti_tid == tid) {
			if (tasklist[i].pvm_tinfo.ti_a_out) {
				free(tasklist[i].pvm_tinfo.ti_a_out);
			}
			tasklist[i].pvm_tinfo.ti_a_out = strdup(exec_name);
      if ((tasklist[i].ckpt_file != NULL) &&
          (*(tasklist[i].ckpt_file) == '\0'))
        tasklist[i].ckpt_file = strdup(gen_ckpt_name(exec_name));
			break;
		}
	}
}


restart_task(otid)
int otid;
{
	struct	pvmtaskinfo		*tinfo;
	int		dest_host;
	int		zero = 0;
	int		one = 1;
	int		arg_count = 2;
	int		new_tid;
	int		sbuf;
  int i;

	for (i=0; i<num_tasks; i++) {
		if (tasklist[i].otid == otid) {
			tinfo = &(tasklist[i].pvm_tinfo); /* freq. used so remember it */

      if((tinfo->ti_flag & T_RESTARTING) == T_RESTARTING)
      {
        /*	dest_host = select_host(PvmTaskDefault, ""); */
        dest_host = tinfo->ti_host;

				if (dest_host) {
					sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataFoo));
					pvm_pkint(&zero, 1, 1);
					pvm_pkstr(tinfo->ti_a_out);
					pvm_pkint(&zero, 1, 1);
					pvm_pkint(&one, 1, 1);	/* Always create 1 task */
					pvm_pkint(&arg_count, 1, 1);
					pvm_pkstr("-_condor_restart");
					pvm_pkstr(tasklist[i].ckpt_file);

          /* XXX Tracing and Output tid Ignored for now!!!! */
          pvm_pkint(&zero, 1, 1);
          pvm_pkint(&zero, 1, 1);
          pvm_pkint(&zero, 1, 1);
          pvm_pkint(&zero, 1, 1);
          pvm_pkint(&zero, 1, 1);

          pvm_send(dest_host | TIDPVMD, SM_EXEC);

          pvm_freebuf(pvm_setsbuf(sbuf));

					/* Synchronous, slow but easy */
					/*		pvm_recv(dest_host | TIDPVMD, SM_EXECACK);  */
					pvm_recv(-1, SM_EXECACK);
					pvm_upkint(&one, 1, 1);
					if (one != 1) {
						printf("sm_spawn: unexpected return count: %d\n", one);
					}

          pvm_upkint(&(tinfo->ti_tid), 1, 1); /* set current tid */

          tinfo->ti_host = dest_host;
        }
      }

      if ((tinfo->ti_flag & T_STAYING) == T_STAYING)
      {
        pvm_recv(-1, SM_EXECACK);
        pvm_upkint(&one, 1, 1);
        if (one != 1)
          printf("sm_spawn: unexpected return count: %d\n", one);

				pvm_upkint(&(tinfo->ti_tid), 1, 1); /* set current tid */
			}
		}
	}
}


int
unpack_string_list(str_list)
char	***str_list;
{
	int		count;
	int		i;
	char	temp_str[1000];

	pvm_upkint( &count, 1, 1 );
	*str_list = (char **) malloc( (count + 1) * sizeof(char *));
	for (i = 0 ; i < count; i++ ) {
		pvm_upkstr( temp_str );
		(*str_list)[i] = strdup( temp_str );
	}
	(*str_list)[i] = 0;
	return count;
}


free_string_list(str_list, count)
char	**str_list;
int		count;
{
	int		i;

	for (i = 0; i < count; i++) {
		free(str_list[i]);
	}
	free(str_list);
}


#if defined(NEED_STRDUP)
char *
strdup(s)
char	*s;
{
	char	*new_s;

	new_s = malloc(strlen(s) + 1);
	strcpy(new_s, s);
	return new_s;
}
#endif

char *basename(f)
char* f;
{
  char *end;

  end = f + strlen(f);

  while (f<=end && *end != '/')
    end--;

  return(end+1);
}

char *
gen_ckpt_name(a_out)
char	*a_out;
{
	static call_number = 0;
	static ckpt_file[1000];
	static char	wd[1000];

	if (call_number == 0) {
		getcwd( wd, sizeof(wd) );
		strcpy(wd, "/tmp");
	}
	sprintf((char *)ckpt_file, "%s/%s.%d", wd, basename(a_out), call_number);
	call_number++;
	return ((char *)ckpt_file);
}


char *
co_check_read_ckpt_line(FILE *ckpt)
{
	static char	buffer[1024];

	for (;;) {
		fgets(buffer, sizeof(buffer), ckpt);
		if (buffer[0] != '#') {
			break;
		}
	}
	buffer[strlen(buffer) - 1] = '\0';  /* Drop the trailing '\n' */
	return buffer;
}


void
co_check_write_ckpt_comment(FILE *ckpt, char *msg)
{
	fprintf(ckpt, "# %s\n", msg);
}


void pvm_rm_write_ckpt(ckpt_file_name, n, tids)
char	*ckpt_file_name;
int n;
int *tids;
{
	FILE	*ckpt_file;
	int		i;
  int j;

#ifdef DEBUG
  printf("Opening master ckpt file: %s\n", ckpt_file_name);
#endif

	ckpt_file = fopen(ckpt_file_name, "w");

	fprintf(ckpt_file, "%d\n", n);

  for (j=0; j<n; j++)
    for (i = 0; i < num_tasks; i++)
      if(tasklist[i].pvm_tinfo.ti_tid == tids[j])
        fprintf(ckpt_file, "%s %s %d %d\n", tasklist[i].pvm_tinfo.ti_a_out,
                tasklist[i].ckpt_file, tasklist[i].otid,
                tasklist[i].pvm_tinfo.ti_host);

  for (j=0; j<n; j++)
    for (i = 0; i < num_tasks; i++)
      if(tasklist[i].pvm_tinfo.ti_tid == tids[j])
        del_task(tasklist[i].pvm_tinfo.ti_tid);

	fclose(ckpt_file);
}


pvm_rm_read_from_ckpt(ckpt_file_name)
char	*ckpt_file_name;
{
	FILE	*ckpt_file;
	int		i;
	char	*buf;
	char	*buf2;
	struct	pvmtaskinfo	tinfo;
	int		ntasks;

  printf("Reading from master ckpt file: %s\n", ckpt_file_name);

	ckpt_file = fopen(ckpt_file_name, "r");

	buf = co_check_read_ckpt_line(ckpt_file);
	ntasks = atoi(buf);
	for (i = 0; i < ntasks; i++) {
		buf = co_check_read_ckpt_line(ckpt_file);
		strtok(buf, " ");
		tinfo.ti_a_out = buf;
		buf2 = strtok(0, " ");
		buf = strtok(0, " ");
		tinfo.ti_tid = atoi(buf);
		buf = strtok(0, " ");
		tinfo.ti_host = atoi(buf);
    tinfo.ti_flag = T_RESTARTING;
		new_task(&tinfo, buf2);
	}

  return(ntasks);
}


int
connect_to_addr( sin )
struct sockaddr_in	sin;
{
	int		sock;

	sock = socket(AF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		perror("connect_to_addr, socket");
	}

	sin.sin_family = AF_INET;

	if (connect(sock, &sin, sizeof(sin)) < 0) {
		perror("connect");
	}
	return sock;
}


/* Really, a lot like get file stream */
copy_bytes_to_file_in_background( src_fd, dest_file )
int		src_fd;
char	*dest_file;
{
	int		dest_fd;
	char	tmp_buf[ 8 * 1024 ];
	int		new_pid;
	int		bytes_read;
	int		tot_read;
	int		magic;

	dest_fd = open(dest_file, O_WRONLY | O_CREAT | O_TRUNC, 0666);
	if (dest_fd < 0) {
		return -1;
	}

	tot_read = 0;
	magic = 0;

	while (magic != CONDOR_CKPT_MAGIC) {
		bytes_read = read(src_fd, &magic, sizeof(magic));
		if (bytes_read <= 0) {
			fprintf(stderr, "NEVER FOUND CONDOR CKPT MAGIC!\n");
			return -1;
		}
		fprintf(stderr, "Read magic value 0x%x\n", magic);
	}

	tot_read +=	write(dest_fd, &magic, sizeof(magic));

	for(;;) {
		bytes_read = read(src_fd, tmp_buf, sizeof(tmp_buf));
		fprintf(stderr, "Read %d bytes\n", bytes_read);
		if (bytes_read <= 0) {
			break;
		}
		if (write(dest_fd, tmp_buf, bytes_read) != bytes_read) {
			perror("write");
		}
		tot_read += bytes_read;
	}
	close(src_fd);
	close(dest_fd);

	fprintf(stderr, "Total size of checkpoint file %s is %d\n",
			dest_file, tot_read);

	if (tot_read == 0) {
		fprintf(stderr, "WARNING!!!!!!!!!!!!! checkpoint is empty\n");
		pvm_halt();
	}
#if 0
	if (new_pid = fork()) {
		/* The Parent */
		close(src_fd);
		close(dest_fd);
		return new_pid;
	} else {
		/* The Child */
		for(;;) {
			bytes_read = read(src_fd, tmp_buf, sizeof(tmp_buf));
			fprintf(stderr, "Read %d bytes\n", bytes_read);
			if (bytes_read < 0) {
				break;
			}
			write(dest_fd, tmp_buf, bytes_read);
		}
		close(src_fd);
		close(dest_fd);
	}
#endif
}


char *
sin_to_string(struct sockaddr_in *sin)
{
	int		i;
	static	char	buf[24];
	char	tmp_buf[10];
	char	*cur_byte;
	unsigned char	this_byte;

	buf[0] = '<';
	buf[1] = '\0';
	cur_byte = (char *) &(sin->sin_addr);
	for (i = 0; i < sizeof(sin->sin_addr); i++) {
		this_byte = (unsigned char) *cur_byte;
		sprintf(tmp_buf, "%u.", this_byte);
		cur_byte++;
		strcat(buf, tmp_buf);
	}
	buf[strlen(buf) - 1] = ':';
	sprintf(tmp_buf, "%d>", ntohs(sin->sin_port));
	strcat(buf, tmp_buf);
	return buf;
}
