/* @TITLE "genPORTION: generators for portioned patterns" */
/* genPORTION.c: Pattern for testing RAPIDFILEs. This pattern has each process
 * reading sequential portions. In random patterns, the length and skip are
 * random (lengths are distributed EXP(PortionLength)). In the S patterns,
 * the portions have fixed length and skip. In local patterns,
 * the processors run until EACH processor reads 'chunks' chunks, whereas in
 * global patterns they cooperate until all processors have read 'chunks'
 * chunks.
 *
 *  9/05/89 - added BAD pattern note
 *  5/22/89 - added SEG pattern
 */

static char rcsid[] = "$Id: genPORTION.c,v 7.1 91/05/09 19:33:24 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "format.h"
#include "refs.h"
#include "pat-intern.h"
#include "gen.h"

static void genForProc();
static boolean RandomPortions;
static float InitialFactor;	/* compute first offset (non-random only) */

/* @SUBTITLE "genLP[RS], genSEG: (Sequential) main control: local patterns" */
void
genLPR()
{
    RandomPortions = TRUE;	/* random portions */
    Share(&RandomPortions);
    generate(FALSE, TRUE, genForProc); /* local pattern, has portions */
}

/* Note that PortionSkip for LPS should be P*x, where P=nprocs and
 * x is the smallest integer that is both
 *   greater than or equal to PortionLength, and
 *   relatively prime to the number of disks in use.
 * This ensures that simultaneous portions start on different disks and
 * have no overlap between portions. See notebook for 5/8/89.
 * For 20 procs and 20 disks, with PortionLength=10, x=11 so PortionSkip=220.
 */
/* BAD is an LPS pattern that starts all portions on the same disk, so
 * that (if the processes stay perfectly synched) all accesses occur
 * simultaneously to the same disk. This is an example of a very unfortunate 
 * disk access pattern. 
 * Note that PortionSkip for BAD should be P*x, where P=nprocs and
 * x is the smallest integer that is both
 *   greater than or equal to PortionLength, and
 *   a multiple of the number of disks in use.
 * This ensures that simultaneous portions start on the same disks and
 * have no overlap between portions. See notebook for 5/8/89, and above genLPS.
 * For 20 procs and 20 disks, with PortionLength=10, x=20 so PortionSkip=200.
 */
void
genLPS()
{
    InitialFactor = (float) PortionSkip / nprocs;
    Share(&InitialFactor);
    RandomPortions = FALSE;
    Share(&RandomPortions);
    filesize = (unsigned) (chunks / PortionLength * PortionSkip) * chunksize;
    Share(&filesize);
    printf("File size is %u bytes\n", filesize);
    generate(FALSE, TRUE, genForProc); /* local pattern, has portions */
}

/* like LPS, but one portion of size 'chunks' */
void
genSEG()
{
    PortionLength = chunks;
    Share(&PortionLength);
    PortionSkip = 0;
    Share(&PortionSkip);
    InitialFactor = PortionLength;	/* contiguous segments */
    Share(&InitialFactor);
    RandomPortions = FALSE;
    Share(&RandomPortions);
    filesize = (unsigned) chunks * nprocs * chunksize;
    Share(&filesize);
    printf("File size is %u bytes\n", filesize);
    generate(FALSE, TRUE, genForProc); /* local pattern, has portions */
}

/* @SUBTITLE "genGP[RS]: (Sequential) main control for global patterns" */

void
genGPR()
{
    RandomPortions = TRUE;	/* random portions */
    generate(TRUE, TRUE, genForProc); /* global pattern, has portions */
}

void
genGPS()
{
    InitialFactor = 0;
    RandomPortions = FALSE;
    filesize = (unsigned) chunks/PortionLength * PortionSkip * chunksize;
    Share(&filesize);
    printf("File size is %u bytes\n", filesize);
    generate(TRUE, TRUE, genForProc); /* global pattern, has portions */
}

/* @SUBTITLE "genForProc: Actual work of generating a string" */
/* (maybe Parallel) */

static void
genForProc(index)
	int index;
{
    unsigned int pos;		/* byte position in the file */
    int length;			/* length of portion (in chunks) */
    int nportions = 0;		/* est. number of portions */
    int c = 0, p = 0;		/* chunk, portion number */
    int chunksleft = chunks;	/* number of chunks still to assign */
    extern double exponential();

    /* choose first portion, estimate nportions */
    if (RandomPortions) {
	   /* choose a random starting place and length */
	   pos = (random() % (filesize/chunksize)) * (unsigned) chunksize;
	   length = exponential(1.0 / (double)PortionLength) + 1;
	   nportions = chunks;	/* worst-case is all 1-chunk portions */
    } else {
	   /* choose starting chunk based on node number */
	   pos = index * InitialFactor * chunksize;
	   length = PortionLength;
	   nportions = chunks / PortionLength + 1;
    }

    /* Allocate portions array; will be too big for random patterns */
    my_refs->portions =
	 (unsigned int *) AllocLocal(nportions * sizeof(unsigned int));

    while (chunksleft > 0) {
	   length = min(length, chunksleft);
	   length = min(length, (filesize - pos) / chunksize);
	   chunksleft -= length;

	   /* write the portion into reference string */
	   do {
		  AddRef(c++, pos, (unsigned) chunksize, writes);
		  pos += chunksize;
	   } while(--length > 0);

	   /* record last index of portion in portion string */
	   my_refs->portions[p++] = c-1;

	   /* choose next portion */
	   if (RandomPortions) {
		  /* choose a random starting place and length */
		  pos = (random() % (filesize/chunksize)) * (unsigned) chunksize;
		  length = exponential(1.0 / (double)PortionLength) + 1;
	   } else {
		  /* use fixed PortionSkip and PortionLength */
		  pos =
		    (pos + filesize + (unsigned) PortionSkip*chunksize
			- (unsigned) PortionLength*chunksize) % filesize;
		  length = PortionLength;
	   }
    }

    /* Put actual nportions into structure */
    my_refs->nportions = p;
    my_refs->portion_limit = RandomPortions; /*  limit if random */

    /* Now add computation time to each chunk */
    genComputation();
}
