/* @TITLE "copycomp - generate comp pattern from plain pattern" */
/*
 * copycomp.c - one-time program; to copy the computation for one
 * pattern into another pattern.
 *
 * In the local case, the work may be done in parallel (using any available
 * processors); in the global case,
 * the work is done only on the main processor and the reference string
 * is shared. The file itself may be written in parallel.
 *
 * David Kotz 5/89
 */

static char rcsid[] = "$Id: copycomp.c,v 7.1 91/05/09 19:33:24 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "format.h"
#include "refs.h"
#include "pat-intern.h"

/* @SUBTITLE "Declarations" */

/* GLOBAL DATA */
char filename[P_FILENAMELEN+1];	/* filename */
int computation;	/* computation time per chunk */
boolean computation_fixed;	/* computation is fixed amount? */

int **comp;		/* array of: comptime for each pattern */

/* LOCAL FUNCTIONS */
static void StoreComp();
static void CopyComp();
static void ReadInfo();
static void ShareInfo();

/* @SUBTITLE "main program" */
main(argc, argv)
	int argc;
	char **argv;
{
    P_HEAD *head;			/* header for file */

    InitializeUs();			/* so we can use parallelism, if there */

    ReadInfo(argc, argv);	/* read pattern parameters */

    ShareInfo();			/* share parameters */

    head = OpenPattern(filename, "r"); /* open the file */
    if (head == (P_HEAD *)NULL)
	 exit(1);

    comp = (int **) UsAlloc(head->nprocs * sizeof(int *));
    Share(&comp);

    /* We use GenOnI so we can use any processors we have */
    GenOnI(StoreComp, head->nprocs);

    computation = head->computation;
    computation_fixed = head->computation_fixed;

    /* close file everywhere */
    ClosePattern();

    ReadInfo(0,0);			/* read new pattern parameters */

    ShareInfo();			/* share parameters */

    head = OpenPattern(filename, "r+"); /* open the file */
    if (head == (P_HEAD *)NULL)
	 exit(1);

    /* We use GenOnI so we can use any processors we have */
    GenOnI(CopyComp, head->nprocs);

    head->computation = computation;
    head->computation_fixed = computation_fixed;
    RewriteHead(head);

    ClosePattern();

    printf("File %s complete\n", filename);
}

/* @SUBTITLE "StoreComp: Remember computation for a pattern" */
static void
StoreComp(dummy, p)
	int dummy; 
	int p;				/* which pattern to do? */
{
    int *comptimes;
    int i;

    ReadPattern(p);			/* read pattern p */
    comptimes =
	 (int *)UsAlloc(sizeof(int) * my_refs->nchunks);
    comp[p] = comptimes;

    for (i = 0; i < my_refs->nchunks; i++) 
	 *comptimes++ = my_refs->chunks[i].comptime;
}

/* @SUBTITLE "CopyComp: Add computation to a pattern" */
static void
CopyComp(dummy, p)
	int dummy; 
	int p;				/* which pattern to do? */
{
    int *comptimes;
    int i;

    ReadPattern(p);			/* read pattern p */
    comptimes = comp[p];
    for (i = 0; i < my_refs->nchunks; i++) 
	 my_refs->chunks[i].comptime = *comptimes++;
    RewritePattern();		/* write it back */
}

/* @SUBTITLE "ReadInfo: Read in the various pattern parameters" */

#define STYLE_LEN 10

static void
ReadInfo(argc, argv)
	int argc;
	char **argv;
{
    char style[STYLE_LEN+1];
    char line[11];

    if (--argc > 0) {
	   strncpy(filename, *++argv, P_FILENAMELEN);
	   printf("Filename is '%s'\n", filename);
    } else {
	   printf("Name of pattern FILE (%d chars): ", P_FILENAMELEN);
	   ReadString(filename, "Filename", P_FILENAMELEN);
    }	   
    if (*filename == '\0') {
	   printf("Must specify filename\n");
	   exit(1);
    }
}

/* @SUBTITLE "ShareInfo: Share the pattern parameters" */
static void
ShareInfo()
{
    Share(&computation);
    Share(&computation_fixed);
}

