/* @TITLE "replace-ws.c - replacement algorithm - working set" */
/* replace-ws.c:	Support for frame replacement algorithm. 
 *             PARALLEL WORKING SET algorithm. 
 *			Includes the following: 
 * 
 *			RT_InitReplaceInode_ws() 
 *			RT_InitReplaceRPD_ws() 
 * and the local functions, called through pointers:
 *			FindFrame() 
 *			DoneReplaceInode() 
 *			DoneReplaceRPD() 
 *			NotifyReplaceA() 
 *			NotifyReplaceB() 
 *			Replaceable() 
 * 
 * The protocols used here are carefully designed to work together to 
 * provide exactly the right mutual exclusion properties and are very 
 * heavily based on the semantics of the problem. Be very careful 
 * if modifying any of these routines. 
 */

static char rcsid[] = "$Id: replace-ws.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

/* DEFINES */

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "replace.h"

/* #define FRAME_DEBUG_FLAG */

/* From globalws.c */
extern void AddToGlobalWS();
extern void RemoveFromGlobalWS();
extern boolean FindFrame_ws();
extern FFDATA *InitGWS();
extern void ReplaceableGWS();
extern void DoneGWS();

/* here */
void RT_PrintWS();

/* Entries in the working-set queue */
typedef struct RT_wsq_struct WSQ_ENTRY;
struct RT_wsq_struct {
    int sector;			/* sector number */
    WSQ_ENTRY *next;		/* pointer to next entry */
    WSQ_ENTRY *prev;		/* pointer to previous entry */
};

struct wsdata {			/* for working-set */
    short wss;				/* local working set size */
    short wsq_used;			/* how many entries used in queue */
    WSQ_ENTRY *wsq_head;		/* head of queue */
    FFDATA *ff;			/* findframe data */
};

static boolean FindFrame();
static void DoneReplaceInode();
static void DoneReplaceRPD();
static boolean NotifyReplaceA();
static void NotifyReplaceB();
static void Replaceable();
static void Dump();

/* @SUBTITLE "RT_InitReplaceInode_ws: initialize inode" */
void
RT_InitReplaceInode_ws(inode_ptr)
	RAPID_INODE *inode_ptr;
{
    /* make data structures needed by findframe */
    inode_ptr->replace.private = (ANYPTR) InitGWS(inode_ptr->num_frames);
    
    inode_ptr->replace.doneinode = DoneReplaceInode;
}

/* @SUBTITLE "RT_InitReplaceRPD_ws: initialize descriptor" */
void
RT_InitReplaceRPD_ws(rpd, wss)
	RAPIDFILE *rpd;
	int wss;
{
    struct wsdata *mydata;

    mydata = (struct wsdata *)AllocLocal(sizeof(struct wsdata));

    mydata->wss = wss;
    mydata->wsq_head = NULL;
    mydata->wsq_used = 0;
    mydata->ff = (FFDATA *)(rpd->inode_ptr->replace.private);

    rpd->replace.private = (ANYPTR)mydata;

    rpd->replace.donerpd = DoneReplaceRPD;
    rpd->replace.notifyA = NotifyReplaceA;
    rpd->replace.notifyB = NotifyReplaceB;
    rpd->replace.findframe = FindFrame;
    rpd->replace.replaceable = Replaceable;
    rpd->replace.dump = Dump;
}

/* @SUBTITLE "FindFrame: find a frame for a sector" */
static boolean
FindFrame(rpd, sector, sme, prefetching)
	RAPIDFILE *rpd;
	int sector;			/* the sector we want */
	SECTOR_MAP_ENTRY *sme;	/* the entry for this sector */
	boolean prefetching;	/* are we doing this for prefetch */
{
    struct wsdata *mydata = (struct wsdata *)(rpd->replace.private);
    
    return(FindFrame_ws(rpd, sector, sme, mydata->ff, prefetching));
}

/* @SUBTITLE "Dump: dump debugging output for private data" */
static void
Dump(rpd)
	RAPIDFILE *rpd;
{
    struct wsdata *wsdata = (struct wsdata *)(rpd->replace.private);

    printf("  Working-set replacement private data:\n");
    if (wsdata != NULL) {
	   printf("   wss = %d\n", wsdata->wss);
	   printf("   Queue: \n");
	   RT_PrintWS(rpd);
	   DumpGWS(wsdata->ff);
    } else
	 printf("   Null wsdata\n");
}

/* @SUBTITLE "DoneReplaceInode: shut down replacement algorithm" */
static void
DoneReplaceInode(inode_ptr)
    RAPID_INODE *inode_ptr;
{
    DoneGWS(inode_ptr->replace.private);	/* findframe info */
    inode_ptr->replace.private = NULL;
}

/* @SUBTITLE "DoneReplaceRPD: shut down replacement algorithm" */
static void
DoneReplaceRPD(rpd)
	RAPIDFILE *rpd;
{
    struct wsdata *mydata = (struct wsdata *)(rpd->replace.private);
    WSQ_ENTRY *w, *next;
    
    /* Take our working set out of global working set */
    for (w = mydata->wsq_head; w != NULL; w = next) {
	   RemoveFromGlobalWS(rpd, w->sector, mydata->ff);
	   next = w->next;
	   free(w);
    }

    UsFree(mydata);

    rpd->replace.private = NULL;
}

/* @SUBTITLE "RT_PrintWS: print the WS for a file for debugging" */
void 
RT_PrintWS(rpd)
	RAPIDFILE *rpd;
{
    struct wsdata *mydata = (struct wsdata *)(rpd->replace.private);
    WSQ_ENTRY *w;
    SECTOR_MAP_ENTRY *sme;
    FRAME_ENTRY *frame;
    
    printf("   Proc %02d: used=%d\n", Vnode, mydata->wsq_used);
    for (w = mydata->wsq_head; w != NULL; w = w->next) {
	   sme = RT_smeForSector(rpd, w->sector);
	   frame = rpd->frame_table[sme->sector_frame];
	   printf("   [sector %4d in frame %2d count %d status %02o]\n",
			w->sector, sme->sector_frame,
			frame->ws_count, frame->frame_status);
    }
    fflush(stdout);
}

/* @SUBTITLE "NotifyReplaceA: first part of replacement recording" */
static boolean
NotifyReplaceA(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct wsdata *mydata = (struct wsdata *)(rpd->replace.private);
    WSQ_ENTRY *w, *wp;
    boolean callB;
    
    /* For efficiency, a few quick cases */
    if (mydata->wsq_head == NULL) { /* The first call to us */
	   w = (WSQ_ENTRY *) AllocPrivate(sizeof(WSQ_ENTRY));
	   if (w == NULL)
		printf("RT_NotifyReplace_ws: Cannot get memory for working-set\n");
	   else {
		  mydata->wsq_head = w;
		  mydata->wsq_used++;
		  w->next = w->prev = NULL;
		  w->sector = sector;
		  return(TRUE);
	   }
	   return(FALSE);
    } else if (sector == mydata->wsq_head->sector)
	 return(FALSE);		/* NO CHANGE IN QUEUE: sector is first */
    
    /* Otherwise, search the rest of list for used sector */
    for (wp = mydata->wsq_head, w = wp->next;
	    w != NULL;
	    wp = w, w = w->next) {
	   if (sector == w->sector) {	/* found it! */
		  /* Move to front (below) */
		  wp->next = w->next; /* cut it out */
		  if (w->next != NULL)
		    w->next->prev = wp;
		  break;
	   }
    }
    
    /* If not found, find an entry to put it in */
    if (w == NULL) {
	   /* Still free entries around? no removal necessary */
	   if (mydata->wsq_used < mydata->wss) {
		  /* grab a free element */
		  w = (WSQ_ENTRY *) AllocPrivate(sizeof(WSQ_ENTRY));
		  mydata->wsq_used++;
	   }
	   /* Queue is full-size or no memory to extend: steal from tail (wp) */
	   if (w == NULL) {
		  w = wp;
    	   	  if (wp->prev != NULL) /* it would be NULL if wss = 1 */
		    wp->prev->next = NULL;	/* cut it out */
		  RemoveFromGlobalWS(rpd, wp->sector, mydata->ff);
	   }
	   /* needs to be added to global WS once there's a frame */
	   callB = TRUE;
	   w->sector = sector;
    } else
	 callB = FALSE;
    
    /* Add w to the queue */
    if (w != mydata->wsq_head) {	/* w not already the head? */
	   /* tack w on the front */
	   mydata->wsq_head->prev = w;
	   w->next = mydata->wsq_head;
	   w->prev = NULL;
	   mydata->wsq_head = w;
    }
    
    return(callB);
}

/* @SUBTITLE "NotifyReplaceB: second part of replacement recording" */
static void
NotifyReplaceB(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    AddToGlobalWS(rpd, sector);
}

/* @SUBTITLE "Replaceable: the given sector is now replaceable" */
/* as long as no one is using it */

static void
Replaceable(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct wsdata *mydata = (struct wsdata *)(rpd->replace.private);

    ReplaceableGWS(rpd, sector, mydata->ff);
}

