/* @TITLE "queue.c - locked queues" */
/* queue.c - simple queue routines
 *
 * We do this with a very simple circular queue with monolithic lock.
 *
 * David Kotz 
 */

static char rcsid[] = "$Id: queue.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $";

#include <stdio.h>
#include <usdfk.h>
#include "queue.h"
#include "error.h"

/* QH is defined already, by queue.h, as follows:
 * typedef struct queue_s *QH;
 * 
 * head==tail implies empty queue 
 */
struct queue_s {
    short size;			/* max number of entries */
    short lock;			/* lock on queue */
    short head;			/* next place to add to back */
    short tail;			/* front of queue */
    TICS minvalue;			/* we keep a value for caller */
    int queue[1];			/* the entries [0..size-1] */
};

#define MAX_QUEUE_ENTRIES (1 << 14 - 1)
#define MAXINT ((TICS)((unsigned int)-1)) /* maximum unsigned integer */

/* This must stay on now, since I've done lots of tests with it on. Sigh. */
#define DEBUG 

typedef boolean (*boolproc)();

/* @SUBTITLE "Make_queue: make a new queue" */
/* Create a queue that will fit the specified number of entries.
 * We allocate room for one extra item, due to the implementation.
 */
QH
Make_queue(length)
	int length;			/* the number that need to fit */
{
    QH new;
    
    if (length <= 0 || length > MAX_QUEUE_ENTRIES)
	 return(NULL);
    
    /* Allocate an object bigger than just the header */
    /* The queue[] array extends past end of defined header */
    new = (QH) AllocGlobal(sizeof(struct queue_s) + length * sizeof(int));

    new->size = length + 1; /* since header contains one entry */
    new->lock = 0;
    new->head = 0;
    new->tail = 0;
    new->minvalue = MAXINT;
    TouchBlock(new->queue, new->size * sizeof(int), TRUE);

    return(new);
}

/* @SUBTITLE "TouchQueue: touch memory in a queue" */
/* Touch all memory involved with a queue */
void
TouchQueue(q)
	QH q;
{
#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif

    TouchBlock(q->queue, q->size * sizeof(int), FALSE);
}

/* @SUBTITLE "InQueue: return fullness of queue" */
/* Returns the number of items currently in the queue. */
int
InQueue(q)
	QH q;
{
#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif
    return((q->head + q->size - q->tail) % q->size);
}


/* @SUBTITLE "EmptyQueue: TRUE if queue is empty" */
/* faster than InQueue(q) == 0 */
boolean					/* true if empty */
EmptyQueue(q)
	QH q;
{
#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif
    return(q->head == q->tail);
}

/* @SUBTITLE "ValQueue: return minvalue of queue" */
TICS
ValQueue(q)
	QH q;
{
#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif
    return(q->minvalue);
}

/* @SUBTITLE "Enqueue: add item to queue" */
/* Insert an entry into the queue.
 * Also take a value for updating minvalue. 
 */
boolean					/* FALSE if full */
Enqueue(q, datum, value)
	QH q;
	int datum;
	TICS value;			/* for comparison with minvalue */
{
    int n;

#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif

    LOCK(&(q->lock),10);
    n = InQueue(q);
    if (n+1 == q->size) {
	   UNLOCK(&(q->lock));
	   return(FALSE);		/* queue is full */
    }
    q->queue[q->head] = datum;
    q->head = (q->head + 1) % q->size;
    if (value < q->minvalue)
	 q->minvalue = value;

    UNLOCK(&(q->lock));

    return(TRUE);
}

/* @SUBTITLE "ForceEnqueue: add item to queue, forced" */
/* Insert an entry into the queue.
 * If the queue is full, dequeue one item and then enqueue.
 * If an item is dequeued, TRUE is returned, and the item is available. 
 * Otherwise FALSE is returned. The whole operation is atomic.
 */
boolean					/* TRUE if was full and item is returned */
ForceEnqueue(q, datum, removed)
	QH q;
	int datum;			/* the item to add */
	int *removed;			/* the item that was removed (output) */
{
    int n;
    boolean status;

#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif

    LOCK(&(q->lock),10);
    n = InQueue(q);
    if (n+1 == q->size) {
	   /* queue is full; dequeue one item */
	   status = TRUE;
	   if (removed != NULL) 
		*removed = q->queue[q->tail];
	   q->tail = (q->tail + 1) % q->size;
    } else
	 status = FALSE;

    /* now enqueue the item */
    q->queue[q->head] = datum;
    q->head = (q->head + 1) % q->size;
    q->minvalue = 0;		/* NOTE this is ignored here */

    UNLOCK(&(q->lock));
    return(status);
}

/* @SUBTITLE "Dequeue: remove an entry from the queue" */
/*  Remove an entry from the queue. */
boolean					/* FALSE if empty */
Dequeue(q, datum)
	QH q;
	int *datum;
{
#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif

    LOCK(&(q->lock),10);
    if (EmptyQueue(q)) {
	   UNLOCK(&(q->lock));
	   return(FALSE);
    } else {
	   *datum = q->queue[q->tail];
	   q->tail = (q->tail + 1) % q->size;
    }
    UNLOCK(&(q->lock));
    
    return(TRUE);
}

/* @SUBTITLE "Peekqueue: peek an element from the queue" */
/* nondestructively return the ith element of the queue */
/* the first (next to be dequeued) element is i=0 */
boolean					/* FALSE if i out of range */
Peekqueue(q, i, x)
	QH q;
	int i;
	int *x;
{
#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif

    LOCK(&(q->lock),10);

    if (i < 0 || i >= InQueue(q)) {
	   UNLOCK(&(q->lock));
	   return (FALSE);
    }

    *x = q->queue[(q->tail+i) % q->size];
    UNLOCK(&(q->lock));

    return(TRUE);
}

/* @SUBTITLE "ScanQueue: scan queue and compress" */
/* We call the given function on each element of the queue.
 * If the function returns TRUE, we keep in queue, else we
 * remove from queue. Queue is compressed to keep it contiguous.
 * All done with queue locked. We also compute the minvalue
 * from the values returned by TRUE function returns. 
 */
void
ScanQueue(q, keep, arg1, arg2)
	QH q;				/* the queue */
	boolproc keep;			/* the decision function */
	ANYPTR arg1;			/* 1st argument to keep */
	ANYPTR arg2;			/* 2nd argument to keep */
{
    int n;				/* number of entries */
    int p;				/* pos in queue */
    int w;				/* next pos to write into */
    unsigned long minvalue = MAXINT; /* new minvalue */
    unsigned long value;		/* one of the values */

#ifdef DEBUG
    if (q == NULL)
	 error("Invalid Queue handle", ERR_DUMP);
#endif

    LOCK(&(q->lock),10);

    /* scan the queue */
    for (p = w = q->tail, n = InQueue(q); n > 0; n--, p = (p+1) % q->size) {
	   if ((*keep)(arg1, arg2, q->queue[p], &value)) {
		  /* keep this one */
		  q->queue[w] = q->queue[p];
		  w = (w+1) % q->size;
		  if (value < minvalue)
		    minvalue = value;
	   }
    }

    q->minvalue = minvalue;	/* reset minvalue */
    q->head = w;			/* compress queue */

    UNLOCK(&(q->lock));
}

/* @SUBTITLE "***  TEST PROGRAM  ***" */
/* This allows this module to be tested alone 
 * compile with -DTEST -lus to get this 
 * It is NOT a parallel test 
 */

#ifdef TEST

main()
{
    QH q;
    int i;
    int n;
    boolean odd();

    InitializeUs();
    
    q = Make_queue(10);
    if (q == NULL)
	 error("Make_queue failed", ERR_DIE);
    if (!EmptyQueue())
	 error("Don't have 0 at start", ERR_DUMP);
    
    /* put on one item and remove */
    Enqueue(q, 999, 0);
    if (InQueue(q) != 1)
	 error("Don't have 1 after one enqueue", ERR_DUMP);
    if (Dequeue(q, &n)) {
	   if (n != 999)
		error("Dequeue wrong value", ERR_DUMP);
    } else
	 error("Can't dequeue only value", ERR_DUMP);

    /* fill and empty */
    for (i=0; i < 5; i++) {
	   Enqueue(q, 900+i, 0);
	   if (InQueue(q) != i+1)
		error("Don't have right count in Enq loop", ERR_DUMP);
    }
    for (i=0; i < 5; i++) {
	   if (Dequeue(q, &n)) {
		  if (n != 900+i)
		    error("Dequeue wrong value in loop", ERR_DUMP);
	   } else
		error("can't dequeue in loop", ERR_DUMP);
	   if (InQueue(q) != 5-i-1)
		error("Don't have right count in DQ loop", ERR_DUMP);
    }
    
    /* test empty handling */
    if (Dequeue(q, &n))
	 error("Queue should be empty\n", ERR_DUMP);

    /* fill and scan */
    for (i=0; i < 5; i++) {
	   Enqueue(q, 900+i, 0);
	   if (InQueue(q) != i+1)
		error("Don't have right count in Enq loop", ERR_DUMP);
    }
    ScanQueue(q, odd, 0, 0);
    if (InQueue(q) != 2)
	 error("Count wrong after scan", ERR_DUMP);

    /* take rest off */
    for (i=0; i < 2; i++) {
	   if (Dequeue(q, &n)) {
		  if (n != 900+2*i+1)
		    error("Dequeue wrong value in cleanup loop", ERR_DUMP);
	   } else
		error("can't dequeue in cleanup loop", ERR_DUMP);
	   if (InQueue(q) != 2-i-1)
		error("Don't have right count in cleanup loop", ERR_DUMP);
    }
    
    /* over-full test */
    for (i=0; i < 10; i++) {
	   Enqueue(q, 900+i, 0);
	   if (InQueue(q) != i+1)
		error("Don't have right count in Filling loop", ERR_DUMP);
    }
    
    if (Enqueue(q, 999, 0) != FALSE)
	 error("Queue overfilled and no error caught\n", ERR_DUMP);

    printf("Silence is golden.\n");
}

boolean
odd(dummy1, dummy2, n, dummy3)
	int n;
{
    return (n%2);
}
#endif TEST

