/* @TITLE "prefetch.c - prefetching sectors" */
/* prefetch.c: Routines for prefetching pages. Uses coroutines.
 *
 * FUNCTIONS:
 *      RT_InitPrefetch
 *      RT_WhatPrefetch
 *      RT_StartPrefetch
 *      RT_CancelPrefetch
 *  	   RT_Prefetch
 *      RT_PrefetchMistake
 *
 * Local:
 *      PrefetchLoop
 *      TryPrefetchSector
 *
 * Prediction Macros (from prefetch.h):
 *      PrefetchWorkAvailable
 *      GetPrefetchWork
 * these call functions given by pointers in the rpd->prefetch structure.
 *
 * David Kotz  February 1988
 * May 1989 - Added FOR_OTHERS prefetching
 * September 1989 - Added OWN_BUFFER prefetching
 * November 1989 - added prediction algorithm switching
 * February 1990 - Added WANTED prefetching
 * March 1990 - added global predictors
 */

static char rcsid[] = "$Id: prefetch.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "replace.h"
#include "stats.h"
#include "rapidelog.h"
#include "others.h"
#include "own.h"
#include "wanted.h"
#include "predict-debug.h"

/* Initialization for various prediction algorithms */
extern void InitPredict_EXACT();
/* local */
extern void InitPredict_OBL();
extern void InitPredict_IBL();
extern void InitPredict_PORT();
extern void InitPredict_ADAPT();
extern void InitPredict_IOBL();
extern void InitPredict_IPORT();
extern void InitPredict_IOPORT();
/* global */
extern void InitPredict_GAPS();
extern void InitPredict_RGAPS();
extern void InitPredict_GW();
extern void InitPredict_SWITCH();

/* LOCAL VARIABLES */
#define STACK_SIZE (8192) /* stack size for coroutines */
  
static void PrefetchLoop();
static boolean TryPrefetchSector();

/* #define PREFETCH_DEBUG_FLAG 
 * a smaller set of debugging: 
 * #define DEBUG_FLAG 
 * #define FRAME_DEBUG_FLAG 
 */
#ifdef DEBUG_FLAG
static int ListUsed = -1;	/* so we can see it in dbx */
#endif

/* @SUBTITLE "RT_InitPrefetch: Initialize for future prefetching" */
void
RT_InitPrefetch()
{
    initialize_coroutines();
}

/* @SUBTITLE "RT_WhatPrefetch: Print prefetch algorithm name" */
void
RT_WhatPrefetch(rpd)
	RAPIDFILE *rpd;
{
    if (rpd->prefetch.cid != 0) {
#ifdef FOR_OTHERS
# ifdef WANTED
	   printf("\nUsing WANTED buffering, FOR_OTHERS prefetching");
# else
	   printf("\nUsing FOR_OTHERS Prefetching");
# endif
#else
# ifdef OWN_BUFFER
	   printf("\nUsing OWN_BUFFER Prefetching");
# else
#  ifdef WANTED
	   printf("\nUsing WANTED buffering, Normal prefetching");
#  else
	   printf("\nUsing Normal Prefetching");
#  endif
# endif 
#endif 
	   switch(rpd->prefetch.alg) {
		  case PFA_NONE: printf(" - no prediction\n"); break;
		  case PFA_EXACT: printf(" - exact (original) algorithm\n"); break;
		  case PFA_OBL: printf(" - OBL prediction\n"); break;
		  case PFA_IBL: printf(" - IBL prediction\n"); break;
		  case PFA_PORT: printf(" - PORT prediction\n"); break;
		  case PFA_ADAPT: printf(" - ADAPT prediction\n"); break;
		  case PFA_IOBL: printf(" - IOBL prediction\n"); break;
		  case PFA_IPORT: printf(" - IPORT prediction\n"); break;
		  case PFA_IOPORT: printf(" - IOPORT prediction\n"); break;
		  case PFA_GAPS: printf(" - GAPS prediction\n"); break;
		  case PFA_RGAPS: printf(" - RGAPS prediction\n"); break;
		  case PFA_GW: printf(" - GW prediction\n"); break;
		  case PFA_SWITCH: printf(" - SWITCH prediction\n"); break;
		  default: printf(" - **UNKNOWN** prediction\n"); break;
	   }
    } else {
	   printf("\nNo prefetching\n");
    }
}

/* @SUBTITLE "RT_StartPrefetch: Set up for prefetching on this file" */
void
RT_StartPrefetch(rpd, alg, parm)
	RAPIDFILE *rpd;
	short alg;
	int parm;
{
    rpd->prefetch.alg = PFA_NONE;
    rpd->prefetch.available = FALSE;
    rpd->prefetch.checkavail = NULL;
    rpd->prefetch.getwork = NULL;
    rpd->prefetch.notify = NULL;
    rpd->prefetch.latenotify = NULL;
    rpd->prefetch.donerpd = NULL;
    rpd->prefetch.private = NULL;

    /* Need to start prefetching coroutine for this rpd? */
    if (rpd->prefetch.cid == 0) {
	   rpd->prefetch.cid = create_coroutine(STACK_SIZE, PrefetchLoop, rpd);
	   if (rpd->prefetch.cid == 0) {
		  printf("Cannot start coroutine, no prefetching will be done.\n");
		  return;
	   } else {
#ifdef PREFETCH_DEBUG_FLAG
		  printf("Started coroutine with id %d\n", rpd->prefetch.cid);
#endif
	   }

	   rpd->prefetch.alg = alg;
	   if (!InitPredict(rpd, alg, parm))
		RT_CancelPrefetch(rpd);
    }
}

/* @SUBTITLE "InitPredict: Initialize the predictor" */
boolean
InitPredict(rpd, alg, parm)
	RAPIDFILE *rpd;
	short alg;
	int parm;
{
    switch(alg) {
	   case PFA_NONE: {
		  return(FALSE);
	   }
	   case PFA_EXACT: {
		  InitPredict_EXACT(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_OBL: {
		  InitPredict_OBL(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_IBL: {
		  InitPredict_IBL(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_PORT: {
		  InitPredict_PORT(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_ADAPT: {
		  InitPredict_ADAPT(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_IOBL: {
		  InitPredict_IOBL(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_IPORT: {
		  InitPredict_IPORT(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_IOPORT: {
		  InitPredict_IOPORT(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_GAPS: {
		  InitPredict_GAPS(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_RGAPS: {
		  InitPredict_RGAPS(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_GW: {
		  InitPredict_GW(rpd, parm);
		  return(TRUE);
	   }
	   case PFA_SWITCH: {
		  InitPredict_SWITCH(rpd, parm);
		  return(TRUE);
	   }
	   default: {
		  printf("Unknown prediction algorithm %d in RT_StartPrefetch()\n",
			    alg);
		  return(FALSE);
	   }
    }
}

/* @SUBTITLE "RT_Prefetch: try to do some prefetching" */
/* We do some prefetching: put one of the prefetching coroutines into action.
 */
void
RT_Prefetch()
{
    TICS start = rtc;
    
#ifdef PREFETCH_DEBUG_FLAG
    printf("Starting prefetching...\n");
#endif
    
    schedule_coroutine();
    
    my_stats->prefetchtime += rtc - start;
    my_stats->prefetchcount++;
    
#ifdef PREFETCH_DEBUG_FLAG
    printf("Ending prefetching...\n");
#endif
}

/* @SUBTITLE "RT_CancelPrefetch: Stop prefetching for file" */
/* Turn off prefetching for this rpd - delete the coroutine. */
void
RT_CancelPrefetch(rpd)
	RAPIDFILE *rpd;
{
    if (rpd->prefetch.cid != 0) {
#ifdef PREFETCH_DEBUG_FLAG
	   printf("Cancelling prefetching id %d\n", rpd->prefetch.cid);
#endif
	   delete_coroutine(rpd->prefetch.cid);
	   rpd->prefetch.cid = 0;
    }

    DonePredict(rpd);

    rpd->prefetch.alg = PFA_NONE;
    rpd->prefetch.available = FALSE;
    rpd->prefetch.checkavail = NULL;
    rpd->prefetch.getwork = NULL;
    rpd->prefetch.notify = NULL;
    rpd->prefetch.latenotify = NULL;
    rpd->prefetch.donerpd = NULL;
    rpd->prefetch.private = NULL;
}

/* @SUBTITLE "RT_PrefetchMistake: try to prefetch a sector" */
/* Called by the prediction algorithm when it realizes a mistake.
 * Sets mistake status of SME, and allows replacement of frame.
 * IF, that is, the sector was actually prefetched.
 *  WARNING: we may be called by the GAPS predictor with an rpd that
 * is correct for executing process, but be doing work for another
 * process. So stats will be misaligned slightly, but it should work ok.
 */
void
RT_PrefetchMistake(rpd, sector)
	RAPIDFILE *rpd;		/* see Warning above */
	int sector;
{
    SECTOR_MAP_ENTRY *sme;
    short oldstat;

    if (sector >= RT_NumSectors(rpd, rpd->size))
	 return;				/* ignore - invalid sector number */

    sme = RT_smeForSector(rpd, sector);

    ELOG_LOG(RTELOG_MISTAKE, sector);
    my_stats->mistake++;		/* we handed out a bad prediction */
#ifdef DEBUGPREDICT
    printf("%d mistake %d\n", Vnode, sector);
#endif DEBUGPREDICT

#ifdef WANTED
	   /* rescind our interest in the sector */
	   Atomic_add(&(sme->misc.want_count), -1);
#endif    

    oldstat = SS_SET(sme, SS_MISTAKE);
    if (!(oldstat & SS_MISTAKE) && (oldstat & SS_PREFETCHED)) {
	   /* we were able to set the mistake bit; and it was prefetched */
	   RT_Replaceable(rpd, sector);
	   /* one less frame to worry about */
#ifdef OWN_BUFFER
	   rpd->prefetch.unused--; /* probably not ok for global predictors  */
#else
	   Atomic_add(&(rpd->inode_ptr->prefetch.unused), -1);
#endif
	   /* the stats are updated at swapout time */
    }
}

/* @SUBTITLE "PrefetchLoop: central loop for prefetching" */
/* (coroutine)
 *	We look for a page to prefetch and a frame to put it in; this may or
 * may not succeed, as we'll only swap out pages that are unlocked.
 * We work as a coroutine, and are only run when we are allowed to.
 * We are allowed to when they call RT_Prefetch.
 *
 * No more than prefetch.limit pages should be outstanding
 * at any time. We must account for the page we plan to prefetch
 * BEFORE we prefetch it. 
*/
static void
PrefetchLoop(rpd)
	RAPIDFILE *rpd;
{
    int sector;
    RAPID_INODE *inode_ptr = rpd->inode_ptr;
    boolean prefetched;
    boolean retry;

    /* Prefetch forever! */
    while (TRUE) {			/* never exits */
	   /* wait until we have something to try to prefetch */
	   while (TRUE) {
#ifndef FOR_OTHERS
		  /* check too see if there is work to be prefetched. */
		  if (PrefetchWorkAvailable(rpd)) {
#endif
#ifdef OWN_BUFFER
			 if (rpd->prefetch.unused < rpd->prefetch.limit) {
				rpd->prefetch.unused++;
				break;	/* OK! We're all set to go! */
			 }
#else
			 /* try to get the right to prefetch */
			 if (inode_ptr->prefetch.unused < inode_ptr->prefetch.limit)
			   if (Atomic_add(&(inode_ptr->prefetch.unused), 1)
				  >= inode_ptr->prefetch.limit)
				Atomic_add(&(inode_ptr->prefetch.unused), -1);
			   else
				break;	/* OK! We're all set to go */
#endif OWN_BUFFER
#ifndef FOR_OTHERS
		  }
#endif
		  /* Nothing to do now; do something else for a while */
		  GOAWAY();
	   }

	   /* Now choose a sector to try to prefetch. We get this from */
	   /* the prefetch queue. If there is one, try to prefetch it. */
	   if (GetPrefetchWork(rpd, &sector)) {
#ifdef DEBUGPREDICT
		  printf("%d got work %d\n", Vnode, sector);
#endif DEBUGPREDICT
		  prefetched = TryPrefetchSector(rpd, sector, &retry);
		  if (!prefetched && retry)
		    GiveBackPrefetchWork(rpd, sector);
	   } else {
		  prefetched = FALSE;
	   }
	   
	   /* If we failed, give up our right to prefetch for now. */
	   /* If success, we have used our right; there */
	   /* is one more unused prefetched page around. */
#ifdef OWN_BUFFER
	   if (!prefetched)
		rpd->prefetch.unused--;
#else
	   if (!prefetched)
		Atomic_add(&(inode_ptr->prefetch.unused), -1);
#endif OWN_BUFFER
	   
	   /* Check the status of the synchronization point */
	   /* and allow for other prefetching to be done */
	   GOAWAY();
    }
}

/* @SUBTITLE "TryPrefetchSector: try to prefetch a sector" */
/* Try to prefetch the given sector. Called by the PrefetchLoop functions,
 * which decide what sectors to try.
 */
static boolean
TryPrefetchSector(rpd, sector, retry)
	RAPIDFILE *rpd;
	int sector;
	boolean *retry;		/* (out) we set true if worth trying again */
{
    SECTOR_MAP_ENTRY *sme;
    
    if (sector >= RT_NumSectors(rpd, rpd->size)) {
	   *retry = FALSE;
	   return(FALSE);
    }

    sme = RT_smeForSector(rpd, sector);
    
#ifdef PREFETCH_DEBUG_FLAG
    printf("Prefetch: considering sector %d status %x\n",
		 sector, sme->sector_status);
#endif
    
#ifdef WANTED
    /* indicate our interest in the sector */
    Atomic_add(&(sme->misc.want_count), 1);
#endif    

    /* Is this sector non-resident, non-zero, and not being changed? */
    if (SS_ISSET(sme, SS_VALID) || SS_ISCLEAR(sme, SS_NONZERO)
	   || IsChangingSector(sme)) {
	   *retry = FALSE;
	   return (FALSE);
    }

#ifdef PREFETCH_DEBUG_FLAG
    printf("Prefetch: Trying to prefetch %d\n", sector);
#endif
    
    /* lock it */
    if (!TryChangeLockSector(sme)) {
	   *retry = FALSE;
	   return(FALSE);
    }
    
    if (SS_ISSET(sme, SS_VALID)) {
	   /* It became valid on us */
	   ChangeUnlockSector(sme);
#ifdef PREFETCH_DEBUG_FLAG
	   printf("TryPrefetchSector: Sector %d became valid\n", sector);
#endif
	   *retry = FALSE;
	   return(FALSE);
    }
    
#ifdef PREFETCH_DEBUG_FLAG
    printf("TryPrefetchSector: FindFrame for sector %d\n", sector);
#endif
    if (!RT_FindFrame(rpd, sector, sme, TRUE)) {
#ifdef WANTED
	   /* rescind our interest in the sector -- we'll be back */
	   Atomic_add(&(sme->misc.want_count), -1);
#endif    
	   ChangeUnlockSector(sme);
#ifdef PREFETCH_DEBUG_FLAG
	   printf("TryPrefetchSector: no ready frames available\n", sector);
#endif
	   *retry = TRUE;		/* may be able to get one later */
	   return(FALSE);
    }
    
    ELOG_LOG(RTELOG_PREFETCH, sector);
    my_stats->prefetch++;
    
    sme->whenready =
	 io_read(sme->sector_ptr,
		    (unsigned) sector * rpd->sector_size,
		    (unsigned) rpd->sector_size);
    my_stats->reads++;

#ifdef OWN_BUFFER
    sme->misc.owner = Vnode;
#endif

    /* Clear the written status bit */
    /* We don't actually do it, since we never prefetch and write */
    /* SS_CLEAR(sme, SS_WRITTEN | SS_NEW); */

    /* Now set the status and unlock it */
    /* we also check to see if it was a mistake */
    if (SS_SET(sme, SS_VALID | SS_PREFETCHED | SS_NONZERO) & SS_MISTAKE) {
	   /* whoops, it turned out to be a mistaken prediction */
	   RT_Replaceable(rpd, sector);
	   /* one less frame to worry about */
#ifdef OWN_BUFFER
	   rpd->prefetch.unused--; /* probably not ok for global predictors */
#else
	   Atomic_add(&(rpd->inode_ptr->prefetch.unused), -1);
#endif
    }
    ChangeUnlockSector(sme);

#ifdef DEBUGPREDICT
    printf("%d: prefetched %d\n", Vnode, sector);
#endif DEBUGPREDICT
#ifdef DEBUG_FLAG
    printf("(%d) (List %d) Prefetched sector %d  frame %d\n",
		 Vnode, ListUsed, sector, sme->sector_frame);
#else
# ifdef PREFETCH_DEBUG_FLAG
    printf("TryPrefetchSector: Fetched sector %d successfully\n",
		 sector);
# endif
#endif
    return(TRUE);
}
