/* @TITLE "predict-sw.c - SWITCH predict algorithm" */
/* predict-sw.c: Switch between a local and a global predictor
 *
 * This predictor is just a startup sequence designed to choose between 
 * local and global patterns, and then start the correct predictor.
 * Two predictors are predefined, one local and one global. They are
 * both initialized when SWITCH is initialized, but then SWITCH takes
 * over until a decision is made.
 *
 * The pattern is watched in a fast, concurrent way until there is enough 
 * to decide local or global. The decision depends on consecutivity: 
 * if all processors have three or more consecutive (contiguous and 
 * ordered) references, then it is local. Otherwise, on the first 
 * non-consecutive reference, it is global.
 *
 * Immediate rereference does not change consecutivity. 
 *
 * FUNCTIONS:
 *      InitPredict_SWITCH
 *
 * 'Local' entry points:
 *      Notify
 *      CheckAvailable
 *      Done
 *      Dump
 *
 * David Kotz  May 1990
 */

static char rcsid[] = "$Id: predict-sw.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "stats.h"
#include "rapidelog.h"

/* DEFINES */

/* Predictors:
 * Must be valid, and not PFA_NONE or PFA_SWITCH.
 * Must supply a LateNotify function.
 * For now, the choice of predictors is hardwired here:
 */
#define LOCAL_PRED PFA_IOPORT
#define GLOBAL_PRED PFA_GAPS

/* how many consecutive refs (on each proc) to call it local? */
#define NUM_CONSEC 3

/* @SUBTITLE "DATA STRUCTURES" */
/* data for one predictor */
struct PredictData {
    voidproc donerpd;		/* cleanup procedure */
    boolproc notify;		/* notification procedure */
    voidproc latenotify;		/* late notification procedure (for SWITCH) */
    boolproc getwork;		/* get a block for prefetching */
    voidproc giveback;		/* give back a block for prefetching */
    boolproc checkavail;		/* is there something to prefetch? */
    voidproc dump;			/* debugging dump of private data */
    ANYPTR private;			/* data particular to algorithm */
};

/* The status for each processor in the SWITCH algorithm */
enum SwitchStatus {
    SWS_UNDET, SWS_GOGLOBAL, SWS_GLOBAL, SWS_LOCAL
};

/* global data */
struct switchdata {
    short nprocs;			/* total number of procs */
    short consecutive;		/* number of consecutive procs */
    
    ALOCK lock;			/* lock on the status */
    enum SwitchStatus status;	/* current global status */

    short UpdateCount;		/* counter for assigning update work */
    struct switchlocal **localdata; /* [nprocs] pointers to all local data */
};

/* local (per-processor) data */
struct switchlocal {
    struct switchdata *common; /* pointer to global data structure */
    short nprocs;			/* total number of procs */

    struct PredictData local;
    struct PredictData global;

    int last;				/* most recent block referenced */
    int length;			/* number of consecutive blocks */
    enum SwitchStatus status;	/* SWITCH status for this processor */
    ALOCK lock;			/* lock for this processor */
};

/* @SUBTITLE "FUNCTIONS" */

static void Done();
static void Dump();
static boolean Notify();
static boolean CheckAvailable();

static void CheckStatus();	/* check the global status */
static void ChangeStatus();	/* change global SWITCH status */
static void HelpGoGlobal();	/* help call LateNotify on all nodes */
static boolean ToGlobal();	/* change a node to global */
static void LocalFuncs();	/* set up function pointers for Local */
static void GlobalFuncs();	/* set up function pointers for Global */


/* @SUBTITLE "InitPredict_SWITCH: Initialize for SWITCH algorithm" */
void
InitPredict_SWITCH(rpd, parm)
	RAPIDFILE *rpd;
	int parm;				/* passed to both local and global predictor */
{
    int nprocs = ProcsInUse();
    struct switchdata *sw;
    struct switchlocal *lsw;

    /* Everyone allocates the local switch structure */
    lsw = (struct switchlocal *)AllocLocal(sizeof(struct switchlocal));

    /* Initialize the local predictor */
    if (LOCAL_PRED == PFA_NONE || LOCAL_PRED == PFA_SWITCH) {
	   fprintf(stderr, "InitPredict_SWITCH: local predictor may not be %d\n",
			 LOCAL_PRED);
	   exit(1);
    }
    (void) InitPredict(rpd, LOCAL_PRED, parm);
    lsw->local.donerpd    = rpd->prefetch.donerpd;
    lsw->local.notify     = rpd->prefetch.notify;
    lsw->local.latenotify = rpd->prefetch.latenotify;
    lsw->local.getwork    = rpd->prefetch.getwork;
    lsw->local.giveback   = rpd->prefetch.giveback;
    lsw->local.checkavail = rpd->prefetch.checkavail;
    lsw->local.dump       = rpd->prefetch.dump;
    lsw->local.private    = rpd->prefetch.private;
    if (lsw->local.latenotify == NULL) {
	   fprintf(stderr, "Local predictor function lacks a LateNotify\n");
	   exit(1);
    }

    /* Initialize the global predictor */
    if (GLOBAL_PRED == PFA_NONE || GLOBAL_PRED == PFA_SWITCH) {
	   fprintf(stderr, "InitPredict_SWITCH: global predictor may not be %d\n",
			 GLOBAL_PRED);
	   exit(1);
    }
    (void) InitPredict(rpd, GLOBAL_PRED, parm);
    lsw->global.donerpd    = rpd->prefetch.donerpd;
    lsw->global.notify     = rpd->prefetch.notify;
    lsw->global.latenotify = rpd->prefetch.latenotify;
    lsw->global.getwork    = rpd->prefetch.getwork;
    lsw->global.giveback   = rpd->prefetch.giveback;
    lsw->global.checkavail = rpd->prefetch.checkavail;
    lsw->global.dump       = rpd->prefetch.dump;
    lsw->global.private    = rpd->prefetch.private;
    if (lsw->global.latenotify == NULL) {
	   fprintf(stderr, "Global predictor function lacks a LateNotify\n");
	   exit(1);
    }

    /* do other initialization */
    lsw->nprocs = nprocs;
    lsw->last = -1;
    lsw->length = 0;
    lsw->status = SWS_UNDET;
    lsw->lock = 0;

    /* Only one process does the global SWITCH initialization */
    if (Vnode == 0) {
	   sw = (struct switchdata *)AllocGlobal(sizeof(struct switchdata));

	   sw->nprocs = nprocs;
	   sw->consecutive = 0;
	   sw->status = SWS_UNDET;
	   sw->lock = 0;
	   sw->UpdateCount = 0;

	   /* the values will be installed by the processors */
	   sw->localdata = (struct switchlocal **)
		AllocGlobal(nprocs * sizeof(struct switchlocal *));

	   rpd->inode_ptr->prefetch.available = FALSE; /* no prediction yet */

	   /* this will release the other processes from a barrier below */ 
	   rpd->inode_ptr->prefetch.private2 = (ANYPTR) sw;
   } else {
	   /* all procs except 0 */
	   while (rpd->inode_ptr->prefetch.private2 == NULL)
		UsWait(50);
    }

    /* @PAGE */
    sw = (struct switchdata *)(rpd->inode_ptr->prefetch.private2);
    lsw->common = sw;

    sw->localdata[Vnode] = lsw;

    /* Record the SWITCH pointer for future access */
    rpd->prefetch.private2 = (ANYPTR)lsw;
    rpd->prefetch.available = FALSE;

    /* Set up function pointers to SWITCH */
    rpd->prefetch.donerpd    = Done;
    rpd->prefetch.notify     = Notify;
    rpd->prefetch.latenotify = NULL;
    rpd->prefetch.getwork    = NULL;
    rpd->prefetch.giveback   = NULL;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.dump       = Dump;
}

/* @SUBTITLE "Notify: notification that a sector is accessed" */
static boolean
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct switchlocal *lsw = (struct switchlocal*)(rpd->prefetch.private2);
    struct switchdata *sw = lsw->common;

    if (sw->status == SWS_UNDET) {
	   if (sector != lsw->last) {
		  LOCK(&(lsw->lock), 10);
		  if (lsw->status == SWS_UNDET) {
			 if (lsw->length == 0 || sector == lsw->last+1) {
				lsw->length++;	
				lsw->last = sector;
				sector = -1;
				if (lsw->length == NUM_CONSEC)
				  if (Atomic_add(&(sw->consecutive), 1)+1 == lsw->nprocs)
				    ChangeStatus(sw, SWS_LOCAL);
			 } else {
				/* non-consecutive: Switch to global */
				ChangeStatus(sw, SWS_GOGLOBAL);
			 }
		  }
		  UNLOCK(&(lsw->lock));
	   }
    }	   

    CheckStatus(rpd, sector, lsw, sw);

    return(FALSE);
}

/* @SUBTITLE "Check the global SWITCH status and act" */
/* This is called by Notify and CheckAvailable. The process may switch 
 * to either a local or a global predictor, or do nothing, as appropriate. 
 */
static void
CheckStatus(rpd, sector, lsw, sw)
	RAPIDFILE *rpd;
	int sector;
	struct switchlocal *lsw;
	struct switchdata *sw;
{
    int count = lsw->length + ((sector >= 0) ? 1 : 0);

    switch (sw->status) {
	   case SWS_UNDET: {
		  break;
	   }
	   case SWS_GOGLOBAL: {
		  /* first we do ourselves */
		  ELOG_LOG(RTELOG_SWITCHCOUNT, count);
		  GlobalFuncs(rpd, lsw, sw);
		  /* From now on the global Notify will be called */

		  if (!ToGlobal(rpd, Vnode, lsw, sector) && sector >= 0) {
			 while(lsw->lock)
			   UsWait(5);
			 NotifyPredict(rpd, sector);
		  }
		  
		  /* then help with the others */
		  HelpGoGlobal(rpd, lsw, sw);
		  break;
	   }
	   case SWS_GLOBAL: {
		  ELOG_LOG(RTELOG_SWITCHCOUNT, count);
		  GlobalFuncs(rpd, lsw, sw);
		  /* From now on the global Notify will be called */

		  if (!ToGlobal(rpd, Vnode, lsw, sector) && sector >= 0) {
			 while(lsw->lock)
			   UsWait(5);
			 NotifyPredict(rpd, sector);
		  }
		  break;
	   }
	   case SWS_LOCAL: {
		  ELOG_LOG(RTELOG_SWITCHCOUNT, count);
		  LocalFuncs(rpd, lsw, sw);
		  /* From now on the local Notify will be called */

		  LateNotifyPredict(Vnode, rpd, lsw->local.private, 
						lsw->last, lsw->length, sector);
		  lsw->status = SWS_LOCAL;
		  break;
	   }
	   default: {
		  printf("%02d: Unknown switch status %d in SWITCH Notify\n",
			    Vnode, (int)(lsw->status));
		  break;
	   }
    }

}

/* @SUBTITLE "ChangeStatus: Decision has been made" */
/* Changes are only allowed from state SWS_UNDET */
/* Do not call with status==SWS_UNDET */

static void
ChangeStatus(sw, status)
	struct switchdata *sw;
	enum SwitchStatus status; /* the new global status */
{
    if (sw->status == SWS_UNDET) {
	   if (TryLock(&(sw->lock))) {
		  if (sw->status == SWS_UNDET) {
			 sw->status = status;
			 ELOG_LOG(RTELOG_SWITCHLOCAL, 
					status == SWS_LOCAL ? 1 : 0);
#ifdef TRACE
			 printf("SWITCHing to %s predictor\n", 
				   status == SWS_LOCAL ? "LOCAL" : "GLOBAL");
#endif /* TRACE */
		  }
		  UNLOCK(&(sw->lock));
	   } else
		while (sw->status == SWS_UNDET)
		  UsWait(10);
    }
}

/* @SUBTITLE "HelpGoGlobal: help in switch to global" */
/* help to get LateNotify() called on all nodes */
/* We assume that GlobalFuncs has already been called on THIS proc. */

static void
HelpGoGlobal(rpd, lsw, sw)
	RAPIDFILE *rpd;
	struct switchlocal *lsw;
	struct switchdata *sw;
{
    int node;
    int nprocs = lsw->nprocs;
    struct switchlocal *rsw;	/* another nodes' lsw */

    do {
	   node = Atomic_add(&(sw->UpdateCount), 1);
	   if (node < nprocs) {
		  rsw = sw->localdata[node];	
		  (void) ToGlobal(rpd, node, rsw, -1);
	   }
    } while (node < nprocs);

    if (node == nprocs) {		/* completed the job */
	   sw->status = SWS_GLOBAL;
    }
}

/* @SUBTITLE "ToGlobal: change a node over to global" */
/* We assume that GlobalFuncs has already been called on THIS proc. */

static boolean				/* TRUE if notify was performed */
ToGlobal(rpd, node, lsw, sector)
	RAPIDFILE *rpd;
	int node;
	struct switchlocal *lsw;	/* may be remote */
	int sector;
{
    boolean notified = FALSE;

    if (lsw->status == SWS_UNDET) {
	   if (TryLock(&(lsw->lock))) {
		  if (lsw->status == SWS_UNDET) {
			 lsw->status = SWS_GLOBAL;
			 /* we pass rpd although it is ours, and possibly not
			  * the rpd for this lsw. The inode pointer will work,
			  * as will access to global predictor data.
			  */
			 LateNotifyPredict(node, rpd, lsw->global.private, 
						    lsw->last, lsw->length, sector);
			 notified = TRUE;
		  }
		  UNLOCK(&(lsw->lock));
	   }
    }

    return(notified);
}

/* @SUBTITLE "LocalFuncs: Switch to local predictor" */
static void
LocalFuncs(rpd, lsw, sw)
	RAPIDFILE *rpd;
	struct switchlocal *lsw;
	struct switchdata *sw;
{
    /* note we don't touch donerpd */
    rpd->prefetch.notify     = lsw->local.notify;
    rpd->prefetch.latenotify = lsw->local.latenotify;
    rpd->prefetch.getwork    = lsw->local.getwork;
    rpd->prefetch.giveback   = lsw->local.giveback;
    rpd->prefetch.checkavail = lsw->local.checkavail;
    rpd->prefetch.dump       = lsw->local.dump;
    rpd->prefetch.private    = lsw->local.private;
}

/* @SUBTITLE "GlobalFuncs: Switch to global predictor" */
static void
GlobalFuncs(rpd, lsw, sw)
	RAPIDFILE *rpd;
	struct switchlocal *lsw;
	struct switchdata *sw;
{
    fflush(stdout);
    /* note we don't touch donerpd */
    rpd->prefetch.notify     = lsw->global.notify;
    rpd->prefetch.latenotify = lsw->global.latenotify;
    rpd->prefetch.getwork    = lsw->global.getwork;
    rpd->prefetch.giveback   = lsw->global.giveback;
    rpd->prefetch.checkavail = lsw->global.checkavail;
    rpd->prefetch.dump       = lsw->global.dump;
    rpd->prefetch.private    = lsw->global.private;
}

/* @SUBTITLE "CheckAvailable: is there any prefetch work?" */
/* This is an Entry Point */
/* This is here for global predictors: the LateNotify procedure may set
 * the inode's prefetch.available flag, causing processes to call CheckAvail.
 * But, some procs may not have switched their funcs to the global predictor,
 * and so may call the CheckAvail for SWITCH. Thus, we must supply a dummy.
 * HOWEVER, since the process is calling us, and obviously has time to 
 * spare, we put it through a status check as we do in Notify.
 * It WILL switch to global (that is why it got here) and the next call
 * to CheckAvailable will land in the global predictor. 
 */
static boolean
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
    struct switchlocal *lsw = (struct switchlocal*)(rpd->prefetch.private2);
    struct switchdata *sw = lsw->common;

    CheckStatus(rpd, -1, lsw, sw);

    return(FALSE);			/* no, there is no prefetch work */
}

/* @SUBTITLE "Done: What happens at end of pattern" */
/* This is an Entry Point */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    struct switchlocal *lsw = (struct switchlocal*)(rpd->prefetch.private2);
    struct switchdata *sw = lsw->common;

    /* Terminate local predictor */
    rpd->prefetch.donerpd = lsw->local.donerpd;
    rpd->prefetch.private = lsw->local.private;
    DonePredict(rpd);

    /* Terminate global predictor */
    rpd->prefetch.donerpd = lsw->global.donerpd;
    rpd->prefetch.private = lsw->global.private;
    DonePredict(rpd);

    UsFree(lsw);

    rpd->prefetch.private2 = NULL;

    if (Vnode == 0) {
	   UsFree(sw->localdata);
	   UsFree(sw);
	   rpd->inode_ptr->prefetch.private2 = NULL;
    }
}

/* @SUBTITLE "Dump: debugging dump" */
static void
Dump(rpd, lsw)
	RAPIDFILE *rpd;
	struct switchlocal *lsw;
{
    struct switchdata *sw;

    if (rpd != NULL)		/* use rpd to find lsw */
	 lsw = (struct switchlocal *)(rpd->prefetch.private2);
    sw = lsw ? lsw->common : NULL;

    printf("  SWITCH Predictor: local\n");
    if (lsw != NULL) {
	   printf("   common = 0x%x\n", lsw->common);
	   printf("   nprocs = %d\n", lsw->nprocs);
	   printf("   local.donerpd    = 0x%x\n", lsw->local.donerpd);
	   printf("   local.notify     = 0x%x\n", lsw->local.notify);
	   printf("   local.latenotify = 0x%x\n", lsw->local.latenotify);
	   printf("   local.getwork    = 0x%x\n", lsw->local.getwork);
	   printf("   local.giveback   = 0x%x\n", lsw->local.giveback);
	   printf("   local.checkavail = 0x%x\n", lsw->local.checkavail);
	   printf("   local.dump       = 0x%x\n", lsw->local.dump);
	   printf("   local.private    = 0x%x\n", lsw->local.private);
	   printf("   global.donerpd    = 0x%x\n", lsw->global.donerpd);
	   printf("   global.notify     = 0x%x\n", lsw->global.notify);
	   printf("   global.latenotify = 0x%x\n", lsw->global.latenotify);
	   printf("   global.getwork    = 0x%x\n", lsw->global.getwork);
	   printf("   global.giveback   = 0x%x\n", lsw->global.giveback);
	   printf("   global.checkavail = 0x%x\n", lsw->global.checkavail);
	   printf("   global.dump       = 0x%x\n", lsw->global.dump);
	   printf("   global.private    = 0x%x\n", lsw->global.private);
	   printf("   last = %d\n", lsw->last);
	   printf("   length = %d\n", lsw->length);
	   printf("   status = %d\n", (int)(lsw->status));
	   printf("   lock = %d\n", lsw->lock);
    } else
	 printf(" NULL local switch data\n");

    printf("  SWITCH Predictor: global\n");
    if (sw != NULL) {
	   printf("   nprocs = %d\n", sw->nprocs);
	   printf("   consecutive = %d\n", sw->consecutive);
	   printf("   lock = %d\n", sw->lock);
	   printf("   status = %d\n", (int)(sw->status));
	   printf("   UpdateCount = %d\n", sw->UpdateCount);
	   printf("   localdata = 0x%x\n", sw->localdata);
    } else
	 printf(" NULL global switch data\n");
}

