/* @TITLE "predict-iobl.c - IOBL predict algorithm" */
/* predict-iobl.c: Infinite-block lookahead
 *
 * Like IBL and OBL. We start out with IBL, but switch to OBL when
 * we see an incorrect prediction.
 *
 * Immediate rereference counts as a correct prediction.
 *
 * FUNCTIONS:
 *      InitPredict_IOBL
 *
 * 'Local' entry points:
 *      LateNotify
 *      Notify
 *      CheckAvailable
 *      GetWork
 *      GiveBackWork
 *      Done
 *      Dump
 *
 * David Kotz  November 1989
 */

static char rcsid[] = "$Id: predict-iobl.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $";

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "stats.h"
#include "rapidelog.h"

static void Done();
static void Dump();
static boolean Notify();
static void LateNotify();
static boolean CheckAvailable();
static boolean GetWork();
static void GiveBackWork();

struct iobldata {
    int last;				/* last reference */
    int pref;				/* last prefetch handed out */
    boolean first;			/* TRUE if in first (only?) run */
};

/* @SUBTITLE "InitPredict_IOBL: Initialize for IOBL algorithm" */
void
InitPredict_IOBL(rpd, ignored)
	RAPIDFILE *rpd;
	int ignored;
{
    struct iobldata *ibl;

    ibl = (struct iobldata *)AllocLocal(sizeof (struct iobldata));

    ibl->last = -10;		/* last reference: impossible */
    ibl->pref = -10;		/* who cares */
    ibl->first = TRUE;

    rpd->prefetch.private = (ANYPTR) ibl;
    rpd->prefetch.available = FALSE; /* no prediction yet */

    rpd->prefetch.donerpd = Done;
    rpd->prefetch.notify = Notify;
    rpd->prefetch.latenotify = LateNotify;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.getwork = GetWork;
    rpd->prefetch.giveback = GiveBackWork;
    rpd->prefetch.dump = Dump;
}

/* @SUBTITLE "LateNotify: Multiple notifications" */
static void
LateNotify(node, rpd, private, last, length, sector)
	int node;				/* processor number */
	RAPIDFILE *rpd;		/* can be trusted: this is local predictor */
	ANYPTR private;		/* private data */
	int last;				/* last block accessed in run */
	int length;			/* length of consecutive run */
	int sector;			/* most recent block accessed (-1 if none) */
{
    struct iobldata *iobl = (struct iobldata *)private;

#ifdef TRACE
    printf("Proc %d (%d) in IOBL LateNotify: last=%d, length=%d, sector=%d\n",
		 Vnode, node, last, length, sector);
    fflush(stdout);
#endif

    if (sector >= 0) {
	   /* record last reference */
	   iobl->last = sector;
	   iobl->pref = sector;
	   rpd->prefetch.available = TRUE; /* we have a prediction */
	   if (length > 0 && sector != last+1)
		iobl->first = FALSE;
    } else if (length > 0) {
	   /* record last reference */
	   iobl->last = last;
	   iobl->pref = last;
	   rpd->prefetch.available = TRUE; /* we have a prediction */
    }
}

/* @SUBTITLE "Notify: notification that a sector is accessed" */
static boolean
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct iobldata *ibl = (struct iobldata*) rpd->prefetch.private;
    int s;
    boolean wanted;

    /* immediate re-reference is correct, but doesn't change prediction */
    if (ibl->last == sector) {
	   my_stats->correct++;
	   return(wanted = FALSE);
    }

    /* correct prediction? */
    if (ibl->last + 1 == sector) {
	   my_stats->correct++;
	   /* record last reference */
	   ibl->last = sector;
	   wanted = (ibl->pref >= ibl->last);
	   if (ibl->last > ibl->pref)
		ibl->pref = ibl->last;
    } else {
	   /* incorrect */
	   wanted = FALSE;
	   if (ibl->first) {
		  /* IBL */
		  for (s = ibl->last+1; s <= ibl->pref; s++)
		    /* we had prefetched s */
		    RT_PrefetchMistake(rpd, s);
		  if (ibl->last >= 0)
		    ibl->first = FALSE; /* not any more */
	   } else {
		  /* OBL */
		  if (rpd->prefetch.available == FALSE && ibl->last >= 0)
		    /* we had prefetched last+1 */
		    RT_PrefetchMistake(rpd, ibl->last+1);
	   }
	   ibl->last = sector;
	   ibl->pref = sector;
    }

    rpd->prefetch.available = TRUE; /* we have a prediction */

    return(wanted);
}

/* @SUBTITLE "CheckAvailable: is there work available?" */
/* Are there any sectors to be prefetched? */

static boolean				/* TRUE if there's work */
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
    return(rpd->prefetch.available); /* this is all we know */
}

/* @SUBTITLE "GetWork: get a sector to prefetch" */
/* If still in first run, use IBL, and give pref+1 as our prediction */
/* Otherwise, use OBL, and give last+1, but only once. */

static boolean				/* TRUE if sector was found */
GetWork(rpd, sector)
	RAPIDFILE *rpd;
	int *sector;	/* returned */
{
    struct iobldata *ibl = (struct iobldata*) rpd->prefetch.private;
    boolean success;

    if (ibl->first) {
	   *sector = ++(ibl->pref); /* IBL */
	   success = TRUE;
    } else {
	   *sector = ibl->last + 1; /* OBL */
	   /* we won't make the same prediction twice */
	   success = rpd->prefetch.available;
	   rpd->prefetch.available = FALSE;
    }
    return(success);
}

/* @SUBTITLE "GiveBackWork: Sector could not be prefetched" */

static void
GiveBackWork(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct iobldata *ibl = (struct iobldata*) rpd->prefetch.private;

    if (ibl->first)
	 /* it should be that sector = ibl->pref */
	 ibl->pref--;			/* undo the above */
    else
	 /* allow us to hand it out again */
	 rpd->prefetch.available = TRUE;
}

/* @SUBTITLE "Done: What happens at end of string" */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    struct iobldata *ibl = (struct iobldata*) rpd->prefetch.private;
    int s;

    if (ibl->first) {
	   /* IBL */
	   for (s = ibl->last+1; s <= ibl->pref; s++)
		/* we had prefetched s */
		RT_PrefetchMistake(rpd, s);
    } else {
	   /* OBL */
	   if (rpd->prefetch.available == FALSE && ibl->last >= 0)
		/* we had prefetched last+1 */
		RT_PrefetchMistake(rpd, ibl->last+1);
    }

    UsFree(ibl);
    rpd->prefetch.private = NULL;
}

/* @SUBTITLE "Dump: debugging dump" */
static void
Dump(rpd)
	RAPIDFILE *rpd;
{
    struct iobldata *iobl = (struct iobldata*) rpd->prefetch.private;

    printf("  IOBL Predictor\n");
    if (iobl != NULL) {
	   printf("   last = %d\n", iobl->last);
	   printf("   pref = %d\n", iobl->pref);
	   printf("   first = %s\n", iobl->first ? "TRUE" : "FALSE");
    } else
	 printf("   NULL iobldata");

}
