/* @TITLE "predict-ibl.c - IBL predict algorithm" */
/* predict-ibl.c: Infinite-block lookahead
 *
 * Like OBL, but we predict infinitely far ahead. Thus there is always
 * work to do.
 *
 * Immediate rereference counts as a correct prediction.
 *
 * FUNCTIONS:
 *      InitPredict_IBL
 *
 * 'Local' entry points:
 *      LateNotify
 *      Notify
 *      CheckAvailable
 *      GetWork
 *      GiveBackWork
 *      Done
 *      Dump
 *
 * David Kotz  November 1989
 */

static char rcsid[] = "$Id: predict-ibl.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "stats.h"
#include "rapidelog.h"

static void Done();
static void Dump();
static boolean Notify();
static void LateNotify();
static boolean CheckAvailable();
static boolean GetWork();
static void GiveBackWork();

struct ibldata {
    int last;				/* last reference */
    int pref;				/* last prefetch handed out */
};

/* @SUBTITLE "InitPredict_IBL: Initialize for IBL algorithm" */
void
InitPredict_IBL(rpd, ignored)
	RAPIDFILE *rpd;
	int ignored;
{
    struct ibldata *ibl;

    ibl = (struct ibldata *)AllocLocal(sizeof (struct ibldata));

    ibl->last = -10;		/* last reference: impossible */
    ibl->pref = -10;		/* who cares */

    rpd->prefetch.private = (ANYPTR) ibl;
    rpd->prefetch.available = FALSE; /* no prediction yet */

    rpd->prefetch.donerpd = Done;
    rpd->prefetch.notify = Notify;
    rpd->prefetch.latenotify = LateNotify;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.getwork = GetWork;
    rpd->prefetch.giveback = GiveBackWork;
    rpd->prefetch.dump = Dump;
}

/* @SUBTITLE "LateNotify: Multiple notifications" */
/* All we care about here is the most recent access. Very simple. */
static void
LateNotify(node, rpd, private, last, length, sector)
	int node;				/* processor number */
	RAPIDFILE *rpd;		/* can be trusted: this is local predictor */
	ANYPTR private;		/* private data */
	int last;				/* last block accessed in run */
	int length;			/* length of consecutive run */
	int sector;			/* most recent block accessed (-1 if none) */
{
    struct ibldata *ibl = (struct ibldata *)private;

#ifdef TRACE
    printf("Proc %d (%d) in IBL LateNotify: last=%d, length=%d, sector=%d\n",
		 Vnode, node, last, length, sector);
    fflush(stdout);
#endif

    if (sector >= 0) {
	   /* record last reference */
	   ibl->last = sector;
	   ibl->pref = sector;
	   rpd->prefetch.available = TRUE; /* we have a prediction */
    } else if (length > 0) {
	   /* record last reference */
	   ibl->last = last;
	   ibl->pref = last;
	   rpd->prefetch.available = TRUE; /* we have a prediction */
    }
}

/* @SUBTITLE "Notify: notification that a sector is accessed" */
static boolean
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct ibldata *ibl = (struct ibldata*) rpd->prefetch.private;
    int s;
    boolean wanted;

    /* immediate re-reference is correct, but doesn't change prediction */
    if (ibl->last == sector) {
	   my_stats->correct++;
	   return(wanted=FALSE);
    }

    /* correct prediction? */
    if (ibl->last + 1 == sector) {
	   my_stats->correct++;
	   /* record last reference */
	   ibl->last = sector;
	   wanted = (ibl->pref >= ibl->last);
	   if (ibl->last > ibl->pref)
		ibl->pref = ibl->last;
    } else {
	   for (s = ibl->last+1; s <= ibl->pref; s++)
		/* we had prefetched s */
		RT_PrefetchMistake(rpd, s);
	   ibl->last = sector;
	   ibl->pref = sector;
	   wanted = FALSE;
    }

    rpd->prefetch.available = TRUE; /* we have a prediction */

    return(wanted);
}

/* @SUBTITLE "CheckAvailable: is there work available?" */
/* Are there any sectors to be prefetched? */

static boolean				/* TRUE if there's work */
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
    return(TRUE);			/* always work available */
}

/* @SUBTITLE "GetWork: get a sector to prefetch" */
/* give pref+1 as our prediction */

static boolean				/* TRUE if sector was found */
GetWork(rpd, sector)
	RAPIDFILE *rpd;
	int *sector;	/* returned */
{
    struct ibldata *ibl = (struct ibldata*) rpd->prefetch.private;

    *sector = ++(ibl->pref);

    return(TRUE);
}

/* @SUBTITLE "GiveBackWork: Sector could not be prefetched" */

static void
GiveBackWork(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct ibldata *ibl = (struct ibldata*) rpd->prefetch.private;

    /* it should be that sector = ibl->pref */
    ibl->pref--;			/* undo the above */
}

/* @SUBTITLE "Done: What happens at end of string" */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    struct ibldata *ibl = (struct ibldata*) rpd->prefetch.private;
    int s;

    for (s = ibl->last+1; s <= ibl->pref; s++)
	 /* we had prefetched s */
	 RT_PrefetchMistake(rpd, s);

    UsFree(ibl);
    rpd->prefetch.private = NULL;
}

/* @SUBTITLE "Dump: debugging dump" */
static void
Dump(rpd)
	RAPIDFILE *rpd;
{
    struct ibldata *ibl = (struct ibldata*) rpd->prefetch.private;

    printf("  IBL Predictor\n");
    if (ibl != NULL) {
	   printf("   last = %d\n", ibl->last);
	   printf("   pref = %d\n", ibl->pref);
    } else
	 printf("   NULL ibldata");

}
