/* @TITLE "predict-GAPS.c -- Global-Access-Pattern Seq" */
/* 
 * GAPS: Global-Access-Pattern Sequentiality.
 * 
 * This is a method to detect sequentiality in global access 
 * patterns. See notebook for 2/27/90 and following.
 *
 * IMPORTANT NOTE: the rpd is used throughout for several purposes, 
 * primarily for sme access for the used and mark bits, and for 
 * passing through to RT_PrefetchMistake. However, when the entry is from 
 * LateNotify, the rpd is for the executing processor, while the node number 
 * and lgaps structure may be for another node. Thus, everything works
 * as long as the rpd is used ONLY for constant data that is the same
 * everywhere, like access to the sector map table.
 *
 * Incompatible with WANTED or OWN_BUFFER flags
 *
 * Define GAPTRACE to print out a trace of activity; note that this will
 * affect the pattern.
 * 
 * Define USEPARM to use the prefetch algorithm parameter for maxdist, 
 * otherwise maxjump is used.
 *
 * David Kotz March-May 1990
 */

static char rcsid[] = "$Id: predict-gaps.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $";

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "prefetch.h"
#include "stats.h"
#include "rapidelog.h"
#include "predict-debug.h"
#include "fifolock.h"
#include "wanted.h"
#include "own.h"
#include "queue.h"

/* @PAGE */

/* #define USEPARM */ /* define to use prefetch parm for maxdist */

/* #define GAPTRACE */

#ifdef GAPTRACE
static char *GapsFormat = "%2d %4d %d %d %d %.3f  %c\n";
# define Trace(node, block, n, cutoff, sample, coef, seq) \
   printf(GapsFormat, \
		(node), (block), (n), (cutoff), (sample), (coef), (seq))
# define TraceFlush() fflush(stdout)
#else
# define Trace(node, block, n, cutoff, sample, coef, seq) 
# define TraceFlush()
#endif

/* Elogs specific to GAPS */
#define GAPSELOG_NOTIFY (RTELOG_PFA+0) /* block */
#define GAPSELOG_START  (RTELOG_PFA+1) /* block */
#define GAPSELOG_END    (RTELOG_PFA+2) /* reason: 0=maxjump, 1=jumpback, 2=compfail*/
#define GAPSELOG_RANDOM (RTELOG_PFA+3) /* Random() time (tics) */
#define GAPSELOG_CONTIN (RTELOG_PFA+4) /* Contin() time (tics) */
#define GAPSELOG_WFIFO  (RTELOG_PFA+5) /* Watch_fifo wait time (tics) */
#define GAPSELOG_WATCH  (RTELOG_PFA+6) /* Watch() time (tics) */
#define GAPSELOG_TOSEQ  (RTELOG_PFA+7) /* ToSequential() time (tics) */

#define CONTEXTPERPROC 10	/* remember this many references per proc */
#define MINSAMPLE 4			/* must have a sample of at least this */
#define MINLEN 2			/* must have a group of at least this */
#define MAXSLOPE 3.0		/* to put a limit on maxjump */

static int RunRep = 2;		/* number of runs in a row to assume regular */
static int SkipRep = 2;		/* number of skips in a row to assume regular */
static int ContextSize = 0;	/* total context size */

/* @PAGE */
/* These structures contain the entire state of the GAPS predictor. */
static struct gapsdata {
    int parm;				/* parameter from command line */
    int nprocs;			/* == NPROCS */

    boolean contin;			/* is any process in Continuation mode */
    boolean random;			/* are we in Random mode */
    boolean restart;		/* restart Continuation mode. */
#define RESTARTCOUNT 3		/* only if prev. portion had 3*nprocs blocks */

    int **last;			/* [NPROCS] pointer to mylast for all procs  */
    boolean **sequential;	/* [NPROCS] pointer to mysequential for all */
    short num_sequential;	/* number of nodes thinking sequential */

    short num_ordered;		/* number of nodes that have order property */
#define LOWORDERPERC 70		/* go to Random when not enough are ordered */
#define HIGHORDERPERC 100	/* go back to Watch when enough are ordered */

    FIFOLOCK *watch_fifo;	/* Watch() entry/exit control */
    QH context;			/* the most recent blocks; FIFO queue of REFs */
    float coef;			/* the coefficient of determination */
#define TOLERANCE (0.8)		/* coef of determination must be greater */
    int maxjump;			/* maximum allowed jump past maxlast */
    int blockuses;			/* # of uses of each block */

    ALOCK minlast_lock;		/* lock for minlast */
    FIFOLOCK *maxlast_fifo;	/* fifo lock for maxlast */
    int start;				/* start of portion */
    int minlast;			/* min over last[i] */
    int maxlast;			/* max over last[i] */
    boolean jumpgap;		/* TRUE when in JumpGap */

    boolean portion1;		/* true if first portion */
    int same_runlen;		/* number of runs with same len */
    int prev_runlen;		/* length of previous run */
    int same_skiplen;		/* number of skips with same len */
    int prev_skiplen;		/* length of previous skip */
    int prev_end;			/* end of previous portion */
    int exp_end;			/* expected end of next portion */

    int runlen;			/* run length if regular, else 0 */
    int skiplen;			/* skip length if regular, else 0 */

    FIFOLOCK *prefetch_fifo;	/* fifo lock for prefetch operations */
    FIFOLOCK *pchange_fifo;	/* fifo lock to change prefetch parameters */
    QH GiveBackQ;			/* queue for work given back */
    int nextpref;			/* last block prefetched or read */
    int startskip;			/* last block before a skip */
    int endskip;			/* first block after a skip */
    boolean prefetch_skip;	/* is there a skip? */
    int size;				/* size of the file in blocks */
    int limit;				/* current limit on prefetching */
    int hardlimit;			/* hard upper limit on prefetching */
    int maxdist;			/* max distance past maxlast that we prefetch */
    int distance;			/* maxdist value when not infinite */
};

/* @PAGE */
/* The local gaps data structure, to replicate and localize constants */
struct gapslocal {
    struct gapsdata *common;	/* pointer to global common data above */

    int nprocs;			/* == NPROCS */

    int mylast;			/* last reference on this proc */
    int **last;			/* [NPROCS] pointer to mylast for all procs */
    boolean mysequential;	/* is this processor in sequential mode */
    boolean **sequential;	/* [NPROCS] pointer to mysequential for all */

    boolean ordered;		/* do we think we are ordered? */
    int ordercount;			/* number of blocks in order */
#define SHORTORDER 3		/* number in order to consider ordered */

    FIFOLOCK *watch_fifo;	/* Watch() entry/exit control */
    QH context;			/* the most recent blocks; FIFO queue of REFs */

    ALOCK *minlast_lock;		/* pointer to lock for minlast */
    FIFOLOCK *maxlast_fifo;	/* fifo lock for maxlast */

    FIFOLOCK *prefetch_fifo;	/* fifo lock for prefetch */
    FIFOLOCK *pchange_fifo;	/* fifo lock for prefetch */
    QH GiveBackQ;			/* queue for work given back */

    int size;				/* size of the file in blocks */
};

/* One of the entries in the context queue */
typedef struct ref_struct REF;
struct ref_struct {
    int block;				/* block number */
    int node;				/* processor node number */
};

/* @PAGE */
/* MACROS */
#define sqr(x) ((x)*(x))

/* This processes one mistake, as determined by other procedures. */
/* Run under minlast lock, or alone, from Finalize. */
#define OneMistake(rpd, block, sme) \
  if (SS_CLEAR((sme), SS_MARKED) & SS_MARKED) /* if was set, now clear */\
      RT_PrefetchMistake((rpd), (block))

/* FUNCTIONS - Entry points */
void InitPredict_GAPS();
static boolean Notify();
static void LateNotify();
static void Done();
static void Dump();
static boolean CheckAvailable();
static boolean GetWork();
static void GiveBackWork();

/* FUNCTIONS - Internal Use Only */
static void NotifyOne();
static void Random();
static boolean Watch();
static boolean CheckComplete();
static boolean CheckSlope();
static void FindMaxjump();
static boolean NotMyREF();
static boolean Contin();
static void ToSequential();
static void FromSequential();
static void Finalize();
static void NewPortion();
static boolean ExtendPortion();
static void JumpGap();
static void EndPortion();
static REF *NewRef();

static void StartPrefetch();
static void StopPrefetch();
static void LimitPrefetch();
static void SoftLimitPrefetch();
static void Mistakes();
static int NextBlock();

extern double fabs();
extern double ceil();

/* @SUBTITLE "InitPredict_GAPS - initialize data structures" */
/* This is an Entry Point */
/* This function allocates and initializes the GAPS data structures. Only 
 * one process will do most of the work (process 0). The others wait
 * until the gaps pointer is ready, and then do per-process initialization.
 */
void
InitPredict_GAPS(rpd, parm)
	RAPIDFILE *rpd;
	int parm;				/* currently ignored */
{
    int nprocs = ProcsInUse();
    struct gapsdata *gaps;
    struct gapslocal *lgaps;

#if defined(WANTED) || defined(OWN_BUFFER)
    fprintf(stderr,
		  "\nERROR: GAPS predictor used with incompatible options!\n");
#endif

    /* Everyone allocates the local gaps structure */
    lgaps = (struct gapslocal *)AllocLocal(sizeof(struct gapslocal));
    /* and a place for local copies of arrays */
    lgaps->last = (int **)
	 AllocLocal(nprocs * sizeof(int *));
    lgaps->sequential = (boolean **)
	 AllocLocal(nprocs * sizeof(boolean *));

    /* and record the pointer for future access */
    rpd->prefetch.private = (ANYPTR)lgaps;
    rpd->prefetch.available = FALSE;

    /* Set up function pointers */
    rpd->prefetch.donerpd = Done;
    rpd->prefetch.notify = Notify;
    rpd->prefetch.latenotify = LateNotify;
    rpd->prefetch.checkavail = CheckAvailable;
    rpd->prefetch.getwork = GetWork;
    rpd->prefetch.giveback = GiveBackWork;
    rpd->prefetch.dump = Dump;

    ContextSize = CONTEXTPERPROC * nprocs;

    /* Only one process does the global initialization */
    if (Vnode == 0) {
	   gaps = (struct gapsdata *)AllocGlobal(sizeof(struct gapsdata));

	   gaps->parm = parm;
	   gaps->nprocs = nprocs;
	   gaps->contin = FALSE;
	   gaps->random = FALSE;	/* we start out ordered */
	   gaps->restart = FALSE;

	   /* the values will be installed by the processors */
	   gaps->last = (int **)
		AllocGlobal(nprocs * sizeof(int *));
	   gaps->sequential = (boolean **)
		AllocGlobal(nprocs * sizeof(boolean *));

	   gaps->num_sequential = nprocs; /* this changes to 0 below */
	   gaps->num_ordered = nprocs; /* we start out ordered */

	   AllocFIFO(gaps->watch_fifo);

	   gaps->context = Make_queue(ContextSize);
	   gaps->coef = 0;
	   gaps->maxjump = 0;
	   gaps->blockuses = 0;

	   gaps->minlast_lock = 0;
	   AllocFIFO(gaps->maxlast_fifo);
	   gaps->start = -1;
	   gaps->minlast = -1;
	   gaps->maxlast = 0;
	   gaps->jumpgap = FALSE;

	   gaps->portion1 = TRUE;
	   gaps->same_runlen = 0;  
	   gaps->prev_runlen = 0;  
	   gaps->same_skiplen = 0; 
	   gaps->prev_skiplen = 0; 
	   gaps->prev_end = 0;     
	   gaps->exp_end = 0;

	   gaps->runlen = 0;
	   gaps->skiplen = 0;

	   AllocFIFO(gaps->prefetch_fifo);
	   AllocFIFO(gaps->pchange_fifo);
	   gaps->GiveBackQ = Make_queue(ContextSize); /* best size? */
	   gaps->nextpref = 0;
	   gaps->startskip = 0;
	   gaps->endskip = 0;
	   gaps->prefetch_skip = FALSE;
	   gaps->size = RT_NumSectors(rpd, rpd->size);
	   gaps->hardlimit = 0;
	   gaps->limit = 0;
	   gaps->maxdist = 0;
	   gaps->distance = 0;

	   rpd->inode_ptr->prefetch.available = FALSE; /* no prediction yet */

	   /* this will release the other processes from a barrier below */
	   rpd->inode_ptr->prefetch.private = (ANYPTR) gaps;
    } else {
	   /* all procs except 0 */
	   while (rpd->inode_ptr->prefetch.private == NULL)
		UsWait(50);
    }

    /* @PAGE */
    /* Copy the constant stuff from global to local */
    gaps = (struct gapsdata *)(rpd->inode_ptr->prefetch.private);
    lgaps->common = gaps;

    lgaps->nprocs = nprocs;
    lgaps->mylast = -1;
    gaps->last[Vnode] = &(lgaps->mylast);
    lgaps->mysequential = FALSE;
    gaps->sequential[Vnode] = &(lgaps->mysequential);
    lgaps->ordered = TRUE;
    lgaps->ordercount = 0;
    lgaps->watch_fifo = gaps->watch_fifo;
    lgaps->context = gaps->context;
    TouchQueue(lgaps->context);
    lgaps->minlast_lock = &(gaps->minlast_lock);
    lgaps->maxlast_fifo = gaps->maxlast_fifo;
    lgaps->prefetch_fifo = gaps->prefetch_fifo;
    lgaps->pchange_fifo = gaps->pchange_fifo;
    lgaps->GiveBackQ = gaps->GiveBackQ;
    lgaps->size = gaps->size;

    /* Indicate that we are done, and wait for the others */
    Atomic_add(&(gaps->num_sequential), -1);
    while(gaps->num_sequential > 0)
	 UsWait(10);
    /* now num_sequential is properly initialized to 0 */

    /* Now copy the arrays of pointers */
    btransfer(gaps->last, lgaps->last, nprocs * sizeof(int *));
    btransfer(gaps->sequential, lgaps->sequential, nprocs * sizeof(boolean *));
}

/* @SUBTITLE "LateNotify: Multiple notifications" */
/* We can't just call Notify since we don't necessarily have the correct 
 * rpd value. We have two choices: we can get the lock once, and then 
 * call Watch many times, or get the lock each time. It is less overhead
 * to get the lock once, but may not give as nice an interleaving of the 
 * blocks in the queue. It also avoids most of the evaluation in Watch().
 *  Here, we choose to get the lock each time, since
 * I have found that in most patterns sequentiality is detected very soon
 * after the first round of blocks (one from each proc). Holding the lock
 * forces all evaluation to wait until the last proc is in LateNotify,
 * and turn-on comes after about 3 blocks per proc, which is too slow.
 */
static void
LateNotify(node, rpd, private, last, length, sector)
	int node;				/* processor number */
	RAPIDFILE *rpd;		/* may not be node's rpd */
	ANYPTR private;		/* private data */
	int last;				/* last block accessed in run */
	int length;			/* length of consecutive run */
	int sector;			/* most recent block accessed (-1 if none) */
{
    struct gapslocal *lgaps = (struct gapslocal *)private;
    struct gapsdata *gaps = lgaps->common;
    int block;

#ifdef GAPTRACE
    printf("Proc %d (%d) in GAPS LateNotify: last=%d, length=%d, sector=%d\n",
		 Vnode, node, last, length, sector);
    TraceFlush();
#endif					/* GAPTRACE */

    if (length > 0)
	 for (block = last-length+1; block <= last; block++)
	   NotifyOne(rpd, node, lgaps, gaps, block);
    if (sector >= 0)
	 NotifyOne(rpd, node, lgaps, gaps, sector);

    rpd->prefetch.available = rpd->inode_ptr->prefetch.available;
}

/* @SUBTITLE "NotifyOne: Notify for one block" */
static void
NotifyOne(rpd, node, lgaps, gaps, sector)
	RAPIDFILE *rpd;		/* may not be rpd for node */
	int node;
	struct gapslocal *lgaps;
	struct gapsdata *gaps;
	int sector;
{
    TICS start = rtc;

    if (gaps->random) {
	   ELOG_LOG(GAPSELOG_NOTIFY, sector);
	   Random(lgaps, (int)sector, node);
	   ELOG_LOG(GAPSELOG_RANDOM, rtc-start); /* Random() time */
	   TraceFlush();
    } else if (gaps->contin && lgaps->mysequential) {
	   /* check sequentiality */
	   ELOG_LOG(GAPSELOG_NOTIFY, sector);
	   if (Contin(rpd, lgaps, (int)sector, node))
		my_stats->correct++;
	   else
		FromSequential(rpd, lgaps, (int)sector, node);
	   /* Contin()/FromSequential() time */
	   ELOG_LOG(GAPSELOG_CONTIN, rtc-start);
	   TraceFlush();
    } else {
	   LockFIFO(lgaps->watch_fifo, 20);
	   ELOG_LOG(GAPSELOG_WFIFO, rtc-start); /* Watch_fifo wait time */
	   ELOG_LOG(GAPSELOG_NOTIFY, sector);
	   start = rtc;
	   if (gaps->random) {
		  /* shifted into Random mode while waiting */
		  UnlockFIFO(lgaps->watch_fifo);
		  Random(lgaps, (int)sector, node);
		  ELOG_LOG(GAPSELOG_RANDOM, rtc-start); /* Random() time */
		  TraceFlush();
	   } else if (lgaps->mysequential) {
		  /* shifted into Continuation mode while we waited. */
		  UnlockFIFO(lgaps->watch_fifo);
		  if (Contin(rpd, lgaps, (int)sector, node))
		    my_stats->correct++;
		  else
		    FromSequential(rpd, lgaps, (int)sector, node);
		  /* Contin()/FromSequential() time */
		  ELOG_LOG(GAPSELOG_CONTIN, rtc-start);
		  TraceFlush();
	   } else {
		  /* what we expected to do: Watching mode */
		  if (Watch(lgaps, (int)sector, node)) {
			 ELOG_LOG(GAPSELOG_WATCH, rtc-start); /* Watch() time */
			 start = rtc;
			 ToSequential(rpd, lgaps);
			 ELOG_LOG(GAPSELOG_TOSEQ, rtc-start); /* ToSeq.() time */
		  } else
		    ELOG_LOG(GAPSELOG_WATCH, rtc-start); /* Watch() time */

		  TraceFlush();
		  UnlockFIFO(lgaps->watch_fifo);
	   }
    }
}

/* @SUBTITLE "Notify - update for a new reference" */
/* This is an Entry Point */
/* This is called with each sector that is accessed. Currently, this function 
 * is protected with a FIFO entry lock so that only one process will be 
 * in the code at a time. The whole predictor is thus serialized. 
 * The return value has to be a boolean for consistency with others,
 * but the return value is irrelevant unless WANTED is defined. This 
 * is not allowed (see above), so our return value never means anything.
 *  If this process is in sequential mode, Contin() is called. Otherwise
 * Watch() is called. Rereferences are NOT ignored here, as Watch needs them.
 */
static boolean				/* irrelevant */
Notify(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct gapslocal *lgaps = (struct gapslocal *)(rpd->prefetch.private);
    struct gapsdata *gaps = lgaps->common;

    NotifyOne(rpd, Vnode, lgaps, gaps, sector);

    /* update the local copy of prefetch.available */
    rpd->prefetch.available = rpd->inode_ptr->prefetch.available;

    return(FALSE);			/* for WANTED, but is inconsistent */
}

/* @SUBTITLE "Random: handle random accesses" */
/* The purpose here is speedy processing of random-access patterns, 
 * with a crude-but-fast measure to detect a switch back from randomness
 * to a possibly sequential pattern. We look at orderedness: a sequential
 * pattern must have all processors doing ordered accesses; that is, 
 * each block number referenced is greater than or equal to the 
 * previous one. Jumpbacks occur when it is less. We come to random mode
 * because several processes experienced jumpbacks after very short 
 * ordered runs; this is a sign of randomness. We only go back to 
 * watching mode from here when a high percentage of processors 
 * have shown ordering. In a random pattern, it is unlikely that many
 * processors will simultaneously have longs strings of ordered references.
 */
static void 
Random(lgaps, block, node)
	struct gapslocal *lgaps;
	int block;
	int node;
{
    struct gapsdata *gaps = lgaps->common;

    if (block != lgaps->mylast) { /* ignore rereferences by same node */
	   if (lgaps->ordered) {
		  /* we were ordered */
		  if (block < lgaps->mylast) {
			 /* jumpback! may not be ordered any more */
			 /* This code seems backward, but it is the original intent. See 
			  * research notes for 4/4/90. 
			  */
			 if (lgaps->ordercount < SHORTORDER) {
				Atomic_add(&(gaps->num_ordered), -1);
				lgaps->ordered = FALSE;
			 }			/* else maybe it was just a long portion */
			 lgaps->ordercount = 1; /* this block */
		  } else {
			 /* in order; increment count */
			 lgaps->ordercount++;
		  }
	   } else {
		  /* we were not ordered */
		  if (block < lgaps->mylast) {
			 /* jumpback! but doesn't change our status */
			 lgaps->ordercount = 1; /* this block */
		  } else {
			 /* in order. Check count */
			 if (++(lgaps->ordercount) >= SHORTORDER) {
				/* becoming ordered */
				Atomic_add(&(gaps->num_ordered), 1);
				lgaps->ordered = TRUE;
			 }
		  }
	   }

	   lgaps->mylast = block;

	   if (gaps->num_ordered >= lgaps->nprocs * HIGHORDERPERC / 100) {
		  gaps->restart = FALSE; /* might be TRUE, which would be bad */
		  gaps->random = FALSE; /* drop out of random mode */
#ifdef GAPTRACE
		  printf("-> leaving random mode; %d are ordered\n", gaps->num_ordered);
#endif GAPTRACE
	   }
    }

#ifdef GAPTRACE
    printf("%2d %4d %d %c %d r\n", node, block, lgaps->ordercount, 
		 lgaps->ordered ? 'y' : 'n', gaps->num_ordered);
#endif
}

/* @SUBTITLE "ToRandom: Change from Watch mode to Random mode" */
/* This is used to convert to random mode. Done under protection
 * of watch_fifo.
 */
ToRandom(lgaps)
	struct gapslocal *lgaps;
{
    struct gapsdata *gaps = lgaps->common;
    REF *r;

    /* empty the context queue */
    while(Dequeue(lgaps->context, &r))
	 UsFree(r);
    /* reset these (probably don't need it) */
    if (!gaps->contin) {		/* there is a small chance of this */
	   gaps->minlast = -1;
	   gaps->start = -1;
    }

#ifdef GAPTRACE
    printf("-> entering random mode (%d are ordered)\n", gaps->num_ordered);
#endif

    /* go into random mode; also allows other processors to come in */
    gaps->random = TRUE;
}

/* @SUBTITLE "Watch: watch the pattern for sequentiality" */
/* This function is for tracking blocks when no sequentiality has been 
 * detected. The blocks are entered into the context queue, which
 * retains the most recent references for each processor, that may be 
 * combined to form a sequential portion. Once all processors have
 * contributed to the queue, two conditions are evaluated to check for
 * sequentiality: 1) completeness, when all of the queue entries less 
 * than the current minlast are contiguous, and 2) slope, when the 
 * ordered queue entries closely approximate a line. 
 */
static boolean
Watch(lgaps, block, node)
	struct gapslocal *lgaps;
	int block;
	int node;
{
    struct gapsdata *gaps = lgaps->common;
    int n;
    REF *dummy;
    int oldlast;
    int cutoff, sample;

    /* update this node's last block */
    oldlast = lgaps->mylast;
    lgaps->mylast = block;

    /* Jumpback? If so, we remove all of this process's entries 
	* from the queue. Orderedness is am important criteria for sequentiality. 
	* If enough processors fail to be ordered, we may switch to random
	* mode and exit here.
	*/
    if (block < oldlast)  {
	   /* used to be JumpBack(lgaps, node); */
	   ScanQueue(lgaps->context, NotMyREF, lgaps, node);
	   if (!gaps->contin) {
		  gaps->start = -1;
		  gaps->minlast = -1;
	   }
	   if (lgaps->ordered && lgaps->ordercount < SHORTORDER) {
		  /* we are no longer ordered */
		  Atomic_add(&(gaps->num_ordered), -1);
		  lgaps->ordered = FALSE;
	   }
	   lgaps->ordercount = 1;

	   /* if we fell out of ordering, go to random mode */
	   if (gaps->num_ordered < lgaps->nprocs * LOWORDERPERC / 100) {
		  ToRandom(lgaps);
		  return(FALSE);
	   }
    } else if (block > oldlast) {
	   /* jump forward */
	   lgaps->ordercount++;
	   if (!lgaps->ordered && lgaps->ordercount >= SHORTORDER) {
		  /* we are now ordered */
		  Atomic_add(&(gaps->num_ordered), 1);
		  lgaps->ordered = TRUE;
	   }
    } /* rereference: no change to order count */

    /* Add to queue. If queue is full, flush the oldest block. */
    if (ForceEnqueue(lgaps->context, NewRef(block, node), &dummy))
	 UsFree(dummy);

    /* All processors must be participating. */
    n = InQueue(lgaps->context);
    if (n <= lgaps->nprocs || gaps->contin) {
	   Trace(node, block, n, 0, 0, 0.0, 'n');
	   return(FALSE);
    }
    /* note here that gaps.sequential[i] is FALSE for all i */

    /* Finalize() may have specified a quick-restart */
    if (gaps->restart) {
	   Trace(node, block, n, 0, 0, 0.0, 'y');
	   /* we need to assign start, minlast, maxjump */
	   return(TRUE);
    }

    /* Now we check for completeness: all of the blocks in the queue
	* that are less than minlast are contiguous in the file.
	*/
    if (!CheckComplete(node, lgaps, block, &cutoff, &sample)) {
	   Trace(node, block, n, cutoff, sample, 0.0, 'n');
	   return(FALSE);
    }

    /* So, completeness has been established. It is possible for 
	* the reference pattern to be complete without having a definite 
	* direction. So now we check the slope (or, more accurately, the
	* coefficient of determination). The block sequence is fit
	* to a straight line; the quality of this fit must be above a certain 
	* tolerance to be considered sequential.
	*/
    if (!CheckSlope(lgaps)) {
	   Trace(node, block, n, cutoff, sample, gaps->coef, 'n');
	   return(FALSE);
    }

    /* Now, we have confirmed that the pattern is sequential. 
	* We will enter Continuation mode, instead of Watching mode, 
	* to follow the progress of the pattern. But first, we 
	* do some more computation on the context queue to find the
	* correct values of maxdist and maxjump.
	*/
    FindMaxjump(lgaps, gaps);

    Trace(node, block, n, cutoff, sample, gaps->coef, 'y');

    return(TRUE);
}

/* @SUBTITLE "CheckComplete: Check for completeness." */
static boolean
CheckComplete(node, lgaps, block, cut, samp)
	int node;
	struct gapslocal *lgaps;
	int block;			/* block in this reference */
	int *cut, *samp;		/* Output: cutoff, sample */
{
    struct gapsdata *gaps = lgaps->common;
    int b;				/* a block number */
    int sample = 0;			/* size of sample <= cutoff */
    int *used;				/* [ContextSize] used counts in sample area */
    int cutoff;			/* minlast */
    boolean complete;		/* is it complete? */
    REF *r;				/* a reference off queue */
    int n;				/* number in queue */
    int i;				/* loop index */
    int minlast = gaps->minlast;

    /* Compute minlast as cutoff. */
    cutoff = *lgaps->last[0];
    for (i = 1; i < lgaps->nprocs; i++) {
	   b = *lgaps->last[i];
	   if (b < cutoff)
		cutoff = b;
    }

    if (cutoff <= 0) {
	   *cut = cutoff;
	   *samp = 0;
	   return(FALSE);
    }

    /* Now we check for completeness. If completeness was noticed before, 
	* the cutoff was stored as gaps->minlast. If the cutoff has not changed, 
	* completeness still holds. Otherwise, we may have to update the
	* completeness.
	*/
    if (cutoff == minlast) {
	   complete = TRUE;
    } else {
	   /* initialize used table to all zeroes */
	   used = (int *)calloc(ContextSize, sizeof(int));

	   /* We look through the entire queue, checking each block against the
	    * cutoff. If the block is less than the new cutoff, but larger 
	    * than the old cutoff, it is part of the sample that is interesting.
	    *  This sample may not fit in the used[] array; if so, then it
	    * cannot be complete, since used[] can hold the whole queue if
	    * it were all one complete sample. Otherwise, we count the occurrences
	    * of each block in the sample.
	    */
	   n = InQueue(lgaps->context);
	   for (i = 0; i < n; i++) {
		  Peekqueue(lgaps->context, i, &r);
		  b = r->block;
		  if (b <= cutoff && b > minlast) {
			 if (cutoff-b < ContextSize) {
				sample++;
				used[cutoff-b]++;
			 } /* else: ignore */
		  }
	   }
	   
	   /*  It is possible that there
	    * is still a gap in the blocks in the sample. So here we 
	    * trace backward from the cutoff to the first unused block,
	    * counting occurrences.  This unused block
	    * represents a gap before which there were other used blocks.
	    * We don't care about those; as long as the complete set is 
	    * large enough. This also gives us the start of the portion.
	    * If we decide it is complete, then we can record the start
	    * of the portion to be the start of the complete area. If 
	    * we are simply updating completeness, then the start was
	    * already recorded and we need only record the cutoff. 
	    */
	   complete = FALSE;	   
	   if (minlast >= 0 || sample >= MINSAMPLE) {
		  /* verify completeness more carefully */
		  for (i = 0, sample = 0; i < ContextSize && used[i] > 0; i++)
		    sample += used[i];
		  /* # uses of each block comes easily here */
		  gaps->blockuses = ceil((float)sample / i);
		  if (minlast >= 0 && cutoff - i > minlast) {
			 /* extension failed */
			 if (i >= MINLEN && sample >= MINSAMPLE) {
				/* but there's enough here for a new area */
				complete = TRUE;
				gaps->start = cutoff-i+1;
				gaps->minlast = cutoff;
			 }
		  } else if (minlast >= 0 || (i >= MINLEN && sample >= MINSAMPLE)) {
			 /* normal case: extend/create completeness area */
			 complete = TRUE;
			 if (gaps->start < 0)
			   gaps->start = cutoff-i+1;
			 gaps->minlast = cutoff;
		  }
	   }
	   
	   cfree(used);
    }

    if (!complete) {
	   gaps->start = -1;
	   gaps->minlast = -1;
    }

    *cut = cutoff;
    *samp = sample;
    return(complete);
}

/* @SUBTITLE "CheckSlope: update the recent-blocks slope" */
/* We treat the context queue as a function, with x represented by count,
 * and y represented by the block number. We are interested in the slope 
 * of this function, hopefully a line. Although we compute the slope,
 * we check the coefficient of determination (see below) to determine
 * whether the function fits the line well. We only include points from
 * the context that are >= the start of the complete area. Other points
 * are assumed to be part of an older portion.
 */
static boolean				/* passes as sequential? */
CheckSlope(lgaps)
	struct gapslocal *lgaps;
{
    struct gapsdata *gaps = lgaps->common;
    int n;				/* number in queue */
    int block;		/* a block number */
    int sumx, sumx2, sumxy, sumy; /* significant sums */
    float slope;			/* the slope */
    float inter;			/* the intercept */
    float y;				/* predicted y value */
    float ymean;			/* the mean block number */
    float sum1, sum2;		/* two parts to the coef of determination */
    REF *r;				/* a reference */
    int start = gaps->start;	/* start of portion, we think */
    int x;				/* x value */
    int i;

    n = InQueue(lgaps->context);
    /* we already know n >= nprocs */

    /* find significant sums */
    sumx = sumx2 = sumxy = sumy = 0;
    for (i = 0, x = 0; i < n; i++) {
	   Peekqueue(lgaps->context, i, &r);
	   block = r->block;
	   if (block >= start) {
		  sumxy += x*block;
		  sumy += block;
		  x++;
	   }
    }
    /* x is now the number of points included */
    sumx = (x-1) * x / 2;	/* sum of i, i=0,1,...(x-1) */
    sumx2 = (x-1)*x*(2*x-1)/6; /* sum of sqr(i), i=0,1,...(x-1) */

    /* find the slope */
    slope = (float)(x * sumxy - sumx*sumy) / (x * sumx2 - sqr(sumx));

    /* negative slope means we are NOT sequentially increasing */
    if (slope < 0) {
	   gaps->coef = 0;
	   return(FALSE);
    }

    /* @PAGE */
    /* Very large slopes imply that either the context contains several
	* portions that are in increasing order, or that the processes are
	* reading many blocks in each self-scheduled chunk. The latter is
	* a legitimate global pattern, but is hard to properly distinguish 
	* other behavior. It can also be handled by a local predictor. So
	* we screen large slopes here. 
	*/
    if (slope > MAXSLOPE) {
	   gaps->coef = 0;
	   return(FALSE);
    }

    /* we no longer compute maxjump here */
/*    gaps->maxjump = ceil(slope * lgaps->nprocs); */
/*    gaps->maxjump = maxjump * 3; */

    /* Compute the coefficient of determination:
	* A measure of the deviation from the linear fit.
	* See Trivedi Pages 546--548.
	*/
    ymean = (float) sumy / x;
    inter = ymean - slope * sumx / x;
    y = inter;				/* start at (0,inter) */
    sum1 = sum2 = 0;
    for (i = 0; i < n; i++) {
	   Peekqueue(lgaps->context, i, &r);
	   block = r->block;
	   if (block >= start) {
		  sum1 += sqr(y - ymean);
		  sum2 += sqr(block - ymean);
		  y += slope;
	   }
    }
    gaps->coef = sum1/sum2;

    /* 0 <= Coef <= 1, and is better near 1. */
    return(gaps->coef >= TOLERANCE);
}

/* @SUBTITLE "FindMaxjump: Compute maxjump and maxdist" */
/* We compute maxjump from the average run length on all processors. We
 * consider all runs within the context queue, ignoring the first and last
 * run on every processor, since they may be only partial runs. A run is
 * a consecutive set of contiguous block references by the same processor.
 * This gives us the chunksize for chunks of one block or more. 
 * Smaller chunksizes are detected by the reference count for each block 
 * in the old zone (courtesy of CheckComplete). Then maxjump is 
 * nprocs*chunksize. Then we compute the maxdist value, which is used 
 * whenever the portions are irregular. This value is empirically 
 * determined from the chunksize. Note that for large chunks the 
 * best policy is maxdist=0.
 *  
 */
static void
FindMaxjump(lgaps, gaps)
	struct gapslocal *lgaps;
	struct gapsdata *gaps;
{
    int *last;				/* [nprocs] last block referenced */
    int *first;			/* [nprocs] first block in run */
    int nprocs = lgaps->nprocs;
    int runlength;			/* sum of run lengths */
    int runcount;			/* number of runs counted */
    int maxjump;			/* possible value */
    float chunksize;		/* in blocks */
    int node;
    int i,n;
    int b;
    REF *r;

    /* initialize last[] and first[] */
    last = (int *)malloc(nprocs * sizeof(int));
    first = (int *)malloc(nprocs * sizeof(int));
    for (node = 0; node < nprocs; node++) {
	   last[node] = -10;
	   first[node] = -10;
    }
    runlength = 0;			/* sum of run lengths */
    runcount = 0;			/* number of runs counted */

    n = InQueue(lgaps->context);
    for (i = 0; i < n; i++) {
	   Peekqueue(lgaps->context, i, &r);
	   b = r->block;
	   node = r->node;

	   if (last[node] >= 0 && last[node]+1 != b) {
		  /* end of run */
		  if (first[node] < 0)
		    first[node] = b; /* ignore first partial run */
		  else {
			 /* count this run */
			 runlength += last[node] - first[node] + 1;
			 runcount++;
			 first[node] = b;
		  }				  
	   }					/* else run continues */
	   last[node] = b;
    }

    /* We ignore all current runs, as they may be incomplete.
	* But we may have some runs to use for finding avg chunk size.
	*/
    if (runcount > 0)
	 chunksize = (float) runlength / runcount;
    else
	 chunksize = 1;

    /* This method does not tell us when chunksize is less than one. 
	* For this we use the number of references to a block; if greater
	* than 1, we consider the chunksize to be smaller than a block.
	*/
    if (chunksize <= 1 && gaps->blockuses > 1)
	 chunksize = 1.0 / gaps->blockuses;

    /* Now maxjump is known! */
    maxjump = ceil(nprocs * chunksize);
    gaps->maxjump = maxjump;

    /* Now decide maxdist. Note that we set gaps->distance; later
	* we decide whether to use this for maxdist. (In certain situations
	* maxdist is infinity).
	*/
    if (chunksize < 1)
	 gaps->distance = nprocs;
    else if (chunksize == 1)
	 gaps->distance = nprocs * 3 / 4;
    else if (chunksize <= 3)
	 gaps->distance = nprocs / 4;
    else
	 gaps->distance = 0;
}

/* @SUBTITLE "ToSequential: change into sequential mode" */
/* When the Watcher notices sequentiality, this function is called to 
 * send all processors into sequential mode. This also involves
 * breaking down the old data structure (the queue) and building the
 * new one (in the sector map table, conveniently). 
 */
static void
ToSequential(rpd, lgaps)
	RAPIDFILE *rpd;		/* rpd and lgaps may be for different procs */
	struct gapslocal *lgaps;
{
    struct gapsdata *gaps = lgaps->common;
    int i;
    int block;
    REF *r;
    int start;
    int minlast;
    int maxlast;
    int last;
    SECTOR_MAP_ENTRY *sme;
    int nprocs = lgaps->nprocs;

    /* 'start', 'minlast', and 'maxjump'' were set by watch(),
	* except when restart is true. So, we compute minlast and start 
	* here again, ignoring start if restart is false. maxjump will
	* just stay the same if restart is true.
	*/

    /* Set the last counters in the table. The last count indicates
	* how many processes have that block as their most-recently-referenced
	* block. The count will be decremented as the processes move on.
	*/
    for (i = 0; i < nprocs; i++) {
	   last = *lgaps->last[i];
	   sme = RT_smeForSector(rpd, last);
	   sme->misc.lastcount++;
	   if (last > maxlast || i == 0)
		maxlast = last;
	   if (last < minlast || i == 0)
		minlast = last;

	   *lgaps->sequential[i] = TRUE;
    }
    gaps->maxlast = maxlast;
    gaps->minlast = minlast;

    gaps->num_sequential = nprocs;

    /* Copy queue to used bits in the table. This indicates which blocks
	* have been used. These will be cleared as the minlast moves through
	* the table, and are used for detecting gaps in the sequence, and
	* for determining which blocks might be prefetched. 
	*/
    start = -1;
    while(Dequeue(lgaps->context, &r)) {
	   block = r->block;
	   UsFree(r);
	   if (block >= minlast) {
		  sme = RT_smeForSector(rpd, block);
		  SS_SET(sme, SS_USED); /* usedcount++ */
	   }
	   if (start < 0 || block < start)
		start = block;
    }
    if (gaps->restart)		/* set the start, if quick restart */
	 gaps->start = start;

    ELOG_LOG(GAPSELOG_START, gaps->start);

#ifdef GAPTRACE
    printf("-> starting sequential from %d, maxjump %d, restart %c\n", 
		 gaps->start, gaps->maxjump, gaps->restart ? 'y' : 'n');
#endif

    /* Record the new portion beginning now. */
    NewPortion(lgaps, gaps->start);

    /* And turn on prefetching, beginning after maxlast. We used to start 
 	* at minlast, and prefetch in the active zone; however, when just
 	* starting, there may be holes in the active zone that were really 
 	* used; this avoids marking mistakes for blocks that were already used. 
 	* Later, once the active zone extends past the current maxlast, we 
 	* will be prefetching in the active zone.
	*/
    StartPrefetch(rpd, gaps, maxlast);

    gaps->num_ordered = nprocs; /* to help when we leave contin mode */
    gaps->restart = FALSE;	/* to avoid restarting too soon */

    /* let people into continuation mode */
    gaps->contin = TRUE;	
}


/* @SUBTITLE "Contin: watch the pattern as it continues to be sequential" */
/*   This is Continuation mode. Once sequentiality is noted, we shift modes and
 * track the sequentiality in a much more efficient manner. Thus, Notify
 * calls this function instead of Watch(). We return FALSE here if we 
 * fall out of sequentiality.
 *   This function, and its children, are executed concurrently
 * by many processors.
 *   The efficiency is possible since we know the pattern will be active
 * in a compact area, and we don't need the ordering information from the
 * queue so much as easy access to minlast/maxlast and completeness
 * information.
 *   The sector map table is used to store our information. We record
 * the lastcount, a count for each block of the number of processes that
 * have that block as their most-recently accessed, and a "used" bit for each 
 * block.
 *   We watch for jumpbacks, sure signs of sequentiality breaking, and 
 * forward jumps that are too far to be explained. In the process of
 * checking the slope we obtained 'maxjump', the furthest any process
 * should extend the maxlast at any one jump. Forward jumps longer
 * than maxjump past maxlast are also taken to be a loss in sequentiality
 * (this is modified when the portions are highly regular, and we can take
 * jumps over portion skip in stride; see ExtendPortion). 
 *   We must also maintain minlast and maxlast in order to check for
 * completeness failures (which arise due to small jumps and portion skips). 
 * Minlast is updated whenever the lastcount of a block goes to zero, and
 * that block was minlast. To find the new minlast, we sequentially scan
 * the table until another block with a nonzero lastcount is found. In the
 * process, we can look for blocks that were not used. Any found are
 * a completeness failure. We let JumpGap handle this case.
 */
static boolean
Contin(rpd, lgaps, block, node)
	RAPIDFILE *rpd;		/* rpd and lgaps may be for different procs */
	struct gapslocal *lgaps;
	int block;
	int node;
{
    struct gapsdata *gaps = lgaps->common;
    int oldlast;
    SECTOR_MAP_ENTRY *sme;
    boolean ok = TRUE;

    if (block == lgaps->mylast) {  /* ignore rereferences by same node */
#ifdef GAPTRACE
	   printf("%2d %4d %d %d  y\n",
			node, block, gaps->minlast, gaps->maxlast);
#endif
	   return(TRUE);
    }
    
    oldlast = lgaps->mylast;
    lgaps->mylast = block;

    if (block < oldlast) {
#ifdef GAPTRACE
	   printf("-> %d jumpback from %d on node %d\n", 
			block, oldlast, node);
#endif
	   sme = RT_smeForSector(rpd, oldlast);
	   Atomic_add(&(sme->misc.lastcount), -1); /* take us out */
	   ELOG_LOG(GAPSELOG_END, 1);
	   return(FALSE);		/* a jump-back: cancel sequential */
    }

    /* now we know that (block > oldlast) */

    sme = RT_smeForSector(rpd, block);
    SS_SET(sme, SS_USED);	/* block.usedcount++ */
    
    if (block > gaps->maxlast) {
	   /* update maxlast */
	   LockFIFO(lgaps->maxlast_fifo, 10);
	   if (block > gaps->maxlast) {
		  if (ExtendPortion(gaps, oldlast, block)) {
			 gaps->maxlast = block;
			 SoftLimitPrefetch(gaps, block + gaps->maxdist);
			 UnlockFIFO(lgaps->maxlast_fifo);
		  } else {
			 UnlockFIFO(lgaps->maxlast_fifo);
#ifdef GAPTRACE
			 printf("-> %d jump too large %d > %d\n",
				   block, block-gaps->maxlast, gaps->maxjump);
#endif
			 sme = RT_smeForSector(rpd, oldlast);
			 Atomic_add(&(sme->misc.lastcount), -1); /* take us out */

			 ELOG_LOG(GAPSELOG_END, 0);
			 return(FALSE);
		  }
	   } else
		UnlockFIFO(lgaps->maxlast_fifo);
    }

    sme = RT_smeForSector(rpd, block);
    Atomic_add(&(sme->misc.lastcount), 1); /* block.lastcount++ */
    
    /* @PAGE */
    /* Here we try to update minlast. If our old last was the minlast, 
	* and no one else currently has that as their last, then 
	* the minlast must change. We have responsibility for updating it. 
	* There may be several processes at this point trying to do
	* this, so we need a lock. However, we do not wait for the
	* lock; it is sufficient for one process to get the lock
	* and do the job. Note also, that since minlast is incremented
	* as the job is performed, a race condition may lead us to
	* believe that we should update minlast, while another
	* process is in the inner loop. The lock helps to insure
	* that only one of us will do it.
	*  The prefetch code may be watching minlast change too. It avoids
	* prefetching during JumpGap, though it may sneak some through.
	* If they are not appropriate, they will be caught as mistakes.
	*/
    sme = RT_smeForSector(rpd, oldlast);
    Atomic_add(&(sme->misc.lastcount), -1); /* decrement oldlast count */
    if (gaps->minlast == oldlast && sme->misc.lastcount == 0) {
	   /* update minlast */
	   while(gaps->minlast == oldlast && ok)
		/* keep trying to update minlast */
		if (TryLock(lgaps->minlast_lock)) {
		    /* we got a lock on the process; may already be done */
		    if (gaps->minlast == oldlast)
			 /* nope; we loop to update minlast */
			 while (sme->misc.lastcount == 0 && ok) {
				if (SS_ISCLEAR(sme, SS_USED)) { /* i.usedcount==0 */
				    /* completeness failure */
				    if (gaps->num_sequential < lgaps->nprocs)
					 ok = FALSE; /* leave continuation mode */
				    else
					 JumpGap(rpd, lgaps); /* changes minlast */
				} else {
				    SS_CLEAR(sme, SS_USED | SS_MARKED);
				    gaps->minlast++; /* ok due to lock */
				}
				sme = RT_smeForSector(rpd, gaps->minlast);
			 }
		    /* in any case, release lock */
		    UNLOCK(lgaps->minlast_lock);
 		    
 		    /* For regular run length, check the expected end of portion */
 		    if (gaps->runlen > 0 && gaps->minlast > gaps->exp_end) {
 			   /* we overshot the expected end of portion */
 			   LockFIFO(lgaps->maxlast_fifo, 5);
 			   gaps->runlen = 0;	/* cancel the regularity for now */
 			   LimitPrefetch(gaps, lgaps->size-1, 
 						  gaps->maxlast + gaps->maxdist);
 			   UnlockFIFO(lgaps->maxlast_fifo);
 		    }
		}
    }

#ifdef GAPTRACE
    printf("%2d %4d %d %d  y\n",
		 node, block, gaps->minlast, gaps->maxlast);
#endif

    if (!ok)
	 ELOG_LOG(GAPSELOG_END, 2);

    return(ok);
}

/* @SUBTITLE "FromSequential: change out of sequential mode" */
/* This function handles any break from sequentiality. It is called for
 * each processor as it leaves the portion, either due to a jumpback or 
 * a too-long jump forward. [Of course, if not all processors leave the 
 * portion in this way, we're stuck.]  First thing we do is to limit the
 * prefetch to maxlast; prefetching to there is safe. Otherwise, 
 * we set our sequentiality flag to FALSE, and enqueue this block on the 
 * context queue to be part of a future portion.
 *  Many processes may execute this concurrently.
 */
static void
FromSequential(rpd, lgaps, block, node)
	RAPIDFILE *rpd;		/* rpd and lgaps may be for different procs */
	struct gapslocal *lgaps;
	int block;
	int node;
{
    struct gapsdata *gaps = lgaps->common;
    int n;
    REF *dummy;

    lgaps->mysequential = FALSE;
#ifdef GAPTRACE
    printf("-> process %d ending sequential\n", node);
#endif

    gaps->maxdist = 0;		/* to keep prefetching below maxlast */
    LockFIFO(lgaps->maxlast_fifo, 5);
    SoftLimitPrefetch(gaps, gaps->maxlast);/* stop prefetching */
    UnlockFIFO(lgaps->maxlast_fifo);
    
    /* Add this latest block and start over. */
    if (ForceEnqueue(lgaps->context, NewRef(block, node), &dummy))
	 UsFree(dummy);

    lgaps->ordered = TRUE;
    lgaps->ordercount = 1;

    /* If we are the last to leave, clean up */
    if (Atomic_add(&(gaps->num_sequential), -1) == 1)
	 Finalize(rpd, gaps);

#ifdef GAPTRACE
    n = InQueue(lgaps->context);
    Trace(node, block, n, 0, 0, 0.0, 'n');
#endif
}

/* @SUBTITLE "Finalize: Finish off continuation mode" */
/* This function makes the final transition from Continuation mode to 
 * Watching mode, as the last processor leaves continuation mode.
 * We have to clear the table of all the used bits and last counts,
 * process all mistakes, stop prefetching, and end the portion. 
 * We also reset all state to be ready for Watching mode.
 *
 * Note that only one process is active here.
 */
static void
Finalize(rpd, gaps)
	RAPIDFILE *rpd;		/* rpd and lgaps may be for different procs */
	struct gapsdata *gaps;
{
    int i;
    int end;
    SECTOR_MAP_ENTRY *sme;
    int maxlast = gaps->maxlast;
    int oldstat;

    /* get that prefetching stopped! */
    StopPrefetch(rpd);

    /* clean out the array */
    end = gaps->minlast;
    sme = RT_smeForSector(rpd, end);
    while (end <= maxlast && SS_ISSET(sme, SS_USED)) {
	   SS_CLEAR(sme, SS_USED | SS_MARKED); /* usedcount = 0 */
	   sme->misc.lastcount = 0;
	   end++;
	   sme = RT_smeForSector(rpd, end);
    }
    for (i = end; i <= maxlast; i++) {
	   sme = RT_smeForSector(rpd, i);
	   /* clear used bit, and if not used, process mistake */
	   oldstat = SS_CLEAR(sme, SS_USED | SS_MARKED);
	   if (!(oldstat & SS_USED) && (oldstat & SS_MARKED))
		RT_PrefetchMistake(rpd, i); /* it was not used, and was marked */
	   sme->misc.lastcount = 0;
    }
    end--;				/* first loop overshoots by one */

    Mistakes(rpd, gaps, maxlast+1);
    EndPortion(gaps, end);

    gaps->minlast = -1;
    gaps->start = -1;

    /* make it a fast one, if this last portion was reasonably long */
    gaps->restart = (gaps->prev_runlen >= RESTARTCOUNT * gaps->nprocs);

    gaps->contin = FALSE;	/* let Watching mode begin */
}    

/* @SUBTITLE "NewPortion: Starting a new portion" */
/* This is the first of several functions for identifying and
 * tracking portions. We record the skip from the previous portion, 
 * if any, and look for regularity. Also, since this occurs at the 
 * start of sequentiality, we use any regularity in the portion
 * length or skip to set up limits for prefetching.
 *
 * This runs alone, from ToSequential, or alone, during minlast update.
 * Thus it is protected by either the watch_fifo lock or the minlast_lock.
 */
static void
NewPortion(lgaps, start)
	struct gapslocal *lgaps;
	int start;			/* starting block number */
{
    struct gapsdata *gaps = lgaps->common;
    int skip = start - gaps->prev_end;
    int runlen = 0;			/* default: irregular run length */
    int skiplen = 0;		/* default: irregular skip length */
    int limit = lgaps->size - 1; /* default hardlimit: EOF */
    int exp_end = 0;		/* expected end of portion */

    /* check skip */
    if (gaps->same_skiplen == 0 || skip != gaps->prev_skiplen) {
	   gaps->same_skiplen = 1; 
	   gaps->prev_skiplen = skip;
    } else
	 gaps->same_skiplen++;

    /* now look for regularity */
    if (gaps->same_runlen >= RunRep) {
	   /* run length is regular */
	   runlen = gaps->prev_runlen;
	   exp_end = start + runlen - 1;
	   if (gaps->same_skiplen >= SkipRep) {
		  /* skip length is regular */
		  skiplen = gaps->prev_skiplen;
#ifdef GAPTRACE
		  printf("-> skip is regular %d\n", skip);
#endif
	   } 
	   /* With regular run length, but no positive regular skip, cut 
	    * off prefetching at end of portion.
	    */
	   if (skiplen <= 0)
		limit = exp_end;
    }

    /* Now compute the value of maxdist.
	*  The prefetch limit is determined from maxdist; maxdist is distance for 
	* irregular portions, or essentially infinite for regular portions.
	* It is also set to zero when we are shutting down continuation mode.
	*/
    if (gaps->num_sequential == lgaps->nprocs)
	 if (gaps->portion1 || skiplen > 0)
	   gaps->maxdist = gaps->size; /* infinite */
	 else 
#ifdef USEPARM
	   gaps->maxdist = gaps->parm; /* limited lookahead */
#else
    	   gaps->maxdist = gaps->distance;	/* limited lookahead */
#endif /* USEPARM */
    else
	 gaps->maxdist = 0;	/* active zone only */

    /* Now store the new values where the prefetchers can see them */
    LockFIFO(lgaps->pchange_fifo, 5);
    gaps->start = start;
    gaps->exp_end = exp_end;
    gaps->runlen = runlen;
    gaps->skiplen = skiplen;
    UnlockFIFO(lgaps->pchange_fifo);

    LockFIFO(lgaps->maxlast_fifo, 5);
    LimitPrefetch(gaps, limit, gaps->maxlast + gaps->maxdist);
    UnlockFIFO(lgaps->maxlast_fifo);
}

/* @SUBTITLE "ExtendPortion: Record this sequential portion" */
/*  If the portion length and skip are regular, then we allow
 * jumps over the skip if they still meet the maxjump criteria. 
 * The skip must be positive and regular, and the length must be regular. 
 * The previous reference must be in the current portion, and the
 * current reference in the expected next portion. 
 *  Then, we allow a jump of maxjump plus the skip length.
 *  If the current reference is in the area we expect to skip, the 
 * regularity is assumed broken and we reset the flag. 
 * Otherwise we do a simple maxjump check on the jump.
 *  In irregular portions, we may allow the jump on faith that it will
 * remain sequential. This is risky, but usually beneficial. We record
 * the jump so the prefetches will avoid the jump.
 *  This function runs alone, under the maxlast_fifo lock.
 * Note there may be concurrent changes in skiplen and exp_end,
 * due to NewPortion activity. These should be ok.
 */

static boolean				/* is this a valid extension? */
ExtendPortion(gaps, last, block)
	struct gapsdata *gaps;
	int last;				/* last block ref'd by this proc */
	int block;			/* the most recent reference */
{
    /* read these things once */
    int maxlast = gaps->maxlast;
    int maxjump = gaps->maxjump;
    int nprocs = gaps->nprocs;
    int skiplen = gaps->skiplen;

    if (skiplen > 0) {
	   /* regular, and positive skip */
	   int exp_end = gaps->exp_end;
	   if (block > exp_end) {
		  if (last <= exp_end)
		    if (block >= exp_end + skiplen)
			 if (block - maxlast < maxjump + skiplen)
			   return(TRUE);
			 else
			   return(FALSE);
		    else
			 gaps->skiplen = 0; /* regularity broken */
	   }
	   /* just a plain maxjump check */
	   if (block - maxlast <= maxjump)
		return(TRUE);
	   else
		return(FALSE);
    } else
	 /* Irregular skip length, anyway */
	 /* just a plain maxjump check */
	 if (block - maxlast <= maxjump)
	   return(TRUE);
	 else {
		/* The jump appears to be too large. We try to ride over it. */
		/* We won't ride gap if other processors have given up. */
		/* Test length of portion to determine our faith in sequentiality */
		if (gaps->num_sequential < nprocs
		    || maxlast - gaps->start < RESTARTCOUNT * nprocs) {
		    /* The portion was short; don't believe sequentiality */
		    return(FALSE);
		} else {
		    /* Fine, the portion was long enough to believe */
		    /* record this skip for the prefetchers */
		    LockFIFO(gaps->pchange_fifo, 5);
		    if (gaps->prefetch_skip) {
			   gaps->endskip = block;
#ifdef GAPTRACE
		    printf("-> Extend prefetch skip: from %d to %d\n", 
				 gaps->startskip, block);
#endif GAPTRACE
		    } else {
			   gaps->startskip = maxlast;
			   gaps->endskip = block;
			   gaps->prefetch_skip = TRUE;
#ifdef GAPTRACE
		    printf("-> Setup prefetch skip: from %d to %d\n", 
				 maxlast, block);
#endif GAPTRACE
		    }
		    UnlockFIFO(gaps->pchange_fifo);
#ifdef USEPARM
		    gaps->maxdist = gaps->parm;	/* change maxdist */
#else
		    gaps->maxdist = gaps->distance; /* change maxdist */
#endif /* USEPARM */
		    return(TRUE);	/* maxlast becomes block */
		}
	 }
}

/* @SUBTITLE "JumpGap: jump over an unused gap in the portion" */
/* This function is used to jump over non-used gaps in the table.
 * These gaps are either a completeness failure or the skip
 * between portions, depending on how you look at it. The gap must 
 * define a new portion, so one thing we do is call EndPortion and
 * NewPortion. We also scan the gap looking for mistakes and the
 * beginning of the next portion (the end of the gap).  It is
 * considered a completeness failure only if it was unexpected.
 *
 * This runs alone, under the minlast lock.
 */
static void
JumpGap(rpd, lgaps)
	RAPIDFILE *rpd;		/* rpd and lgaps may be for different procs */
	struct gapslocal *lgaps;
{
    struct gapsdata *gaps = lgaps->common;
    SECTOR_MAP_ENTRY *sme;
    int block;
    int end = gaps->minlast;

    /* This flag will stop prefetching temporarily */
    gaps->jumpgap = TRUE;

    /* As we enter, minlast is a non-used block. Thus, the portion
	* ended with the previous block.
	*/
    EndPortion(gaps, gaps->minlast-1);

    /* it would be nice to find a way to optimize when we expect regularity */
    block = gaps->minlast;
    sme = RT_smeForSector(rpd, block);
    while (SS_ISCLEAR(sme, SS_USED)) {	/* minlast.usedcount==0 */
	   OneMistake(rpd, block, sme);
	   block = ++gaps->minlast;		/* ok, due to lock */
	   sme = RT_smeForSector(rpd, block);
    }
    /* block (minlast) is now the next used block */

    NewPortion(lgaps, block);

    gaps->jumpgap = FALSE;

#ifdef GAPTRACE
    if (gaps->skiplen == 0)
	 printf("-> completeness failure: skipping blocks %d-%d\n", 
		   end, gaps->minlast-1);
#endif
}


/* @SUBTITLE "EndPortion: Record this sequential portion" */
/* Here we record the end of a portion, remembering its length
 * and comparing the length to previous portions.
 *  This runs alone, called either by Finalize, or under the minlast
 * lock. 
 */
static void
EndPortion(gaps, end)
	struct gapsdata *gaps;
	int end;
{
    int length, skip;

    length = end - gaps->start + 1;
    skip = gaps->start - gaps->prev_end;
    gaps->prev_end = end;
    gaps->portion1 = FALSE;

#ifdef GAPTRACE
    printf("-> finished portion: %d to %d length %d skip %d\n", 
		 gaps->start, end, length, skip);
#endif

    if (gaps->same_runlen == 0 || length != gaps->prev_runlen) {
	   gaps->same_runlen = 1;		/* reset length */
	   gaps->prev_runlen = length;
	   gaps->same_skiplen = 0;	/* reset skip */
	   gaps->prev_skiplen = 0;
    } else {
	   gaps->same_runlen++;		/* same length */
    }

#ifdef GAPTRACE
    if (gaps->same_runlen >= RunRep) {
	   printf("-> portion length is regular %d\n", length);
    }
#endif

    /* we look at the skip at the start of the portion */
}

/* @SUBTITLE "Start/Stop/LimitPrefetch: turn on and off" */
/* This turns prefetching on, to begin after "block."  */
/* Always runs alone, from ToSequential */
static void
StartPrefetch(rpd, gaps, block)
	RAPIDFILE *rpd;		/* rpd and lgaps may be for different procs */
	struct gapsdata *gaps;
	int block;
{
    QH giveback = gaps->GiveBackQ;
    int dummy;

    gaps->nextpref = block;	/* will start after block */
    gaps->prefetch_skip = FALSE;
    gaps->endskip = 0;
    /* we assume prefetch limits have already been set up */

    /* empty the give-back queue */
    while(Dequeue(giveback, &dummy))
	 ;

    rpd->inode_ptr->prefetch.available = TRUE;
}

/* This turns prefetching off. */
/* Run alone from Finalize */
static void
StopPrefetch(rpd)
	RAPIDFILE *rpd;		/* rpd and lgaps may be for different procs */
{
    rpd->inode_ptr->prefetch.available = FALSE;
}

/* This puts a hard and soft upper limit on prefetching. */
/* This done under maxlast lock */
static void
LimitPrefetch(gaps, hardlimit, softlimit)
	struct gapsdata *gaps;
	int hardlimit;			/* block number */
	int softlimit;			/* block number */
{
    gaps->hardlimit = hardlimit;
    if (softlimit <= hardlimit)
	 gaps->limit = softlimit;
    else
	 gaps->limit = hardlimit;
}

/* This puts an upper limit on prefetching. */
/* This done under maxlast lock. Too bad that it needs a lock. */
static void
SoftLimitPrefetch(gaps, softlimit)
	struct gapsdata *gaps;
	int softlimit;			/* block number */
{
    if (softlimit <= gaps->hardlimit)
	 gaps->limit = softlimit;
}


/* @SUBTITLE "Mistakes: Handle mistakes" */
/* This scans the table, starting at "begin" and ending at the
 * prefetching high-water mark (the last block prefetched), to 
 * find and process mistakes. 
 *  Run alone, from Finalize, after prefetching has stopped.
 */
static void 
Mistakes(rpd, gaps, begin)
	RAPIDFILE *rpd;		/* rpd and lgaps may be for different procs */
	struct gapsdata *gaps;
	int begin;			/* first block that may be a mistake */
{
    int i;
    int end = min(gaps->nextpref, gaps->size-1);
    SECTOR_MAP_ENTRY *sme;

    for (i = begin; i <= end; i++) {
	   sme = RT_smeForSector(rpd, i);
	   OneMistake(rpd, i, sme);
    }
}

/* @SUBTITLE "CheckAvailable: is there any prefetch work?" */
/* This is an Entry Point */
/* This tells whether prefetching is possible. */
static boolean
CheckAvailable(rpd)
	RAPIDFILE *rpd;
{
/* 
    struct gapslocal *lgaps = (struct gapslocal *)(rpd->prefetch.private);
    struct gapsdata *gaps = lgaps->common;
*/
    boolean available;

    /* note that rpd->prefetch.available is always TRUE on entry */
    available = rpd->inode_ptr->prefetch.available; /* just a hint, really */
    rpd->prefetch.available = available; /* propagate hint locally */

    return(available);
}

/* @SUBTITLE "GetWork: get prefetch work" */
/* This is an Entry Point */
/* This hands out prefetching work. It uses the prefetching state in the gaps
 * structure, information about the current portion, and state data
 * in the sector map (but only GAPS state), to find a block to prefetch.
 * We will not try to prefetch a block that has already been tried,
 * or that has been used. We also may limit our prefetching according to the
 * limit in rgaps->limit.
 *  We also pay attention to the minlast lock, and avoid prefetching
 * while that is set. This avoids spurious mistakes and wasted time.
 *  This uses its own lock. The lock is also used in NewPortion.
 */
static boolean
GetWork(rpd, sector)
	RAPIDFILE *rpd;
	int *sector;	/* returned */
{
    struct gapslocal *lgaps = (struct gapslocal *)(rpd->prefetch.private);
    struct gapsdata *gaps = lgaps->common;
    int block;
    SECTOR_MAP_ENTRY *sme;
    boolean ok = FALSE;
    boolean overlimit = FALSE;
    int nextpref;

    do {
	   LockFIFO(lgaps->prefetch_fifo,10);
	   LockFIFO(lgaps->pchange_fifo, 1);
	   if (!rpd->inode_ptr->prefetch.available || gaps->jumpgap) {
		  UnlockFIFO(lgaps->pchange_fifo);
		  UnlockFIFO(lgaps->prefetch_fifo);
		  return(FALSE);	/* give up NOW */
	   }

	   /* Check for any blocks to be retried */
	   if (Dequeue(lgaps->GiveBackQ, &block)) {
		  /* Possibly retry this block number */
		  ok = (block > gaps->minlast && block >= gaps->endskip
			   && block <= gaps->limit);
	   } else { 
		  /* get the next block in sequence */
		  nextpref = gaps->nextpref; 

		  /* Should we jump an irregular portion gap? */
		  if (gaps->prefetch_skip && nextpref >= gaps->startskip)
		    /* we have gone past a skip */
		    if (nextpref < gaps->endskip) {
#ifdef GAPTRACE
			   printf("-> took the prefetch skip %d to %d\n", 
					nextpref, gaps->endskip);
#endif
			   /* jump the skip */
			   nextpref = gaps->nextpref = gaps->endskip;
			   gaps->prefetch_skip = FALSE;
		    } else
			 gaps->prefetch_skip = FALSE;

		  if (nextpref < gaps->minlast)
		    /* prefetching fell behind */
		    block = NextBlock(gaps->minlast, gaps->start,
						  gaps->runlen, gaps->skiplen);
		  else
		    /* find next block in sequence */
		    block = NextBlock(nextpref, gaps->start,
						  gaps->runlen, gaps->skiplen);

		  /* Check the prefetch limits */
		  if (block <= gaps->limit) {
			 gaps->nextpref = block; /* commit */
			 ok = TRUE;
		  } else
		    overlimit = TRUE;
	   }

	   UnlockFIFO(lgaps->pchange_fifo);
	   UnlockFIFO(lgaps->prefetch_fifo);

	   if (ok && !gaps->jumpgap) {
		  /* Try to mark this block for prefetch */
		  sme = RT_smeForSector(rpd, block);
		  if (SS_ISCLEAR(sme, SS_USED | SS_MARKED)) {
			 /* This block is good to prefetch. */
			 /* mark it as work handed out */
			 SS_SET(sme, SS_MARKED);
			 /* check to see if minlast changed */
			 if (block <= gaps->minlast) {
				/* oh well, minlast changed */
				SS_CLEAR(sme, SS_MARKED); 
				ok = FALSE;
			 }			/* else ok stays TRUE: prefetch it! */
		  } else
		    ok = FALSE;
	   } else
		ok = FALSE;
    } while (!ok && !overlimit);

    if (ok) {
	   *sector = block;

#ifdef GAPTRACE
	   printf("-> node %d prefetch work block %d\n", Vnode, block);
#endif
    }

    return(ok);
}

/* @SUBTITLE "NextBlock: compute the next block in sequence" */
/*   This computes the next block number in the sequence of blocks
 * in a regular-portioned pattern.
 *   This function is simplified by the assumption the skip is nonnegative
 * and any limits on the portion are handled elsewhere.
 * Thus all of our parameters (and return values) are nonnegative integers. 
 */
static int				/* next block number */
NextBlock(block, start, runlen, skiplen)
	int block;			/* current block number >= 0 */
	int start;			/* start of portion >= 0 */
	int runlen;			/* run length >= 0 */
	int skiplen;			/* skip length >= 0*/
{
    if (block < start)
	 return(start);

    if (runlen == 0 || skiplen == 0)
	 return(block + 1);
    else
	 if ((block - start) % (runlen+skiplen-1) < runlen - 1)
	   return(block + 1);
	 else
	   return(block + skiplen);
}

/* @SUBTITLE "GiveBackWork: give back some prefetch work" */
/* This is an Entry Point */
/* This function is used by the prefetchers to give back prefetch
 * work that could not be completed. These are unmarked, and
 * GAPS's prefetch pointer is reset so these may be tried again. */
static void
GiveBackWork(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    struct gapslocal *lgaps = (struct gapslocal *)(rpd->prefetch.private);
    struct gapsdata *gaps = lgaps->common;
    SECTOR_MAP_ENTRY *sme;

    sme = RT_smeForSector(rpd, sector);
    SS_CLEAR(sme, SS_MARKED | SS_MISTAKE);

    /* Put this block number on a queue, to be retried */
    /* (push out oldest entry on overflow) */
    if (rpd->inode_ptr->prefetch.available)
	 ForceEnqueue(lgaps->GiveBackQ, sector, NULL);
}

/* @SUBTITLE "Done: Finished with reference pattern" */
/* This is an Entry Point */
/* This deallocates the gaps data structures. Only once. */
static void
Done(rpd)
	RAPIDFILE *rpd;
{
    struct gapslocal *lgaps = (struct gapslocal *)(rpd->prefetch.private);
    struct gapsdata *gaps = lgaps->common;

    if (Vnode == 0) {
	   if (gaps->num_sequential > 0)
		Finalize(rpd, gaps);

	   FreeFIFO(gaps->watch_fifo);
	   FreeFIFO(gaps->maxlast_fifo);
	   FreeFIFO(gaps->prefetch_fifo);
	   FreeFIFO(gaps->pchange_fifo);
	   UsFree(gaps->context);
	   UsFree(gaps->GiveBackQ);
	   UsFree(gaps->last);
	   UsFree(gaps->sequential);
	   UsFree(gaps);
	   rpd->inode_ptr->prefetch.private = NULL;
	   rpd->inode_ptr->prefetch.available = FALSE;
	   rpd->prefetch.available = FALSE;
    }

    UsFree(lgaps->sequential);
    UsFree(lgaps->last);
    UsFree(lgaps);
}


/* @SUBTITLE "NewRef: allocate a new REF structure" */
static REF *
NewRef(block, node)
	int block;
	int node;
{
    REF *r;

    r = (REF *)AllocGlobal(sizeof(REF));
    if (r == (REF *)NULL) {
	   fprintf(stderr, "out of memory\n");
	   exit(1);
    }
    r->block = block;
    r->node = node;

    return(r);
}

/* @SUBTITLE "NotMyREF: is this REF not one of mine?" */
/* When in watching mode, if a processor exhibits a jump-back
 * (referencing a block whose number is lower than that processor's 
 * previous reference), we must remove all of that processor's 
 * references from the context queue. This is because, a globally sequential
 * portion does not have processors doing jumpbacks. Since we are looking 
 * for portions, we can safely assume the references before the jumpback
 * were for another portion and can be removed. 
 *  This is called by ScanQueue to determine which REFs to throw out.
 * It will only keep REFs when this function returns TRUE.
 *  This returns TRUE if the given REF is not one of the given node's refs.
 */
static boolean
NotMyREF(lgaps, node, r, value)
	struct gapslocal *lgaps;
	int node;
	REF *r;
	int *value;			/* we ignore */
{
    return (r->node != node);
}

/* @SUBTITLE "Dump: debugging dump" */
/* Internally, we can call Dump(NULL, lgaps) instead. */
static void
Dump(rpd, lgaps)
	RAPIDFILE *rpd;
	struct gapslocal *lgaps; /* may be supplied if rpd==NULL */
{
    struct gapsdata *gaps;

    if (rpd != NULL)		/* use rpd to find lgaps */
	 lgaps = (struct gapslocal *)(rpd->prefetch.private);
    gaps = lgaps ? lgaps->common : NULL;

    printf("  GAPS Predictor: local\n");
    if (lgaps != NULL) {
	   printf("   common = 0x%x\n", lgaps->common);
	   printf("   nprocs = %d\n", lgaps->nprocs);
	   printf("   mylast = %d\n", lgaps->mylast);
	   printf("   last = 0x%x\n", lgaps->last);
	   printf("   mysequential = %d\n", lgaps->mysequential);
	   printf("   seqeuential = 0x%x\n", lgaps->sequential);
	   printf("   ordered = %s\n", lgaps->ordered ? "TRUE" : "FALSE");
	   printf("   ordercount = %d\n", lgaps->ordercount);
	   printf("   watch_fifo = 0x%x\n", lgaps->watch_fifo);
	   printf("   context = 0x%x\n", lgaps->context,
			lgaps->context ? InQueue(lgaps->context) : 0);
	   printf("   minlast_lock = 0x%x\n", lgaps->minlast_lock);
	   printf("   maxlast_fifo = 0x%x\n", lgaps->maxlast_fifo);
	   printf("   prefetch_fifo = 0x%x\n", lgaps->prefetch_fifo);
	   printf("   pchange_fifo = 0x%x\n", lgaps->pchange_fifo);
	   printf("   GiveBackQ = 0x%x (%d)\n", lgaps->GiveBackQ,
			lgaps->GiveBackQ ? InQueue(lgaps->GiveBackQ) : 0);
	   printf("   size = %d\n", lgaps->size);
    } else
	 printf("   NULL gapslocal data");

    printf("  GAPS Predictor: global\n");
    if (gaps != NULL) {
	   printf("   parm = %d\n", gaps->parm);
	   printf("   nprocs = %d\n", gaps->nprocs);
	   printf("   contin = %s\n", gaps->contin ? "TRUE" : "FALSE");
	   printf("   random = %s\n", gaps->random ? "TRUE" : "FALSE");
	   printf("   restart = %s\n", gaps->restart ? "TRUE" : "FALSE");
	   printf("   last = 0x%x\n", gaps->last);
	   printf("   sequential = 0x%x\n", gaps->sequential);
	   printf("   num_sequential = %d\n", gaps->num_sequential);
	   printf("   num_ordered = %d\n", gaps->num_ordered);
	   printf("   watch_fifo = 0x%x\n", gaps->watch_fifo);
	   printf("   context = 0x%x\n", lgaps->context,
			lgaps->context ? InQueue(lgaps->context) : 0);
	   printf("   coef = %g (tolerance %g)\n", gaps->coef, TOLERANCE);
	   printf("   maxjump = %d\n", gaps->maxjump);
	   printf("   blockuses = %d\n", gaps->blockuses);
	   printf("   minlast_lock = %s\n", gaps->minlast_lock ? "SET" : "CLEAR");
	   printf("   maxlast_fifo = 0x%x\n", gaps->maxlast_fifo);
	   printf("   start = %d\n", gaps->start);
	   printf("   minlast = %d\n", gaps->minlast);
	   printf("   maxlast = %d\n", gaps->maxlast);
	   printf("   jumpgap = %s\n", gaps->jumpgap ? "TRUE" : "FALSE");
	   printf("   portion1 = %s\n", gaps->portion1 ? "TRUE" : "FALSE");
	   printf("   same_runlen = %d\n", gaps->same_runlen);
	   printf("   prev_runlen = %d\n", gaps->prev_runlen);
	   printf("   same_skiplen = %d\n", gaps->same_skiplen);
	   printf("   prev_skiplen = %d\n", gaps->prev_skiplen);
	   printf("   prev_end = %d\n", gaps->prev_end);
	   printf("   runlen = %d\n", gaps->runlen);
	   printf("   skiplen = %d\n", gaps->skiplen);
	   printf("   prefetch_fifo = 0x%x\n", gaps->prefetch_fifo);
	   printf("   pchange_fifo = 0x%x\n", gaps->pchange_fifo);
	   printf("   GiveBackQ = 0x%x (%d)\n", lgaps->GiveBackQ,
			lgaps->GiveBackQ ? InQueue(lgaps->GiveBackQ) : 0);
	   printf("   nextpref = %d\n", gaps->nextpref);
	   printf("   startskip = %d\n", gaps->startskip);
	   printf("   endskip = %d\n", gaps->endskip);
	   printf("   prefetch_skip = %s\n", 
			gaps->prefetch_skip ? "TRUE" : "FALSE");
	   printf("   size = %d\n", gaps->size);
	   printf("   limit = %d\n", gaps->limit);
	   printf("   maxdist = %d\n", gaps->maxdist);
	   printf("   distance = %d\n", gaps->distance);
    } else
	 printf("   NULL gapsdata");

}
