/* @TITLE "io.c - I/O simulation package" */
/* io.c: Simulates real I/O by sleeping
 *
 * FUNCTIONS:
 *  Completely unaware of rapid internals
 *   io_init, io_init_global
 *   io_open
 *   io_read, io_readwait
 *   io_write, io_writewait
 *   io_close
 *  These last three know about inodes:
 *   io_touch
 *
 * David Kotz   April 1988
 */

static char rcsid[] = "$Id: io.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

#include <stdio.h>
#include <usdfk.h>
#include "io.h"
#include "rapidelog.h"

/* @SUBTITLE "Definitions" */
#define msec  * TICKperMSEC	/* convert msec to tics */
#define READ_TIME  (TICS)(30 msec)	/* default only, see below */
#define WRITE_TIME (TICS)(30 msec)	/* default only, see below */
#define OPEN_TIME  (TICS)(0 msec) /* no reason to wait */
#define CLOSE_TIME (TICS)(0 msec) /* no reason to wait */

#define Sleep(time) UsWait((int)((time) * USECperTICK / 10))

/* Description for each disk drive */
typedef struct disk DISK;
struct disk {
    lock_t lock;			/* lock for whenfree */
    TICS whenfree;			/* rtc when the disk will be available again */
    TICS idle;				/* total idle time */
    TICS wait;				/* total wait time */
    unsigned long count;		/* number of accesses */
};

static int numdisks = 0;		/* number of disks we have */
static DISK **disks = NULL;	/* array of disks, scattered and shared */
static TICS startstat;		/* start of statistics */
static TICS endstat;		/* end of statistics */
static int blocksize;		/* blocksize on disk (shared) */
static TICS readtime;		/* time for one read - a constant */
static TICS writetime;		/* time for one write - a constant */

/* Which disk does a given address reside on? 
 * This mapping represents interleaving with granularity 'blocksize' 
 */
#define DISKMAP(addr, blocksize) 	(((addr) / (blocksize)) % numdisks)

static TICS DiskOp();

/* @SUBTITLE "io_init_global: initialize I/O package" */
/*	Global initialization for I/O handler - call on primary  processor.
 * Can be called many times, but no files should be open when called.
 * The disk access times can be 0 to use the built-in defaults.
 */

void
io_init_global(n, bsize, rtime, wtime)
	int n;				/* number of disks */
	int bsize;			/* blocksize */
	int rtime;			/* read-access time (msec) */
	int wtime;			/* write-access time (msec) */
{
    if (n <= 0) {
	   printf("io_init_global: number of disks (%d) must be > 0\n", n);
	   CoreDump();
    }

    if (n > numdisks) {
	   numdisks = n;
	   Share(&numdisks);

	   if (disks != NULL)
		UsFree(disks);

	   /* Allocate disk descriptors */
	   disks = (DISK **) AllocScatterMatrix(numdisks, 1, sizeof(DISK));
	   if (disks == NULL) {
		  printf("io_init_global: not enough memory for disk descriptors\n");
		  exit(0);
	   }
	   ShareScatterMatrix(&disks, numdisks);
    } else {
	   numdisks = n;
	   Share(&numdisks);
    }

    /* Share the other constants */
    blocksize = bsize;
    Share(&blocksize);

    readtime = (rtime > 0) ? rtime * TICKperMSEC : READ_TIME;
    Share(&readtime);
    writetime = (wtime > 0) ? wtime * TICKperMSEC : WRITE_TIME;
    Share(&writetime);

    /* Initialize stats */
    io_clearstat();
}

/* @SUBTITLE "io_getstat, io_clearstat: Read, Clear disk statistics." */

void
io_getstat(util, idletime, waittime)
	float *util;			/* disk utilization (fraction [0,1]) */
	float *idletime;		/* total idle time in msec */
	float *waittime;		/* average wait time in msec */
{
    int d;
    TICS idle = 0;	/* global idle time */
    TICS wait = 0;	/* global wait time */
    unsigned long count = 0;	/* number of accesses */

    for (d = 0; d < numdisks; d++) {
	   idle += disks[d]->idle;
	   if (disks[d]->whenfree < endstat)
		idle += endstat - disks[d]->whenfree;
	   wait += disks[d]->wait;
	   count += disks[d]->count;
    }

    if (idletime)
	 *idletime = (float) idle * MSECperTICK;
    if (util)
	 *util = 1 - (float)idle / (float)(endstat - startstat) / (float)numdisks;
    if (waittime)
	 *waittime = (float) wait / (float) count * MSECperTICK;
}

#define LOOPTIME(d) (d)	/* (tics) predicted time for the loop */

void
io_clearstat()
{
    int d;

    startstat = rtc + LOOPTIME(numdisks);

    for (d = 0; d < numdisks; d++) {
	   disks[d]->lock = 0;
	   disks[d]->idle = 0;
	   disks[d]->wait = 0;
	   disks[d]->count = 0;
	   disks[d]->whenfree = startstat;
    }

    while (rtc < startstat);
}

/* @SUBTITLE "io_init: Initialization for I/O handler." */
/* call on each processor */

void
io_init()
{
}

/* @SUBTITLE "io_open: open a file" */
void
io_open()
{
    Sleep(OPEN_TIME);
}

/* @SUBTITLE "io_read: initiate a read operation" */
/*	Queue a read on the appropriate disk from the given position in the file.
 * Returns the time when the I/O should be finished.
 */
/* ARGSUSED */
TICS
io_read(buf, disk_addr, size)
	char *buf;			/* frame object */
	unsigned int disk_addr;	/* disk address */
	unsigned int size;		/* size of frame */
{
    int disknum;
    TICS finished;

    disknum = DISKMAP(disk_addr, blocksize);

    finished = DiskOp(disknum, readtime);

    return (finished);
}

/* @SUBTITLE "io_readwait: issue read and wait for completion" */
/*	Queue a read on the appropriate disk from the given position in the file.
 * Waits for the I/O to complete.
 */

void
io_readwait(buf, disk_addr, size)
	char *buf;			/* frame object */
	unsigned int disk_addr;	/* disk address */
	unsigned int size;		/* size of frame */
{
    TICS finished;

    /* Queue the request */
    finished = io_read(buf, disk_addr, size);

    /* Wait for it to finish */
    while (rtc < finished)
	 ;
}

/* @SUBTITLE "io_write: initiate a write operation" */
/*	Queue a write on the appropriate disk to the given position in the file.
 * Returns the time when the I/O should be finished.
 */
/* ARGSUSED */
TICS
io_write(buf, disk_addr, size)
	char *buf;			/* frame object */
	unsigned int disk_addr;	/* disk address */
	unsigned int size;		/* size of frame */
{
    int disknum;
    TICS finished;

    disknum = DISKMAP(disk_addr, blocksize);

    finished = DiskOp(disknum, writetime);

    return (finished);
}

/* @SUBTITLE "io_writewait: issue a write and wait for completion" */
/*	Queue a write on the appropriate disk to the given position in the file.
 * Waits for the I/O to complete.
 */

void
io_writewait(buf, disk_addr, size)
	char *buf;			/* frame object */
	unsigned int disk_addr;	/* disk address */
	unsigned int size;		/* size of frame */
{
    TICS finished;

    /* Queue the request */
    finished = io_write(buf, disk_addr, size);

    /* Wait for it to finish */
    while (rtc < finished)
	 ;
}

/* @SUBTITLE "DiskOp: Queue an operation on the given disk" */
static
TICS
DiskOp(disknum, duration)
	int disknum;
	TICS duration;
{
    DISK *disk = disks[disknum];
    TICS finished;
    TICS diskfree;

    LOCK(&(disk->lock), 10);

    /* Compute when our read will finish */
    diskfree = disk->whenfree;
    if (diskfree <= rtc) {
	   /* disk currently idle */
	   disk->idle += rtc - diskfree;
	   finished = (disk->whenfree = rtc + duration);
    } else
	 /* disk currently busy */
	 finished = (disk->whenfree = diskfree + duration);

    disk->wait += finished - rtc;
    disk->count++;

    ELOG_LOG(RTELOG_DISKUSED, disknum);
    ELOG_LOG(RTELOG_DISKWAIT, finished-rtc);

    UNLOCK(&(disk->lock));

    return (finished);
}

/* @SUBTITLE "io_close: Close a file." */
void
io_close()
{
    /* mark the finish time for statistics */
    endstat = GetRtc();
    Sleep(CLOSE_TIME);
}

/* @SUBTITLE "io_touch: touch all memory in I/O system" */
/* Touch all points in disks[] array. */

void
io_touch()
{
    int d;
    TICS dummy;

    /* look at each disk structure */
    for (d = 0; d < numdisks; d++) {
	   dummy = disks[d]->idle;
    }
    d = dummy;				/* shuts up lint */
}

