/* @TITLE "interface.c - interface function" */
/* interface.c: RAPIDFILE management routines.  These are the subroutines 
 *		invoked by application programs: 
 * 
 *			RT_access
 *			RT_close 
 *             RT_dump 
 *             RT_flush 
 *             RT_swapoutall 
 *			RT_lseek 
 *			RT_open 
 *			RT_read 
 *			RT_write 
 */

static char rcsid[] = "$Id: interface.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

/* INCLUDES */

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "replace.h"
#include "stats.h"
#include "wanted.h"
#include "rapidelog.h"
#include "queue.h"
#include "write.h"

/* @SUBTITLE "DEFINES" */

static void RT_swapoutall();
void RT_dump();

/* Diagnostic flags (print diagnostic messages of non-fatal errors) */

#define CLOSE_DIAG_FLAG
#define CREAT_DIAG_FLAG
#define LSEEK_DIAG_FLAG
#define OPEN_DIAG_FLAG
#define READ_DIAG_FLAG
#define STAT_DIAG_FLAG
#define TRUNC_DIAG_FLAG
#define WRITE_DIAG_FLAG

/* Debugging flags (print messages for debugging aids) */

/* #define CLOSE_DEBUG_FLAG 
 * #define CREAT_DEBUG_FLAG 
 * #define LSEEK_DEBUG_FLAG 
 * #define OPEN_DEBUG_FLAG 
 * #define READ_DEBUG_FLAG 
 * #define STAT_DEBUG_FLAG 
 * #define TRUNC_DEBUG_FLAG 
 * #define WRITE_DEBUG_FLAG 
 */

static void CloseWorkerA(), CloseWorkerB();
	
/* @SUBTITLE "Definitions for RT_open" */

/* Information shared by RT_open to its parallel workers */
struct OpenInfo_s {
    RAPIDFILE **my_rpd_ptr;	/* pointer to my_rpd in all procs */
    RAPID_INODE *inode_ptr;	/* inode pointer being opened */
    int p_lead;			/* minimum prefetching lead */
    int p_mintime;			/* minimum prefetching time */
    boolean p_demand;		/* prefetch while waiting for demand? */
    boolean p_bufhit;		/* prefetch while waiting for buffer hit? */
    int write_style;		/* the writeback algorithm */
    int local_wss;			/* size of each working set */
    ANYPTR *my_refs_ptr;		/* pointer to my_refs in all procs */
} OpenInfo;

/* parallel worker procedures */
static void OpenMakeRPD();

/* @SUBTITLE "RT_access: access a sector of the file" */
/* simply use the sector in question, and perform a block transfer 
 * of the appropriate size. This avoids all the complexity of RT_read
 * or RT_write.
 */
void
RT_access (rpd, buf, sector, writeflag)
	RAPIDFILE *rpd;		/* the file */
	char *buf;			/* buffer to copy into/from */
	int sector;	/* sector number */
	boolean writeflag;		/* is this a write? */
{
    TICS start = rtc;

    if (writeflag) {
	   /* WRITE ACCESS */
	   SECTOR dest = UseSector(rpd, sector, TRUE, TRUE);
	   btransfer(buf, dest, rpd->sector_size);
	   DoneSector(rpd, sector);
	   my_stats->writetime += rtc - start;
	   my_stats->nwrite++;
    } else {
	   /* READ ACCESS */
	   SECTOR src = UseSector(rpd, sector, FALSE, FALSE);
	   if (src != (SECTOR)NULL)
		btransfer(src, buf, rpd->sector_size);
	   else
		bzero(buf, rpd->sector_size);
	   DoneSector(rpd, sector);
	   my_stats->readtime += rtc - start;
	   my_stats->nread++;
    }
}

/* @SUBTITLE "RT_close: close a file" */
void
RT_close (my_rpd_ptr)
	RAPIDFILE **my_rpd_ptr;	/* pointer to my_rpd on all procs */
						/* also, a way to get to the inode_ptr */
{
    RAPIDFILE *rpd = *my_rpd_ptr; /* pointer to rpd on this node */
    RAPID_INODE *inode_ptr = rpd->inode_ptr;

    /* shut down prefetching/prediction */
    GenTaskForEachProc(CloseWorkerA, my_rpd_ptr);

    /* swapout all buffers - dirty or not */
    RT_swapoutall(rpd);			/* may change some stats */

    io_close();			/* cuts off timing of disks */

    /* now processor-specific stuff  */
    GenTaskForEachProc(CloseWorkerB, my_rpd_ptr);

    /* shut down replacement algorithm */
    RT_DoneReplaceInode(inode_ptr);

    /* no RemoveSectors needed */
    UsFree(inode_ptr->sector_map_table);
    UsFree(inode_ptr->buffers); /* frame buffers */
    UsFree(inode_ptr->frame_table);
    UsFree(inode_ptr);
}

/* @SUBTITLE "CloseWorker[AB]: close for each process" */
/* parallel */
static void
CloseWorkerA(my_rpd_ptr)
	RAPIDFILE **my_rpd_ptr;
{
    RAPIDFILE *rpd = *my_rpd_ptr; /* pointer to rpd on this node */

    /* Clean up prefetching */
    RT_CancelPrefetch(rpd);
}

/* parallel */
static void
CloseWorkerB(my_rpd_ptr)
	RAPIDFILE **my_rpd_ptr;
{
    RAPIDFILE *rpd = *my_rpd_ptr; /* pointer to rpd on this node */

    /* Clean up replacement stuff */
    RT_DoneReplaceRPD(rpd);

    /* free RPD structures */
    if (!rpd->caching)
	 free(rpd->buffer);
    free(rpd->frame_table);
    free(rpd->sector_map_table);
    free(rpd);
}

/* @SUBTITLE "RT_open: open a file" */
/* All the parameters needed to fully specify the open file are given.
 * Uniform system generators are used to set up the open file on all 
 * processors. All processors will store the new rpd in *my_rpd, so
 * my_rpd should be a pointer to a RAPIDFILE* in process private static 
 * memory. Also, all processors will read *my_refs to get the REFS* 
 * that is their reference string. Thus the pattern file should be read
 * before RT_open.
 */
void
RT_open (my_rpd_ptr, 
	    sector_size, sector_map_size, size, num_frames, 
	    caching,
	    p_limit, p_lead, p_mintime, p_demand, p_bufhit,
	    write_style, FileIsNew,
	    replace_style, local_wss, my_refs_ptr)
	RAPIDFILE **my_rpd_ptr;	/* the address of my_rpd in all procs */
	int sector_size;		/* size of sector in bytes */
	int sector_map_size;	/* size of sector map in bytes */
	unsigned int size;		/* file size in bytes */
	int num_frames;		/* number of frames to allocate */
	int p_limit;			/* prefetching limit */
	int p_lead;			/* minimum prefetching lead */
	int p_mintime;			/* minimum prefetching time */
	boolean p_demand;		/* prefetch while waiting for demand? */
	boolean p_bufhit;		/* prefetch while waiting for buffer hit? */
	int write_style;		/* writeback algorithm */
	boolean FileIsNew;		/* is the file newly created? */
	int replace_style;		/* replace algorithm to use */
	int local_wss;	/* size of each working set */
	ANYPTR *my_refs_ptr;	/* address of references pointer in all procs */
{
    RAPID_INODE *inode_ptr;	/* the new inode */
    int num_maps;			/* number of sector maps */
    int entries_per_map;		/* number of sector map entries per sector map */
    int num_sectors;		/* number of sectors */

    /* COMPUTATIONS */
    sector_size = Next_Larger_Power_of_2(sector_size);
    sector_map_size = Next_Larger_Power_of_2(sector_map_size);
    /* number of sectors in file */
    num_sectors = size / sector_size;
    if ((unsigned) num_sectors * sector_size < size)
	 num_sectors++;
    /* number of sector map entries per sector map */
    entries_per_map = sector_map_size / sizeof(SECTOR_MAP_ENTRY);
    /* number of sector maps */
    num_maps = num_sectors / entries_per_map;
    if (num_maps * entries_per_map < num_sectors)
	 num_maps++;

    /* BUILD THE INODE */
    inode_ptr = (RAPID_INODE *) AllocGlobal(sizeof(RAPID_INODE));

    inode_ptr->sector_size = sector_size;
    inode_ptr->sector_size_exp = Binary_log2(sector_size);
    inode_ptr->byte_offset_mask = sector_size - 1;
    inode_ptr->sector_map_size = entries_per_map;
    inode_ptr->sector_map_size_exp = Binary_log2 (entries_per_map);
    inode_ptr->map_sector_offset_mask = entries_per_map - 1;
    inode_ptr->size = size;

    /* SECTOR TABLE */
    inode_ptr->num_sectors = num_sectors;
    inode_ptr->num_maps = num_maps;
    inode_ptr->sector_map_table = (SECTOR_MAP_ENTRY **)
	 AllocScatterMatrix(num_maps, entries_per_map,
					sizeof(SECTOR_MAP_ENTRY));
    AddSectors(inode_ptr, num_sectors, FileIsNew);

#ifdef INTERLEAVED_MAPS
    printf("Interleaved Sector Maps in use\n");
#endif

    inode_ptr->caching = caching;

    /* FRAME TABLE */
    inode_ptr->num_frames = num_frames;
    if (caching) {
	   inode_ptr->frame_table = (FRAME_ENTRY **) 
		AllocScatterMatrix(num_frames, 1, sizeof(FRAME_ENTRY));
	   AddFrames(inode_ptr, num_frames);
    } else {
	   inode_ptr->frame_table = (FRAME_ENTRY **) NULL;
	   inode_ptr->buffers = (SECTOR *) NULL;
    }

    /* PREFETCH STUFF */
    inode_ptr->prefetch.limit = p_limit;
    inode_ptr->prefetch.unused = 0;
    inode_ptr->prefetch.available = FALSE;
    inode_ptr->prefetch.private = NULL;
    inode_ptr->prefetch.private2 = NULL;
    
    inode_ptr->write_style = write_style;

    /* REPLACEMENT ALGORITHM STUFF */
    inode_ptr->replace.style = replace_style;
    RT_InitReplaceInode(inode_ptr);

    /* Store things to share to other procs */
    OpenInfo.my_rpd_ptr = my_rpd_ptr;
    OpenInfo.inode_ptr = inode_ptr;
    OpenInfo.p_lead = p_lead;
    OpenInfo.p_mintime = p_mintime;
    OpenInfo.p_demand = p_demand;
    OpenInfo.p_bufhit = p_bufhit;
    OpenInfo.write_style = write_style;
    OpenInfo.local_wss = local_wss;
    OpenInfo.my_refs_ptr = my_refs_ptr;
    ShareBlk(&OpenInfo, sizeof(OpenInfo));

    /* BUILD DESCRIPTORS */
    GenTaskForEachProc(OpenMakeRPD, 0);

    /* Open Backing Store */
    io_open();
}

/* @SUBTITLE "OpenMakeRPD: worker for RT_open" */
/* parallel - build descriptor for this processor */
static void
OpenMakeRPD()
{
    RAPIDFILE *rpd;
    RAPID_INODE *inode_ptr = OpenInfo.inode_ptr;
    int i;

    TouchBlock(inode_ptr, sizeof(RAPID_INODE), FALSE);

    /* allocate and record the rpd for this process */
    rpd = (RAPIDFILE *) AllocPrivate(sizeof(RAPIDFILE));
    bzero(rpd, sizeof(RAPIDFILE));
    *(OpenInfo.my_rpd_ptr) = rpd;

    rpd->inode_ptr = inode_ptr;

    /* Copy the duplicated fields in one fell swoop using block transfer. 
	*	int	 	    sector_size			(4) 
	*	int	 	    sector_size_exp			(4) 
	*	int	 	    byte_offset_mask		(4) 
	*	int	 	    sector_map_size			(4) 
	*	int	 	    sector_map_size_exp		(4) 
	*	int	 	    map_sector_offset_mask	(4) 
	*	unsigned int  size					(4)
	*	int	 	    num_sectors			(4) 
	*	int	 	    num_maps				(4) 
	*	boolean 	    caching				(4) 
	*	int	 	    num_frames				(4) 
	*/
    btransfer (&(inode_ptr->sector_size), &(rpd->sector_size), 44);

    /* make a local copy of the sector map table */
    rpd->sector_map_table = (SECTOR_MAP_ENTRY **) 
	 AllocPrivate(rpd->num_maps * sizeof(SECTOR_MAP_ENTRY *));
    btransfer(inode_ptr->sector_map_table, rpd->sector_map_table, 
		    rpd->num_maps * sizeof(SECTOR_MAP_ENTRY *));
    
    TouchSectorMaps(rpd);

    if (rpd->caching) {
	   /* make a local copy of the frame table pointers */
	   rpd->frame_table = (FRAME_ENTRY **) 
		AllocPrivate(rpd->num_frames * sizeof(FRAME_ENTRY *));
	   btransfer(inode_ptr->frame_table, rpd->frame_table, 
			   rpd->num_frames * sizeof(FRAME_ENTRY *));
	   TouchFrames(rpd);
    } else {				/* not caching */
	   rpd->frame_table = (FRAME_ENTRY **) NULL;

	   /* make a single local buffer */
	   rpd->buffer = (SECTOR) AllocPrivate(rpd->sector_size);
    }

    /* prefetching stuff */
    rpd->prefetch.cid = 0; /* start later if desired */
    rpd->prefetch.lead = OpenInfo.p_lead;
    rpd->prefetch.mintime = OpenInfo.p_mintime;
    rpd->prefetch.demand = OpenInfo.p_demand;
    rpd->prefetch.bufhit = OpenInfo.p_bufhit;
    rpd->prefetch.refs = *OpenInfo.my_refs_ptr;
    rpd->prefetch.limit = inode_ptr->prefetch.limit / ProcsInUse();
    rpd->prefetch.unused = 0;
    
    rpd->prefetch.available = FALSE;
    rpd->prefetch.checkavail = NULL;
    rpd->prefetch.getwork = NULL;
    rpd->prefetch.notify = NULL;
    rpd->prefetch.latenotify = NULL;
    rpd->prefetch.donerpd = NULL;
    rpd->prefetch.dump = NULL;
    rpd->prefetch.private = NULL;
    rpd->prefetch.private2 = NULL;
    
    rpd->write.thru = (OpenInfo.write_style == WTHRU);
    rpd->write.full = (OpenInfo.write_style == WFULL);
    rpd->write.newfull = (OpenInfo.write_style == WNFULL);
    rpd->write.leavegws = (OpenInfo.write_style == WLEAVE);

    /* set up replacement algorithm */
    rpd->replace.style = inode_ptr->replace.style;
    RT_InitReplaceRPD(rpd, OpenInfo.local_wss);
}

/* @SUBTITLE "RT_dump: dump a file descriptor for debugging" */

#define FLAG(x) ((x) ? "TRUE" : "FALSE")
#define PMUTEX(m) printf("%s\n", (m) ? "LOCKED" : "CLEAR")
  
char *frame_status[] = {
    "NONFRAME", "OLDFRAME", "IN_WS", "PREFETCHED", "TRANSITION", "WANTED"
};

char *writestyles[] = {
    "WriteBack", "WriteThrough", "write when leaving GWS",
    "Write when full", "Write when new and full"
#define MAXWRITESTYLE 4
};

static int Nitems();		/* a protected call to InQueue() */

void
RT_dump (rpd)
	RAPIDFILE *rpd;
{
    RAPID_INODE *inode_ptr = rpd->inode_ptr;
    FRAME_ENTRY *frame;
    int index;
    int sector;
    SECTOR_MAP_ENTRY *sme;
    int ans;
    int write_style = inode_ptr->write_style;

    printf("\n\n RAPIDFILE DUMP:\n");
    
    printf("\nINODE (Ptr %0x):\n", inode_ptr);
    printf("=>sector_size: %d\n", inode_ptr->sector_size);
    printf("  sector_size_exp: %d\n", inode_ptr->sector_size_exp);
    printf("  byte_offset_mask: %d\n", inode_ptr->byte_offset_mask);
    printf("=>sector_map_size: %d\n", inode_ptr->sector_map_size);
    printf("  sector_map_size_exp: %d\n", inode_ptr->sector_map_size_exp);
    printf("  map_sector_offset_mask: %d\n",
		 inode_ptr->map_sector_offset_mask);
    printf("=>size: %u\n", inode_ptr->size);
    printf("  num_sectors: %d\n", inode_ptr->num_sectors);
    printf("  caching: %s\n", FLAG(inode_ptr->caching));
    printf("  num_frames: %d\n", inode_ptr->num_frames);
    printf("  sector_map_table: %0x\n", inode_ptr->sector_map_table);
    printf("  buffers: %0x\n", inode_ptr->buffers);
    printf("  frame_table: %0x\n", inode_ptr->frame_table);
    printf("=>prefetch.limit: %d\n", inode_ptr->prefetch.limit);
    printf("=>prefetch.unused: %d\n", inode_ptr->prefetch.unused);
    printf("  prefetch.private: 0x%0x\n", inode_ptr->prefetch.private);
    printf("  prefetch.private2: 0x%0x\n", inode_ptr->prefetch.private2);
    if (inode_ptr->replace.style == RT_REPLACE_NONE) {
	   printf("=>replace.style: NONE\n");
    } else if (inode_ptr->replace.style == RT_REPLACE_WS) {
	   printf("=>replace.style: WORKING-SET\n");
    } else if (inode_ptr->replace.style == RT_REPLACE_TOSS) {
	   printf("=>replace.style: TOSS_IMMEDIATE\n");
    } else {
	   printf("=>replace.style: UNKNOWN %d\n", inode_ptr->replace.style);
    }
    printf("  replace.private: 0x%0x\n", inode_ptr->replace.private);

    if (write_style >= 0 &&  write_style <= MAXWRITESTYLE)
	 printf("  write_style: %s\n", writestyles[write_style]);
    else 
	 printf("  write_style: UNKNOWN (%d)\n", write_style);
    
    printf("\nDESCRIPTOR: (Ptr %0x):\n", rpd);
    printf("=>sector_size: %d\n", rpd->sector_size);
    printf("  sector_size_exp: %d\n", rpd->sector_size_exp);
    printf("  byte_offset_mask: %d\n", rpd->byte_offset_mask);
    printf("=>sector_map_size: %d\n", rpd->sector_map_size);
    printf("  sector_map_size_exp: %d\n", rpd->sector_map_size_exp);
    printf("  map_sector_offset_mask: %d\n",
		 rpd->map_sector_offset_mask);
    printf("=>size: %u\n", rpd->size);
    printf("  num_sectors: %d\n", rpd->num_sectors);
    printf("  caching: %s\n", FLAG(rpd->caching));
    printf("  num_frames: %d\n", rpd->num_frames);
    printf("  sector_map_table: %0x\n", rpd->sector_map_table);
    printf("  buffer: %0x\n", rpd->buffer);
    printf("  frame_table: %0x\n", rpd->frame_table);
    printf("  prefetch.cid: %d\n", rpd->prefetch.cid);
    printf("  prefetch.lead: %d\n", rpd->prefetch.lead);
    printf("  prefetch.mintime: %d\n", rpd->prefetch.mintime);
    printf("  prefetch.bufhit: %s\n", FLAG(rpd->prefetch.bufhit));
    printf("  prefetch.demand: %s\n", FLAG(rpd->prefetch.demand));
    printf("  prefetch.refs: %0x\n", rpd->prefetch.refs);
    printf("  prefetch.alg: %d\n", rpd->prefetch.alg);
    printf("  prefetch.private: 0x%0x\n", rpd->prefetch.private);
    printf("  prefetch.private2: 0x%0x\n", rpd->prefetch.private2);
    printf("  prefetch.limit: %d\n", rpd->prefetch.limit);
    printf("  prefetch.unused: %d\n", rpd->prefetch.unused);
    printf("=>posn: %ld\n", rpd->posn);
    if (rpd->replace.style == RT_REPLACE_NONE) {
	   printf("  replace.style: NONE\n");
    } else if (rpd->replace.style == RT_REPLACE_WS) {
	   printf("  replace.style: WORKING-SET\n");
    } else if (rpd->replace.style == RT_REPLACE_TOSS) {
	   printf("  replace.style: TOSS-IMMEDIATE\n");
    } else {
	   printf("=> replace.style: UNKNOWN %d\n", rpd->replace.style);
    }
    printf("  replace.private: 0x%0x\n", rpd->replace.private);
    if (rpd->replace.dump)
	 (*(rpd->replace.dump))(rpd);
    else
	 printf("\n  No Replacement algorithm dump procedure\n");

    if (rpd->prefetch.dump)
	 (*(rpd->prefetch.dump))(rpd);
    else
	 printf("\n  No Predictor dump procedure\n");
    printf("  write.thru: %s\n", FLAG(rpd->write.thru));
    printf("  write.full: %s\n", FLAG(rpd->write.full));
    printf("  write.newfull: %s\n", FLAG(rpd->write.newfull));
    printf("  write.leavegws: %s\n", FLAG(rpd->write.leavegws));

    printf("\nFRAME TABLE:\n");
    for (index = 0; index < rpd->num_frames; index++) {
	   frame = rpd->frame_table[index];
	   if (frame->frame_ptr != (SECTOR)NULL)	/* a normal, in-use frame */
		printf("%d: buf %0x, sector %d  ws_count %d  %s\n",
			  index, frame->frame_ptr, frame->frame_sector,
			  frame->ws_count, frame_status[frame->frame_status],
			  frame->frame_lock ? "LOCKED" : "");
	   else if (frame->frame_sector > 0 /* not-in-use frame; check it out */
			  || frame->frame_lock != 0
			  || frame->ws_count != 0
			  || frame->frame_status != FS_NONFRAME)
		printf("%d: buf %0x, sector %d  ws_count %d  %s (SOMETHING WRONG)\n",
			  index, frame->frame_ptr, frame->frame_sector,
			  frame->ws_count, frame_status[frame->frame_status],
			  frame->frame_lock ? "LOCKED" : "");
	   
    }
    
    printf("\nDo you want the sector map (y or n)?");
    ans = getchar();
    while (getchar() != '\n')
	 ;
    
    if (ans == 'y') {
	   printf("\SECTOR MAP:\n");
	   for (sector = 0; sector < rpd->num_sectors; sector++) {
		  sme = RT_smeForSector(rpd, sector);
		  printf(" %d: buf %0x  frame %d  status %c%c%c%c%c%c%c%c%c  use_count %d \n\t  ready %ld  misc %d  %s\n",
			    sector, sme->sector_ptr, sme->sector_frame,
			    SS_ISSET(sme, SS_VALID) ? 'v' : '-',
			    SS_ISSET(sme, SS_DIRTY) ? 'd' : '-',
			    SS_ISSET(sme, SS_NONZERO) ? 'n' : '-',
			    SS_ISSET(sme, SS_CHANGING) ? 'c' : '-',
			    SS_ISSET(sme, SS_PREFETCHED) ? 'p' : '-',
			    SS_ISSET(sme, SS_MISTAKE) ? 'm' : '-',
			    SS_ISSET(sme, SS_MARKED) ? 'x' : '-',
			    SS_ISSET(sme, SS_NEW) ? 'N' : '-',
			    SS_ISSET(sme, SS_WRITTEN) ? 'w' : '-',
			    sme->use_count,
			    (long) (sme->whenready - rtc),
			    sme->misc.owner,
			    IsChangingSector(sme) ? "CHANGE LOCKED" : "");
	   }
    }
    
    printf("\n");
    fflush(stdout);
}

#undef FLAG
#undef PMUTEX

static int
Nitems(queue)
	QH queue;
{
    if (queue)
	 return(InQueue(queue));
    else
	 return(0);
}

/* @SUBTITLE "RT_WhatWrite: what writeback alg are we using */
void
RT_WhatWrite(rpd)
	RAPIDFILE *rpd;
{
#define FLAG(x) ((x) ? "TRUE" : "FALSE")
    int write_style = rpd->inode_ptr->write_style;

    if (write_style >= 0 &&  write_style <= MAXWRITESTYLE)
	 printf("  WriteAlg: %s\n", writestyles[write_style]);
    else 
	 printf("  WriteAlg: UNKNOWN (%d)\n", write_style);
    printf("  write.thru: %s\n", FLAG(rpd->write.thru));
    printf("  write.full: %s\n", FLAG(rpd->write.full));
    printf("  write.newfull: %s\n", FLAG(rpd->write.newfull));
    printf("  write.leavegws: %s\n", FLAG(rpd->write.leavegws));
#undef FLAG
}

/* @SUBTITLE "RT_flush: flush buffer cache for file" */
/* This is only meant for use at end-of-pattern time.
 * Parallel entry possible. Trashes frame queues.
 * Second argument controls the parallel access to the frames. 
 * Procedure returns when all frames handled by this processor have 
 * all I/O completed, and all frames have been handled by some processor.
 */
void
RT_flush (rpd, index)
	RAPIDFILE *rpd;
	short *index;			/* must be initialized to zero before flush */
{
    int sector;			/* the sector in this frame */
    int frame_index;		/* a frame we are responsible for */
    SECTOR_MAP_ENTRY *sme;	/* the sector in the frame */
    FRAME_ENTRY *frame;		/* the frame */
    TICS ready = rtc;		/* when all I/O is complete */
    int num_frames = rpd->num_frames;

    ELOG_LOG(100, 0);
    /* For each of the frames ... */
    while (*index < num_frames) {
	   frame_index = Atomic_add(index, 1);
	   if (frame_index >= num_frames) 
		break;
	   frame = rpd->frame_table[frame_index];
	   sector = frame->frame_sector;
	   if (sector >= 0) {	/* a normal, in-use frame */
		  sme = RT_smeForSector(rpd, sector);
		  if (SS_ISSET(sme, SS_VALID)) {
			 WriteBack(rpd, sme, sector);
			 if (sme->whenready > ready) /* any I/O to wait for? */
			   ready = sme->whenready;
		  }
	   }
    }

    ELOG_LOG(100, 1);
    /* Now wait for I/O to complete */
    while (rtc < ready)
	 ;
    ELOG_LOG(100, 2);
}

/* @SUBTITLE "RT_swapoutall: empty buffer cache for file" */
/* This is only meant for use at close time.
 * Non-parallel. Trashes frame queues.
 * We assume flush has already been run, so we complain if there's
 * any outstanding I/O.
 */
static void
RT_swapoutall (rpd)
	RAPIDFILE *rpd;
{
    int sector;
    SECTOR_MAP_ENTRY *sme;
    FRAME_ENTRY *frame;
    int index;

    ELOG_LOG(101, 0);
    /* For each of the frames ... */
    for (index = 0; index < rpd->num_frames; index++) {
	   frame = rpd->frame_table[index];
	   sector = frame->frame_sector;
	   if (sector >= 0) {	/* a normal, in-use frame */
		  sme = RT_smeForSector(rpd, sector);
#ifdef WANTED
		  if (sme->misc.want_count > 0) {
			 printf("warning: sector %d wanted by %d procs\n",
				    sector, sme->misc.want_count);
			 sme->misc.want_count = 0;
		  }
#endif WANTED
		  if (frame->frame_status == FS_PARTFULL) {
			 printf("warning: sector %d: only %d bytes written\n",
				    sector, sme->misc.written);
		  }
		  if (SS_ISSET(sme, SS_VALID)) {
			 frame->ws_count = 0;
			 (void) SwapSectorOut(rpd, frame, NULL);
			 /* should not be any I/O to wait for */
			 if (rtc < sme->whenready) {
				printf("warning: sector %d not ready for %u more tics\n",
					  sector, sme->whenready-rtc);
				while(rtc < sme->whenready)
				  ;
			 }
		  }
	   }
    }

    ELOG_LOG(101, 1);
}

/* @SUBTITLE "RT_lseek: adjust position in file" */
long
RT_lseek (rpd, offset, whence)
	RAPIDFILE *rpd;
	long    offset;
	int	    whence;
{
    register	long	posn;
    
    switch (whence)	{
	   case 0: {			/* set (from start) */
		  posn = offset;
		  break;
	   }
	   case 1: {			/* extend (from end) */
		  posn = rpd->size + offset;
		  break;
	   }
	   case 2: {			/* increment (from current) */
		  posn = rpd->posn + offset;
		  break;
	   }
	   default: {
		  return ((long) (-1));
	   }
    }
    
    if (posn < 0 || posn >= rpd->size)	{
#ifdef LSEEK_DIAG_FLAG
	   printf ("RT_lseek: Requested offset is illegal\n");
	   printf ("         posn = %d   offset = %ld   relative = %d\n",
			 rpd->posn, offset, whence);
	   printf ("          Leaving current pointer unchanged\n");
#endif
	   return ((long) (-1));
    }
    else	{
	   rpd->posn = posn;
	   return (posn);
    }
}

/* @SUBTITLE "RT_read: read from file" */
int
RT_read (rpd, buf, nbytes)
	RAPIDFILE *rpd;
	char *buf;
	int nbytes;
{
    
    long    nbytes_read = 0;
    long    nbytes_remaining_this_sector;
    long    nbytes_to_read;
    char    *src;
    char    *dest;
    int    sector;
    int    offset;
    
    TICS start = rtc; /* start time */
    
    /* if posn is still past end of file, have real end of file error, so 
	* simply return 0 as number of bytes read. 
	*/
    
    if (rpd->posn >= rpd->size)
	 return (0L);
    
    /* if nbytes requested takes past end of file, reduce to read to end of 
	* file only 
	*/
    
    if (nbytes > rpd->size - rpd->posn)
	 nbytes = rpd->size - rpd->posn;
    
#ifdef READ_DEBUG_FLAG
    printf ("RT_read: nbytes = %x (= %d)\n", nbytes, nbytes);
#endif
    
    /* if nbytes takes us past end of sector, must do in parts */
    
    /* while nbytes > 0 ...
	  copy what can to dest (buf)
	  update dest
	  update posn
	  update nbytes
	  update nbytes_read
	  */
    
    dest = buf;
    while (nbytes != 0)	{
	   sector = RT_SectorNum (rpd, rpd->posn);
	   offset = RT_SectorByteOffset (rpd, rpd->posn);
	   nbytes_remaining_this_sector = rpd->sector_size - offset;
	   
#ifdef READ_DEBUG_FLAG
	   printf ("RT_read: sector = %d, offset = %d, bytes remaining = %d\n",
			 sector, offset, nbytes_remaining_this_sector);
#endif
	   
	   src = UseSector (rpd, sector, FALSE, FALSE) + offset;
	   nbytes_to_read = min (nbytes, nbytes_remaining_this_sector);
	   
	   if (src != NULL) {
		  btransfer(src, dest, nbytes_to_read);
	   } else
		bzero(dest, nbytes_to_read);
	   
	   DoneSector(rpd, sector, 0); /* finished with this sector */
	   
	   dest += nbytes_to_read;
	   rpd->posn += nbytes_to_read;
	   nbytes -= nbytes_to_read;
	   nbytes_read += nbytes_to_read;
    }
    
    /* Update statistics */
    my_stats->readtime += rtc - start;
    my_stats->nread++;
    
    return (nbytes_read);
}

/* @SUBTITLE "RT_write: write to file" */
int
RT_write (rpd, buf, nbytes)
	RAPIDFILE *rpd;
	char *buf;
	int nbytes;
{
    long    nbytes_written = 0;
    long	nbytes_remaining_this_sector;
    long	nbytes_to_write;
    char    *src;
    char    *dest;
    int	sector;
    int	offset;
    
    TICS start = rtc;
    
    /* if nbytes takes us past end of file, we get an error */
    
    if (nbytes > rpd->size - rpd->posn)	{
#ifdef WRITE_DIAG_FLAG
	   printf ("RT_write: write past end of file: error\n");
	   printf ("          posn = %d, nbytes = %d, size = %d\n",
			 rpd->posn, nbytes, rpd->size);
#endif
	   return(0);
    }
    
#ifdef WRITE_DEBUG_FLAG
    printf ("RT_write: nbytes = %x (= %d)\n", nbytes, nbytes);
#endif
    
    /* if nbytes takes us past end of sector, must do in parts */
    
    /* while nbytes > 0 ...
	  copy what can from src (buf)
	  update src (buf)
	  update posn
	  update nbytes
	  update nbytes_written
	  */
    
    src = buf;
    while (nbytes != 0)	{
	   sector = RT_SectorNum (rpd, rpd->posn);
	   offset = RT_SectorByteOffset (rpd, rpd->posn);
	   nbytes_remaining_this_sector = rpd->sector_size - offset;
	   
#ifdef WRITE_DEBUG_FLAG
	   printf ("RT_write: sector = %d, offset = %d, bytes remaining = %d\n",
			 sector, offset, nbytes_remaining_this_sector);
#endif
	   
	   nbytes_to_write = min (nbytes, nbytes_remaining_this_sector);
	   if (nbytes_to_write == rpd->sector_size) /* full page write? */
		dest = UseSector (rpd, sector, TRUE, TRUE) + offset;
	   else
		dest = UseSector (rpd, sector, TRUE, FALSE) + offset;
	   
	   btransfer (src, dest, nbytes_to_write);
	   
	   /* finished with this sector */
	   DoneSector(rpd, sector, nbytes_to_write);
	   
	   src += nbytes_to_write;
	   rpd->posn += nbytes_to_write;
	   nbytes -= nbytes_to_write;
	   nbytes_written += nbytes_to_write;
    }
    
    /* Update statistics */
    my_stats->writetime += rtc - start;
    my_stats->nwrite++;
    
    return (nbytes_written);
}

