/* @TITLE "globalws.c - manage global working set" */
/* globalws.c:	Support for global working set
 *		Includes the following: 
 *  	   	  	InitGWS(num_frames)
 *  	   	  	DoneGWS(ff)
 *             AddToGlobalWS(rpd, sector) 
 *             RemoveFromGlobalWS(rpd, sector, oldq) 
 *             FindFrame_ws(rpd, sector, sme, ff, prefetching)
 *
 * The protocols used here are carefully designed to work together to 
 * provide exactly the right mutual exclusion properties and are very 
 * heavily based on the semantics of the problem. Be very careful 
 * if modifying any of these routines. 
 */

static char rcsid[] = "$Id: globalws.c,v 7.1 91/05/09 19:31:27 dfk Tape2 $"; 

/* INCLUDES */

#include <stdio.h>
#include <usdfk.h>
#include "internal.h"
#include "rapidelog.h"
#include "error.h"
#include "replace.h"
#include "stats.h"
#include "wanted.h"
#include "queue.h"

/* @SUBTITLE "DEFINES" */

/* #define FRAME_DEBUG_FLAG */

/* are there any ready frames on unready queue? */
#define AnyReady(ff) (rtc >= ValQueue((ff)->unreadyq))

/* LOCAL TYPES */
/* This data structure is made by InitWS and stored in inode_ptr. */
/* It is passed to FindFrame. */
/* typedef struct ffdata_s FFDATA; */
struct ffdata_s {
    QH readyq;				/* the queue of IO-ready frames */
    QH unreadyq;			/* the queue of IO-unready frames */
    lock_t updating;		/* lock controlling update */
};

/* LOCAL FUNCTIONS */
	
static boolean CheckReady();

/* @SUBTITLE "InitWS: initialization for this module" */
/* create a ffdata structure, to be passed to FindFrame later on */
FFDATA *
InitGWS(num_frames)
	int num_frames;		/* number of frames in system */
{
    FFDATA *ff;			/* the new structure */
    int index;

    ff = (FFDATA *)AllocGlobal(sizeof(FFDATA));
    if (ff == (FFDATA *)NULL)
	 error("No memory for frame queues", ERR_DIE);

    /* make queue of ready frames */
    ff->readyq = Make_queue(num_frames+1);
    
    /* put all frames on queue */
    for (index = 0; index < num_frames; index++)
	 Enqueue(ff->readyq, index, 0);

    /* make queue of unready frames */
    ff->unreadyq = Make_queue(num_frames+1);
    
    ff->updating = 0;

    return(ff);
}

/* @SUBTITLE "DoneWS: cleanup for this module" */
void
DoneGWS(ff)
	FFDATA *ff;			/* findframe data */
{
    UsFree(ff->readyq);
    UsFree(ff->unreadyq);
    UsFree(ff);
}

/* @SUBTITLE "AddToGlobalWS: Add sector to global WS" */
void
AddToGlobalWS(rpd, sector)
	RAPIDFILE *rpd;
	int sector;
{
    SECTOR_MAP_ENTRY *sme;	/* the entry for this sector */
    FRAME_ENTRY *frame;
    
    sme = RT_smeForSector(rpd, sector);
    
    frame = rpd->frame_table[sme->sector_frame];
    
    LockFrame(frame);
    
    if (frame->ws_count++ == 0) {
	   if (frame->frame_status == FS_PREFETCHED
		  || frame->frame_status == FS_TRANSITION
		  || frame->frame_status == FS_PARTFULL)
		frame->frame_status = FS_IN_WS;
#ifdef WANTED
	   if (frame->frame_status == FS_WANTED)
		frame->frame_status = FS_IN_WS;
#endif
    }
    UnlockFrame(frame);
}

/* @SUBTITLE "RemoveFromGlobalWS: remove sector from global WS" */
void
RemoveFromGlobalWS(rpd, sector, ff)
	RAPIDFILE *rpd;
	int sector;
	FFDATA *ff;			/* findframe data */
{
    SECTOR_MAP_ENTRY *sme;	/* the entry for this sector */
    FRAME_ENTRY *frame;
    
    sme = RT_smeForSector(rpd, sector);
    
    frame = rpd->frame_table[sme->sector_frame];
    
    LockFrame(frame);
    
    if (--(frame->ws_count) == 0)
#ifdef WANTED
	 if (sme->misc.want_count > 0)
	   /* just move to state WANTED */
	   frame->frame_status = FS_WANTED;
	 else
#endif WANTED
	   if (rpd->write.full && sme->misc.written < rpd->sector_size) {
		  frame->frame_status = FS_PARTFULL; /* not on queues, not in GWS */
	   } else
		if (frame->frame_status != FS_OLDFRAME) {
#ifdef FRAME_DEBUG_FLAG
		    fprintf(stderr, "(%02d) RemoveFromGlobalWS: Enq onto oldq\n",
				  Vnode);
#endif
		    ELOG_LOG(RTELOG_LEFTGWS, sector);
		    if (rpd->write.leavegws)
			 WriteBack(rpd, sme, sector);

		    if (rtc < sme->whenready)
			 Enqueue(ff->unreadyq, sme->sector_frame, 0);
		    else
			 Enqueue(ff->readyq, sme->sector_frame, 0);
		    frame->frame_status = FS_OLDFRAME;
		}
    UnlockFrame(frame);
}

/* @SUBTITLE "ReplaceableGWS: make a sector replaceable" */
/* like RemoveFromGlobalWS except that it's not already in our ws,
 * so we don't decrement the ws_count first. We also don't deal with
 * writebacks. This is meant for prefetching mistakes.
 */
void
ReplaceableGWS(rpd, sector, ff)
	RAPIDFILE *rpd;
	int sector;
	FFDATA *ff;			/* findframe data */
{
    SECTOR_MAP_ENTRY *sme;	/* the entry for this sector */
    FRAME_ENTRY *frame;

    sme = RT_smeForSector(rpd, sector);
    
    frame = rpd->frame_table[sme->sector_frame];

    LockFrame(frame);
    
    if (frame->ws_count == 0)
#ifdef WANTED
	 if (sme->misc.want_count > 0)
	   /* just move to state WANTED */
	   frame->frame_status = FS_WANTED;
	 else
#endif WANTED
	   if (frame->frame_status != FS_OLDFRAME) {
#ifdef FRAME_DEBUG_FLAG
		  fprintf(stderr, "(%02d) ReplaceableGWS: Enq onto old queue\n",
				Vnode);
#endif
		  if (rtc < sme->whenready)
		    Enqueue(ff->unreadyq, sme->sector_frame, 0);
		  else
		    Enqueue(ff->readyq, sme->sector_frame, 0);
		  frame->frame_status = FS_OLDFRAME;
	   }
    UnlockFrame(frame);
}

/* @SUBTITLE "FindFrame_ws: get frame to use" */
boolean
FindFrame_ws(rpd, sector, sme, ff, prefetching)
	RAPIDFILE *rpd;
	int sector;			/* the sector we want */
	SECTOR_MAP_ENTRY *sme;	/* the entry for this sector */
	FFDATA *ff;			/* frame queues etc */
	boolean prefetching;	/* are we doing this for prefetch */
{
    int frame_index;
    FRAME_ENTRY *frame;
    boolean found = FALSE;	/* found a frame yet? */
    boolean empty = FALSE;	/* is the frame empty? */

    do {
	   found = FALSE;
	   /* Look for ready frame that isn't in the global working set */
	   do {
		  if (!EmptyQueue(ff->readyq)) {
			 /* something on ready queue */
#ifdef FRAME_DEBUG_FLAG
			 fprintf(stderr, "(%02d) FindFrame: Deq from old\n", 
				    Vnode);
#endif
			 if (Dequeue(ff->readyq, &frame_index)) {
				frame = rpd->frame_table[frame_index];
				
				LockFrame(frame);
				if (frame->ws_count > 0)
				  frame->frame_status = FS_IN_WS;
				else {
				    frame->frame_status =
					 prefetching ? FS_PREFETCHED : FS_TRANSITION;
				    found = TRUE;
				}
				UnlockFrame(frame);
			 }			/* else queue had become empty */
		  } else {
			 /* ready queue is empty */
			 if (AnyReady(ff)) {
				/* some ready on unready queue */
				/* we'll try to move them to ready queue */
				if (TryLock(&(ff->updating))) {
#ifdef FRAME_DEBUG_FLAG
				    fprintf(stderr, "(%02d) FindFrame: updating\n",
						  Vnode);
#endif
				    my_stats->scanq++;
				    ELOG_LOG(RTELOG_SCANQ, 0);
				    ScanQueue(ff->unreadyq, CheckReady, rpd, ff);
				    UNLOCK(&(ff->updating));
				}		/* else someone is already updating */
			 } else {
				/* none ready on unready queue */
				if (prefetching)
				  return(FALSE); /* failure */
				else
				  /* wait until something becomes ready */
				  while(!AnyReady(ff) && EmptyQueue(ff->readyq)) 
				    UsWait(10);
			 }
		  }
	   } while (!found);
	   
	   /* We found a ready frame! */
	   /* Any sector in this frame must first be swapped out: */
	   /* This could still fail if it reenters global WS. */
	   /* also fails if writeback is needed */
	   if (frame->frame_sector != -1) {
		  if (SwapSectorOut(rpd, frame, ff->unreadyq)) {
			 empty = TRUE;
		  }
	   } else empty = TRUE;
    } while (!empty);
    
    /* The frame is now empty; we have committed to replacing the frame. */
    /* Connect the sector map entry and frame table entries */
    frame->frame_sector = sector;
    sme->sector_frame = frame_index;
    sme->sector_ptr = frame->frame_ptr;

    return(TRUE);
}

/* @SUBTITLE "CheckReady: check one frame" */
/* we can take three actions:
 * 1) if frame re-entered GWS, let it go
 * 2) if frame is ready, put on ready queue
 * 3) if not ready, leave on unready queue
 */
static boolean
CheckReady(rpd, ff, frame_index, ready)
	RAPIDFILE *rpd;
	FFDATA *ff;
	int frame_index;
	TICS *ready;			/* (out) if keep, when it'll be ready */
{
    FRAME_ENTRY *frame = rpd->frame_table[frame_index];
    boolean keep = FALSE;
    SECTOR_MAP_ENTRY *sme;

    LockFrame(frame);

    if (frame->ws_count > 0)
	 /* returned to global working set */
	 frame->frame_status = FS_IN_WS;
    else {
	   /* Looks dead. Is the I/O finished? */
	   sme = RT_smeForSector(rpd, frame->frame_sector);
	   if (rtc > sme->whenready) {
#ifdef FRAME_DEBUG_FLAG
		  fprintf(stderr, "(%02d) CheckReady: frame %d is ready\n",
				Vnode, frame_index);
#endif
		  /* this is ready! */
		  Enqueue(ff->readyq, frame_index, 0);
	   } else {
		  /* not ready, put it back */
		  keep = TRUE;
		  *ready = sme->whenready;
	   }
    }
    
    UnlockFrame(frame);

    return(keep);
}

/* @SUBTITLE "DumpGWS: print information about FF data structure" */
void
DumpGWS(ff)
	FFDATA *ff;			/* findframe data */
{
    printf("   readyq (0x%x): %d entries\n", 
		 ff->readyq, ff->readyq ? InQueue(ff->readyq) : 0);
    printf("   unreadyq (0x%x): %d entries, value %d\n", 
		 ff->unreadyq, 
		 ff->unreadyq ? InQueue(ff->unreadyq) : 0,
		 ff->unreadyq ? ValQueue(ff->unreadyq) : 0);
}

