/* @TITLE "internal.h - data structure definitions" */
/* $Id: internal.h,v 7.1 91/05/09 19:29:51 dfk Tape2 $ */

#include "vnode.h"			/* Vnode numbering system */

/* @SUBTITLE "MACROS" */

/* #define INTERLEAVED_MAPS */

/* Macros for obtaining appropriate map table, map table entry, sector, and 
 * byte offset. All of these will work if passed an inode_ptr as well,
 * since the necessary fields have the same names.
 */

/* Absolute sector number */

#define RT_SectorNum(rpd,p)\
	((p) >> (rpd)->sector_size_exp)

#ifdef INTERLEAVED_MAPS
/* Absolute sector map number (index into rpd->sector_map_table) */
#define RT_SectorMapNum(rpd,p)\
	(RT_SectorNum((rpd),p) % (rpd)->num_maps)

#define RT_SectorMapNumForSector(rpd,s)\
	((s) % (rpd)->num_maps)

/* Gives a SECTOR_MAP_ENTRY* for the given sector number */
#define RT_smeForSector(rpd,s)\
	(&((rpd)->sector_map_table[RT_SectorMapNumForSector((rpd),(s))]\
	   [(s) / (rpd)->num_maps]))
#else
/* Absolute sector map number (index into rpd->sector_map_table) */
#define RT_SectorMapNum(rpd,p)\
	(RT_SectorNum((rpd),p) >> (rpd)->sector_map_size_exp)

#define RT_SectorMapNumForSector(rpd,s)\
	((s) >> (rpd)->sector_map_size_exp)

/* Gives a SECTOR_MAP_ENTRY* for the given sector number */
#define RT_smeForSector(rpd,s)\
	(&((rpd)->sector_map_table[RT_SectorMapNumForSector((rpd),(s))]\
	   [(s) & (rpd)->map_sector_offset_mask]))
#endif INTERLEAVED_MAPS

/* Byte offset in the appropriate sector */

#define RT_SectorByteOffset(rpd,p)\
	((p) & (rpd)->byte_offset_mask)

/* Number of sectors and sector maps for a requested size */

#define RT_NumSectors(rpd,s)\
	((s == 0) ? 0 : ((s - 1) >> (rpd)->sector_size_exp) + 1)

#define RT_NumSectorMaps(rpd,s)\
	((s == 0) ? 0 : (((s - 1) >> (rpd)->sector_size_exp) >>\
			 (rpd)->sector_map_size_exp) + 1)

/* @SUBTITLE "TYPE DEFINITIONS" */

typedef short ALOCK;		/* lock */

typedef boolean (*boolproc)();
typedef void (*voidproc)();

/* @SUBTITLE "SECTOR MAP" */

typedef char* SECTOR;		/* sector buffer */
typedef struct Sector_Map_Entry_Struct SECTOR_MAP_ENTRY;
struct Sector_Map_Entry_Struct {
    SECTOR sector_ptr;		/* frame object containing sector */
    short sector_status;		/* status of the sector */
    short sector_frame;		/* index of frame containing sector, or -1 */
    short use_count;		/* count of number of users */
    union {
	   short owner;		/* owner for OWN_BUFFERS */
	   short want_count;	/* want_count for WANTED */
	   short lastcount;		/* for GAPS predictor */
	   short written;		/* number of bytes written to sector */
	   short dummy;		/* dummy filler otherwise */
    } misc;
    TICS whenready;			/* time when I/O completes for this sector */
};
#define SS_VALID 001		/* status bits */
#define SS_DIRTY 002		/* (all 0 is initial state for new sectors) */
#define SS_NONZERO 004
#define SS_CHANGING 010		/* a lock for the sector */
#define SS_PREFETCHED 020	/* this sector was prefetched but not yet used */
#define SS_MISTAKE 040		/* this sector was mistakenly prefetched */
#define SS_USED 0100		/* this sector was used (GAPS predictor) */
#define SS_MARKED 0200		/* this sector was given as work (GAPS) */
#define SS_NEW 0400			/* this sector is zero on disk */
#define SS_WRITTEN 01000		/* sector has been written since last read */

#define SS_SET(sme, bits) (Atomic_ior(&((sme)->sector_status), (bits)))
#define SS_ISSET(sme, bits) (((sme)->sector_status & (bits)) == (bits))
#define SS_CLEAR(sme, bits) (Atomic_and(&((sme)->sector_status), ~(bits)))
#define SS_ISCLEAR(sme, bits) (!((sme)->sector_status & (bits)))

#define IsChangingSector(sme) SS_ISSET((sme), SS_CHANGING)
#define TryChangeLockSector(sme) \
  ((SS_SET((sme), SS_CHANGING) & SS_CHANGING) == 0)
#define ChangeLockSector(sme) \
  {while (!TryChangeLockSector(sme)) UsWait(10);}
#define ChangeUnlockSector(sme) SS_CLEAR((sme), SS_CHANGING)

/* @SUBTITLE "FRAME TABLE" */
/* The entries in the frame table */
typedef struct Frame_Entry_Struct FRAME_ENTRY;
struct Frame_Entry_Struct {
    SECTOR frame_ptr;		/* frame buffer */
    int frame_sector;		/* sector in this frame (-1 is empty) */
    ALOCK frame_lock;		/* ALOCK is a short */
    short frame_status;		/* (ws only, see below) status of frame */
    short ws_count;			/* (ws only) number of WS we are in */
};

/* Frame statuses (mutually exclusive) */
#define FS_NONFRAME   0		/* initial state */
#define FS_OLDFRAME   1
#define FS_IN_WS      2
#define FS_PREFETCHED 3
#define FS_TRANSITION 4
#define FS_WANTED     5		/* only used with #define WANTED */
#define FS_PARTFULL   6		/* only used with write_style wfull */

/* Try to set a lock - compatible with LOCK & UNLOCK 
 * TRUE if lock obtained, false if not 
 */
#define TryLockFrame(frame) \
  (Atomic_ior(&((frame)->frame_lock), 1)  ==  0)
/* Wait for the lock to be set */
#define LockFrame(frame)		LOCK(&((frame)->frame_lock), 1)
/* Unlock the frame */
#define UnlockFrame(frame)	UNLOCK(&((frame)->frame_lock))

/* @SUBTITLE "Inode Structure" */
typedef struct RAPID_inode RAPID_INODE;
struct RAPID_inode	{
    int sector_size;		/* Sector size of this RAPIDFILE (even */
						/*  power of 2) */
    int sector_size_exp;		/* Log2 of sector size (even power) */
    int byte_offset_mask;	/* Mask for retrieving byte offset */
						/*  into sector */
    int sector_map_size;		/* Size of the partial sector maps */
						/*  (number of entries - again, an even */
						/*  power of 2) */
    int sector_map_size_exp;
						/* Log2 of sector map size (even power) */
    int map_sector_offset_mask;
						/* Mask for retrieving offset of */
						/*  sector in partial sector map */
    unsigned int size;		/* Size of the RAPIDFILE (in bytes). */
    int num_sectors;		/* number of sectors in file */
    int num_maps;			/* number of sector maps in file */
    boolean caching;		/* are we caching? */
    int num_frames;			/* number of frames in table */
    FRAME_ENTRY **frame_table; /* table listing frame objects */
    SECTOR *buffers;		/* list of frame buffers */
    SECTOR_MAP_ENTRY **sector_map_table; /* Table of sector descriptors */
    struct {
	   short limit;		/* amount of frames we can use for prefetch */
	   short unused;		/* number of frames with FS_PREFETCHED set */
	   boolean available;	/* might there be something to prefetch? */
	   ANYPTR private;		/* private data for global predictors */
	   ANYPTR private2;		/* private data for SWITCH predictor */
    } prefetch;
    struct {				/* Replacement algorithm */
	   short style;		/* type of replacement algorithm */
	   ANYPTR private;		/* eg findframe data */
	   voidproc doneinode;	/* function to clean up at inode level */
    } replace;
    int write_style;		/* see write.h */
};

/* @SUBTITLE "Descriptor structure" */

/* The Rapid_descriptor structure is what is given to processes through 
 * RT_open calls.  It is the descriptor that is provided to the appropriate 
 * RAPIDFILE calls.  Each process gets its own 
 * descriptor.  Hence, the descriptor contains process private information 
 * (such as the current file position), copies of some of the RAPID_INODE 
 * structure members (the read only members are copied for efficiency of 
 * access), and a pointer to the RAPID_INODE structure.
 */
typedef struct RAPID_descriptor RAPIDFILE;
struct RAPID_descriptor {
    RAPID_INODE   *inode_ptr;	/* Pointer to the inode structure */
						/*  which houses the global info */
						/*  for the RAPIDFile. */
    /* BEGIN COPY OF INODE */
    int sector_size;		/* Sector size of this RAPIDFILE (even */
						/*  power of 2) */
    int sector_size_exp;		/* Log2 of sector size (even power) */
    int byte_offset_mask;	/* Mask for retrieving byte offset */
						/*  into sector */
    int sector_map_size;		/* Size of the partial sector maps */
						/*  (number of entries - again, an even */
						/*  power of 2) */
    int sector_map_size_exp;
						/* Log2 of sector map size (even power) */
    int map_sector_offset_mask;
						/* Mask for retrieving offset of */
						/*  sector in partial sector map */
    unsigned int size;		/* Size of the RAPIDFILE (in bytes). */
    int num_sectors;		/* number of sectors in file */
    int num_maps;			/* number of sector maps in file */
    boolean caching;		/* are we caching? */
    int num_frames;			/* number of frames in table */
    /* END COPY OF INODE */
    FRAME_ENTRY **frame_table; /* table listing frame objects */
    SECTOR buffer;			/* a single local buffer, for !caching case */
    SECTOR_MAP_ENTRY **sector_map_table; /* 2D array of SMEs */
    struct {				/* for prefetching algorithm */
	   int cid;			/* coroutine id; 0 if none exists */
	   short lead;			/* extra lead used when prefetching */
	   short mintime;		/* minimum time necessary for prefetch call */
	   boolean demand;		/* prefetch in UseSector during demand fetch? */
	   boolean bufhit;		/* prefetch in UseSector during buffer hit? */
	   ANYPTR refs;		/* actually REFS* - reference string */
	   /* Prediction algorithm */
	   short alg;			/* which algorithm is in use */
	   voidproc donerpd;	/* cleanup procedure */
	   boolproc notify;		/* notification procedure */
	   voidproc latenotify;	/* late notification procedure (for SWITCH) */
	   boolproc getwork;	/* get a block for prefetching */
	   voidproc giveback;	/* give back a block for prefetching */
	   boolproc checkavail;	/* is there something to prefetch? */
	   voidproc dump;		/* debugging dump of private data */
	   boolean available;	/* might there be something to prefetch? */
	   ANYPTR private;		/* data particular to algorithm */
	   ANYPTR private2;		/* private data for SWITCH predictor */
	   /* for OWN_BUFFER prefetching */
	   short limit;		/*  in place of inode copy */
	   short unused;		/*  in place of inode copy */
    } prefetch;
    struct {				/* Replacement algorithm */
	   short style;		/* type of replacement algorithm */
	   ANYPTR private;		/* data particular to alg; process-private */
	   voidproc donerpd;	/* breakdown at rpd level */
	   boolproc notifyA;	/* first notify proc */
	   voidproc notifyB;	/* second notify proc */
	   boolproc findframe;	/* find a frame to use */
	   voidproc replaceable;	/* to say that a sector is replaceable */
	   voidproc dump;		/* debugging dump of private data */
    } replace;
    struct {				/* writeback style */
	   boolean thru;		/* write-through */
	   boolean full;		/* write when full */
	   boolean newfull;		/* write when full and new */
	   boolean leavegws;	/* write when leaving GWS */
    } write;
    unsigned int posn;		/* Current file posn offset (bytes) */
};

/* @SUBTITLE "FUNCTIONS" */

/* from alloc.c */
extern ANYPTR AllocGlobal();	/* safely allocate shared memory */
extern ANYPTR AllocScatterMatrix();	/* safely allocate scattered memory */
extern ANYPTR AllocLocal();	/* safely allocate local shared memory */
extern ANYPTR AllocPrivate();	/* safely allocate private memory */

/* from binarith.c */
extern	int	Binary_log2();
extern	long	Next_Larger_Power_of_2();

/* from suppt.c */
extern void AddSectors();
extern void TouchSectorMaps();
extern void AddFrames();
extern void TouchFrames();

/* From frame.c */
extern SECTOR UseSector();
extern void DoneSector();
extern boolean SwapSectorOut();
extern void WriteBack();

/* From error.c: print error and die */
extern void error();
